
/*******************************************************************************//**
 * @file    usbg_buspower_core.c
 * @brief   USB Gadget BusPower Driver - core
 * 
 * Copyright 2011 Sony Corporation
 * Copyright 2018, 2019 Sony Imaging Products and Solutions Incorporated.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 *******************************************************************************/

/*=============================================================================
/  Includes
/==============================================================================*/

#include <linux/uaccess.h>
#include <asm/page.h>
#include <asm/errno.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/version.h>

#include <linux/usb/ch9.h>
#include <linux/device.h>
#include <linux/usb/gadget.h>

#include <linux/mm.h>
#include <linux/usb/gcore/usb_event.h>
#include <linux/usb/gcore/usb_gadgetcore.h>

#include <linux/usb/specific_gadget/usbg_cmn.h>
#include <linux/usb/specific_gadget/usbg_type.h>
#include <linux/usb/specific_gadget/usbg_still.h>
#include <linux/usb/specific_gadget/usb_kdbg.h>
#include <linux/usb/specific_gadget/usb_ioctl.h>
#include <linux/usb/specific_gadget/usb_ioctl_still.h>

#define FNAME "usbg_buspower_core.c"
#include "usbg_buspower_pvt.h"


/*=============================================================================
/  Enumerations
/==============================================================================*/


/*=============================================================================
/  Structs
/==============================================================================*/


/*=============================================================================
/  Prototypes
/==============================================================================*/


/*=============================================================================
/  Variable Declarations
/==============================================================================*/


/*=============================================================================
/  Functions
/==============================================================================*/

/**
 *    @brief        EP0 transfer callback
 *    @param        ep        EP0 ep struct
 *    @param        req       request struct
 */
static void usbg_buspower_ep0_comp( struct usb_ep *ep, struct usb_request *req )
{
    
    USBG_BUSPWR_PDEBUG( "Calling: %s() ", __FUNCTION__ );
    
    if( !req ) {
        
        USBG_BUSPWR_PERR( "%s(): req==NULL.", __FUNCTION__ );
        goto SUB_RET;
        
    }
    
    usb_ep_free_request( ep, req );
    
SUB_RET:
    
    USBG_BUSPWR_PDEBUG( "Leaving: %s()", __FUNCTION__ );
    
    return;
    
}

#define USBG_BUSPWR_GET_REPORT_RESPONSE ( 0x00 )
#define USBG_BUSPWR_GET_REPORT_RESPONSE_SIZE ( 1 )
/**
 *    @brief        Get Report
 *    @param        p_ctrl                  USB setup packet
 *    @param        ep0                     EP0 ep struct
 *    @retval       0                       success
 *    @retval       USB_GADGETCORE_STALL    STALL
 */
static int usbg_buspower_class_get_report( const struct usb_ctrlrequest* p_ctrl, struct usb_ep* ep0 )
{
    int ret;
    struct usb_request *req;
    u8 *tmp_p, *buf_p;
    
    USBG_BUSPWR_PDEBUG( "Calling: %s() ", __FUNCTION__ );
    
    // Check data direction
    if( p_ctrl->bRequestType & USB_DIR_OUT ) {
        
        ret = USB_GADGETCORE_STALL;
        goto SUB_RET;
        
    }
    
    // Allocate Request
    req = usb_ep_alloc_request( ep0, GFP_ATOMIC );
    
    if( !req ) {
        
        USBG_BUSPWR_PERR( "%s(): usb_ep_alloc_request.", __FUNCTION__ );
        ret = USB_GADGETCORE_STALL;
        goto SUB_RET;
        
    }
    
    buf_p = tmp_p = usbg_buspower_get_ep0_buffer();
    
    if( !buf_p ){
        
        USBG_BUSPWR_PERR( "%s(): ep0 buffer is NULL.", __FUNCTION__ );
        ret = USB_GADGETCORE_STALL;
        goto SUB_RET;
        
    }
    
    // Put response data to buffer
    usbg_buspower_put_u8( &tmp_p, USBG_BUSPWR_GET_REPORT_RESPONSE );
    
    // Prepare request info
    req->buf          = buf_p;
    req->length       = USBG_BUSPWR_GET_REPORT_RESPONSE_SIZE;
    req->dma          = (dma_addr_t)NULL;
    req->zero         = 0;
    req->complete     = usbg_buspower_ep0_comp;
    req->no_interrupt = 0;
    req->short_not_ok = 0;
    req->context      = NULL;
    
    // Queue request
    ret = usb_ep_queue( ep0, req, GFP_ATOMIC );
    
    if( ret ) {
        
        USBG_BUSPWR_PERR( "%s(): usb_ep_queue err._error code is %d.", __FUNCTION__, ret );
        usb_ep_free_request( ep0, req );
        ret = USB_GADGETCORE_STALL;
        
    }
    
SUB_RET:
    USBG_BUSPWR_PDEBUG( "Leaving: %s() , return val is %d ", __FUNCTION__, ret );
    
    return ret;
    
}

/**
 *    @brief        GadgetCore callback - No Operation
 *    @param        all unused.
 */
static void core_cb_nop( struct usb_gadget_func_driver* p_drv )
{
    
    USBG_BUSPWR_PDEBUG( "Calling: %s()", __FUNCTION__ );
    
    USBG_BUSPWR_PDEBUG( "Leaving: %s()", __FUNCTION__ );
    
    return;
    
}

/**
 *    @brief        GadgetCore callback - start
 *    @param        p_drv       function driver struct
 *    @param        alt         alternative interface no.
 *    @param        list        Endpoint list
 */
static void core_cb_start( struct usb_gadget_func_driver* p_drv,
                        unsigned char alt,
                        struct usb_gadget_ep_list list )
{
    
    int i;
    struct func_data    *p_this    = usbg_buspower_get_this();
    struct func_data_ex *p_this_ex = usbg_buspower_get_this_ex();
    struct usb_ep *ep;
    struct usbg_buspower_ep_info *epinf;
    
    USBG_BUSPWR_PDEBUG( "Calling: %s()", __FUNCTION__ );
    
    USBG_BUSPWR_EP_LOCK_ON();
    
    if( usbg_cmn_make_ep_list( p_this, alt, list ) ){
        
        USBG_BUSPWR_PERR( "%s(): Failed to usbg_cmn_make_ep_list().", __FUNCTION__ );
        
    }
    
    // Initialize ep table
    ep                = p_this->ptbl_ep[ USBG_BUSPWR_EP_IDX_INT_IN ];
    epinf             = &( p_this_ex->ep_info[ USBG_BUSPWR_EP_IDX_INT_IN ] );
    epinf->ep_no      = 0x81;
    epinf->stalled    = FALSE;
    epinf->requesting = FALSE;
    epinf->req        = usb_ep_alloc_request( ep, GFP_ATOMIC );
    ep->name          = USBG_BUSPWR_EP_NAME_INT_IN;
    ep->driver_data   = epinf;
    
    // Print EP info
    USBG_BUSPWR_PDEBUG( "---EP information--------" );
    
    for( i = 0; i < USBG_BUSPWR_EP_IDX_MAX; i++ ) {
        
        ep = p_this->ptbl_ep[i];
        USBG_BUSPWR_PDEBUG( "%s(): EP%d: usage=%s, ep=%p, maxp=%d", __FUNCTION__, i + 1, ep->name, ep, ep->maxpacket );
        
    }
    
    USBG_BUSPWR_EP_LOCK_OFF();
    
    usbg_buspower_call_user_cb( USBG_BUSPWR_CBID_START, USBG_KEVENT_ID_CMN_START, NULL, 0 );
    
    USBG_BUSPWR_PDEBUG( "Leaving: %s()", __FUNCTION__ );
    
    return;
    
}

/**
 *    @brief        GadgetCore callback - stop
 *    @param        p_drv        function driver struct
 */
static void core_cb_stop( struct usb_gadget_func_driver* p_drv )
{
    
    struct func_data    *p_this    = usbg_buspower_get_this();
    struct func_data_ex *p_this_ex = usbg_buspower_get_this_ex();
    
    USBG_BUSPWR_PDEBUG( "Calling: %s()", __FUNCTION__ );
    
    USBG_BUSPWR_EP_LOCK_ON();
    
    usbg_buspower_ep_dequeue_all();
    
    if( usbg_buspower_is_requesting( USBG_BUSPWR_EP_IDX_INT_IN ) ) {
        // stop usb_ep_queue()
        p_this_ex->stop_usb_request = true;
        // temporarily cancel irqsave if there are pending requests
        USBG_BUSPWR_EP_LOCK_OFF();
        if (wait_for_completion_interruptible(&(p_this_ex->req_wait_completion))) {
            USBG_BUSPWR_PINF( "interrupted completion" );
        }
        USBG_BUSPWR_EP_LOCK_ON();
        // usb_ep_queue() is started but exclusive by spinlock
        p_this_ex->stop_usb_request = false;
    }
    
    // Initialize ep info
    usb_ep_free_request( p_this->ptbl_ep[ USBG_BUSPWR_EP_IDX_INT_IN ],
                         p_this_ex->ep_info[ USBG_BUSPWR_EP_IDX_INT_IN ].req );
    
    memset( p_this_ex->ep_info, 0, sizeof(struct usbg_buspower_ep_info) * USBG_BUSPWR_EP_IDX_MAX );
    
    usbg_cmn_clear_ep_list( p_this );
    
    USBG_BUSPWR_EP_LOCK_OFF();
    
    usbg_buspower_call_user_cb( USBG_BUSPWR_CBID_STOP, USBG_KEVENT_ID_CMN_STOP, NULL, 0 );
    
    USBG_BUSPWR_PDEBUG( "Leaving: %s()", __FUNCTION__ );
    
    return;
    
}

/**
 *    @brief        GadgetCore callback - set halt
 *    @param        p_drv       function driver struct
 *    @param        p_ep        stalled ep
 *    @retval       0                       Success
 *    @retval       USB_GADGETCORE_STALL    STALL
 *    @note         Called when SetFeature issued to this EP
 */
static int core_cb_set_halt( struct usb_gadget_func_driver* p_drv,
                      struct usb_ep* p_ep )
{
    
    int err = 0;
    struct usbg_buspower_ep_info *ep_info = usbg_buspower_get_ep_info( p_ep );
    
    USBG_BUSPWR_PDEBUG( "Calling: %s()", __FUNCTION__ );
    
    USBG_BUSPWR_EP_LOCK_ON();
    
    // stall flag <- ON
    ep_info = ( struct usbg_buspower_ep_info * )p_ep->driver_data;
    ep_info->stalled = TRUE;
    
    // dequeue requests
    usbg_buspower_ep_dequeue( p_ep );
    
    if( usb_ep_set_halt( p_ep ) != 0 ){
        
        USBG_BUSPWR_PERR( "%s(): usb_ep_set_halt( p_ep ) != 0.", __FUNCTION__ );
        USBG_BUSPWR_PERR( "%s(): retry usb_ep_set_halt( p_ep ) = %d.", __FUNCTION__, usb_ep_set_halt( p_ep ) );
        err = USB_GADGETCORE_STALL;
        
    }
    
    USBG_BUSPWR_EP_LOCK_OFF();
    
    usbg_buspower_call_user_cb( USBG_BUSPWR_CBID_EP_SET_HALT, 0, NULL, 0 );
    
    USBG_BUSPWR_PDEBUG( "Leaving: %s() , return val is %d ", __FUNCTION__, err );
    
    return err;
    
}

/**
 *    @brief        GadgetCore callback - clear halt
 *    @param        p_drv        function driver struct
 *    @param        p_ep        stalled ep
 *    @retval       0                      success
 *    @retval       USB_GADGETCORE_STALL   STALL
 *    @note         Called when ClearFeature issued to this EP
 */
static int core_cb_clear_halt( struct usb_gadget_func_driver* p_drv,
                        struct usb_ep* p_ep )
{
    
    int err = 0;
    struct usbg_buspower_ep_info *ep_info = usbg_buspower_get_ep_info( p_ep );
    struct func_data_ex *p_this_ex = usbg_buspower_get_this_ex();
    
    USBG_BUSPWR_PDEBUG( "Calling: %s()", __FUNCTION__ );
    
    USBG_BUSPWR_EP_LOCK_ON();
    
    // stall flag <- OFF
    if( ep_info ) {
        
        ep_info->stalled = FALSE;
        
    }
    
    /* <framework> */
    if( usb_ep_clear_halt( p_ep ) != 0 ){
        
        err = USB_GADGETCORE_STALL;
        
    }
    
    if( p_this_ex->ep_info[ USBG_BUSPWR_EP_IDX_INT_IN ].stalled == FALSE ) {
        
        // all EP stall cleared
        usbg_buspower_call_user_cb( USBG_BUSPWR_CBID_EP_CLR_HALT, 0, NULL, 0 );
        
    }
    
    USBG_BUSPWR_EP_LOCK_OFF();
    
    USBG_BUSPWR_PDEBUG( "Leaving: %s() , return val is %d ", __FUNCTION__, err );
    
    return err;
    
}

/**
 *    @brief        GadgetCore callback - class request
 *    @param        p_drv        function driver struct
 *    @param        p_ctrl        USB setup packet
 *    @param        ep0            EP0 ep struct
 *    @retval       0                     success
 *    @retval       USB_GADGETCORE_STALL  STALL
 *    @retval       USB_GADGETCORE_THRU   hand over this request to another driver 
 */
static int core_cb_class( struct usb_gadget_func_driver* p_drv,
                       const struct usb_ctrlrequest* p_ctrl,
                       struct usb_ep* ep0 )
{
    
    int err = 0;
    
    USBG_BUSPWR_PDEBUG( "Calling: %s()", __FUNCTION__ );
    
    // Is this class request for me?
    if( !( p_ctrl->bRequestType & USB_TYPE_CLASS ) || 
        !( p_ctrl->bRequestType & USB_RECIP_INTERFACE ) ) {
        
        USBG_BUSPWR_PDEBUG( "%s(): bmRequestType is not for USB Gadget BusPower", __FUNCTION__ );
        return USB_GADGETCORE_THRU;
        
    }
    
    USBG_BUSPWR_PDEBUG( "%s(): bRequest is 0x%x", __FUNCTION__, p_ctrl->bRequest );
    
    switch( p_ctrl->bRequest ) {
        
        case USBG_BUSPWR_B_REQUEST_GET_REPORT:
            // Get Report
            USBG_BUSPWR_PINF( "%s(): GetReport request. bRequest is 0x%x", __FUNCTION__, p_ctrl->bRequest );
            err = usbg_buspower_class_get_report( p_ctrl, ep0 );
            break;
        default:
            // Unknown request
            USBG_BUSPWR_PDEBUG( "%s(): Unknown request. bRequest is 0x%x", __FUNCTION__, p_ctrl->bRequest );
            err = USB_GADGETCORE_STALL;
            break;
        
    }
    
    return err;
    
}

/**
 *    @brief        USBG Buspower core ioctl - probe
 *    @param        file        file struct
 *    @param        arg         ioctl arguments
 *    @retval       0           Success
 *    @retval       -ENOBUFS    No buffer
 */
static int ioctl_probe( struct file *file, unsigned long arg )
{
    
    int err;
    struct usb_gadget_func_driver* reginfo;
    struct probe_info    *p_info    = (struct probe_info *)arg;
    struct func_data    *p_this    = usbg_buspower_get_this();
    int ifnum;
    
    USBG_BUSPWR_PINF( "Calling: %s()", __FUNCTION__ );
    
    // we do not allow multi I/F per 1 instance
    p_info->uc_func_info_interface_num = 1;
    
    if( usbg_cmn_copy_probe_info( p_this, p_info ) ){
        
        USBG_BUSPWR_PERR( "%s(): Failed to usbg_cmn_copy_probe_info().", __FUNCTION__ );
        USBG_BUSPWR_PERR( "%s(): retry usbg_cmn_copy_probe_info() = %d.", __FUNCTION__, usbg_cmn_copy_probe_info( p_this, p_info ) );
        return -ENOBUFS;
        
    }
    
    ifnum                  = p_this->info.uc_func_info_interface_num;
    reginfo                = &( p_this->core_reg_info );
    reginfo->function      = USBG_BUSPWR_FUNC_NAME;
    reginfo->context       = p_this;
    reginfo->start         = core_cb_start;
    reginfo->stop          = core_cb_stop;
    reginfo->class         = core_cb_class;
    reginfo->ep_set_halt   = core_cb_set_halt;
    reginfo->ep_clear_halt = core_cb_clear_halt;
    reginfo->suspend       = core_cb_nop;
    reginfo->resume        = core_cb_nop;
    reginfo->config        = p_this->info.tbl_interface[0].uc_configuration;
    reginfo->interface     = p_this->info.tbl_interface[0].uc_interface;
    
    USBG_BUSPWR_PDEBUG( "%s(): register %s, cfg=%d, if=%d",
                __FUNCTION__,
                reginfo->function,
                reginfo->config,
                reginfo->interface );
    
    err = usb_gadgetcore_register_driver( reginfo );
    
    if( err ) {
        
        USBG_BUSPWR_PERR( "%s(): Failed to usb_gadgetcore_register_driver(). error code is %d.", __FUNCTION__, err );
        
    }
    
    USBG_BUSPWR_PDEBUG( "Leaving: %s() , return val is %d ", __FUNCTION__, err );
    
    return err;
    
}

/**
 *    @brief        USBG Buspower core ioctl - remove
 *    @param        file        file struct
 *    @param        arg         ioctl arguments
 *    @retval       0           Success
 *    @retval       -ENOBUFS    No buffer
 */
static int ioctl_remove( struct file *file, unsigned long arg )
{
    
    int ret;
    struct func_data* p_this = usbg_buspower_get_this();
    
    USBG_BUSPWR_PINF( "Calling: %s()", __FUNCTION__ );
    
    ret = usb_gadgetcore_unregister_driver( &(p_this->core_reg_info) );
    usbg_cmn_clear_probe_info( p_this );
    
    USBG_BUSPWR_PDEBUG( "Leaving: %s() , return val is %d ", __FUNCTION__, ret );
    
    return ret;
    
}

/**
 *    @brief        USBG Buspower core ioctl - Gadget functin driver common
 *    @param        file        file struct
 *    @param        arg         ioctl arguments
 *    @retval       0           success
 *    @retval       -ENOBUFS    no buffer
 *    @retval       -EFAULT     segmentation fault
 *    @retval       -ENOTTY     invalid ioctl no.
 */
static int ioctl_cmn( struct file *file, unsigned int cmd, unsigned long arg )
{
    
    int err = 0;
    
    USBG_BUSPWR_PDEBUG( "Calling: %s()", __FUNCTION__ );
    
    // valid ioctl no?
    if( _IOC_NR(cmd) > USB_IOC_CMN_NBROF ){
        
        USBG_BUSPWR_PERR( "%s(): Invalid ioctl number.", __FUNCTION__ );
        return -ENOTTY;
        
    }
    
    // read/write access?
    err = !access_ok( (void *)arg, _IOC_SIZE(cmd) );
    
    // access ok?
    if( err ){
        
        USBG_BUSPWR_PERR( "%s(): ioctl verify error. error code is %d", __FUNCTION__, err );
        return -EFAULT;
        
    }
    
    // ioctl for what?
    switch ( cmd ) {
        
        case USB_IOC_CMN_PROBE:
            err = ioctl_probe( file, arg );
            break;
        case USB_IOC_CMN_REMOVE:
            err = ioctl_remove( file, arg );
            break;
        default:
            USBG_BUSPWR_PERR( "%s(): Invalid ioctl command id. cmd code is %d", __FUNCTION__, cmd );
            err = -ENOTTY;
            break;
        
    }
    
    USBG_BUSPWR_PDEBUG( "Leaving: %s() , return val is %d ", __FUNCTION__, err );
    
    return err;
    
}


/**
 *    @brief        USBG Buspower core driver open
 *    @param        inode       inode struct
 *    @param        file        file struct
 *    @retval       0           success
 */
UDIF_ERR usbg_buspower_core_open(UDIF_FILE *filp)
{
    struct file fd;
    struct file *file = &fd;
    UDIF_ERR ret = UDIF_ERR_OK;
    
    USBG_BUSPWR_PINF( "Calling: %s()", __FUNCTION__ );
    
    usbg_buspower_link_this_to_file( file );
    udif_file_data(filp) = file->private_data;
    
    USBG_BUSPWR_PDEBUG( "Leaving: %s() , return val is %d ", __FUNCTION__, ret );
    
    return ret;
}

/**
 *    @brief        USBG Buspower core driver release
 *    @param        inode       inode struct
 *    @param        file        file struct
 *    @retval       0           success
 */
UDIF_ERR usbg_buspower_core_release(UDIF_FILE *filp)
{
    UDIF_ERR ret = UDIF_ERR_OK;
    
    USBG_BUSPWR_PINF( "Calling: %s()", __FUNCTION__ );
    
    USBG_BUSPWR_PDEBUG( "Leaving: %s() , return val is %d ", __FUNCTION__, ret );
    
    return ret;
}

/**
 *    @brief        USBG Buspower core ioctl
 *    @param        inode       inode struct
 *    @param        file        file struct
 *    @param        cmd         ioctl no.
 *    @param        arg         ioctl arguments
 *    @retval       0           success
 *    @retval       -ENOTTY     wrong ioctl no.
 */
UDIF_ERR usbg_buspower_core_ioctl(UDIF_FILE *filp, UDIF_IOCTL *param)
{
    struct file fd;
    struct file *file = &fd;
    unsigned int cmd = param->cmd;
    unsigned long arg = param->arg;
    int err;
    
    USBG_BUSPWR_PDEBUG( "Calling: %s()", __FUNCTION__ );
    
    file->private_data = udif_file_data(filp);
    if( _IOC_TYPE( cmd ) == USB_IOC_CMN ){
        
        // Common ioctl
        err = ioctl_cmn( file, cmd, arg );
        udif_file_data(filp) = file->private_data;
        
    } else {
        
        // Invalid ioctl
        USBG_BUSPWR_PERR( "%s(): Invalid ioctl command id. cmd code is %d", __FUNCTION__, cmd );
        err = UDIF_ERR_NOTTY;
        
    }
    
    USBG_BUSPWR_PDEBUG( "Leaving: %s() , return val is %d ", __FUNCTION__, err );
    
    return (UDIF_ERR)err;

}

/**
 *    @brief        USBG Buspower write complete callback
 *    @param        ep            bulk-in ep
 *    @param        req           request struct
 */
static void usbg_buspower_core_write_comp( struct usb_ep *ep, struct usb_request *req )
{
    
    USBG_BUSPWR_PDEBUG( "Calling: %s()", __FUNCTION__ );
    
    usbg_buspower_request_stop( USBG_BUSPWR_EP_IDX_INT_IN );

    /* Suppress ep_queue when the status is outlier because unnecessary waits for COMP may occur. */
    if(0 == req->status){
        usbg_buspower_call_user_cb( USBG_BUSPWR_CBID_VENDOR, USBG_KEVENT_ID_CMN_WRITE_CMP, NULL, 0 );
    }else{
        USBG_BUSPWR_PINF( "%s(): end of write loop status=%d", __FUNCTION__, req->status );
    }

    complete(&(usbg_buspower_get_this_ex()->req_wait_completion));
    
    USBG_BUSPWR_PDEBUG( "Leaving: %s()", __FUNCTION__ );
    
}

/**
 *    @brief        USBG Buspower write 
 *    @param        file        file struct
 *    @param        data        data buffer
 *    @param        size        data size
 *    @param        off         offset
 *    @retval       0           success
 *    @retval       -EINVAL     invalid parameter
 *    @retval       -ENOMEM     short of memory
 *    @retval       -EIO        I/O error
 */
UDIF_SSIZE usbg_buspower_core_write(UDIF_FILE *filp, UDIF_CDEV_WRITE *param)
{
    const char *data = param->uaddr;
    size_t size = param->count;
    UDIF_ERR err;
    struct usb_request *req;
    struct usb_ep *ep;
    u8 *buf;
    
    USBG_BUSPWR_PDEBUG( "Calling: %s()", __FUNCTION__ );
    
    // Check Param
    if( !data || ( size > USBG_BUSPWR_DATA_BUFSIZE ) ) {
        
        USBG_BUSPWR_PERR( "%s(): data==NULL", __FUNCTION__ );
        err = UDIF_ERR_PAR;
        goto SUB_RET;
        
    }
    
    USBG_BUSPWR_EP_LOCK_ON();

    if( usbg_buspower_is_stop_usb_reqest() ) {
        USBG_BUSPWR_EP_LOCK_OFF();
        USBG_BUSPWR_PERR( "%s(): request not allowed", __FUNCTION__ );
        err = UDIF_ERR_BUSY;
        goto SUB_RET;
    }

    ep = usbg_buspower_get_ep( USBG_BUSPWR_EP_IDX_INT_IN );
    
    if( !ep ) {
        
        USBG_BUSPWR_EP_LOCK_OFF();
        USBG_BUSPWR_PERR( "%s(): ep==NULL", __FUNCTION__ );
        err = UDIF_ERR_IO;
        goto SUB_RET;
        
    }
    
    if( usbg_buspower_is_requesting( USBG_BUSPWR_EP_IDX_INT_IN ) ) {
        
        USBG_BUSPWR_EP_LOCK_OFF();
        USBG_BUSPWR_PERR( "%s(): usbg_buspower_is_requesting() error.", __FUNCTION__ );
        err = UDIF_ERR_BUSY;
        goto SUB_RET;
        
    }
    
    // Alloc request
    req = usbg_buspower_get_request( ep );
    
    if( !req ) {
        
        USBG_BUSPWR_EP_LOCK_OFF();
        USBG_BUSPWR_PERR( "%s(): Could not alloc request.", __FUNCTION__ );
        err = UDIF_ERR_NOMEM;
        goto SUB_RET;
        
    }
    
    // Alloc buffer
    buf = usbg_buspower_get_data_buffer();
    
    if( !buf ) {
        
        USBG_BUSPWR_EP_LOCK_OFF();
        USBG_BUSPWR_PERR( "%s(): Failed ) usbg_buspower_get_data_buffer().", __FUNCTION__ );
        err = UDIF_ERR_NOMEM;
        goto SUB_RET;
        
    }
    
    err = copy_from_user( buf, data, size );
    
    if( err ) {
        
        USBG_BUSPWR_EP_LOCK_OFF();
        USBG_BUSPWR_PERR( "%s(): Failed ) copy_from_user(). error code is %d ", __FUNCTION__, err );
        err = UDIF_ERR_BUSY;
        goto SUB_RET;
        
    }
    
    // Fill request
    req->buf            = buf;
    req->length         = size;
    req->dma            = (dma_addr_t)NULL;
    req->zero           = req->length ? 0 : 1;
    req->complete       = usbg_buspower_core_write_comp;
    req->no_interrupt   = 0;
    req->short_not_ok   = 0;
    req->context        = NULL;
    
    init_completion(&(usbg_buspower_get_this_ex()->req_wait_completion));
    usbg_buspower_request_start( USBG_BUSPWR_EP_IDX_INT_IN );
    
    // Queue request
    err = usb_ep_queue( ep, req, GFP_ATOMIC );
    
    USBG_BUSPWR_EP_LOCK_OFF();
    
    if( err ) {
        
        USBG_BUSPWR_PERR( "%s(): Failed ) usb_ep_queue(). error code is %d ", __FUNCTION__, err );
        usbg_buspower_request_stop( USBG_BUSPWR_EP_IDX_INT_IN );
        goto SUB_RET;
        
    }
    
SUB_RET:
    USBG_BUSPWR_PDEBUG( "Leaving: %s() , return val is %d ", __FUNCTION__, err );
    
    return err;
    
}
