
/*******************************************************************************//**
 * @file    usbg_buspower_pvt.h
 * @brief   USB Gadget BusPower Driver - private header
 * 
 * Copyright 2011 Sony Corporation
 * Copyright 2018, 2019 Sony Imaging Products and Solutions Incorporated.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 *******************************************************************************/

#ifndef __USBG_BUSPOWER_PVT_H__
#define __USBG_BUSPOWER_PVT_H__

/*=============================================================================
/  Includes
/==============================================================================*/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <asm/page.h>

#include <linux/udif/spinlock.h>
#include <linux/udif/cdev.h>

/*=============================================================================
/  On Debug
/==============================================================================*/

#ifndef MY_NAME
#define MY_NAME "usbg_buspower"
#endif

// #define USBG_BUSPWR_DEBUG

#ifdef USBG_BUSPWR_DEBUG
 #define USBG_BUSPWR_PDEBUG( fmt,args... )  printk( MY_NAME ": " fmt "\n", ## args )
#else
 #define USBG_BUSPWR_PDEBUG( fmt,args... )  do { } while (0)
#endif

#define USBG_BUSPWR_PINF( fmt,args... )     printk( MY_NAME ": " fmt "\n", ## args )
#define USBG_BUSPWR_PERR( fmt,args... )     printk( MY_NAME ": " fmt "\n", ## args )

/*=============================================================================
/  Definitions
/==============================================================================*/

#define USBG_BUSPWR_VER	"02.00.000"

#ifndef FNAME
#define FNAME "usbg_buspower"
#endif

#define USBG_BUSPWR_LOCK_INI()             { udif_spin_lock_init(    &( usbg_buspower_get_this_ex()->ep_lock )         ); \
                                             udif_spin_lock_init(    &( usbg_buspower_get_this_ex()->requesting_lock ) ); }

#define USBG_BUSPWR_EP_LOCK_ON()           { USBG_BUSPWR_PDEBUG( "%s(%d):### EP LOCK ON  ---> ", __FUNCTION__, __LINE__                                      ); \
                                             udif_spin_lock_irqsave(    &( usbg_buspower_get_this_ex()->ep_lock ), usbg_buspower_get_this_ex()->ep_lock_flag ); }

#define USBG_BUSPWR_EP_LOCK_OFF()          { USBG_BUSPWR_PDEBUG( "%s(%d):### EP LOCK OFF <--- ", __FUNCTION__, __LINE__                                         ); \
                                             udif_spin_unlock_irqrestore(  &( usbg_buspower_get_this_ex()->ep_lock ), usbg_buspower_get_this_ex()->ep_lock_flag ); }

#define USBG_BUSPWR_REQUESTING_LOCK_ON()   { USBG_BUSPWR_PDEBUG( "%s(%d):### REQUESTING LOCK ON <--- ", __FUNCTION__, __LINE__                                               ); \
                                             udif_spin_lock_irqsave(    &( usbg_buspower_get_this_ex()->requesting_lock ), usbg_buspower_get_this_ex()->requesting_lock_flag ); }

#define USBG_BUSPWR_REQUESTING_LOCK_OFF()  { USBG_BUSPWR_PDEBUG( "%s(%d):### REQUESTING LOCK OFF <--- ", __FUNCTION__, __LINE__                                                 ); \
                                             udif_spin_unlock_irqrestore(  &( usbg_buspower_get_this_ex()->requesting_lock ), usbg_buspower_get_this_ex()->requesting_lock_flag ); }
                                             

#ifndef TRUE
#define TRUE 1
#endif

#ifndef FALSE
#define FALSE 0
#endif

#define USBG_BUSPWR_FUNC_NAME            "USB Gadget BusPower Device"
#define USBG_BUSPWR_EP_NAME_INT_IN       "USB Gadget Buspower (Int.-in)"

#define USBG_BUSPWR_EP0_BUFSIZE             (   32 )   ///< Buffer size (EP0)
#define USBG_BUSPWR_DATA_BUFSIZE            (   32 )   ///< Buffer size (Data)
#define USBG_BUSPWR_B_REQUEST_GET_REPORT    ( 0x01 )   ///< GET_REPORT


/*=============================================================================
/  Enumerations
/==============================================================================*/


/*=============================================================================
/  Structs
/==============================================================================*/

/*
 *    @brief    USB Gadget Buspower Endpoint Information
 */
struct usbg_buspower_ep_info {
    u8 ep_no;                               ///< EP no. (eg.0x81)
    u8 stalled;                             ///< stall? or not?
    u8 requesting;                          ///< requesting?
    struct usb_request *req;                ///< usb request
};

/*
 *    @brief    Callback ID (to userspace)
 */
enum {
    USBG_BUSPWR_CBID_START = 0,                    ///< started
    USBG_BUSPWR_CBID_STOP,                         ///< stopped
    USBG_BUSPWR_CBID_EP_SET_HALT,                  ///< EP halt set
    USBG_BUSPWR_CBID_EP_CLR_HALT,                  ///< EP halt cleared
    USBG_BUSPWR_CBID_SUSPEND,                      ///< USB bus suspened
    USBG_BUSPWR_CBID_RESUME,                       ///< USB bus resumed
    USBG_BUSPWR_CBID_CLASS,                        ///< class request
    USBG_BUSPWR_CBID_VENDOR,                       ///< vendor request
    USBG_BUSPWR_CBID_NBROF,                        ///< callback id MAX
};

/*
 *    @brief    USB Gadget Buspower Driver ep ID
 */
enum {
    USBG_BUSPWR_EP_IDX_INT_IN = 0,                 ///< ID: Int.-in EP
    USBG_BUSPWR_EP_IDX_MAX,                        ///< ID: MAX
};

/*
 *    @brief    USB Gadget Buspower Driver information (extension)
 */
struct func_data_ex {
    void*            buffer;                                       ///< kernel-virtual address
    dma_addr_t       buffer_dma;                                   ///< dma address
    uintptr_t        buffer_phys;                                  ///< physical address
    void*            buffer_user;                                  ///< user-virtual address
    struct usbg_buspower_ep_info ep_info[USBG_BUSPWR_EP_IDX_MAX];  ///< EP information
    u8               ep0_buffer[USBG_BUSPWR_EP0_BUFSIZE];          ///< EP0 buffer
    u8               data_buffer[USBG_BUSPWR_DATA_BUFSIZE];        ///< Data buffer
    bool             stop_usb_request;                             ///< Stop new usb requests
    UDIF_SPINLOCK    ep_lock;                                      ///< lock for endpoint
    unsigned long    ep_lock_flag;
    UDIF_SPINLOCK    requesting_lock;                              ///< lock for requesting flag
    unsigned long    requesting_lock_flag;
    struct completion req_wait_completion;                         ///< ep request comp completion
};


/*=============================================================================
/  Prototypes
/==============================================================================*/

//----------------------------------------------------------------------
// usbg_buspower_core file operations
UDIF_ERR usbg_buspower_core_open(UDIF_FILE *filp);
UDIF_ERR usbg_buspower_core_release(UDIF_FILE *filp);
UDIF_ERR usbg_buspower_core_ioctl(UDIF_FILE *filp, UDIF_IOCTL *param);
UDIF_SSIZE usbg_buspower_core_write(UDIF_FILE *filp, UDIF_CDEV_WRITE *param);

//----------------------------------------------------------------------
// usbg_buspower local functions
struct func_data*                usbg_buspower_get_this(void);
struct func_data_ex*             usbg_buspower_get_this_ex(void);
struct usb_ep*                   usbg_buspower_get_ep(int ep_id);
struct usbg_buspower_ep_info*    usbg_buspower_get_ep_info(struct usb_ep *ep);
void*                            usbg_buspower_get_ep0_buffer(void);

int  usbg_buspower_create_this(void);
void usbg_buspower_delete_this(void);
void usbg_buspower_link_this_to_file(struct file *file);
void usbg_buspower_put_u8(u8 **p, u8 val);
void usbg_buspower_call_user_cb(int cb_id, int evt_id, void *arg, int size);
void usbg_buspower_ep_dequeue(struct usb_ep *ep);
void usbg_buspower_ep_dequeue_all(void);
int usbg_buspower_is_requesting( int ep_id );
struct usb_request *usbg_buspower_get_request( struct usb_ep *ep );
void *usbg_buspower_get_data_buffer( void );
void usbg_buspower_set_requesting_flag( int ep_id, int flag );
void usbg_buspower_request_start( int ep_id );
void usbg_buspower_request_stop( int ep_id );
bool usbg_buspower_is_stop_usb_reqest(void);

#endif /* __USBG_BUSPOWER_PVT_H__ */

