/**
 *	@file	usbg_still_cmn.c
 *	@brief	USB SICD(Still Image Capture Device class) - common
 *	
 *		Copyright 2005,2006,2011 Sony Corporation
 * Copyright 2018 Sony Imaging Products and Solutions Incorporated.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <asm/errno.h>
#include <asm/unaligned.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/version.h>

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 21)
#include <linux/usb/ch9.h>
#else
#include <linux/usb_ch9.h>
#endif

#include <linux/device.h>

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 29)
#include <linux/usb/gadget.h>
#else
#include <linux/usb_gadget.h>
#endif

#include <linux/usb/gcore/usb_event.h>
#include <linux/usb/gcore/usb_gadgetcore.h>

#ifndef CONFIG_OSAL_UDIF
#include <asm/arch/sonymisc.h>
#endif

#include <linux/usb/specific_gadget/usbg_cmn.h>
#include <linux/usb/specific_gadget/usbg_type.h>
#include <linux/usb/specific_gadget/usbg_still.h>
#include <linux/usb/specific_gadget/usb_kdbg.h>

#define FNAME "usbg_still_cmn.c"
#include "usbg_still_conf.h"
#include "usbg_still_pvt.h"

#include <linux/udif/malloc.h>

/**
 *	@brief		SICD driver instance
 *	@note		usbg_dev == 0 means "SICD driver is invalid"
 *				usbg_dev != 0 means "SICD driver is valid"
 */
static struct func_data *sicd_this = NULL;

/**
 *	@brief		Create driver instance
 *	@retval		0			success
 *	@retval		-ENOMEM		short of memory
 */
int sicd_create_this(void)
{
    struct func_data*    p_this;
    struct func_data_ex* p_this_ex;
    
//    SICD_FNC();
    
    if(sicd_this != NULL) {
		// instance is already created
		// nothint to do
		return 0;
	}
	
    // alloc driver instance area
	p_this = kmalloc(sizeof(struct func_data), GFP_ATOMIC);
	if(!p_this) {
		SICD_ERR("can not create instance\n");
		return -ENOMEM;
	}
	
    // alloc driver instance area (SICD extention)
	p_this_ex = kmalloc(sizeof(struct func_data_ex), GFP_ATOMIC);
	if(!p_this_ex) {
		SICD_ERR("can not create instance\n");
		kfree(p_this);
		return -ENOMEM;
	}
	
    // 0 fill 
	memset(p_this,    0, sizeof(struct func_data));
	memset(p_this_ex, 0, sizeof(struct func_data_ex));

	// SICD status --> BUSY
	p_this_ex->sicd_status = SICD_STATUS_BUSY;

	// init wait completion obj
	p_this_ex->req_comp_waiting = false;
	init_completion(&(p_this_ex->req_wait_completion));

	// new usb request stop flag
	p_this_ex->stop_usb_request = false;

	// link informations this->this_ex
	p_this->vp_user = p_this_ex;
	sicd_this = p_this;
	
	return 0;
}

/**
 *	@brief		Free driver instance
 */
void sicd_delete_this(void)
{
	struct func_data*    p_this;
	struct func_data_ex* p_this_ex;
	
//	SICD_FNC();
	
	p_this    = sicd_get_this();
	p_this_ex = sicd_get_this_ex();
	
	// free driver informations
	if(p_this_ex) {
		kfree(p_this_ex);
	}
	
	if(p_this) {
		kfree(p_this);
	}
	
	sicd_this = NULL;
}

/**
 *	@brief		Get driver instance
 *	@retval		driver instance
 */
struct func_data *sicd_get_this(void)
{
	return sicd_this;
}

/**
 *	@brief		Get driver instance (SICD extention)
 *	@retval		driver instance (SICD extention)
 */
struct func_data_ex *sicd_get_this_ex(void)
{
	if(!sicd_this) {
		return NULL;
	}
	
	return (struct func_data_ex *)sicd_this->vp_user;
}

/**
 *	@brief		Link driver instance to struct file
 *	@param		file	struct file
 */
void sicd_link_this_to_file(struct file *file)
{
	if(file) {
		file->private_data = sicd_this;
	}
}

/**
 *	@brief		Put U8 data
 *	@param		p		address
 *	@param		val		data
 */
void sicd_put_u8(u8 **p, u8 val)
{
	*(u8 *)(*p) = val;
	(*p) += 1;
}

/**
 *	@brief		Put U16 data
 *	@param		p		address
 *	@param		val		data
 */
void sicd_put_u16(u8 **p, u16 val)
{
	put_unaligned(cpu_to_le16(val), (u16*)*p);
	(*p) += 2;
}

/**
 *	@brief		Put U32 data
 *	@param		p		address
 *	@param		val		data
 */
void sicd_put_u32(u8 **p, u32 val)
{
	put_unaligned(cpu_to_le32(val), (u32*)*p);
	(*p) += 4;
}


/**
 *	@brief		Convert address (User-virtual --> DMA)
 *	@param		uaddr	user-virtual address
 *	@retval		DMA address
 */
dma_addr_t sicd_user_to_dma(void *uaddr)
{
	uintptr_t dma;
	
	struct func_data_ex *p_still = sicd_get_this_ex();
	dma = (uintptr_t)uaddr - (uintptr_t)p_still->buffer_user + (uintptr_t)p_still->buffer_dma;
	
	return (dma_addr_t)dma;
}

/**
 *	@brief		Convert address (User-virtual --> Kernel-virtual)
 *	@param		uaddr	user-virtual address
 *	@retval		Kernel-virtual address
 */
void *sicd_user_to_kvirt(void *uaddr)
{
	uintptr_t kaddr;
	
	struct func_data_ex *p_still = sicd_get_this_ex();
	kaddr = (uintptr_t)uaddr - (uintptr_t)p_still->buffer_user + (uintptr_t)p_still->buffer;
	
	return (void*)kaddr;
}

/**
 *	@brief		Call user area callback function
 *	@param		cb_id	callback ID
 *	@param		evt_id	event ID
 *	@param		arg		arguments
 *	@param		size	argument size
 */
void sicd_call_user_cb(int cb_id, int evt_id, void *arg, int size)
{
	struct func_data *p_this;
	void *user_cb;
	usb_hndl_t handle;
	
//	SICD_FNC();
	
	p_this = sicd_get_this();
	handle = (usb_hndl_t)p_this->info.hndl;
	
	switch(cb_id) {
		case SICD_CBID_START:
			user_cb = p_this->info.event.start;
			break;
		case SICD_CBID_STOP:
			user_cb = p_this->info.event.stop;
			break;
		case SICD_CBID_EP_SET_HALT:
			user_cb = p_this->info.event.ep_set_halt;
			break;
		case SICD_CBID_EP_CLR_HALT:
			user_cb = p_this->info.event.ep_clear_halt;
			break;
		case SICD_CBID_SUSPEND:
			user_cb = p_this->info.event.suspend;
			break;
		case SICD_CBID_RESUME:
			user_cb = p_this->info.event.resume;
			break;
		case SICD_CBID_CLASS:
			user_cb = p_this->info.event.class_req;
			break;
		case SICD_CBID_VENDOR:
			user_cb = p_this->info.event.vendor_req;
			break;
		default:
			SICD_ERR("Unknown Callback ID\n");
			return;
	}
    
	usb_event_add_queue(USB_EVENT_PRI_NORMAL,
			user_cb, handle, evt_id, size, arg);
}

/**
 *	@brief		Get SICD status
 *	@retval		SICD status
 */
int sicd_get_status(void)
{
	struct func_data_ex *p_this_still = sicd_get_this_ex();

	if(p_this_still) {
		return p_this_still->sicd_status;
	}
	else {
		return 0;
	}
}

/**
 *	@brief		Set SICD status
 *	@param		status	SICD status
 */
void sicd_set_status(int status)
{
	struct func_data_ex *p_this_still = sicd_get_this_ex();

	if(p_this_still) {
		p_this_still->sicd_status = status;
	}
}


/**
 *	@brief		Get ep0 buffer
 *	@retval		ep0 buffer address
 */
void *sicd_get_ep0_buffer(void)
{
	struct func_data_ex* p_this_ex = sicd_get_this_ex();
	
	if(!p_this_ex) {
		return NULL;
	}
	else {
		return p_this_ex->ep0_buffer;
	}
}

/**
 *	@brief		Get ep struct
 *	@param		ep_id	Endpoint ID
 *	@retval		struct ep
 */
struct usb_ep *sicd_get_ep(int ep_id)
{
	struct func_data_ex *p_this_still = sicd_get_this_ex();
	
	if(ep_id >= p_this_still->ep_max_num) {
		// invalid EP no.
		return NULL;
	}
	
	if(!sicd_this ||
	   !sicd_this->ptbl_ep) {
		SICD_INF("p_this->ptbl_ep==NULL\n");
		return NULL;
	}
	
	return sicd_this->ptbl_ep[ep_id];
}

/**
 *	@brief		Get ep information
 *	@param		ep	struct ep
 *	@retval		ep information
 */
struct sicd_ep_info *sicd_get_ep_info(struct usb_ep *ep)
{
	if(!ep) return NULL;
	
	return (struct sicd_ep_info*)ep->driver_data;
}

/**
 *	@brief		Start request
 *	@param		ep_id	Endpoint ID
 */
void sicd_request_start(int ep_id)
{
    
    sicd_set_requesting_flag( ep_id, TRUE );
    
}

/**
 *	@brief		Stop request
 *	@param		ep_id	Endpoint ID
 */
void sicd_request_stop(int ep_id)
{
    
    sicd_set_requesting_flag( ep_id, FALSE );
    
}

/**
 *	@brief		Requesting?
 *	@param		ep_id	Endpoint ID
 */
int  sicd_is_requesting(int ep_id)
{
    
    struct func_data_ex *p_this_still = sicd_get_this_ex();
    
    if( !p_this_still || !(p_this_still->ep_info)) {
        
        SICD_INF( "p_this_still->ep_info is NULL\n" );
        
    }else{
        
        SICD_REQUESTING_LOCK_ON();
        
        if( p_this_still->ep_info[ep_id].u.data.requesting ){
            
            SICD_REQUESTING_LOCK_OFF();
            
            return TRUE;
            
        }
        
        SICD_REQUESTING_LOCK_OFF();
        
    }
    
    return FALSE;
    
}

/**
 *	@brief		Set requesting flag
 *	@param		ep_id	Endpoint ID
  *	@param		flag	requesting flag to be set. TRUE or FALSE.
 */
void sicd_set_requesting_flag( int ep_id, int flag )
{
    
    struct func_data_ex *p_this_still = sicd_get_this_ex();
    
    if( !p_this_still || !(p_this_still->ep_info)) {
        
        SICD_INF( "p_this_still->ep_info is NULL\n" );
        
    }else{
        
        if( ( flag == TRUE ) && sicd_is_requesting( ep_id ) ){
            
            SICD_INF("request already queued\n");
            
        }
        
        SICD_REQUESTING_LOCK_ON();
        
        p_this_still->ep_info[ep_id].u.data.requesting = flag;
        
        SICD_REQUESTING_LOCK_OFF();
        
    }
    
}


/**
 *	@brief		Dequeue request
 *	@param		ep		struct ep
 */
void sicd_ep_dequeue(struct usb_ep *ep)
{
    int i;
    struct sicd_ep_info *ep_inf = sicd_get_ep_info(ep);
    
    SICD_FNC();
    
    if (!ep_inf) {
        SICD_INF("ep_inf==NULL\n");
        return;
    }
    
    switch (ep_inf->ep_no) {
    case 0x81: /* data - BULK_IN */
        SICD_REQUESTING_LOCK_ON();
        if (ep_inf->u.data.req && ep_inf->u.data.requesting) {
            SICD_REQUESTING_LOCK_OFF();
            usb_ep_dequeue(ep, ep_inf->u.data.req);
        }
        else {
            SICD_REQUESTING_LOCK_OFF();
            SICD_INF("request==NULL\n");
        }
        if (ep_inf->u.evt != NULL) { /* bulk event mode */
            for (i=0; i<SICD_MAX_EVT_REQ_NUM; i++) {
                if(ep_inf->u.evt[i].req && ep_inf->u.evt[i].requesting) {
                    usb_ep_dequeue(ep, ep_inf->u.evt[i].req);
                }
            }
        }
        break;
    case 0x02: /* data - BULK_OUT */
        SICD_REQUESTING_LOCK_ON();
        if (ep_inf->u.data.req && ep_inf->u.data.requesting) {
            SICD_REQUESTING_LOCK_OFF();
            usb_ep_dequeue(ep, ep_inf->u.data.req);
        }
        else {
            SICD_REQUESTING_LOCK_OFF();
            SICD_INF("request==NULL\n");
        }
        break;
    case 0x83: /* event - INT_IN */
        /* 呼び出し元のSICD_EP_LOCKでアクセス排他済 */
        for (i=0; i<SICD_MAX_EVT_REQ_NUM; i++) {
            if(ep_inf->u.evt[i].req && ep_inf->u.evt[i].requesting) {
                usb_ep_dequeue(ep, ep_inf->u.evt[i].req);
            }
        }
        break;
    default:
        SICD_INF("unknown ep_no:%X\n", ep_inf->ep_no);
        break;
    }
}

/**
 *	@brief		Dequeue request from bulk ep
 */
void sicd_ep_dequeue_bulk(void)
{
	struct usb_ep *ep;
	
	SICD_EP_LOCK_ON();

	ep = sicd_get_ep(SICD_EP_IDX_BULK_IN);
	if(ep) {
		sicd_ep_dequeue(ep);
	}

	ep = sicd_get_ep(SICD_EP_IDX_BULK_OUT);
	if(ep) {
		sicd_ep_dequeue(ep);
	}

	SICD_EP_LOCK_OFF();

}

/**
 *	@brief		Dequeue request from all ep
 */
void sicd_ep_dequeue_all(void)
{
	int i;
	struct func_data_ex *p_this_still = sicd_get_this_ex();
	struct usb_ep **ep_list;
	
	if(!sicd_this) {
		// no driver instance
		return;
	}
	
	ep_list = sicd_this->ptbl_ep;
	if(!ep_list) {
		// no ep table
		return;
	}
	
	for(i = 0; i < p_this_still->ep_max_num; i++) {
		if(ep_list[i]) {
			sicd_ep_dequeue(ep_list[i]);
		}
	}
}

/**
 *	@brief	first, check request complete status for completion
 */
bool sicd_check_req_comp_waiting(void)
{
    int ep_no, i;
    struct func_data_ex *p_this_still = sicd_get_this_ex();

    SICD_REQUESTING_LOCK_ON();
    SICD_EVT_QUE_LOCK_ON();

    for (ep_no = 0; ep_no < p_this_still->ep_max_num; ep_no++) {
        if (p_this_still->ep_info[ep_no].u.data.requesting == TRUE) {
            goto found_waiting;
        }
        if (p_this_still->ep_info[ep_no].u.evt != NULL) {
            for (i=0; i < SICD_MAX_EVT_REQ_NUM; i++) {
                if (p_this_still->ep_info[ep_no].u.evt[i].requesting == TRUE)
                    goto found_waiting;
            }
        }
    }

    SICD_EVT_QUE_LOCK_OFF();
    SICD_REQUESTING_LOCK_OFF();

    return false;

found_waiting:
    p_this_still->req_comp_waiting = true;

    SICD_EVT_QUE_LOCK_OFF();
    SICD_REQUESTING_LOCK_OFF();

    return true;
}

/**
 *	@brief	recheck request complete status
 */
void sicd_recheck_req_comp_waiting(void)
{
    int ep_no, i;
    struct func_data_ex *p_this_still = sicd_get_this_ex();

    SICD_REQUESTING_LOCK_ON();
    SICD_EVT_QUE_LOCK_ON();
    if (p_this_still->req_comp_waiting) { // only when core_cb_stop() is running.

        for (ep_no = 0; ep_no < p_this_still->ep_max_num; ep_no++) {
            if (p_this_still->ep_info[ep_no].u.data.requesting == TRUE) {
                goto found_waiting;
            }
            if (p_this_still->ep_info[ep_no].u.evt != NULL) {
                for (i=0; i < SICD_MAX_EVT_REQ_NUM; i++) {
                    if (p_this_still->ep_info[ep_no].u.evt[i].requesting == TRUE)
                        goto found_waiting;
                }
            }
        }

        p_this_still->req_comp_waiting = false;
        complete(&p_this_still->req_wait_completion);
    }

found_waiting:
    SICD_EVT_QUE_LOCK_OFF();
    SICD_REQUESTING_LOCK_OFF();

    return;
}

/**
 *	@brief		Halt Bulk-out ep
 *	@retval		0		success
 *	@retval		-EIO	IO error
 */
int sicd_bulk_out_halt(void)
{
	int err;
	struct usb_ep       *ep;
	struct sicd_ep_info *ep_info;

	SICD_FNC();
	
	SICD_EP_LOCK_ON();
	
	ep = sicd_get_ep(SICD_EP_IDX_BULK_OUT);
	if(!ep) {
		SICD_EP_LOCK_OFF();
		return -EIO;
	}
	
	ep_info = sicd_get_ep_info(ep);
	if(!ep_info) {
		SICD_EP_LOCK_OFF();
		return -EIO;
	}
	
	ep_info->stalled = TRUE;
	err = usb_ep_set_halt(ep);
	
	sicd_ep_dequeue(ep);
	
	SICD_EP_LOCK_OFF();
	
	return err;
}

/**
 *	@brief		Halt Bulk-in ep
 *	@retval		0		success
 *	@retval		-EIO	IO error
 */
int sicd_bulk_in_halt(void)
{
	int err;
	struct usb_ep       *ep;
	struct sicd_ep_info *ep_info;

	SICD_FNC();

	SICD_EP_LOCK_ON();

	ep = sicd_get_ep(SICD_EP_IDX_BULK_IN);
	if(!ep) {
		SICD_EP_LOCK_OFF();
		return -EIO;
	}

	ep_info = sicd_get_ep_info(ep);
	if(!ep_info) {
		SICD_EP_LOCK_OFF();
		return -EIO;
	}

	ep_info->stalled = TRUE;
	sicd_ep_dequeue(ep);
	
	err = usb_ep_set_halt(ep);
	
	SICD_EP_LOCK_OFF();
	
	return err;

}

/**
 *	@brief		Halt clear Bulk-out ep
 *	@retval		0		success
 *	@retval		-EIO	IO error
 */
int  sicd_bulk_out_halt_clear(void)
{
	int err;
	struct usb_ep       *ep;
	struct sicd_ep_info *ep_info;

	SICD_FNC();

	SICD_EP_LOCK_ON();

	ep = sicd_get_ep(SICD_EP_IDX_BULK_OUT);
	if(!ep) {
		SICD_EP_LOCK_OFF();
		return -EIO;
	}

	ep_info = sicd_get_ep_info(ep);
	if(!ep_info) {
		SICD_EP_LOCK_OFF();
		return -EIO;
	}
	
	ep_info->stalled = FALSE;
	err = usb_ep_clear_halt(ep);
	
	SICD_EP_LOCK_OFF();
	
	return err;
}

/**
 *	@brief		Halt clear Bulk-in ep
 *	@retval		0		success
 *	@retval		-EIO	IO error
 */
int  sicd_bulk_in_halt_clear(void)
{
	int err;
	struct usb_ep       *ep;
	struct sicd_ep_info *ep_info;

	SICD_FNC();

	SICD_EP_LOCK_ON();

	ep = sicd_get_ep(SICD_EP_IDX_BULK_IN);
	if(!ep) {
		SICD_EP_LOCK_OFF();
		return -EIO;
	}

	ep_info = sicd_get_ep_info(ep);
	if(!ep_info) {
		SICD_EP_LOCK_OFF();
		return -EIO;
	}

	ep_info->stalled = FALSE;
	err = usb_ep_clear_halt(ep);

	SICD_EP_LOCK_OFF();
	
	return err;
}

/**
 *	@brief		Get usb_request area
 *	@retval		usb_request address
 */
struct usb_request *sicd_get_request(struct usb_ep *ep)
{
	struct sicd_ep_info *ep_inf = sicd_get_ep_info(ep);
	
	SICD_FNC();
	
	if(ep_inf) {
		return ep_inf->u.data.req;
	}
	else {
		SICD_INF("request==NULL\n");
		return NULL;
	}
}


/**
 *	@brief		alloc event request and buffer
 *	@retval		event request info
 */
struct sicd_evt_req *sicd_alloc_evt_req_info(struct usb_ep *ep)
{
    int i;
    struct sicd_ep_info *ep_inf = sicd_get_ep_info(ep);
    
    SICD_FNC();
    SICD_EVT_QUE_LOCK_ON();
    
    if(ep_inf) {
        // return not used buffer pointer from buffer list
        for (i=0; i<SICD_MAX_EVT_REQ_NUM; i++) {
            if (ep_inf->u.evt[i].requesting == FALSE) {
                ep_inf->u.evt[i].evt_buffer = udif_kmalloc("evt_buf", SICD_EVT_BUFSIZE, 0);
                ep_inf->u.evt[i].req        = usb_ep_alloc_request(ep, GFP_ATOMIC);
                ep_inf->u.evt[i].requesting = TRUE;
                SICD_EVT_QUE_LOCK_OFF();
                return &(ep_inf->u.evt[i]);
            }
        }
        
        if (i == SICD_MAX_EVT_REQ_NUM) {
            SICD_EVT_QUE_LOCK_OFF();
            SICD_INF("buffer full\n");
            return NULL;
        }
        
    }
    else {
        SICD_EVT_QUE_LOCK_OFF();
        SICD_INF("request==NULL\n");
        return NULL;
    }
    
    SICD_EVT_QUE_LOCK_OFF();
    return NULL;
}

/**
 *	@brief		free event request and buffer
 */
void sicd_free_evt_req_info(struct usb_ep *ep, struct sicd_evt_req *evt_req)
{
    SICD_FNC();
    SICD_EVT_QUE_LOCK_ON();
    
    if(ep && evt_req) {
        evt_req->requesting = FALSE;
        
        if (evt_req->evt_buffer) {
            udif_kfree(evt_req->evt_buffer);
            evt_req->evt_buffer = NULL;
        }
        if (evt_req->req) {
            usb_ep_free_request(ep, evt_req->req);
            evt_req->req = NULL;
        }
    }
    
    SICD_EVT_QUE_LOCK_OFF();
    return;
}

/**
 *	@brief		buffer address check
 *	@param		p		buffer address
 *	@param		size	buffer size
 *	@retval		TRUE	buffer is valid
 *	@retval		FALSE	buffer is invalid
 */
int  sicd_is_valid_buffer(u8 *buf, size_t size)
{
	struct func_data_ex *p_still = sicd_get_this_ex();
	
	SICD_FNC();
	
	if(buf < (u8*)p_still->buffer_user) {
		return FALSE;
	}
	
	if((buf + size) > ((u8*)p_still->buffer_user + p_still->UnitSize)) {
		return FALSE;
	}
	
	return TRUE;
}

/**
 *	@brief		Dump
 *	@param		p		dump address
 *	@param		size	dump size
 *	@param		fname	file name
 *	@param		line	line
 */
void sicd_dump(u8 *p, int size, char *fname, int line) 
{
	printk("%s(%d): [D]addr=%p, size=%d\n", fname, line, p, size);

#ifdef SICD_DEBUG_DMP_HEADER_ONLY
	size = SICD_HEADER_LEN;
#endif

	while(size>0) {
		printk("%02x %02x %02x %02x %02x %02x %02x %02x - %02x %02x %02x %02x %02x %02x %02x %02x\n",
			p[0],  p[1],  p[2],  p[3],  p[4],  p[5],  p[6],  p[7],
			p[8],  p[9],  p[10], p[11], p[12], p[13], p[14], p[15]);
		size-=16;
		p+=16;
	}
}

/**
 *	@brief		new ep_queue allowed check
 *	@retval		TRUE	ep_queue not allowed
 *	@retval		FALSE	ep_queue allowed
 */
bool  sicd_is_stop_usb_reqest(void)
{
	struct func_data_ex *p_still = sicd_get_this_ex();

	SICD_FNC();

	return p_still->stop_usb_request;
}
