/**
 *	@file	usbg_still_data.c
 *	@brief	USB SICD(Still Image Capture Device class) - data
 *	
 *		Copyright 2005,2006,2008,2011,2018 Sony Corporation
 * Copyright 2018, 2019 Sony Imaging Products and Solutions Incorporated.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <asm/errno.h>
#include <asm/memory.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/version.h>


#include <linux/udif/cdev.h>
#include <linux/udif/cache.h>
#include <mach/noncache.h> 

#include <linux/usb/ch9.h>

#include <linux/device.h>
#include <linux/dma-mapping.h>

#include <linux/usb/gadget.h>

#include <linux/usb/gcore/usb_event.h>
#include <linux/usb/gcore/usb_gadgetcore.h>

#ifndef CONFIG_OSAL_UDIF
#include <asm/arch/sonymisc.h>
#endif

#include <linux/usb/specific_gadget/usbg_cmn.h>
#include <linux/usb/specific_gadget/usbg_still.h>
#include <linux/usb/specific_gadget/usbg_type.h>
#include <linux/usb/specific_gadget/usb_kdbg.h>
#include <linux/usb/specific_gadget/usb_config.h>
#include <linux/usb/specific_gadget/usb_ioctl.h>
#include <linux/usb/specific_gadget/usb_ioctl_still.h>

#define FNAME "usbg_still_data.c"
#include "usbg_still_conf.h"
#include "usbg_still_pvt.h"

#ifdef USB_SICD_USE_RAMMAP
#include <sony/conf/bid_g_systemspec_rammap.h>
#endif

static uint32_t g_SicdDatBufSize = 0;

/*==============================================================*/
/*                     Xbhp`                           */
/*==============================================================*/
/******************************/
/*           Define           */
/******************************/
#if !defined(VM_RESERVED)
#define VM_RESERVED		(VM_DONTEXPAND | VM_DONTDUMP)
#endif

/* bZ[Wf[^p\ */
typedef struct _sicd_header_data {
	unsigned int length;
	unsigned short type;
	unsigned short opcode;
	unsigned int transactionid;
	unsigned int param1;
	unsigned int param2;
	unsigned int param3;
	unsigned int param4;
} SICD_HEADER_DATA;

#define SICD_EE_PTPHEADERLEN    (12)    /* PTP header */
#define SICD_EE_PTPEEHEADERLEN	(12+16)	/* PTP header + Vender Param 4 */
#define SICD_EE_EXT_SIZE_MAX	(32 * 1024)	/* gf[^őTCY */
/* ߂l` */
#define SICD_EE_RET_OK			(0)
#define SICD_EE_RET_ERR			(-1)
/* SIC Data Type */
#define SICD_EE_DATATYPE_DATA	(2)
#define SICD_EE_DATATYPE_RESP	(3)
/* SIC Response Code */
#define SICD_EE_RESCODE_OK				(0x2001)
#define SICD_EE_RESCODE_ACCESSDENIED	(0x200F)

/******************************/
/*       Variable             */
/******************************/
static uint64_t     g_SicdEeHeadPhyAddr  = 0;	/* PTPwb_f[^pAhX */
static unsigned int g_SicdEeHeadSize     = 0;	/* PTPwb_f[^pTCY   */
static void *       g_SicdEeHeadVirtAddr = NULL;/* PTPwb_f[^p_AhX */
static uint64_t     g_SicdEeImgPhyAddr   = 0;	/* f[^pAhX          */
static unsigned int g_SicdEeImgSize      = 0;	/* f[^pTCY            */
static unsigned int g_SicdEeImgJpgNum    = 0;	/* f[^p擾                  */
static unsigned int g_SicdEeExtSize      = 0;		/* gf[^TCY                    */
static uint64_t     g_SicdEeExtPhyAddr   = 0;		/* gf[^pAhX */
static void *       g_SicdEeExtVirtAddr  = NULL;	/* gf[^p_AhX */
static unsigned int g_SicdEeExtSizeMax   = 0;		/* gf[^pTCY   */
static unsigned int g_SicdEeTotalSize    = 0;		/* f[^pvTCY              */
static unsigned int g_SicdEeSentSize     = 0;		/* f[^pMς݃TCY          */
static SICD_HEADER_DATA g_SicdEePtpHeadData;

/******************************/
/*       Prototype            */
/******************************/
static int ioctl_ee_open(unsigned long arg);
static int ioctl_ee_picture(struct file *, unsigned long);
static int ioctl_ee_close(void);
static int ioctl_ee_picture_write(void*f , unsigned int , unsigned int);
static int ioctl_ee_picture_header(unsigned long arg);
static int ioctl_ee_picture_data(unsigned long arg);
static int ioctl_ee_picture_response(unsigned long arg);
static int ioctl_ee_alloc_buf_extinfo(void);
static int ioctl_ee_free_buf_extinfo(void);
static int ioctl_ee_picture_extinfo(unsigned long arg);
static int ioctl_ee_picture_prepare(unsigned long arg);
static int ioctl_ee_picture_data_direct(unsigned long arg);
static int ioctl_ee_picture_extinfo_direct(unsigned long arg);

/**
 *	@brief		SICD data ioctl - deliver ext buf
 *	@param		file		file struct
 *	@param		arg			ioctl arguments
 *	@retval		0			success
 */
static int ioctl_deliver_buf_info(struct file *file, unsigned long arg)
{
	int err;
	struct func_data_ex *p_this_ex = sicd_get_this_ex();
	struct usbg_still_ext_buff_info st;

	SICD_FNC();

	err = copy_from_user(&st, (struct usbg_still_ext_buff_info *)arg, sizeof(struct usbg_still_ext_buff_info));
	if(err != 0) {
		SICD_ERR("copy_from_user failed./ err=[%d]\n",err);
		return -ENOBUFS;
	}

	p_this_ex->UnitSize = st.size;
	g_SicdDatBufSize = ( p_this_ex->UnitSize / USBG_STILL_BUFFER_NUM);


	p_this_ex->buffer = (void *)phys_to_virt(st.phyaddr);

	if(!p_this_ex->buffer) {
		SICD_ERR("sicd_data_buf no mem...\n");
		return -ENOMEM;
	}
	p_this_ex->buffer_dma = st.phyaddr;

	SICD_INF("sicd_data_buf allocated./ virtAddr=[%p], phyAddr=[%p], size=[%d]\n",
				p_this_ex->buffer, (void *)p_this_ex->buffer_dma, USBG_STILL_BUFFER_SIZE);

	return 0;
}

/**
 *	@brief		SICD data ioctl - clear current buf
 *	@param		file		file struct
 *	@param		arg			ioctl arguments
 *	@retval		0			success
 */
static int ioctl_clear_current_buf(struct file *file, unsigned long arg)
{
	struct func_data_ex* p_this_ex = sicd_get_this_ex();

	SICD_API();

	// EP dequeue
	sicd_ep_dequeue_bulk();

	p_this_ex->UnitSize = 0;
	p_this_ex->buffer      = (void*)NULL;
	p_this_ex->buffer_dma  = (dma_addr_t)NULL;

	return 0;
}

/**
 *	@brief		SICD data ioctl - SICD protocol
 *	@param		file		file struct
 *	@param		arg			ioctl arguments
 *	@retval		0			success
 *	@retval		-ENOBUFS	no buffer
 *	@retval		-EFAULT		segmentation fault
 *	@retval		-ENOTTY		invalid ioctl no.
 */
static int ioctl_still_data(struct file *file, unsigned int cmd, unsigned long arg)
{
	int err = 0;

//	SICD_FNC();

	// valid ioctl no?
	if( _IOC_NR(cmd) > USBG_IOC_STILL_NBROF ){
		SICD_ERR("invalid ioctl seq. number\n");
		return -ENOTTY;
	}

	// read/write access?
	if( _IOC_DIR(cmd) & _IOC_READ ){
		err = !access_ok((void *)arg, _IOC_SIZE(cmd) );
	}

	// access ok?
	if(err){
		SICD_ERR("invalid ioctl dir\n");
		return -EFAULT;
	}
	
	if( _IOC_DIR(cmd) & _IOC_WRITE ){
		err = !access_ok((void *)arg, _IOC_SIZE(cmd) );
	}
	
	// access ok?
	if(err){
		SICD_ERR("invalid ioctl dir\n");
		return -EFAULT;
	}
	
	// ioctl for what?
	switch (cmd) {
		case USBG_IOC_STILL_DELIVER_BUF_INFO:
			err = ioctl_deliver_buf_info(file, arg);
			break;
		case USBG_IOC_STILL_CLEAR_CURRENT_BUF:
			err = ioctl_clear_current_buf(file, arg);
			break;
		case USBG_IOC_STILL_EE_OPEN:
			err = ioctl_ee_open(arg);
			break;
		case USBG_IOC_STILL_EE_PICTURE:
			err = ioctl_ee_picture(file,arg);
			break;
		case USBG_IOC_STILL_EE_CLOSE:
			err = ioctl_ee_close();
			break;
		default:
			SICD_ERR("invalid ioctl seq. number\n");
			err = -ENOTTY;
			break;
	}
	
	return err;
}

/**
 *	@brief		SICD data driver open
 *	@param		inode		inode struct
 *	@param		file		file struct
 *	@retval		0			success
 *	@retval		-ENOMEM		short of memory
 *	@retval		-EBUSY		this driver is opened already
 */
UDIF_ERR sicd_data_open(UDIF_FILE *filp)
{
	struct file fd;
	struct file *file = &fd;
	struct func_data_ex *p_this_ex = sicd_get_this_ex();

	SICD_API();

	sicd_link_this_to_file(file);
    udif_file_data(filp) = file->private_data;

	if (p_this_ex->buffer) {
		SICD_ERR("already opened\n");
		return UDIF_ERR_BUSY;
	}
	else {
		p_this_ex->UnitSize = 0;
		g_SicdDatBufSize = 0;
	}

	return UDIF_ERR_OK;
}

/**
 *	@brief		SICD data driver release
 *	@param		inode		inode struct
 *	@param		file		file struct
 *	@retval		none
 */
UDIF_ERR sicd_data_release(UDIF_FILE *filp)
{
    struct func_data_ex* p_this_ex = sicd_get_this_ex();

	SICD_API();
	
	// EP dequeue
	sicd_ep_dequeue_bulk();

#if defined(USB_SICD_USE_RAMMAP)

	iounmap(p_this_ex->buffer);
	
#elif defined(USB_SICD_DMA_ALLOC)
	//dma_free_coherent(NULL, USBG_STILL_BUFFER_SIZE,p_this_ex->buffer, p_this_ex->buffer_dma);

	p_this_ex->UnitSize = 0;
	g_SicdDatBufSize = 0;

#else

	#error "USB_SICD_USE_RAMMAP? USB_SICD_DMA_ALLOC?"
	
#endif


	p_this_ex->buffer      = (void*)NULL;
	p_this_ex->buffer_dma  = (dma_addr_t)NULL;
	p_this_ex->buffer_user = (void*)NULL;
	
	return 0;
}

/**
 *	@brief		SICD data write(send) complete callback
 *	@param		ep			bulk-in ep
 *	@param		file		request struct
 *	@retval		none
 */
static void sicd_data_write_comp(struct usb_ep *ep, struct usb_request *req)
{
	struct usbg_io_comp_arg arg;
	
	SICD_API();
	
	SICD_INF("status=%d, length=%d, actual=%d\n", req->status, req->length, req->actual);
	
	if (PA2NC_CA == VA_TO_CACHETYPE((uintptr_t)req->buf)) {
		udif_cache_ctrl(req->buf,req->length, UDIF_CACHE_INVALIDATE);
	}
	
	sicd_request_stop( SICD_EP_IDX_BULK_IN );

	if(req->status) {
		// error!
		arg.addr = req->context;
		arg.size = req->status;
	}
	else {
		// success!
//		SICD_DMP(req->buf, req->actual);
		arg.addr = req->context;
		arg.size = req->actual;
	}
	
	sicd_call_user_cb(SICD_CBID_VENDOR, USBG_STILL_EVENT_WRITE_COMP,
						&arg, sizeof(struct usbg_io_comp_arg));
	
	sicd_recheck_req_comp_waiting();
}

/**
 *	@brief		SICD data write(send)
 *	@param		file		file struct
 *	@param		data		data buffer
 *	@param		size		data size
 *	@param		off			offset
 *	@retval		0			success
 *	@retval		-EINVAL		invalid parameter
 *	@retval		-ENOMEM		short of memory
 *	@retval		-EIO		I/O error
 */
UDIF_SSIZE sicd_data_write(UDIF_FILE *filp, UDIF_CDEV_WRITE *param)
{
	const char *data = param->uaddr;
	size_t size = param->count;
	int err;
	struct usb_request *req;
	struct usb_ep *ep;
	
	SICD_API();
	
	if(!sicd_is_valid_buffer((u8*)data, size)) {
		SICD_ERR("invalid buffer\n");
		return UDIF_ERR_PAR;
	}
	
	SICD_STATUS_LOCK_ON();
	
	if(sicd_get_status() != SICD_STATUS_OK) {
        SICD_STATUS_LOCK_OFF();
		SICD_ERR("status!=OK\n");
		return UDIF_ERR_IO;
	}
	SICD_STATUS_LOCK_OFF();
	
	SICD_EP_LOCK_ON();

	if( sicd_is_stop_usb_reqest() ) {
		SICD_EP_LOCK_OFF();
		SICD_ERR( "%s(): request not allowed", __FUNCTION__ );
		return UDIF_ERR_BUSY;
	}

	ep = sicd_get_ep(SICD_EP_IDX_BULK_IN);
	if(!ep) {
		SICD_EP_LOCK_OFF();
		SICD_ERR("ep==NULL\n");
		return UDIF_ERR_IO;
	}

	if( sicd_is_requesting(SICD_EP_IDX_BULK_IN) ) {
		SICD_EP_LOCK_OFF();
		SICD_ERR("ep==NULL\n");
		return UDIF_ERR_BUSY;
	}
	
	// alloc request
	req = sicd_get_request(ep);
	if(!req) {
		SICD_EP_LOCK_OFF();
		SICD_ERR("can not alloc request\n");
		return UDIF_ERR_NOMEM;
	}
	
	// fill request
	req->buf			= sicd_user_to_kvirt((void*)data);
	req->length			= size;
	req->complete		= sicd_data_write_comp;
	req->no_interrupt	= 0;
	req->short_not_ok	= 0;
	req->context		= (void*)data;
#ifdef SICD_TX_DMA
	req->dma			= sicd_user_to_dma((void*)data);
#else
	req->dma			= (dma_addr_t)NULL;
#endif
	
	if((req->length == 0) ||
	   (req->length != g_SicdDatBufSize &&
		req->length % ep->maxpacket == 0)) {
		req->zero = 1;	// append NULL packet
	}
	else {
		req->zero = 0;
	}

	if(!req->buf) {
		SICD_EP_LOCK_OFF();
		SICD_ERR("buffer area err\n");
		return UDIF_ERR_FAULT;
	}
	
	SICD_DMP((void*)req->buf, size);
	
	sicd_request_start( SICD_EP_IDX_BULK_IN );
	
	if (PA2NC_CA == VA_TO_CACHETYPE((uintptr_t)req->buf)) {
		udif_cache_ctrl(req->buf,req->length, UDIF_CACHE_FLUSH);
	}
                                
	// queue request
	err = usb_ep_queue(ep, req, GFP_ATOMIC);

	SICD_EP_LOCK_OFF();

	if(err) {
		// usb_ep_queue error
		SICD_ERR("usb_ep_queue err_code=%d\n", err);
		sicd_request_stop( SICD_EP_IDX_BULK_IN );
		sicd_recheck_req_comp_waiting();
		return err;
	}
	
	return req->length;
}

/**
 *	@brief		SICD data read(receive) complete callback
 *	@param		ep			bulk-out ep
 *	@param		file		request struct
 *	@retval		none
 */
static void sicd_data_read_comp(struct usb_ep *ep, struct usb_request *req)
{
	struct usbg_io_comp_arg arg;
	struct func_data_ex *p_this_still = sicd_get_this_ex();
	
	SICD_API();
	
	SICD_INF("status=%d, length=%d, actual=%d\n", req->status, req->length, req->actual);
    
	sicd_request_stop( SICD_EP_IDX_BULK_OUT );

	if(req->status) {
		// error!
		arg.addr = req->context;
		arg.size = req->status;
	}
	else {
		// success!
		SICD_DMP(req->buf, req->actual);
		arg.addr = req->context;
		arg.size = req->actual;
	}

    SICD_RST_REQ_LOCK_ON();
    
	if( p_this_still->reset_request_flag == TRUE ){
		p_this_still->reset_request_flag = FALSE;
		SICD_RST_REQ_LOCK_OFF();
		sicd_call_user_cb(SICD_CBID_VENDOR, USBG_STILL_EVENT_READ_COMP_WITH_RESET,
			&arg, sizeof(struct usbg_io_comp_arg));
	}else{
        SICD_RST_REQ_LOCK_OFF();
    	sicd_call_user_cb(SICD_CBID_VENDOR, USBG_STILL_EVENT_READ_COMP,
						&arg, sizeof(struct usbg_io_comp_arg));
	}
	
	sicd_recheck_req_comp_waiting();
}

/**
 *	@brief		SICD data read(receive)
 *	@param		file		file struct
 *	@param		data		data buffer
 *	@param		size		data size
 *	@param		off			offset
 *	@retval		0			success
 *	@retval		-EINVAL		invalid parameter
 *	@retval		-ENOMEM		short of memory
 *	@retval		-EIO		I/O error
 */
UDIF_SSIZE sicd_data_read(UDIF_FILE *filp, UDIF_CDEV_READ *param)
{
	char *data = param->uaddr;
	size_t size = param->count;
	int err;
	struct usb_request *req;
	struct usb_ep *ep;
	
	SICD_API();

	if(!sicd_is_valid_buffer((u8*)data, size)) {
		SICD_ERR("invalid buffer\n");
		return UDIF_ERR_PAR;
	}
	
    SICD_STATUS_LOCK_ON();
    
	if(sicd_get_status() != SICD_STATUS_OK) {
		SICD_STATUS_LOCK_OFF();
		SICD_ERR("status!=OK\n");
		return UDIF_ERR_IO;
	}
	
	SICD_STATUS_LOCK_OFF();
	
	SICD_EP_LOCK_ON();

	if( sicd_is_stop_usb_reqest() ) {
		SICD_EP_LOCK_OFF();
		SICD_ERR( "%s(): request not allowed", __FUNCTION__ );
		return UDIF_ERR_BUSY;
	}

	ep = sicd_get_ep(SICD_EP_IDX_BULK_OUT);
	if(!ep) {
		SICD_EP_LOCK_OFF();
		SICD_ERR("ep==NULL\n");
		return UDIF_ERR_IO;
	}
	if (!usb_gadgetcore_is_ep_alive(ep)) {
		SICD_EP_LOCK_OFF();
		SICD_ERR("ep is not alive\n");
		return UDIF_ERR_IO;
	}

	if( sicd_is_requesting(SICD_EP_IDX_BULK_OUT) ) {
		SICD_EP_LOCK_OFF();
		SICD_ERR("ep==NULL\n");
		return UDIF_ERR_BUSY;
	}
	
	// alloc request
	req = sicd_get_request(ep);
	if(!req) {
		SICD_EP_LOCK_OFF();
		SICD_ERR("can not alloc request\n");
		return UDIF_ERR_NOMEM;
	}
	

	// fill request
	req->buf			= sicd_user_to_kvirt(data);
	req->length			= usb_gadgetcore_ep_align_maybe(ep, size);
	SICD_INF("%s: size=%ld, aligned=%d\n", __FUNCTION__, size, req->length);
	req->zero			= 0;	// we NEVER ignore NULL packet
	req->complete		= sicd_data_read_comp;
	req->no_interrupt	= 0;
	req->short_not_ok	= 0;
	req->context		= (void*)data;
#ifdef SICD_RX_DMA
	req->dma			= sicd_user_to_dma(data);
#else
	req->dma			= (dma_addr_t)NULL;
#endif
	
	if(!req->buf) {
		SICD_EP_LOCK_OFF();
		SICD_ERR("buffer area err\n");
		return UDIF_ERR_FAULT;
	}

	sicd_request_start( SICD_EP_IDX_BULK_OUT );

	if (PA2NC_CA == VA_TO_CACHETYPE((uintptr_t)req->buf)) {
		udif_cache_ctrl(req->buf,req->length, UDIF_CACHE_INVALIDATE);
	}

	// queue request
	err = usb_ep_queue(ep, req, GFP_ATOMIC);

	SICD_EP_LOCK_OFF();

	if(err) {
		// usb_ep_queue error
		SICD_ERR("usb_ep_queue err_code=%d\n", err);
		sicd_request_stop( SICD_EP_IDX_BULK_OUT );
		sicd_recheck_req_comp_waiting();
		return err;
	}
	
	return (UDIF_SSIZE)req->length;
}

/**
 *	@brief		SICD data buffer mmap
 *	@param		file		file struct
 *	@param		vm			vm struct for userspace
 *	@retval		0			success
 *	@retval		!0			fail
 */
UDIF_ERR sicd_data_mmap(UDIF_FILE *filp, UDIF_VMAREA *vma)
{
	struct vm_area_struct *vm = (struct vm_area_struct*)vma;
	int err;
	struct func_data	*p_this;
	struct func_data_ex *p_this_still;
	ptrdiff_t size;
	uintptr_t phys;
	void *vadr;
	
	SICD_API();

	p_this		 = sicd_get_this();
	p_this_still = sicd_get_this_ex();

	size = vm->vm_end - vm->vm_start;
	if(size > p_this_still->UnitSize) {
		SICD_ERR("mmap size > UnitSize\n");
		return UDIF_ERR_PAR;
	}
	
	vadr = p_this_still->buffer;
	phys = p_this_still->buffer_dma;

	//check Phy address whether or not cache area
	if(PHYS_TO_CACHETYPE((unsigned long)phys) == PA2NC_UC){
		// use noncache 
		vm->vm_page_prot = udif_pgprot_noncached(vm->vm_page_prot);
	}
	vm->vm_flags |= VM_IO;
	vm->vm_flags |= VM_RESERVED;
	
	// remap buffer area for user-land
	err = udif_remap_pfn_range(vm, vm->vm_start, phys >> PAGE_SHIFT, size, vm->vm_page_prot);
	
	if(err < 0) {
		SICD_ERR("mmap fail %d\n", err);
		return err;
	}
	
	// remember user address
	p_this_still->buffer_user = (void*)vm->vm_start;

	SICD_INF("sicd_data_mmap done\n");
	SICD_INF("kern-virt:%p, user-virt:%p, dma:%p\n",
				(void*)p_this_still->buffer,
				(void*)p_this_still->buffer_user,
				(void*)p_this_still->buffer_dma);
	
	return err;
}


/**
 *	@brief		SICD data ioctl
 *	@param		inode		inode struct
 *	@param		file		file struct
 *	@param		cmd			ioctl no.
 *	@param		arg			ioctl arguments
 *	@retval		0			success
 *	@retval		-ENOTTY		wrong ioctl no.
 */
UDIF_ERR sicd_data_ioctl(UDIF_FILE *filp, UDIF_IOCTL *param)
{
	struct file fd;
	struct file *file = &fd;
	unsigned int cmd = param->cmd;
	unsigned long arg = param->arg;
	int err;

//	SICD_API();
	
    file->private_data = udif_file_data(filp);
	if( _IOC_TYPE(cmd) == USB_IOC_SICD_DATA ) {
		// still image ioctl
		err = ioctl_still_data(file, cmd, arg );
		udif_file_data(filp) = file->private_data;
	}
	else {
		// invalid ioctl
		SICD_ERR("invalid ioctl magic number\n");
		err = UDIF_ERR_NOTTY;
	}
	
	return (UDIF_ERR)err;
}


/**
 *	@brief		Get the buffer to send PTP header
 *	@param		none
 *	@retval		0			success
 *	@retval		-1			error
 */
static int ioctl_ee_open(unsigned long arg)
{
	int err = 0;
	struct usbg_still_ext_buff_info st;

	SICD_FNC();

	/* O[oϐ */
	g_SicdEeHeadPhyAddr  = 0;
	g_SicdEeHeadSize     = 0;
	g_SicdEeHeadVirtAddr  = NULL;
	g_SicdEeImgPhyAddr   = 0;
	g_SicdEeImgSize      = 0;
	g_SicdEeImgJpgNum    = 0;
	g_SicdEeExtPhyAddr   = 0;
	g_SicdEeExtVirtAddr  = NULL;
	g_SicdEeExtSizeMax   = 0;
	g_SicdEeTotalSize    = 0;
	g_SicdEeSentSize     = 0;

	memset(&g_SicdEePtpHeadData,0x00,sizeof(g_SicdEePtpHeadData));

	err = copy_from_user(&st, (struct usbg_still_ext_buff_info *)arg, sizeof(struct usbg_still_ext_buff_info));
	if (err != 0) {
		SICD_ERR("copy_from_user %d\n",err);
		return -ENOBUFS;
	}
	
	g_SicdEeHeadPhyAddr = st.phyaddr;
	g_SicdEeHeadSize = st.size;
	g_SicdEeHeadVirtAddr = (void *)PHYS_TO_VA( (uint64_t)g_SicdEeHeadPhyAddr );

	SICD_INF("sicd_data_buf allocated./ virtAddr=[%p], phyAddr=[%p], size=[%d]\n",
		g_SicdEeHeadVirtAddr, (void *)g_SicdEeHeadPhyAddr, USBG_STILL_BUFFER_SIZE);

	err = ioctl_ee_alloc_buf_extinfo();

	if (err != SICD_EE_RET_OK) {
		SICD_ERR("ee alloc extended infomation buf error!\n");
		return -ENOBUFS;
	}
	
	
	return err;
}

/**
 *	@brief		Send the EE image
 *	@param		file		file struct
 *	@param		arg			ioctl arguments
 *	@retval		0			success
 *	@retval		-1			error
 */
static int ioctl_ee_picture(struct file *file, unsigned long arg)
{
	struct usbg_still_ee_info *eeinfo = (struct usbg_still_ee_info*)arg;
	int                        ret = SICD_EE_RET_OK;

	SICD_INF("ioctl_ee_picture, ee_phase =0x%x\n", eeinfo->ee_phase);
	SICD_INF("  ee_imgsize =0x%x, ee_addr    =%p,   ee_size     =0x%x, ext_addr =%p,     ext_size  =0x%x\n", eeinfo->ee_imgsize, eeinfo->ee_addr, eeinfo->ee_size, eeinfo->ext_addr, eeinfo->ext_size);
	SICD_INF("  ee_jpgnum =0x%x,  total_size =0x%x, packet_size =0x%x, is_last  =0x%x\n", eeinfo->ee_jpgnum, eeinfo->total_size, eeinfo->packet_size, eeinfo->is_last);

	if (eeinfo == NULL) {
		SICD_ERR("sicd ee picture eeinfo NULL\n");
		ret = SICD_EE_RET_ERR;
	} else {
		if (eeinfo->ee_phase == PTPT_EE_STATE_PREPARE) {
			/* ] */
			ret = ioctl_ee_picture_prepare(arg);
			if (ret != SICD_EE_RET_OK) {
				SICD_ERR("sicd ee picture ptp, send header error\n");
				ret = SICD_EE_RET_ERR;
			}

		} else if (eeinfo->ee_phase == PTPT_EE_STATE_HEAD) {

			/* PTPwb_M */
			ret = ioctl_ee_picture_header(arg);
			if (ret != SICD_EE_RET_OK) {
				SICD_ERR("sicd ee picture ptp, send header error\n");
				ret = SICD_EE_RET_ERR;
			}

		} else if (eeinfo->ee_phase == PTPT_EE_STATE_EEDATA) {
			/* f[^M */
			ret = ioctl_ee_picture_data(arg);
			if (ret != SICD_EE_RET_OK) {
				SICD_ERR("sicd ee picture, send image data error\n");
				ret = SICD_EE_RET_ERR;
			}
		} else if (eeinfo->ee_phase == PTPT_EE_STATE_EEDATA_DIRECT) {
			/* f[^M */
			ret = ioctl_ee_picture_data_direct(arg);
			if (ret != SICD_EE_RET_OK) {
				SICD_ERR("sicd ee picture, send image data error\n");
				ret = SICD_EE_RET_ERR;
			}
		} else if (eeinfo->ee_phase == PTPT_EE_STATE_EXTINFO) {
			/* g񑗐M */
			ret = ioctl_ee_picture_extinfo(arg);
			if (ret != SICD_EE_RET_OK) {
				SICD_ERR("sicd ee picture, send ectended infomation error\n");
				ret = SICD_EE_RET_ERR;
			}
		} else if (eeinfo->ee_phase == PTPT_EE_STATE_EXTINFO_DIRECT) {
			/* g񑗐M */
			ret = ioctl_ee_picture_extinfo_direct(arg);
			if (ret != SICD_EE_RET_OK) {
				SICD_ERR("sicd ee picture, send ectended infomation error\n");
				ret = SICD_EE_RET_ERR;
			}
		} else if (eeinfo->ee_phase == PTPT_EE_STATE_RES) {
			/* X|XM */
			ret = ioctl_ee_picture_response(arg);
			if (ret != SICD_EE_RET_OK) {
				SICD_ERR("sicd ee picture, send response error\n");
				ret = SICD_EE_RET_ERR;
			}
		} else {
			SICD_ERR("sicd ee picture parameter error\n");
			ret = SICD_EE_RET_ERR;
		}
	}

	return (ret);
}

/**
 *	@brief		Free the buffer to send PTP header
 *	@param		none
 *	@retval		0			success
 *	@retval		-1			error
 */
static int ioctl_ee_close(void)
{
	int ret = SICD_EE_RET_OK;
	
	SICD_FNC();

	/* gpobt@̉ */
	ret = ioctl_ee_free_buf_extinfo();
	if (ret != SICD_EE_RET_OK) {
		SICD_ERR("ee ptp free extended info error!\n");
	}
	// EP dequeue
	sicd_ep_dequeue_bulk();

	return ret;
}

/**
 *	@brief		the ee picture data transmission(There is a need to review NULL packet addition to the general functions)
 *	@param		data_buf	send data buffer address(physical)
 *	@param		data_size	data size
 *	@retval		0			success
 *	@retval		-1			error
 */
static int ioctl_ee_picture_write(void* data_buf , unsigned int data_size , unsigned int check_nullpacket)
{
	int err;
	struct usb_request *req;
	struct usb_ep *ep;
	int ret = SICD_EE_RET_OK;

	for (;;) {
		SICD_EP_LOCK_ON();

		if( sicd_is_stop_usb_reqest() ) {
			SICD_EP_LOCK_OFF();
			SICD_ERR( "%s(): request not allowed", __FUNCTION__ );
			return UDIF_ERR_BUSY;
		}

		/* Gh|Cg\[Xm */
		ep = sicd_get_ep(SICD_EP_IDX_BULK_IN);
		if(!ep) {
			SICD_EP_LOCK_OFF();
			SICD_ERR("ee ep==NULL\n");
			ret = SICD_EE_RET_ERR;
			break;
		}

		if( sicd_is_requesting(SICD_EP_IDX_BULK_IN) ) {
			SICD_EP_LOCK_OFF();
			SICD_ERR("ee ep==NULL\n");
			ret = SICD_EE_RET_ERR;
			break;
		}

		/* Gh|Cg\[Xv */
		req = sicd_get_request(ep);
		if(!req) {
			SICD_EP_LOCK_OFF();
			SICD_ERR("ee can not alloc request\n");
			ret = SICD_EE_RET_ERR;
			break;
		}

		/* 쐬 */
		req->buf			= (void *)PHYS_TO_VA((unsigned long)data_buf);
		req->length			= data_size;
		req->complete		= sicd_data_write_comp;
		req->no_interrupt	= 0;
		req->short_not_ok	= 0;
		req->context		= req->buf;
	#ifdef SICD_TX_DMA
		req->dma			= (dma_addr_t)data_buf;
	#else
		req->dma			= (dma_addr_t)NULL;
	#endif

#if 1
		/* NULL Packet */
		/* ̑M̂Ƃ̂ݍsAobt@1ʂő肫̂MaxPacketSizȅ]`FbN̂ */
		req->zero = 0;
		if (check_nullpacket == 1) {
			if (req->length % ep->maxpacket == 0) {
				req->zero = 1;	// append NULL packet
			}
		}
#else
		/* NULL Packet */
		if((req->length == 0) ||
		   (req->length != g_SicdEeHeadSize &&
			req->length % ep->maxpacket == 0)) {
			req->zero = 1;	// append NULL packet
		}
		else {
			req->zero = 0;
		}
#endif

		/* Mobt@NULL Pointer`FbN */
		if(!req->buf) {
			SICD_EP_LOCK_OFF();
			SICD_ERR("ee buffer area err\n");
			ret = SICD_EE_RET_ERR;
			break;
		}

		sicd_request_start( SICD_EP_IDX_BULK_IN );

		if(PHYS_TO_CACHETYPE((unsigned long)data_buf) == PA2NC_CA){
				udif_cache_ctrl(req->buf,req->length, UDIF_CACHE_FLUSH);
		}
		/* MJn */
		err = usb_ep_queue(ep, req, GFP_ATOMIC);

		SICD_EP_LOCK_OFF();

		if(err) {
			// usb_ep_queue error
			SICD_ERR("ee usb_ep_queue err_code=%d\n", err);
			sicd_request_stop( SICD_EP_IDX_BULK_IN );
			sicd_recheck_req_comp_waiting();
			ret = SICD_EE_RET_ERR;
			break;
		}

		/* I */
		break;
	}

	return(ret);
}

/**
 *	@brief		SICD data ioctl - alloc ee picture ptp extended infomation buffer
 *	@param		none
 *	@retval		0			success
 *	@retval		-1			error
 */
static int ioctl_ee_alloc_buf_extinfo(void)
{
    int ret = SICD_EE_RET_OK;

    if (NULL != g_SicdEeExtVirtAddr) {
        SICD_ERR("EE extended buffer is already Acquired !\n");
        return (ret);
    }

    /*kmallocŎ擾*/
    g_SicdEeExtVirtAddr = kmalloc(SICD_EE_EXT_SIZE_MAX, GFP_KERNEL | GFP_DMA);
    if (NULL != g_SicdEeExtVirtAddr) {
        g_SicdEeExtPhyAddr  = VA_TO_PHYS((unsigned long)g_SicdEeExtVirtAddr);
        g_SicdEeExtSizeMax  = SICD_EE_EXT_SIZE_MAX;
        SICD_INF("sicd_data_buf alloced. virt_addr=0x%p, phys_addr=0x%p, size=%d\n",
                    g_SicdEeHeadVirtAddr, (void *)g_SicdEeHeadPhyAddr, SICD_EE_EXT_SIZE_MAX);
    } else {
        SICD_ERR("ioctl_ee_alloc_buf_extinfo: kmalloc err\n");
        ret = SICD_EE_RET_ERR;
    }

    return (ret);
}

/**
 *	@brief		SICD data ioctl - free ptp extended infomation buf
 *	@param		none
 *	@retval		0			success
 *	@retval		-1			error
 */
static int ioctl_ee_free_buf_extinfo(void)
{
	int ret = SICD_EE_RET_OK;

	SICD_API();

	// EP dequeue
	sicd_ep_dequeue_bulk();

	/* obt@mۍς݂free */
    if (NULL != g_SicdEeExtVirtAddr) {
        kfree(g_SicdEeExtVirtAddr);
        g_SicdEeExtVirtAddr = NULL;
        g_SicdEeExtPhyAddr  = 0;
        g_SicdEeExtSizeMax  = 0;
    }

	return (ret);
}

/**
 *	@brief		SICD data ioctl - send ptp header
 *	@param		arg			ioctl arguments
 *	@retval		0			success
 *	@retval		-1			error
 */
static int ioctl_ee_picture_prepare(unsigned long arg)
{
	struct usbg_still_ee_info *eeinfo = (struct usbg_still_ee_info*)arg;
	int                        ret = SICD_EE_RET_OK;
	struct usb_ep *ep;
	uint32_t maxpacketsize = 0;

	SICD_EP_LOCK_ON();
	ep = sicd_get_ep(SICD_EP_IDX_BULK_IN);
	if(!ep) {
		SICD_ERR("ee ep==NULL\n");
		ret = SICD_EE_RET_ERR;
	} else {
		maxpacketsize = ep->maxpacket;
	}
	SICD_EP_LOCK_OFF();

	eeinfo->packet_size = maxpacketsize;
	return (ret);
}

/**
 *	@brief		SICD data ioctl - send ptp header
 *	@param		arg			ioctl arguments
 *	@retval		0			success
 *	@retval		-1			error
 */
static int ioctl_ee_picture_header(unsigned long arg)
{
	struct usbg_still_ee_info *eeinfo = (struct usbg_still_ee_info*)arg;
	int                        ret = SICD_EE_RET_OK;
	int                        sendsize;
	int                        check_null = 0;
	
	for (;;) {
		/* PTPwb_M */

		/* ۑ */
		g_SicdEeImgSize     = eeinfo->ee_imgsize;
		g_SicdEeImgJpgNum   = eeinfo->ee_jpgnum;
		g_SicdEeExtSize     = eeinfo->ext_size;
		g_SicdEeTotalSize   = eeinfo->total_size;
		g_SicdEeSentSize    = 0;
		SICD_INF("g_SicdEeImgSize=[%d], g_SicdEeImgJpgNum=[%d]\n", g_SicdEeImgSize, g_SicdEeImgJpgNum);

		if (g_SicdEeImgJpgNum == 0) {
		/* 擾łȂꍇ */
			/* PTPwb_쐬 */
			/* Length~TransactionID܂ł12oCgŒ蒷ŃRs[ */
			memcpy(&g_SicdEePtpHeadData , (void*)eeinfo->ee_addr , SICD_EE_PTPEEHEADERLEN);
			/* Mf[^XV HeaderBuffer(512byte)+TCY */
			g_SicdEePtpHeadData.length = SICD_EE_PTPEEHEADERLEN;
			g_SicdEePtpHeadData.type   = SICD_EE_DATATYPE_DATA;
			g_SicdEePtpHeadData.param1 = 0;				/* x_p[^͂Ƃ0 */
			g_SicdEePtpHeadData.param2 = 0;
			g_SicdEePtpHeadData.param3 = 0;
			g_SicdEePtpHeadData.param4 = 0;
			/* ]pobt@ɃRs[ */
			memcpy(g_SicdEeHeadVirtAddr , &g_SicdEePtpHeadData , SICD_EE_PTPEEHEADERLEN);
			/* ]TCYݒ */
			sendsize = SICD_EE_PTPEEHEADERLEN;
			check_null = 1;
		} else {
		/* 擾łꍇ */

			struct usb_ep *ep;
			uint32_t maxpacketsize = 0;
			uint64_t padding_size = 0, temp = 0;

			SICD_EP_LOCK_ON();
			ep = sicd_get_ep(SICD_EP_IDX_BULK_IN);
			if(!ep) {
				SICD_EP_LOCK_OFF();
				SICD_ERR("ee ep==NULL\n");
				ret = SICD_EE_RET_ERR;
				break;
			}
			maxpacketsize = ep->maxpacket;
			SICD_EP_LOCK_OFF();

			// ShortpPbgɂȂȂ悤ɑMTCY킹.
			temp = g_SicdEeExtSize % maxpacketsize;  // [Zo.
			if (0 != temp ){
				padding_size = maxpacketsize - temp;
				g_SicdEeExtSize   += padding_size;
				g_SicdEeTotalSize += padding_size;
			}
			/* PTPwb_쐬 */
			/* Length~TransactionID܂ł12oCgŒ蒷ŃRs[ */
			memcpy(&g_SicdEePtpHeadData , (void*)eeinfo->ee_addr , SICD_EE_PTPEEHEADERLEN);
			/* Mf[^XV HeaderBuffer(512byte)+TCY */
			g_SicdEePtpHeadData.length = g_SicdEeHeadSize + g_SicdEeTotalSize;
			g_SicdEePtpHeadData.type   = SICD_EE_DATATYPE_DATA;
			g_SicdEePtpHeadData.param3 = g_SicdEeHeadSize - SICD_EE_PTPHEADERLEN;
			g_SicdEePtpHeadData.param4 = eeinfo->ext_size;                              // f̊gf[^TCYi[.
			g_SicdEePtpHeadData.param1 = g_SicdEePtpHeadData.param3 + g_SicdEeExtSize;  // pfBO␳gf[^TCYZ.
			g_SicdEePtpHeadData.param2 = g_SicdEeImgSize;
			/* ]pobt@ɃRs[ */
			memcpy(g_SicdEeHeadVirtAddr , &g_SicdEePtpHeadData , SICD_EE_PTPEEHEADERLEN);
			/* ]TCYݒ */
			sendsize = g_SicdEeHeadSize;
		}

		/* PTPwb_f[^MÂƑăf[^𑗂̂NULLpPbg`FbNȂ */
		ret = ioctl_ee_picture_write((void*)g_SicdEeHeadPhyAddr , sendsize , check_null);
		if (ret != SICD_EE_RET_OK) {
			SICD_ERR("ee ptp header send error!\n");
			ret = SICD_EE_RET_ERR;
			break;
		}

		/* I */
		break;
	}

	return (ret);

}

/**
 *	@brief		SICD data ioctl - send ee picture
 *	@param		arg			ioctl arguments
 *	@retval		0			success
 *	@retval		-1			error
 */
static int ioctl_ee_picture_data(unsigned long arg)
{
	int                        ret = SICD_EE_RET_OK;
	struct usbg_still_ee_info *eeinfo = (struct usbg_still_ee_info*)arg;
    unsigned int check_nullpacket = 0;

	g_SicdEeImgPhyAddr	 = (uint64_t)eeinfo->ee_addr;
	SICD_INF("g_SicdEeImgPhyAddr=[%p]\n", (void*)g_SicdEeImgPhyAddr);

	if (g_SicdEeImgJpgNum == 0) {
	/* 擾łȂꍇ̓XLbv */
		ret = SICD_EE_RET_ERR;

	} else {
		/* MTCYXVANULLpPbg`FbNL𔻒 */
		g_SicdEeSentSize += g_SicdEeImgSize;
		if (g_SicdEeSentSize >= g_SicdEeTotalSize ) {
		    check_nullpacket = 1;
		}
		/* f[^MANULLpPbg`FbN */
		ret = ioctl_ee_picture_write((void*)g_SicdEeImgPhyAddr , g_SicdEeImgSize , check_nullpacket);
	}

	return (ret);
}

/**
 *	@brief		SICD data ioctl - send ee picture
 *	@param		arg			ioctl arguments
 *	@retval		0			success
 *	@retval		-1			error
 */
static int ioctl_ee_picture_data_direct(unsigned long arg)
{
	int                        ret = SICD_EE_RET_OK;
	struct usbg_still_ee_info *eeinfo = (struct usbg_still_ee_info*)arg;
    unsigned int check_nullpacket = 0;

	SICD_INF("ee_addr=[%p], ee_imgsize=[%d]\n", (void*)eeinfo->ee_addr, eeinfo->ee_imgsize);

	/* MTCYXVANULLpPbg`FbNL𔻒 */
	if (0 != eeinfo->is_last) {
		check_nullpacket = 1;
	}
	/* f[^MANULLpPbg`FbN */
	ret = ioctl_ee_picture_write((void*)eeinfo->ee_addr, eeinfo->ee_imgsize , check_nullpacket);

	return (ret);
}

/**
 *  @brief      SICD data ioctl - send ee picture
 *  @param      arg         ioctl arguments
 *  @retval     0           success
 *  @retval     -1          error
 */
static int ioctl_ee_picture_extinfo(unsigned long arg)
{
	int err = 0;
    int ret = SICD_EE_RET_OK;
    struct usbg_still_ee_info *eeinfo = (struct usbg_still_ee_info*)arg;
    unsigned int check_nullpacket = 0;

    if (0 == eeinfo->ext_size || eeinfo->ext_size > g_SicdEeExtSizeMax || NULL == eeinfo->ext_addr) {
        /* f[^s̏ꍇXLbv*/
        SICD_ERR("eeinfo error!  addr=0x%p, size=0x%x, ExtSize=0x%x\n", eeinfo->ext_addr, eeinfo->ext_size, g_SicdEeExtSizeMax);
        ret = SICD_EE_RET_ERR;

    } else {
        /* gRs[ */
        if(NULL == g_SicdEeExtVirtAddr){
            SICD_ERR("EE extended buffer is NULL\n");
            return -EFAULT;
        }
        err = copy_from_user(g_SicdEeExtVirtAddr, eeinfo->ext_addr, eeinfo->ext_size);
        if(err != 0) {
            SICD_ERR("copy_from_user error %d\n",err);
            return -ENOBUFS;
        }
        /* MTCYXVANULLpPbg`FbNL𔻒 */
        g_SicdEeSentSize += g_SicdEeExtSize;
        if (g_SicdEeSentSize >= g_SicdEeTotalSize ) {
            check_nullpacket = 1;
        }
        
        /* gf[^M */
        ret = ioctl_ee_picture_write((void*)g_SicdEeExtPhyAddr , g_SicdEeExtSize , check_nullpacket);   // pfBO␳TCY].
    }

    return (ret);
}

/**
 *  @brief      SICD data ioctl - send ee picture
 *  @param      arg         ioctl arguments
 *  @retval     0           success
 *  @retval     -1          error
 */
static int ioctl_ee_picture_extinfo_direct(unsigned long arg)
{
	int err = 0;
    int ret = SICD_EE_RET_OK;
    struct usbg_still_ee_info *eeinfo = (struct usbg_still_ee_info*)arg;
    unsigned int check_nullpacket = 0;

    if (0 == eeinfo->ext_size || eeinfo->ext_size > SICD_EE_EXT_SIZE_MAX || NULL == eeinfo->ext_addr) {
        /* f[^s̏ꍇXLbv*/
        SICD_ERR("eeinfo error!  addr=0x%p, size=0x%x, ExtSize=0x%x\n", eeinfo->ext_addr, eeinfo->ext_size, SICD_EE_EXT_SIZE_MAX);
        ret = SICD_EE_RET_ERR;

    } else {
        /* gRs[ */
        if(NULL == g_SicdEeExtVirtAddr){
            SICD_ERR("EE extended buffer is NULL\n");
            return -EFAULT;
        }
        err = copy_from_user(g_SicdEeExtVirtAddr, eeinfo->ext_addr, eeinfo->ext_size);
        if(err != 0) {
            SICD_ERR("copy_from_user error %d\n",err);
            return -ENOBUFS;
        }
        /* MTCYXVANULLpPbg`FbNL𔻒 */
        if (0 != eeinfo->is_last) {
            check_nullpacket = 1;
        }
        
        /* gf[^M */
        ret = ioctl_ee_picture_write((void*)g_SicdEeExtPhyAddr , eeinfo->ext_size, check_nullpacket);   // pfBO␳TCY].
    }

    return (ret);
}

/**
 *	@brief		SICD data ioctl - send response data
 *	@param		arg			ioctl arguments
 *	@retval		0			success
 *	@retval		-1			error
 */
static int ioctl_ee_picture_response(unsigned long arg)
{
	int ret = SICD_EE_RET_OK;


	/* L̏G[łM */
	/* X|Xf[^쐬 */
	g_SicdEePtpHeadData.length = SICD_EE_PTPHEADERLEN;
	g_SicdEePtpHeadData.type   = SICD_EE_DATATYPE_RESP;
	if (g_SicdEeImgJpgNum != 0) {
		g_SicdEePtpHeadData.opcode = SICD_EE_RESCODE_OK;
	} else {
	/* 擾łȂꍇACCESS DENIEDԂ */
		g_SicdEePtpHeadData.opcode = SICD_EE_RESCODE_ACCESSDENIED;
	}
	/* TransactionID͂̂܂܎g */

	/* wb_pɃRs[ */
	memcpy(g_SicdEeHeadVirtAddr , &g_SicdEePtpHeadData , SICD_EE_PTPHEADERLEN);

	/* X|Xf[^MANULLpPbg`FbNȂ */
	ret = ioctl_ee_picture_write((void*)g_SicdEeHeadPhyAddr , SICD_EE_PTPHEADERLEN , 0);
	if (ret != SICD_EE_RET_OK) {
		SICD_ERR("ee ptp response send error!\n");
	}

	return (ret);
}

/* end of file */
