/**
 *	@file	usbg_still_main.c
 *	@brief	USB SICD(Still Image Capture Device class) - main
 *	
 *		Copyright 2005,2006,2011 Sony Corporation
 * Copyright 2018, 2019 Sony Imaging Products and Solutions Incorporated.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/version.h>

#include <linux/usb/ch9.h>

#include <linux/device.h>

#include <linux/usb/gadget.h>

#include <linux/usb/gcore/usb_event.h>
#include <linux/usb/gcore/usb_gadgetcore.h>

#ifdef CONFIG_OSAL_UDIF
#include <linux/udif/cdev.h>
#include <mach/udif/devno.h>
#else
#include <asm/arch/sonymisc.h>
#endif

#include <linux/usb/specific_gadget/usbg_cmn.h>
#include <linux/usb/specific_gadget/usbg_still.h>
#include <linux/usb/specific_gadget/usbg_type.h>
#include <linux/usb/specific_gadget/usbg_still.h>
#include <linux/usb/specific_gadget/usb_kdbg.h>

#define FNAME "usbg_still_main.c"
#include "usbg_still_conf.h"
#include "usbg_still_pvt.h"








/*
 *	@brief	Licence
 */
MODULE_AUTHOR("Sony Corporation");								///< author
MODULE_DESCRIPTION(MY_DESC);									///< description
MODULE_VERSION(MY_VER);											///< version
MODULE_LICENSE("GPL");											///< license


/**
 *	@brief	File operations (/dev/sicd_core)
 */
static struct UDIF_CDEV_OPS sicd_core_fops = {
    .open  = sicd_core_open,
    .close = sicd_core_release,
    .ioctl = sicd_core_ioctl,
};

/**
 *	@brief	File operations (/dev/sicd_data)
 */
static struct UDIF_CDEV_OPS sicd_data_fops = {
    .open  = sicd_data_open,
    .close = sicd_data_release,
    .read  = sicd_data_read,
    .write = sicd_data_write,
    .mmap  = sicd_data_mmap,
    .ioctl = sicd_data_ioctl,
};

/**
 *	@brief	File operations (/dev/sicd_event)
 */
static struct UDIF_CDEV_OPS sicd_event_fops = {
    .open  = sicd_event_open,
    .close = sicd_event_release,
    .write = sicd_event_write,
};


/**
 *	@brief	Core driver information for sonymisc
 */
#ifdef CONFIG_OSAL_UDIF
static struct usbg_cdev sicd_core_dev = {
    .node = udif_device_node(UDIF_NODE_USB_SICD_CORE),
    .fops = &sicd_core_fops,
};
#else
static struct sonymisc_device sicd_core_dev = 
{
    .class  = DEV_CLASS_CHR,
    .minor  = USB_MINOR_SICD_CORE,
    .name   = MY_NAME,
    .fops   = &sicd_core_fops,
    .gd     = NULL,
};
#endif

/**
 *	@brief	Data driver information for sonymisc
 */
#ifdef CONFIG_OSAL_UDIF
static struct usbg_cdev sicd_data_dev = {
    .node = udif_device_node(UDIF_NODE_USB_SICD_DATA),
    .fops = &sicd_data_fops,
};
#else
static struct sonymisc_device sicd_data_dev = 
{
    .class  = DEV_CLASS_CHR,
    .minor  = USB_MINOR_SICD_DATA,
    .name   = MY_NAME,
    .fops   = &sicd_data_fops,
    .gd     = NULL,
};
#endif

/**
 *	@brief	Event driver information for sonymisc
 */
#ifdef CONFIG_OSAL_UDIF
static struct usbg_cdev sicd_event_dev = {
    .node = udif_device_node(UDIF_NODE_USB_SICD_EVENT),
    .fops = &sicd_event_fops,
};
#else
static struct sonymisc_device sicd_event_dev = 
{
    .class  = DEV_CLASS_CHR,
    .minor  = USB_MINOR_SICD_EVENT,
    .name   = MY_NAME,
    .fops   = &sicd_event_fops,
    .gd     = NULL,
};
#endif

/*
 *	@brief	Driver load
 */
static int __init start_module(void)
{
	int err;

    SICD_API();
	// register core driver
	err = usbg_cmn_reg(&sicd_core_dev);
	if(err){
		SICD_ERR( "register core driver %d\n", err );
		return err;
	}
	
	// register data driver
	err = usbg_cmn_reg(&sicd_data_dev);
	if(err){
		SICD_ERR( "register data driver %d\n", err );
		usbg_cmn_dereg(&sicd_core_dev);
		return err;
	}
	
	// register event driver
	err = usbg_cmn_reg(&sicd_event_dev);
	if(err){
		SICD_ERR( "register event driver %d\n", err );
		usbg_cmn_dereg(&sicd_core_dev);
		usbg_cmn_dereg(&sicd_data_dev);
		return err;
	}
	
	// create sicd instance
	err = sicd_create_this();

	SICD_LOCK_INI();

	sicd_get_this_ex()->req_comp_waiting = false;
	init_completion(&(sicd_get_this_ex()->req_wait_completion));
	sicd_get_this_ex()->activated = false;

	SICD_INF("driver loaded!!\n");

	return err;
}

/*
 *	@brief	Driver unload
 */
static void __exit stop_module(void)
{
    SICD_API();

	// create sicd instance
	sicd_delete_this();

	usbg_cmn_dereg(&sicd_core_dev);
	usbg_cmn_dereg(&sicd_data_dev);
	usbg_cmn_dereg(&sicd_event_dev);
	
    SICD_INF("driver unloaded!!\n");
}


// module_init/module_exit
module_init(start_module);
module_exit(stop_module);

