/*
 * usbg_bdio.c
 *
 * USB Mass Storage Gadget Function Driver
 *
 * Copyright 2005,2006,2011 Sony Corporation
 * Copyright 2018 Sony Imaging Products and Solutions Incorporated.
 *
 * << Block-direct I/O (implementation) >>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/device.h>
#include <linux/bio.h>
#include <linux/version.h>
#include <linux/blkdev.h>
#include "usbg_bdio.h"
#include "usbg_storage_debug.h"

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 3, 0)
#define f_dentry	f_path.dentry
#endif

static struct bio* usbg_bio = NULL;

/**
 *	@brief		BIO complete callback
 *	@param		bio		struct bio
 *	@param		num		data size transferd
 *	@retval		0		success (always)
 */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 3, 0)
static void bdio_end_io(struct bio * bio)
{
	GS_DET( GS_OUT__BDIO, "func Req" );
	GS_DBG( GS_OUT__BDIO, "bio=%p, err=%d\n", bio, bio->bi_status );

	complete((struct completion*)bio->bi_private);
}
#elif LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 29)
static void bdio_end_io(struct bio * bio, int err)
{
	GS_DET( GS_OUT__BDIO, "func Req" );
	GS_DBG( GS_OUT__BDIO, "bio=%p, err=%d\n", bio, err);
	
	complete((struct completion*)bio->bi_private);
}
#else
static int bdio_end_io(struct bio * bio, unsigned int num, int err)
{
	GS_DET( GS_OUT__BDIO, "func Req" );
	GS_DBG( GS_OUT__BDIO, "bio=%p, num=%d, err=%d\n", bio, num, err);

	if (bio->bi_size){
		return 1;
	}

	complete((struct completion*)bio->bi_private);

	return 0;
}
#endif

/**
 *	@brief		RE-init BIO structure
 *	@param		bio		struct bio
 *	@retval		none
 */
static void bdio_reuse(struct bio * bio)
{
	GS_DET( GS_OUT__BDIO, "func Req" );

	if(!bio) {
		return;
	}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 3, 0)
	bio_reset(bio);
#else
	bio->bi_next = NULL;
	bio->bi_flags |= 1 << BIO_UPTODATE;
	bio->bi_rw = 0;
	bio->bi_vcnt = 0;
	bio->bi_idx = 0;
	bio->bi_phys_segments = 0;
	bio->bi_size = 0;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 29)
	bio->bi_seg_front_size = 0;
	bio->bi_seg_back_size = 0;
#else
	bio->bi_hw_segments = 0;
	bio->bi_hw_front_size = 0;
	bio->bi_hw_back_size = 0;
#endif

	atomic_set(&bio->bi_cnt, 1);
#endif
}


/**
 *	@brief		Submit BIO request
 *	@param		rw		read? or write?
 *	@param		bdev	struct block device
 *	@param		buf		buffer address
 *	@param		size	data size
 *	@param		off		offset
 *	@retval		positive value or 0		success -> indicates transferrd size
 *	@retval		negative value			error   -> indicates error code
 */
static int bdio_submit(int rw, struct block_device *bdev, u8 *buf, int size, loff_t off)
{
	struct completion comp;
	struct page *cur_page;
	int i, err, cur_size, page_num;
	u8 *p = buf;
	int rest = size;

	GS_DET( GS_OUT__BDIO, "func Req" );
	GS_DBG( GS_OUT__BDIO, "rw=%d, bdev=%p, buf=%p, size=%d, off=%d\n", rw, bdev, buf, size, (int)off);

	if(!usbg_bio) {
		GS_ERR( GS_OUT__BDIO, "usbg_bio==NULL");
		return -EIO;
	}

	init_completion(&comp);
	page_num = (size-1) / PAGE_SIZE + 1;

	// re-init struct BIO
	bdio_reuse(usbg_bio);

	// fill struct bio
	bio_get(usbg_bio);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 3, 0)
	usbg_bio->bi_iter.bi_sector  = off >> 9;
#else
	usbg_bio->bi_sector  = off >> 9;
#endif
	usbg_bio->bi_bdev    = bdev;
	usbg_bio->bi_end_io  = bdio_end_io;
	usbg_bio->bi_private = &comp;

	// add page (I/O buffer)
	for(i = 0; i < page_num; i++) {
		cur_page = virt_to_page(p);
		cur_size = (rest < PAGE_SIZE) ? rest : PAGE_SIZE;

		err= bio_add_page(usbg_bio, cur_page, cur_size, 0);
		if(err < cur_size) {
			GS_ERR( GS_OUT__BDIO, "bio_add_page fail...");
			GS_ERR( GS_OUT__BDIO, "size=%d, page_num=%d, bi_vcnt=%d, bi_max_vecs=%d, err=%d",
					size, page_num, usbg_bio->bi_vcnt, usbg_bio->bi_max_vecs, err);
			return -EFAULT;
		}

		if(rw & WRITE) {
			set_page_dirty(cur_page);
		}

		p    += cur_size;
		rest -= cur_size;
	}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 3, 0)
	if (rw == REQ_SYNC)
		bio_set_op_attrs(usbg_bio, REQ_OP_WRITE, REQ_SYNC);
	else
		/* case rw == READ_SYNC */
		bio_set_op_attrs(usbg_bio, REQ_OP_READ, 0);

	// submit bio & wait completion
	submit_bio(usbg_bio);
	wait_for_completion(&comp);

	/*
	 * expire BIO_UPTODATE and use bi_error instead
	 */
	return usbg_bio->bi_status;
#else
	// submit bio & wait completion
	submit_bio(rw, usbg_bio);
	wait_for_completion(&comp);

	if (!test_bit(BIO_UPTODATE, &usbg_bio->bi_flags)) {
		return -EIO;
	}

	return 0;	// success
#endif
}

/**
 *	@brief		Init BIO structure
 *	@param		none
 *	@retval		0	success
 *	@retval		!0	fail
 */
int bdio_init(void)
{
	int page_num = (BDIO_MAX_SIZE-1) / PAGE_SIZE + 1;

	GS_DET( GS_OUT__BDIO, "func Req" );

	if(!usbg_bio) {
		usbg_bio = bio_alloc(GFP_ATOMIC, page_num);
		if (!usbg_bio) {
			GS_ERR( GS_OUT__BDIO, "bio_alloc fail.");
			GS_ERR( GS_OUT__BDIO, "MAX_SIZE=%d, page_num=%d", BDIO_MAX_SIZE, page_num);
			return -ENOMEM;
		}
	}


	return 0;
}

/**
 *	@brief		Fin BIO structure
 *	@param		none
 *	@retval		none
 */
void bdio_fin(void)
{
	GS_DET( GS_OUT__BDIO, "func Req" );

	if(usbg_bio) {
		bio_put(usbg_bio);
	}
}

/**
 *	@brief		Write data via BIO layer
 *	@param		fp		struct file
 *	@param		buf		buffer address
 *	@param		size	data size
 *	@param		off		offset
 *	@retval		positive value or 0		success -> indicates transferd size
 *	@retval		negative value			error   -> indicates error code
 */
int bdio_write(struct file *fp, u8 *buf, int size, loff_t off)
{
	int err;
	struct block_device *bdev;
	struct inode        *inode = NULL;

	GS_DET( GS_OUT__BDIO, "func Req" );
	GS_DBG( GS_OUT__BDIO, "fp=%p, buf=%p, size=%d, off=%d\n", fp, buf, size, (int)off);

	if(size < 0 || BDIO_MAX_SIZE < size || size % BDIO_BLOCK_SIZE ||
		(uintptr_t)buf % BDIO_BLOCK_SIZE) {
		GS_ERR( GS_OUT__BDIO, "buf or size is invalid, buf=%p, size=%d\n", buf, size);
		return -EINVAL;
	}

	inode = fp->f_mapping->host;
	bdev = I_BDEV(inode);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 3, 0)
	if (!bdev) {
		GS_ERR( GS_OUT__BDIO, "err bdev(NULL), fd=%p, d_inode=%p\n", fp, fp->f_dentry->d_inode);
		return -EAGAIN;
	}
#endif
	err  = bdio_submit(REQ_SYNC, bdev, buf, size, off);

	if(err) {
		GS_ERR( GS_OUT__BDIO, "bdio_submit err=%d\n", err);
		return err;
	}
	else {
		return size;
	}
}

/**
 *	@brief		Read data via BIO layer
 *	@param		fp		struct file
 *	@param		buf		buffer address
 *	@param		size	data size
 *	@param		off		offset
 *	@retval		positive value or 0		success -> indicates transferd size
 *	@retval		negative value			error   -> indicates error code
 */
int bdio_read(struct file *fp, u8 *buf, int size, loff_t off)
{
	int err;
	struct block_device *bdev;
	struct inode        *inode = NULL;

	GS_DET( GS_OUT__BDIO, "func Req" );
	GS_DBG( GS_OUT__BDIO, "fp=%p, buf=%p, size=%d, off=%d\n", fp, buf, size, (int)off);

	if(size < 0 || BDIO_MAX_SIZE < size || size % BDIO_BLOCK_SIZE ||
		(uintptr_t)buf % BDIO_BLOCK_SIZE) {
		GS_ERR( GS_OUT__BDIO, "buf or size is invalid, buf=%p, size=%d\n", buf, size);
		return -EINVAL;
	}

	inode = fp->f_mapping->host;
	bdev = I_BDEV(inode);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 3, 0)
	if (!bdev) {
		GS_ERR( GS_OUT__BDIO, "err bdev(NULL), fd=%p, d_inode=%p\n", fp, fp->f_dentry->d_inode);
		return -EAGAIN;
	}
#endif
	err  = bdio_submit(0, bdev, buf, size, off);

	if(err) {
		GS_ERR( GS_OUT__BDIO, "bdio_submit err=%d\n", err);
		return err;
	}
	else {
		return size;
	}
}

/**
 *	@brief		IOCTL via BIO layer
 *	@param		fp		struct file
 *	@param		cmd		IOCTL command
 *	@param		arg		IOCTL args
 *	@retval		0		success
 *	@retval		!0		error
 */
int bdio_ioctl(struct file *fp, unsigned cmd, unsigned long arg)
{
	int err;
	struct block_device *bdev;
	struct inode        *inode = NULL;

	GS_DET( GS_OUT__BDIO, "func Req" );
	GS_DBG( GS_OUT__BDIO, "fp=%p, cmd=%d, arg=%d\n", fp, cmd, (int)arg);

	inode = fp->f_mapping->host;
	bdev = I_BDEV(inode);
	err = blkdev_ioctl(bdev, 0, cmd, arg);
	if(err) {
		GS_ERR( GS_OUT__BDIO, "ioctl_by_bdev err=%d\n", err);
	}

	return err;
}


