/*
 * usbg_storage_cmd_std.c
 *
 * USB Mass Storage Gadget Function Driver
 *
 * Copyright 2005,2006,2008,2009,2010,2011 Sony Corporation
 * Copyright 2018 Sony Imaging Products and Solutions Incorporated.
 *
 * << MassStorage Command Processing (standard) >>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#define __USBG_STORAGE_CMD_C__
/*
┏━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┓
┃ インクルードファイル                                                 ┃
┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛*/
#include <linux/uaccess.h>
#include <asm/page.h>
#include <asm/errno.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/hdreg.h>
#include <linux/version.h>

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 36)
#include <generated/autoconf.h>
#else
#include <linux/autoconf.h>
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 21)
#include <linux/usb/ch9.h>
#else
#include <linux/usb_ch9.h>
#endif

#include <linux/device.h>
#include <linux/mm.h>
#include <linux/usb/gcore/usb_event.h>
#include <linux/usb/gcore/usb_gadgetcore.h>



#include <linux/usb/specific_gadget/usb_config.h>
#include <linux/usb/specific_gadget/usbg_cmn.h>
#include <linux/usb/specific_gadget/usb_extcmd.h>
#include <linux/usb/specific_gadget/usb_serial.h>

#include "usbg_storage_cmd.h"
#include "usbg_storage_cmd_extern.h"
#include "usbg_bdio.h"

#include "usbg_storage_conf.h"

/*
┏━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┓
┃ 定数定義                                                             ┃
┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛*/

/*
┏━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┓
┃ 変数定義                                                             ┃
┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛*/

/*
┏━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┓
┃ 関数プロトタイプ宣言                                                 ┃
┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛*/

/*
┏━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┓
┃ 関数定義                                                             ┃
┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛*/
/*■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■*
    コマンド実行
 *■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■*/
/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * mstrg_brdgdrv_ms_cmdPacket
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：コマンド実行
 *        【MassStorage Drive Controller から呼ばれる】
 * 入力  ：GSTORAGE_DRV__CmdContext のポインタ
 * 出力  ：GSTORAGE_DRV__CmdContext
 *         ( GSTORAGE_DRV__DeviceContext, GSTORAGE_DRV__UnitContext )
 * 返り値：GSTORAGE__SUCCESS/GSTORAGE__ERROR
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
GSTORAGE__Status storage_cmd_cmdPacket( GSTORAGE_DRV__CmdContext *pCmdCtx )
{
    GSTORAGE__Status ret = GSTORAGE__SUCCESS;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    pCmdCtx->pUnitCtx->continueInfo.extcmd = GSTORAGE__OFF;

    switch( pCmdCtx->pUnitCtx->pDrive->drvType ){
      case GSTORAGE__DRIVE_ID_MS   :
        ret = execPacketCmd_ms( pCmdCtx );
        break;
      case GSTORAGE__DRIVE_ID_MSB   :
      case GSTORAGE__DRIVE_ID_HDD  :
      case GSTORAGE__DRIVE_ID_CF   :
      case GSTORAGE__DRIVE_ID_NAND :
      case GSTORAGE__DRIVE_ID_MMC  :
      case GSTORAGE__DRIVE_ID_SXS  :
      case GSTORAGE__DRIVE_ID_SXS_ADPT :
        ret = execPacketCmd_hdd( pCmdCtx );
        break;
      case GSTORAGE__DRIVE_ID_DVD  :
        ret = execPacketCmd_dvd( pCmdCtx );
        break;
      default :
        GS_ERR( GS_OUT__CMD_STD, "unsupported drive type [opcode=%d]", pCmdCtx->pCdb->opCode );
        break;
    }

    return ret;
}

/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * storage_cmd_cmdContinue
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
GSTORAGE__Status storage_cmd_cmdContinue( GSTORAGE_DRV__CmdContext *pCmdCtx )
{
    GSTORAGE__Status ret = GSTORAGE__SUCCESS;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    if ( ( pCmdCtx->pUnitCtx->media.mediaStatus  == GSTORAGE__MEDIA_READY ||
           pCmdCtx->pUnitCtx->continueInfo.extcmd == GSTORAGE__ON ) &&
         ( pCmdCtx->pUnitCtx->continueInfo.pProc != NULL                  ) ){
        ret = pCmdCtx->pUnitCtx->continueInfo.pProc( pCmdCtx );
    }
    else {
        if ( pCmdCtx->pUnitCtx->media.changed == GSTORAGE__ON ){
            /* media changed */
            setMediaChanged( pCmdCtx, pCmdCtx->totalTransLen );
        }
        else if ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_NOT_READY ){
            /* no media */
            setMediaEjected( pCmdCtx, pCmdCtx->totalTransLen );
        }
        else {
            /* media error */
            pCmdCtx->status = GSTORAGE_CMD__STATUS_FAIL;
            setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
        }
        pCmdCtx->cb = NULL;
    }

    return ret;
}


/*■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■*
    コマンド実行
 *■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■*/
/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * execPacketCmd_ms
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：PACKET コマンド実行( for MS )
 * 入力  ：GSTORAGE_DRV__CmdContext のポインタ
 * 出力  ：GSTORAGE_DRV__CmdContext
 *         ( GSTORAGE_DRV__DeviceContext, GSTORAGE_DRV__UnitContext )
 * 返り値：GSTORAGE__SUCCESS/GSTORAGE__ERROR
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
GSTORAGE__Status execPacketCmd_ms( GSTORAGE_DRV__CmdContext *pCmdCtx )
{
    GSTORAGE__Status ret = GSTORAGE__SUCCESS;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    switch( pCmdCtx->pCdb->opCode ){
      case GSTORAGE_CMD__OPCODE_INQUIRY :
        ret = inquiry_std( pCmdCtx );
        break;
      case GSTORAGE_CMD__OPCODE_MODE_SENSE6 :
        ret = modeSense6_ms( pCmdCtx );
        break;
      case GSTORAGE_CMD__OPCODE_MODE_SENSE10 :
        ret = modeSense10_ms( pCmdCtx );
        break;
      case GSTORAGE_CMD__OPCODE_PREVENT_ALLOW :
        ret = preventAllow_std( pCmdCtx );
        break;
      case GSTORAGE_CMD__OPCODE_READ6 :
      case GSTORAGE_CMD__OPCODE_READ10 :
      case GSTORAGE_CMD__OPCODE_READ12 :
        ret = read_std( pCmdCtx );
        break;
      case GSTORAGE_CMD__OPCODE_READ_CAPACITY :
        ret = readCapa_std( pCmdCtx );
        break;
      case GSTORAGE_CMD__OPCODE_READ_FORMAT_CAPACITIES :
        ret = readFormatCapa_ms( pCmdCtx );
        break;
      case GSTORAGE_CMD__OPCODE_REQUEST_SENSE :
        ret = requestSense_std( pCmdCtx );
        break;
      case GSTORAGE_CMD__OPCODE_START_STOP_UNIT :
        ret = startStop_std( pCmdCtx );
        break;
      case GSTORAGE_CMD__OPCODE_SYNCHRONIZE_CACHE :
        ret = synchronize_cache_std( pCmdCtx );
        break;
      case GSTORAGE_CMD__OPCODE_UNIT_READY :
        ret = testUnit_std( pCmdCtx );
        break;
      case GSTORAGE_CMD__OPCODE_VERIFY :
        ret = verify_std( pCmdCtx );
        break;
      case GSTORAGE_CMD__OPCODE_WRITE10 :
      case GSTORAGE_CMD__OPCODE_WRITE12 :
        ret = write_std( pCmdCtx );
        break;
      default :
        ret = extCmd_std( pCmdCtx );
        break;
    }

    return ret;
}

/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * execPacketCmd_hdd
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：コマンド実行( for HDD )
 * 入力  ：GSTORAGE_DRV__CmdContext のポインタ
 * 出力  ：GSTORAGE_DRV__CmdContext
 *         ( GSTORAGE_DRV__DeviceContext, GSTORAGE_DRV__UnitContext )
 * 返り値：GSTORAGE__SUCCESS/GSTORAGE__ERROR
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
GSTORAGE__Status execPacketCmd_hdd( GSTORAGE_DRV__CmdContext *pCmdCtx )
{
    GSTORAGE__Status ret = GSTORAGE__SUCCESS;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    switch( pCmdCtx->pCdb->opCode ){
      case GSTORAGE_CMD__OPCODE_INQUIRY :
        ret = inquiry_std( pCmdCtx );
        break;
      case GSTORAGE_CMD__OPCODE_MODE_SENSE6 :
        ret = modeSense6_ata( pCmdCtx );
        break;
      case GSTORAGE_CMD__OPCODE_MODE_SENSE10 :
        ret = modeSense10_ata( pCmdCtx );
        break;
      case GSTORAGE_CMD__OPCODE_PREVENT_ALLOW :
        ret = preventAllow_std( pCmdCtx );
        break;
      case GSTORAGE_CMD__OPCODE_READ6 :
      case GSTORAGE_CMD__OPCODE_READ10 :
      case GSTORAGE_CMD__OPCODE_READ12 :
        ret = read_std( pCmdCtx );
        break;
      case GSTORAGE_CMD__OPCODE_READ_CAPACITY :
        ret = readCapa_std( pCmdCtx );
        break;
      case GSTORAGE_CMD__OPCODE_READ_FORMAT_CAPACITIES :
        ret = readFormatCapa_std( pCmdCtx );
        break;
      case GSTORAGE_CMD__OPCODE_REQUEST_SENSE :
        ret = requestSense_std( pCmdCtx );
        break;
      case GSTORAGE_CMD__OPCODE_START_STOP_UNIT :
        ret = startStop_std( pCmdCtx );
        break;
      case GSTORAGE_CMD__OPCODE_SYNCHRONIZE_CACHE :
        ret = synchronize_cache_std( pCmdCtx );
        break;
      case GSTORAGE_CMD__OPCODE_UNIT_READY :
        ret = testUnit_std( pCmdCtx );
        break;
      case GSTORAGE_CMD__OPCODE_VERIFY :
        ret = verify_std( pCmdCtx );
        break;
      case GSTORAGE_CMD__OPCODE_WRITE10 :
      case GSTORAGE_CMD__OPCODE_WRITE12 :
        ret = write_std( pCmdCtx );
        break;
      default :
        ret = extCmd_std( pCmdCtx );
        break;
    }

    return ret;
}

/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * inquiry_std
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：INQUIRY コマンド処理【 Data IN 】
 * 入力  ：GSTORAGE_DRV__CmdContext のポインタ
 * 出力  ：GSTORAGE_DRV__CmdContext
 *         ( GSTORAGE_DRV__DeviceContext, GSTORAGE_DRV__UnitContext )
 * 返り値：GSTORAGE__SUCCESS/GSTORAGE__ERROR
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
GSTORAGE__Status inquiry_std( GSTORAGE_DRV__CmdContext *pCmdCtx )
{
    GSTORAGE__Status         ret = GSTORAGE__SUCCESS;
    GSTORAGE_CMD__CdbInquiry *pCdb;
    GSTORAGE_CMD__DatVPD *pData;
    unsigned int bufLen = pCmdCtx->expectDatLen;
    unsigned int execLen;
    unsigned int allocLen;
    unsigned int pageCode;
    unsigned int maxLen = 0;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    /* set buffer pointer */
    pCdb = ( GSTORAGE_CMD__CdbInquiry * )pCmdCtx->pCdb;

    allocLen = ( (unsigned int)pCdb->AL[ 0 ] << 8 ) + 
                 (unsigned int)pCdb->AL[ 1 ];

    if ( ( pCmdCtx->dio == DATA_IN ) || ( pCmdCtx->dio == DATA_NON ) ){
        if ( ( ( pCdb->f1_evpd == 0 ) && ( pCdb->pageCode != 0x00 ) )
            || ( pCdb->f1_link == 1 ) || ( pCdb->f1_naca == 1 ) ){
            /* invalid filed in cdb */
            pCmdCtx->totalTransLen = 0;
            pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
            setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_FIELD );
        }
        else {
            if ( pCdb->f1_evpd == 0 ) {
                execLen = ( allocLen < pCmdCtx->pUnitCtx->pDrive->inquiryDataLen ) ?
                    allocLen : pCmdCtx->pUnitCtx->pDrive->inquiryDataLen;
                if ( bufLen < execLen ){
                    /* PhaseError設定 */
                    pCmdCtx->totalTransLen = 0;
                    pCmdCtx->status = GSTORAGE_CMD__STATUS_FATAL;
                    setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
                }
                else {
                    memcpy( pCmdCtx->pData, pCmdCtx->pUnitCtx->pDrive->pInquiryData, execLen );
                    /* set output data */
                    pCmdCtx->totalTransLen = execLen;
                    pCmdCtx->status        = GSTORAGE_CMD__STATUS_PASS;
                    setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
                }
            }
            else {
                pageCode = pCdb->pageCode;
                if ( ( pageCode == GSTORAGE_VPD__PAGECODE_SUPPORTED_VPD ) || 
                     ( pageCode == GSTORAGE_VPD__PAGECODE_SERIAL_NUM ) || 
                     ( pageCode == GSTORAGE_VPD__PAGECODE_DEVICE_IDENTIFIER ) ) {
                    int i;
                    pData = ( GSTORAGE_CMD__DatVPD * )pCmdCtx->pData;
                    memset( pData, 0x00, sizeof( GSTORAGE_CMD__DatVPD ) );
                    pData->vpdhd.pageCode = pageCode;
                    switch ( pageCode ) {
                        case GSTORAGE_VPD__PAGECODE_SUPPORTED_VPD:
                            pData->vpdhd.pageLen[1] = sizeof( GSTORAGE_CMD__DatVPD00 );
                            pData->page.vpd00.pageList[0] = GSTORAGE_VPD__PAGECODE_SUPPORTED_VPD;
                            pData->page.vpd00.pageList[1] = GSTORAGE_VPD__PAGECODE_SERIAL_NUM;
                            pData->page.vpd00.pageList[2] = GSTORAGE_VPD__PAGECODE_DEVICE_IDENTIFIER;
                            maxLen = sizeof( GSTORAGE_CMD__DatVPDHead ) + 
                                     sizeof( GSTORAGE_CMD__DatVPD00 );
                            break;
                        case GSTORAGE_VPD__PAGECODE_SERIAL_NUM:
                            pData->vpdhd.pageLen[1] = sizeof( GSTORAGE_CMD__DatVPD80 );
                            for (i = 0; i < USB_SERIAL_A_SIZE; i++ ) {
                                pData->page.vpd80.SN[i] = pCmdCtx->pUnitCtx->pDrive->vpdSerialData[i];
                            }
                            maxLen = sizeof( GSTORAGE_CMD__DatVPDHead ) + 
                                     sizeof( GSTORAGE_CMD__DatVPD80 );
                            break;
                        case GSTORAGE_VPD__PAGECODE_DEVICE_IDENTIFIER:
                            pData->vpdhd.pageLen[0] = ( sizeof( GSTORAGE_CMD__DatVPD83 ) >> 8 ) & 0xFF;
                            pData->vpdhd.pageLen[1] = sizeof( GSTORAGE_CMD__DatVPD83 ) & 0xFF;
                            pData->page.vpd83.f4_codeSet = GSTORAGE_VPD__DEVICE_IDENTIFIER_CODE_SET;
                            pData->page.vpd83.f4_protocolID = GSTORAGE_VPD__DEVICE_IDENTIFIER_PROTOCOL_ID;
                            pData->page.vpd83.f4_IDType = GSTORAGE_VPD__DEVICE_IDENTIFIER_ID_TYPE;
                            pData->page.vpd83.id_length = GSTORAGE_VPD__DEVICE_IDENTIFIER_ID_LEN;
                            pData->page.vpd83.f4_NAA = GSTORAGE_VPD__DEVICE_IDENTIFIER_NAA;
                            pData->page.vpd83.ieeeCompanyID[0] = (GSTORAGE_VPD__IEEE_COMPANY_ID_SONY >> 16) & 0xFF;
                            pData->page.vpd83.ieeeCompanyID[1] = (GSTORAGE_VPD__IEEE_COMPANY_ID_SONY >> 8) & 0xFF;
                            pData->page.vpd83.ieeeCompanyID[2] = GSTORAGE_VPD__IEEE_COMPANY_ID_SONY & 0xFF;
                            maxLen = sizeof( GSTORAGE_CMD__DatVPDHead ) + 
                                     sizeof( GSTORAGE_CMD__DatVPD83 );
                            break;
                        default:
                            /* F/W error */
                            break;
                    }
                    execLen = min( allocLen, maxLen );
                    if ( bufLen < execLen ){
                        /* PhaseError設定 */
                        pCmdCtx->totalTransLen = 0;
                        pCmdCtx->status = GSTORAGE_CMD__STATUS_FATAL;
                        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
                    }
                    else {
                        /* set output data */
                        pCmdCtx->totalTransLen = execLen;
                        pCmdCtx->status        = GSTORAGE_CMD__STATUS_PASS;
                        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
                    }
                }
                else {
                    /* invalid filed in cdb */
                    pCmdCtx->totalTransLen = 0;
                    pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
                    setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_FIELD );
                }
            }
        }
    }
    else {
        /* invalid command */
        pCmdCtx->totalTransLen = 0;
        pCmdCtx->status        = GSTORAGE_CMD__STATUS_FATAL;
        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_COMMAND );
    }

    return ret;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * modeSense6_ms
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：MODE SENSE 6 コマンド処理【 Data IN 】
 * 入力  ：GSTORAGE_DRV__CmdContext のポインタ
 * 出力  ：GSTORAGE_DRV__CmdContext
 *         ( GSTORAGE_DRV__DeviceContext, GSTORAGE_DRV__UnitContext )
 * 返り値：GSTORAGE__SUCCESS/GSTORAGE__ERROR
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
GSTORAGE__Status modeSense6_ms( GSTORAGE_DRV__CmdContext *pCmdCtx )
{
    GSTORAGE__Status            ret = GSTORAGE__SUCCESS;
    GSTORAGE__Cdb               *pCdb;
    GSTORAGE_CMD__DatModeSense6 *pData;
    signed   int pageCode;
    unsigned int subPageCode;
    signed   int pc;
    unsigned int bufLen = pCmdCtx->expectDatLen;
    unsigned int listLen;
    unsigned int maxLen;
    unsigned int execLen;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    /* drive reset ? */
    if ( pCmdCtx->pUnitCtx->driveReset == GSTORAGE__ON ){
        setDriveReset( pCmdCtx, 0 );
        return ret;
    }
    /* media changed ? */
    if ( pCmdCtx->pUnitCtx->media.changed == GSTORAGE__ON ){
        setMediaChanged( pCmdCtx, 0 );
        return ret;
    }

    if ( ( pCmdCtx->dio == DATA_IN ) || ( pCmdCtx->dio == DATA_NON ) ){
        pCdb  = pCmdCtx->pCdb;
        pData = ( GSTORAGE_CMD__DatModeSense6 * )pCmdCtx->pData;
        memset( pData, 0x00, sizeof( GSTORAGE_CMD__DatModeSense6 ) );
        listLen  = ( ( GSTORAGE_CMD__CdbModeSense6 * )pCdb )->AL;
        pageCode = ( ( GSTORAGE_CMD__CdbModeSense6 * )pCdb )->f6_pageCode;
        subPageCode = ( ( GSTORAGE_CMD__CdbModeSense6 * )pCdb )->subPageCode;
        pc       = ( ( GSTORAGE_CMD__CdbModeSense6 * )pCdb )->f2_pc;

        if ( ( ( pc == 0x00 ) || ( pc == 0x02 ) ) &&
             ( ( pageCode == 0x00 ) || ( pageCode == 0x20 ) || ( pageCode == 0x3F ) ) &&
             ( ( subPageCode == 0x00 ) || ( subPageCode == 0xFF ) ) &&
             ( ( ( ( GSTORAGE_CMD__CdbModeSense6 * )pCdb )->f1_link == 0 ) && 
               ( ( ( GSTORAGE_CMD__CdbModeSense6 * )pCdb )->f1_naca == 0 ) ) ) {
             
            pData->modeDataLen = ( pageCode == 0x00 ) ? 0x03 : 0x67;

            if ( ( pCmdCtx->pUnitCtx->media.mediaStatus != GSTORAGE__MEDIA_READY          ) ||
                 ( pCmdCtx->pUnitCtx->media.unitStatus  != GSTORAGE__UNIT_NOT_EJECT_START ) ){
                pData->mediumType = 0x00;
                pData->f1_wp      = 0;
            }
            else {
                switch( pCmdCtx->pUnitCtx->media.mediumType ){
                  case GSTORAGE_MEDIA__MS_V1  :
                  case GSTORAGE_MEDIA__MS_MG  :
                  case GSTORAGE_MEDIA__MS_R   :
                  case GSTORAGE_MEDIA__MS_PRO :
                  case GSTORAGE_MEDIA__MS_XC  :
                    pData->mediumType = pCmdCtx->pUnitCtx->media.mediumType;
                    break;
                  case GSTORAGE_MEDIA__MS_HG  :
                    pData->mediumType = GSTORAGE_MEDIA__MS_MG;
                    break;
                  default :
                    pData->mediumType = 0x00;
                    break;
                }
                if ( pCmdCtx->pUnitCtx->media.writeProtect || pCmdCtx->pUnitCtx->pDrive->forceWP ){
                    pData->f1_wp  = 1;
                }
            }

            if ( ( pData->mediumType != 0x00 ) && ( pageCode != 0x00 ) ){
                pData->page.ms.f1_ps       = 0;
                pData->page.ms.f6_pageCode = pageCode;
                pData->page.ms.pageLen     = 0x62;
                pData->page.ms.f1_ac0      = 0;
                pData->page.ms.f1_ac1      = 0;
                pData->page.ms.f1_sf       = 0;
                memcpy( pData->page.ms.msSystemInfo,
                        pCmdCtx->pUnitCtx->mediaExt.pMsInfo,
                        sizeof( pData->page.ms.msSystemInfo ) );
            }
            maxLen = ( pageCode == 0x00 ) ? 0x04 : 0x68;
            execLen = ( listLen < maxLen ) ? listLen : maxLen;
            if ( bufLen < execLen ){
                /* PhaseError設定 */
                pCmdCtx->totalTransLen = 0;
                pCmdCtx->status = GSTORAGE_CMD__STATUS_FATAL;
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
            }
            else {
                /* set output data */
                pCmdCtx->totalTransLen = execLen;
                pCmdCtx->status        = GSTORAGE_CMD__STATUS_PASS;
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
            }
        }
        else {
            pCmdCtx->totalTransLen = 0;
            pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
            if ( pc == 0x03 ){
                /* saving parameters not supports */
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_SAVING_PARAM_NOT_SUPPORTED );
            }
            else {
                /* invalid field */
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_FIELD );
            }
        }
    }
    else {
        /* invalid command */
        pCmdCtx->totalTransLen = 0;
        pCmdCtx->status        = GSTORAGE_CMD__STATUS_FATAL;
        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_COMMAND );
    }

    return ret;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * modeSense10_ms
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：MODE SENSE 10 コマンド処理【 Data IN 】
 * 入力  ：GSTORAGE_DRV__CmdContext のポインタ
 * 出力  ：GSTORAGE_DRV__CmdContext
 *         ( GSTORAGE_DRV__DeviceContext, GSTORAGE_DRV__UnitContext )
 * 返り値：GSTORAGE__SUCCESS/GSTORAGE__ERROR
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
GSTORAGE__Status modeSense10_ms( GSTORAGE_DRV__CmdContext *pCmdCtx )
{
    GSTORAGE__Status             ret = GSTORAGE__SUCCESS;
    GSTORAGE__Cdb                *pCdb;
    GSTORAGE_CMD__DatModeSense10 *pData;
    signed   int pageCode;
    unsigned int subPageCode;
    signed   int pc;
    unsigned int bufLen = pCmdCtx->expectDatLen;
    unsigned int listLen;
    unsigned int maxLen;
    unsigned int execLen;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    /* drive reset ? */
    if ( pCmdCtx->pUnitCtx->driveReset == GSTORAGE__ON ){
        setDriveReset( pCmdCtx, 0 );
        return ret;
    }
    /* media changed ? */
    if ( pCmdCtx->pUnitCtx->media.changed == GSTORAGE__ON ){
        setMediaChanged( pCmdCtx, 0 );
        return ret;
    }

    if ( ( pCmdCtx->dio == DATA_IN ) || ( pCmdCtx->dio == DATA_NON ) ){
        pCdb  = pCmdCtx->pCdb;
        pData = ( GSTORAGE_CMD__DatModeSense10 * )pCmdCtx->pData;
        memset( pData, 0x00, sizeof( GSTORAGE_CMD__DatModeSense10 ) );

        listLen  = ( (unsigned int)( ( GSTORAGE_CMD__CdbModeSense10 * )pCdb )->AL[ 0 ] << 8 )
                 + ( (unsigned int)( ( GSTORAGE_CMD__CdbModeSense10 * )pCdb )->AL[ 1 ]      );
        pageCode = ( ( GSTORAGE_CMD__CdbModeSense10 * )pCdb )->f6_pageCode;
        subPageCode = ( ( GSTORAGE_CMD__CdbModeSense10 * )pCdb )->subPageCode;
        pc       = ( ( GSTORAGE_CMD__CdbModeSense10 * )pCdb )->f2_pc;

        if ( ( ( pc == 0x00 ) || ( pc == 0x02 ) ) &&
             ( ( pageCode == 0x00 ) || ( pageCode == 0x20 ) || ( pageCode == 0x3F ) ) && 
             ( ( subPageCode == 0x00 ) || ( subPageCode == 0xFF ) ) &&
             ( ( ( ( GSTORAGE_CMD__CdbModeSense10 * )pCdb )->f1_link == 0 ) && 
               ( ( ( GSTORAGE_CMD__CdbModeSense10 * )pCdb )->f1_naca == 0 ) ) ) {

            pData->modeDataLen[0] = 0x00;
            pData->modeDataLen[1] = ( pageCode == 0x00 ) ? 0x06 : 0x6A;

            if ( ( pCmdCtx->pUnitCtx->media.mediaStatus != GSTORAGE__MEDIA_READY          ) ||
                 ( pCmdCtx->pUnitCtx->media.unitStatus  != GSTORAGE__UNIT_NOT_EJECT_START ) ){
                pData->mediumType = 0x00;
                pData->f1_wp      = 0;
            }
            else {
                switch( pCmdCtx->pUnitCtx->media.mediumType ){
                  case GSTORAGE_MEDIA__MS_V1  :
                  case GSTORAGE_MEDIA__MS_MG  :
                  case GSTORAGE_MEDIA__MS_R   :
                  case GSTORAGE_MEDIA__MS_PRO :
                  case GSTORAGE_MEDIA__MS_XC  :
                    pData->mediumType = pCmdCtx->pUnitCtx->media.mediumType;
                    break;
                  case GSTORAGE_MEDIA__MS_HG  :
                    pData->mediumType = GSTORAGE_MEDIA__MS_MG;
                    break;
                  default :
                    pData->mediumType = 0x00;
                    break;
                }
                if ( pCmdCtx->pUnitCtx->media.writeProtect || pCmdCtx->pUnitCtx->pDrive->forceWP ){
                    pData->f1_wp  = 1;
                }
            }

            if ( ( pData->mediumType != 0x00 ) && ( pageCode != 0x00 ) ){
                pData->page.ms.f1_ps       = 0;
                pData->page.ms.f6_pageCode = pageCode;
                pData->page.ms.pageLen     = 0x62;
                pData->page.ms.f1_ac0      = 0;
                pData->page.ms.f1_ac1      = 0;
                pData->page.ms.f1_sf       = 0;
                memcpy( pData->page.ms.msSystemInfo,
                        pCmdCtx->pUnitCtx->mediaExt.pMsInfo,
                        sizeof( pData->page.ms.msSystemInfo ) );
            }
            maxLen = ( pageCode == 0x00 ) ? 0x08 : 0x6C;
            execLen = ( listLen < maxLen ) ? listLen : maxLen;
            if ( bufLen < execLen ){
                /* PhaseError設定 */
                pCmdCtx->totalTransLen = 0;
                pCmdCtx->status = GSTORAGE_CMD__STATUS_FATAL;
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
            }
            else {
                /* set output data */
                pCmdCtx->totalTransLen = execLen;
                pCmdCtx->status        = GSTORAGE_CMD__STATUS_PASS;
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
            }
        }
        else {
            pCmdCtx->totalTransLen = 0;
            pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
            if ( pc == 0x03 ){
                /* saving parameters not supports */
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_SAVING_PARAM_NOT_SUPPORTED );
            }
            else {
                /* invalid field */
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_FIELD );
            }
        }
    }
    else {
        /* invalid command */
        pCmdCtx->totalTransLen = 0;
        pCmdCtx->status        = GSTORAGE_CMD__STATUS_FATAL;
        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_COMMAND );
    }

    return ret;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * modeSense6_ata
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：MODE SENSE 6 コマンド処理【 Data IN 】
 * 入力  ：GSTORAGE_DRV__CmdContext のポインタ
 * 出力  ：GSTORAGE_DRV__CmdContext
 *         ( GSTORAGE_DRV__DeviceContext, GSTORAGE_DRV__UnitContext )
 * 返り値：GSTORAGE__SUCCESS/GSTORAGE__ERROR
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
GSTORAGE__Status modeSense6_ata( GSTORAGE_DRV__CmdContext *pCmdCtx )
{
    GSTORAGE__Status            ret = GSTORAGE__SUCCESS;
    GSTORAGE__Cdb               *pCdb;
    GSTORAGE_CMD__DatModeSense6 *pData;
    signed   int pageCode;
    unsigned int subPageCode;
    signed   int pc;
    unsigned int bufLen = pCmdCtx->expectDatLen;
    unsigned int listLen;
    unsigned int maxLen = 0;
    unsigned int execLen;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    /* drive reset ? */
    if ( pCmdCtx->pUnitCtx->driveReset == GSTORAGE__ON ){
        setDriveReset( pCmdCtx, 0 );
        return ret;
    }
    /* media changed ? */
    if ( pCmdCtx->pUnitCtx->media.changed == GSTORAGE__ON ){
        setMediaChanged( pCmdCtx, 0 );
        return ret;
    }

    if ( ( pCmdCtx->dio == DATA_IN ) || ( pCmdCtx->dio == DATA_NON ) ){
        pCdb  = pCmdCtx->pCdb;
        pData = ( GSTORAGE_CMD__DatModeSense6 * )pCmdCtx->pData;
        memset( pData, 0x00, sizeof( GSTORAGE_CMD__DatModeSense6 ) );
        listLen  = ( ( GSTORAGE_CMD__CdbModeSense6 * )pCdb )->AL;
        pageCode = ( ( GSTORAGE_CMD__CdbModeSense6 * )pCdb )->f6_pageCode;
        subPageCode = ( ( GSTORAGE_CMD__CdbModeSense6 * )pCdb )->subPageCode;
        pc       = ( ( GSTORAGE_CMD__CdbModeSense6 * )pCdb )->f2_pc;

        if ( ( ( pc == 0x00 ) || ( pc == 0x02 ) ) && ( pageCode == 0x3F ) &&
             ( ( subPageCode == 0x00 ) || ( subPageCode == 0xFF ) ) &&
             ( ( ( ( GSTORAGE_CMD__CdbModeSense6 * )pCdb )->f1_link == 0 ) && 
               ( ( ( GSTORAGE_CMD__CdbModeSense6 * )pCdb )->f1_naca == 0 ) ) ) {
            pData->modeDataLen = 0x03;
            if ( ( pCmdCtx->pUnitCtx->media.mediaStatus != GSTORAGE__MEDIA_READY          ) ||
                 ( pCmdCtx->pUnitCtx->media.unitStatus  != GSTORAGE__UNIT_NOT_EJECT_START ) ){
                pData->mediumType = 0x00;
                pData->f1_wp      = 0;
            }
            else {
                pData->mediumType = pCmdCtx->pUnitCtx->media.mediumType;
                if ( pCmdCtx->pUnitCtx->media.writeProtect || pCmdCtx->pUnitCtx->pDrive->forceWP ){
                    pData->f1_wp  = 1;
                }
            }
            maxLen = 0x04;
            execLen = ( listLen < maxLen ) ? listLen : maxLen;
            
            if ( bufLen < execLen ){
                /* PhaseError設定 */
                pCmdCtx->totalTransLen = 0;
                pCmdCtx->status = GSTORAGE_CMD__STATUS_FATAL;
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
            }
            else {
                /* set output data */
                pCmdCtx->totalTransLen = execLen;
                pCmdCtx->status        = GSTORAGE_CMD__STATUS_PASS;
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
            }
        }
        else {
            pCmdCtx->totalTransLen = 0;
            pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
            if ( pc == 0x03 ){
                /* saving parameters not supports */
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_SAVING_PARAM_NOT_SUPPORTED );
            }
            else {
                /* invalid field */
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_FIELD );
            }
        }
    }
    else {
        /* invalid command */
        pCmdCtx->totalTransLen = 0;
        pCmdCtx->status        = GSTORAGE_CMD__STATUS_FATAL;
        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_COMMAND );
    }

    return ret;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * modeSense10_ata
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：MODE SENSE 10 コマンド処理【 Data IN 】
 * 入力  ：GSTORAGE_DRV__CmdContext のポインタ
 * 出力  ：GSTORAGE_DRV__CmdContext
 *         ( GSTORAGE_DRV__DeviceContext, GSTORAGE_DRV__UnitContext )
 * 返り値：GSTORAGE__SUCCESS/GSTORAGE__ERROR
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
GSTORAGE__Status modeSense10_ata( GSTORAGE_DRV__CmdContext *pCmdCtx )
{
    GSTORAGE__Status             ret = GSTORAGE__SUCCESS;
    GSTORAGE__Cdb                *pCdb;
    GSTORAGE_CMD__DatModeSense10 *pData;
    signed   int pageCode;
    unsigned int subPageCode;
    signed   int pc;
    unsigned int bufLen = pCmdCtx->expectDatLen;
    unsigned int listLen;
    unsigned int maxLen = 0;
    unsigned int execLen;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    /* drive reset ? */
    if ( pCmdCtx->pUnitCtx->driveReset == GSTORAGE__ON ){
        setDriveReset( pCmdCtx, 0 );
        return ret;
    }
    /* media changed ? */
    if ( pCmdCtx->pUnitCtx->media.changed == GSTORAGE__ON ){
        setMediaChanged( pCmdCtx, 0 );
        return ret;
    }

    if ( ( pCmdCtx->dio == DATA_IN ) || ( pCmdCtx->dio == DATA_NON ) ){
        pCdb  = pCmdCtx->pCdb;
        pData = ( GSTORAGE_CMD__DatModeSense10 * )pCmdCtx->pData;
        memset( pData, 0x00, sizeof( GSTORAGE_CMD__DatModeSense10 ) );

        listLen  = ( (unsigned int)( ( GSTORAGE_CMD__CdbModeSense10 * )pCdb )->AL[ 0 ] << 8 )
                 + ( (unsigned int)( ( GSTORAGE_CMD__CdbModeSense10 * )pCdb )->AL[ 1 ]      );
        pageCode = ( ( GSTORAGE_CMD__CdbModeSense10 * )pCdb )->f6_pageCode;
        subPageCode = ( ( GSTORAGE_CMD__CdbModeSense10 * )pCdb )->subPageCode;
        pc       = ( ( GSTORAGE_CMD__CdbModeSense10 * )pCdb )->f2_pc;

        if ( ( ( pc == 0x00 ) || ( pc == 0x02 ) ) && ( pageCode == 0x3F ) &&
             ( ( subPageCode == 0x00 ) || ( subPageCode == 0xFF ) ) &&
             ( ( ( ( GSTORAGE_CMD__CdbModeSense10 * )pCdb )->f1_link == 0 ) && 
               ( ( ( GSTORAGE_CMD__CdbModeSense10 * )pCdb )->f1_naca == 0 ) ) ) {
            pData->modeDataLen[0] = 0x00;
            pData->modeDataLen[1] = 0x06;
            if ( ( pCmdCtx->pUnitCtx->media.mediaStatus != GSTORAGE__MEDIA_READY          ) ||
                 ( pCmdCtx->pUnitCtx->media.unitStatus  != GSTORAGE__UNIT_NOT_EJECT_START ) ){
                pData->mediumType = 0x00;
                pData->f1_wp      = 0;
            }
            else {
                pData->mediumType = pCmdCtx->pUnitCtx->media.mediumType;
                if ( pCmdCtx->pUnitCtx->media.writeProtect || pCmdCtx->pUnitCtx->pDrive->forceWP ){
                    pData->f1_wp  = 1;
                }
            }
            maxLen = 0x08;
            execLen = ( listLen < maxLen ) ? listLen : maxLen;
            if ( bufLen < execLen ){
                /* PhaseError設定 */
                pCmdCtx->totalTransLen = 0;
                pCmdCtx->status = GSTORAGE_CMD__STATUS_FATAL;
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
            }
            else {
                /* set output data */
                pCmdCtx->totalTransLen = execLen;
                pCmdCtx->status        = GSTORAGE_CMD__STATUS_PASS;
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
            }
        }
        else {
            pCmdCtx->totalTransLen = 0;
            pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
            if ( pc == 0x03 ){
                /* saving parameters not supports */
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_SAVING_PARAM_NOT_SUPPORTED );
            }
            else {
                /* invalid field */
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_FIELD );
            }
        }
    }
    else {
        /* invalid command */
        pCmdCtx->totalTransLen = 0;
        pCmdCtx->status        = GSTORAGE_CMD__STATUS_FATAL;
        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_COMMAND );
    }

    return ret;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * preventAllow_std
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：MEDIUM REMOVAL コマンド処理【 Data NON 】
 * 入力  ：GSTORAGE_DRV__CmdContext のポインタ
 * 出力  ：GSTORAGE_DRV__CmdContext
 *         ( GSTORAGE_DRV__DeviceContext, GSTORAGE_DRV__UnitContext )
 * 返り値：GSTORAGE__SUCCESS/GSTORAGE__ERROR
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
GSTORAGE__Status preventAllow_std( GSTORAGE_DRV__CmdContext *pCmdCtx )
{
    GSTORAGE_CMD__CdbPreventAllow *pCdb;
    GSTORAGE__Status              ret = GSTORAGE__SUCCESS;
    int          result = 0;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    /* drive reset ? */
    if ( pCmdCtx->pUnitCtx->driveReset == GSTORAGE__ON ){
        setDriveReset( pCmdCtx, 0);
        return ret;
    }
    /* media changed ? */
    if ( pCmdCtx->pUnitCtx->media.changed == GSTORAGE__ON ){
        setMediaChanged( pCmdCtx, 0 );
        return ret;
    }

    pCdb = ( GSTORAGE_CMD__CdbPreventAllow * )pCmdCtx->pCdb;

    /* Invalid field ? */
    if ( ( pCdb->f2_prevent > 1 ) || ( pCdb->f1_link == 1 ) || ( pCdb->f1_naca == 1 ) ) {
        pCmdCtx->totalTransLen = 0;
        pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_FIELD );
        return ret;
    }
    
    if ( pCdb->f2_prevent == 1 ){
        /* prevent 処理 */
        if ( pCmdCtx->pUnitCtx->pDrive->preventEnable ){
            if ( pCmdCtx->pUnitCtx->pDrive->driveLock == GSTORAGE__LOCK_REAL ){
                if ( pCmdCtx->pUnitCtx->currentLock == GSTORAGE__OFF ){
                    pCmdCtx->pUnitCtx->currentLock = GSTORAGE__ON;
                }
            }
            else if ( pCmdCtx->pUnitCtx->pDrive->driveLock == GSTORAGE__LOCK_VIRTUAL ){
                if ( pCmdCtx->pUnitCtx->currentLock == GSTORAGE__OFF ){
                    pCmdCtx->pUnitCtx->currentLock = GSTORAGE__ON;
                }
            }
            pCmdCtx->totalTransLen = 0;
            pCmdCtx->status        = GSTORAGE_CMD__STATUS_PASS;
            setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
        }
        else {
            /* invalid field */
            pCmdCtx->totalTransLen = 0;
            pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
            setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_FIELD );
        }
    }
    else {
        /* Allow 処理 */
        if ( pCmdCtx->pUnitCtx->pDrive->flushEnable ){
            if ( ( pCmdCtx->pUnitCtx->pDrive->drvType == GSTORAGE__DRIVE_ID_HDD  ) ||
                ( pCmdCtx->pUnitCtx->pDrive->drvType == GSTORAGE__DRIVE_ID_CF   ) ){
                if ( ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_READY          ) &&
                    ( pCmdCtx->pUnitCtx->media.unitStatus  == GSTORAGE__UNIT_NOT_EJECT_START ) ){
                    result = flushCache_ata_cmn( pCmdCtx->pUnitCtx, pCmdCtx->pUnitCtx->fd );
                }
            }
        }
        if ( result == 0 ){
            if ( pCmdCtx->pUnitCtx->pDrive->driveLock == GSTORAGE__LOCK_REAL ){
                if ( pCmdCtx->pUnitCtx->currentLock == GSTORAGE__ON ){
                    pCmdCtx->pUnitCtx->currentLock = GSTORAGE__OFF;
                }
            }
            else if ( pCmdCtx->pUnitCtx->pDrive->driveLock == GSTORAGE__LOCK_VIRTUAL ){
                if ( pCmdCtx->pUnitCtx->currentLock == GSTORAGE__ON ){
                    pCmdCtx->pUnitCtx->currentLock = GSTORAGE__OFF;
                }
            }
            pCmdCtx->status = GSTORAGE_CMD__STATUS_PASS;
        }
        else {
            pCmdCtx->status = GSTORAGE_CMD__STATUS_FAIL;
        }
        pCmdCtx->totalTransLen = 0;
        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
    }

    return ret;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * read_std
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：READ10/READ12 コマンド処理【 Data IN 】
 * 入力  ：GSTORAGE_DRV__CmdContext のポインタ
 * 出力  ：GSTORAGE_DRV__CmdContext
 *         ( GSTORAGE_DRV__DeviceContext, GSTORAGE_DRV__UnitContext )
 * 返り値：GSTORAGE__SUCCESS/GSTORAGE__ERROR
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
GSTORAGE__Status read_std( GSTORAGE_DRV__CmdContext *pCmdCtx )
{
    GSTORAGE__Status ret = GSTORAGE__SUCCESS;
    GSTORAGE__Cdb    *pCdb = pCmdCtx->pCdb;
    signed long long transLen;
    signed long long bufLen = (signed long long)pCmdCtx->expectDatLen;
    signed long long maxLen;
    signed long long execLen;
    signed long long lastLba;
    signed long long readLba;
    signed long long readSize;
    signed long long readOffset;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    /* drive reset ? */
    if ( pCmdCtx->pUnitCtx->driveReset == GSTORAGE__ON ){
        setDriveReset( pCmdCtx, 0 );
        return ret;
    }
    /* media changed ? */
    if ( pCmdCtx->pUnitCtx->media.changed == GSTORAGE__ON ){
        setMediaChanged( pCmdCtx, 0 );
        return ret;
    }
    /* unit eject ? */
    if ( pCmdCtx->pUnitCtx->media.unitStatus == GSTORAGE__UNIT_EJECT ){
        setMediaEjected( pCmdCtx, 0 );
        return ret;
    }
    /* drive busy ? ( = media unknown ? ) */
    if ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_UNKNOWN ){
        setDriveNotReady( pCmdCtx, 0 );
        return ret;
    }

    if ( ( pCmdCtx->dio == DATA_IN ) || ( pCmdCtx->dio == DATA_NON ) ){

        /* Invalid Field ? */
        if ( pCmdCtx->pCdb->opCode == GSTORAGE_CMD__OPCODE_READ6 ){
            /* READ(6) */
            if ( ( ( ( GSTORAGE_CMD__CdbRead6 * )pCdb )->f1_link != 0 ) ||
                ( ( ( GSTORAGE_CMD__CdbRead6 * )pCdb )->f1_naca != 0 ) ) {
                pCmdCtx->totalTransLen = 0;
                pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_FIELD );
                return ret;
            }
        }
        else if ( pCmdCtx->pCdb->opCode == GSTORAGE_CMD__OPCODE_READ10 ){
            /* READ(10) */
            if ( ( ( ( GSTORAGE_CMD__CdbRead10 * )pCdb )->f1_fua_nv != 0 ) ||
                ( ( ( GSTORAGE_CMD__CdbRead10 * )pCdb )->f1_fua != 0 ) ||
                ( ( ( GSTORAGE_CMD__CdbRead10 * )pCdb )->f1_dpo != 0 ) ||
                //( ( ( ( GSTORAGE_CMD__CdbRead10 * )pCdb )->f3_rdprotect >= 1 ) &&
                //  ( ( ( GSTORAGE_CMD__CdbRead10 * )pCdb )->f3_rdprotect <= 4 ) ) ||
                ( ( ( GSTORAGE_CMD__CdbRead10 * )pCdb )->f1_link != 0 ) ||
                ( ( ( GSTORAGE_CMD__CdbRead10 * )pCdb )->f1_naca != 0 ) ) {
                pCmdCtx->totalTransLen = 0;
                pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_FIELD );
                return ret;
            }
        }
        else {
            /* READ(12) */
            if ( ( ( ( GSTORAGE_CMD__CdbRead12 * )pCdb )->f1_fua_nv != 0 ) ||
                ( ( ( GSTORAGE_CMD__CdbRead12 * )pCdb )->f1_fua != 0 ) ||
                ( ( ( GSTORAGE_CMD__CdbRead12 * )pCdb )->f1_dpo != 0 ) ||
                //( ( ( ( GSTORAGE_CMD__CdbRead12 * )pCdb )->f3_rdprotect >= 1 ) &&
                //  ( ( ( GSTORAGE_CMD__CdbRead12 * )pCdb )->f3_rdprotect <= 4 ) ) ||
                ( ( ( GSTORAGE_CMD__CdbRead12 * )pCdb )->f1_link != 0 ) ||
                ( ( ( GSTORAGE_CMD__CdbRead12 * )pCdb )->f1_naca != 0 ) ) {
                pCmdCtx->totalTransLen = 0;
                pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_FIELD );
                return ret;
            }
        }

        if ( pCmdCtx->pCdb->opCode == GSTORAGE_CMD__OPCODE_READ6 ){
            transLen = ( ( GSTORAGE_CMD__CdbRead6 * )pCdb )->transLen;
            readLba  = ( (unsigned int)( ( GSTORAGE_CMD__CdbRead6 * )pCdb )->f5_LBA << 16 )
                     | ( (unsigned int)( ( GSTORAGE_CMD__CdbRead6 * )pCdb )->LBA[ 0 ] <<  8 )
                     | ( (unsigned int)( ( GSTORAGE_CMD__CdbRead6 * )pCdb )->LBA[ 1 ] );
        }
        else if ( pCmdCtx->pCdb->opCode == GSTORAGE_CMD__OPCODE_READ10 ){
            transLen = ( (unsigned int)( ( GSTORAGE_CMD__CdbRead10 * )pCdb )->transLen[ 0 ] << 8 )
                     | ( (unsigned int)( ( GSTORAGE_CMD__CdbRead10 * )pCdb )->transLen[ 1 ] );
            readLba  = ( (unsigned int)( ( GSTORAGE_CMD__CdbRead10 * )pCdb )->LBA[ 0 ] << 24 )
                     | ( (unsigned int)( ( GSTORAGE_CMD__CdbRead10 * )pCdb )->LBA[ 1 ] << 16 )
                     | ( (unsigned int)( ( GSTORAGE_CMD__CdbRead10 * )pCdb )->LBA[ 2 ] <<  8 )
                     | ( (unsigned int)( ( GSTORAGE_CMD__CdbRead10 * )pCdb )->LBA[ 3 ] );
        }
        else {
            transLen = ( (unsigned int)( ( GSTORAGE_CMD__CdbRead12 * )pCdb )->transLen[ 0 ] << 24 )
                     | ( (unsigned int)( ( GSTORAGE_CMD__CdbRead12 * )pCdb )->transLen[ 1 ] << 16 )
                     | ( (unsigned int)( ( GSTORAGE_CMD__CdbRead12 * )pCdb )->transLen[ 2 ] <<  8 )
                     | ( (unsigned int)( ( GSTORAGE_CMD__CdbRead12 * )pCdb )->transLen[ 3 ]       );
            readLba  = ( (unsigned int)( ( GSTORAGE_CMD__CdbRead12 * )pCdb )->LBA[ 0 ] << 24 )
                     | ( (unsigned int)( ( GSTORAGE_CMD__CdbRead12 * )pCdb )->LBA[ 1 ] << 16 )
                     | ( (unsigned int)( ( GSTORAGE_CMD__CdbRead12 * )pCdb )->LBA[ 2 ] <<  8 )
                     | ( (unsigned int)( ( GSTORAGE_CMD__CdbRead12 * )pCdb )->LBA[ 3 ]       );
        }
        lastLba    = (signed long long)pCmdCtx->pUnitCtx->media.lastLBA;
        maxLen     = (signed long long)pCmdCtx->pUnitCtx->media.blocks
                   * (signed long long)pCmdCtx->pUnitCtx->media.blockLength;
        readSize   = transLen * (signed long long)pCmdCtx->pUnitCtx->media.blockLength;
        readOffset = readLba * (signed long long)pCmdCtx->pUnitCtx->media.blockLength;
        execLen    = ( readSize < maxLen ) ? readSize : maxLen;
        if ( bufLen < execLen ){
            /* PhaseError設定 */
            pCmdCtx->totalTransLen = 0;
            pCmdCtx->status = GSTORAGE_CMD__STATUS_FATAL;
            setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
        }
        else {
            if ( ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_READY          ) &&
                 ( pCmdCtx->pUnitCtx->media.unitStatus  == GSTORAGE__UNIT_NOT_EJECT_START ) ){
                if ( ( pCmdCtx->dio == DATA_IN ) && ( transLen > 0 ) ) {
                    /* check Access LBA */
                    if ( readLba + transLen <= lastLba + 1 ){
                        pCmdCtx->pUnitCtx->continueInfo.address     = readOffset;
                        pCmdCtx->pUnitCtx->continueInfo.remainLen   = execLen;
                        pCmdCtx->pUnitCtx->continueInfo.transferLen = 0;
                        pCmdCtx->pUnitCtx->continueInfo.pProc       = read_std_continue;
                        if( execLen >= GSTORAGE__PREREAD_SIZE ){
                            pCmdCtx->cb = preRead_std;
                        }
                        else {
                            pCmdCtx->cb = NULL;
                        }
                        pCmdCtx->pProcParam = pCmdCtx->pUnitCtx;
                        GS_DBG( GS_OUT__CMD_RW, "bufLen=%llu, execLen=%llu, address=%llu, remainLen=%d, transferLen=%d",
                                bufLen, execLen, pCmdCtx->pUnitCtx->continueInfo.address,
                                pCmdCtx->pUnitCtx->continueInfo.remainLen, pCmdCtx->pUnitCtx->continueInfo.transferLen );
                        ret = read_std_continue( pCmdCtx );
                    }
                    else {
                        /* LBA error */
                        pCmdCtx->totalTransLen = 0;
                        pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
                        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_LBA );
                    }
                }
                else {
                    pCmdCtx->totalTransLen = 0;
                    pCmdCtx->status        = GSTORAGE_CMD__STATUS_PASS;
                    setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
                }
            }
            else {
                /* メディア共通エラー処理 */
                setMediaCommonError( pCmdCtx, 0 );
            }
        }
    }
    else {
        /* invalid command */
        pCmdCtx->totalTransLen = 0;
        pCmdCtx->status        = GSTORAGE_CMD__STATUS_FATAL;
        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_COMMAND );
    }

    return ret;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * read_std_continue
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：Read 処理
 * 入力  ：GSTORAGE_DRV__CmdContext のポインタ
 * 出力  ：GSTORAGE_DRV__CmdContext
 *         ( GSTORAGE_DRV__DeviceContext, GSTORAGE_DRV__UnitContext )
 * 返り値：GSTORAGE__SUCCESS/GSTORAGE__ERROR
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
GSTORAGE__Status read_std_continue( GSTORAGE_DRV__CmdContext *pCmdCtx )
{
    GSTORAGE__Status ret = GSTORAGE__SUCCESS;
    GSTORAGE__Status result;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    result = readMedia( pCmdCtx );
    pCmdCtx->status = result;

    switch( result ){
      case GSTORAGE_CMD__STATUS_PASS:
        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
        break;
      case GSTORAGE_CMD__STATUS_CONTINUE:
        break;
      case GSTORAGE_CMD__STATUS_FAIL:
      case GSTORAGE_CMD__STATUS_FATAL:
      default:
        pCmdCtx->pUnitCtx->preRead.pAccessBuf = pCmdCtx->pUnitCtx->pSendReadBuf[0];
        pCmdCtx->pUnitCtx->preRead.startAddr  = 0x0000000000000000;
        pCmdCtx->pUnitCtx->preRead.nextAddr   = 0x0000000000000000;
        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_READ_SECTOR_ERROR );
        break;
    }

    return ret;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * readMedia
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：Read 処理
 * 入力  ：GSTORAGE_DRV__CmdContext のポインタ
 * 出力  ：GSTORAGE_DRV__CmdContext
 *         ( GSTORAGE_DRV__DeviceContext, GSTORAGE_DRV__UnitContext )
 * 返り値：GSTORAGE__SUCCESS/GSTORAGE__ERROR
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
GSTORAGE__Status readMedia( GSTORAGE_DRV__CmdContext *pCmdCtx )
{
    GSTORAGE__Status ret = GSTORAGE_CMD__STATUS_PASS;
    unsigned int  result;
    unsigned int  transLen;
    unsigned char *pAccessBuf;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    if ( pCmdCtx->pUnitCtx->continueInfo.remainLen > GSTORAGE__SEND_READ_BUF_SIZE ){
        transLen = GSTORAGE__SEND_READ_BUF_SIZE;
    }
    else {
        transLen = pCmdCtx->pUnitCtx->continueInfo.remainLen;
    }

    if ( ( pCmdCtx->pUnitCtx->preRead.startAddr <= pCmdCtx->pUnitCtx->continueInfo.address            ) &&
         ( pCmdCtx->pUnitCtx->preRead.nextAddr  >= pCmdCtx->pUnitCtx->continueInfo.address + transLen ) ){
        pCmdCtx->pData  = pCmdCtx->pUnitCtx->preRead.pAccessBuf;
        pCmdCtx->pData += (unsigned int)( pCmdCtx->pUnitCtx->continueInfo.address
                                     - pCmdCtx->pUnitCtx->preRead.startAddr );
        GS_DBG( GS_OUT__CMD_RW, "preRead ON" );
        result = transLen;
    }
    else {
        pAccessBuf = ( pCmdCtx->pUnitCtx->preRead.pAccessBuf == pCmdCtx->pUnitCtx->pSendReadBuf[ 0 ] ) ?
                     pCmdCtx->pUnitCtx->pSendReadBuf[ 1 ] : pCmdCtx->pUnitCtx->pSendReadBuf[ 0 ];
        storage_drvcnt_setMediaAccess( pCmdCtx->pUnitCtx );
        result = storage_read_cmn( pCmdCtx->pUnitCtx,
                                   pCmdCtx->pUnitCtx->fd,
                                   pAccessBuf,
                                   transLen,
                                   pCmdCtx->pUnitCtx->continueInfo.address );
        pCmdCtx->pData                        = pAccessBuf;
        pCmdCtx->pUnitCtx->preRead.pAccessBuf = pAccessBuf;
        pCmdCtx->pUnitCtx->preRead.startAddr  = pCmdCtx->pUnitCtx->continueInfo.address;
        pCmdCtx->pUnitCtx->preRead.nextAddr   = pCmdCtx->pUnitCtx->continueInfo.address + transLen;
    }

    if ( result == transLen ){
        pCmdCtx->totalTransLen                      += transLen;
        pCmdCtx->pUnitCtx->continueInfo.remainLen   -= transLen;
        pCmdCtx->pUnitCtx->continueInfo.address     += transLen;
        pCmdCtx->pUnitCtx->continueInfo.transferLen += transLen;
        if ( pCmdCtx->pUnitCtx->continueInfo.remainLen == 0 ){
            ret = GSTORAGE_CMD__STATUS_PASS;
        }
        else {
            ret = GSTORAGE_CMD__STATUS_CONTINUE;
        }
    }
    else {
        ret = GSTORAGE_CMD__STATUS_FAIL;
        GS_INF( GS_OUT__CMD_RW, "read error : result=%d", result );
    }

    return ret;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * preRead_std
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：Read 処理
 * 入力  ：GSTORAGE_DRV__CmdContext のポインタ
 * 出力  ：GSTORAGE_DRV__CmdContext
 *         ( GSTORAGE_DRV__DeviceContext, GSTORAGE_DRV__UnitContext )
 * 返り値：GSTORAGE__SUCCESS/GSTORAGE__ERROR
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
GSTORAGE__Status preRead_std( void *pParam )
{
    GSTORAGE__Status          ret = GSTORAGE__SUCCESS;
    GSTORAGE_DRV__UnitContext *pUnitCtx;
    unsigned int     result = 0;
    unsigned int     readSize;
    signed long long readOffset;
    signed long long maxAddr;
    unsigned char    *pAccessBuf;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    pUnitCtx   = ( GSTORAGE_DRV__UnitContext * )pParam;
    readSize   = (pUnitCtx->continueInfo.transferLen > GSTORAGE__SEND_READ_BUF_SIZE)
               ? GSTORAGE__SEND_READ_BUF_SIZE : pUnitCtx->continueInfo.transferLen;
    readOffset = pUnitCtx->preRead.nextAddr;
    maxAddr    = (signed long long)( pUnitCtx->media.lastLBA + 1 )
               * (signed long long)pUnitCtx->media.blockLength;

    GS_DBG( GS_OUT__CMD_RW, "readSize=%d, readOffset=%llu, maxAddr=%llu", readSize, readOffset, maxAddr );

    if ( readOffset >= maxAddr ){
        GS_DBG( GS_OUT__CMD_RW, "preRead : readOffset >= maxAddr, readOffset=%llu", readOffset );
        return ret;
    }
    if ( readOffset + readSize >= maxAddr ){
        readSize = maxAddr - readOffset;
    }
    pAccessBuf = ( pUnitCtx->preRead.pAccessBuf == pUnitCtx->pSendReadBuf[ 0 ] ) ?
                 pUnitCtx->pSendReadBuf[ 1 ] : pUnitCtx->pSendReadBuf[ 0 ];
    storage_drvcnt_setMediaAccess( pUnitCtx );
    result = storage_read_cmn( pUnitCtx,pUnitCtx->fd, pAccessBuf, readSize, readOffset );
    if ( result == readSize ){
        pUnitCtx->preRead.pAccessBuf = pAccessBuf;
        pUnitCtx->preRead.startAddr  = readOffset;
        pUnitCtx->preRead.nextAddr   = readOffset + readSize;
    }

    return ret;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * readCapa_std
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：READ CAPACITY コマンド処理【 Data IN 】
 * 入力  ：GSTORAGE_DRV__CmdContext のポインタ
 * 出力  ：GSTORAGE_DRV__CmdContext
 *         ( GSTORAGE_DRV__DeviceContext, GSTORAGE_DRV__UnitContext )
 * 返り値：GSTORAGE__SUCCESS/GSTORAGE__ERROR
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
GSTORAGE__Status readCapa_std( GSTORAGE_DRV__CmdContext *pCmdCtx )
{
    GSTORAGE_CMD__CdbReadCapa *pCdb;
    GSTORAGE__Status ret = GSTORAGE__SUCCESS;
    unsigned int bufLen = pCmdCtx->expectDatLen;
    unsigned int execLen;
    signed long long lastLba;
    signed long long readCapaLba;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    /* set buffer pointer */
    pCdb = ( GSTORAGE_CMD__CdbReadCapa * )pCmdCtx->pCdb;

    /* drive reset ? */
    if ( pCmdCtx->pUnitCtx->driveReset == GSTORAGE__ON ){
        setDriveReset( pCmdCtx, 0 );
        return ret;
    }
    /* media changed ? */
    if ( pCmdCtx->pUnitCtx->media.changed == GSTORAGE__ON ){
        setMediaChanged( pCmdCtx, 0 );
        return ret;
    }
    /* media eject ? */
    if ( pCmdCtx->pUnitCtx->media.unitStatus == GSTORAGE__UNIT_EJECT ){
        setMediaEjected( pCmdCtx, 0 );
        return ret;
    }
    /* drive busy ? ( = media unknown ? ) */
    if ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_UNKNOWN ){
        setDriveNotReady( pCmdCtx, 0 );
        return ret;
    }

    if ( ( pCmdCtx->dio == DATA_IN ) || ( pCmdCtx->dio == DATA_NON ) ){

        readCapaLba  = ( (unsigned int)pCdb->LBA[ 0 ] << 24 ) |
                       ( (unsigned int)pCdb->LBA[ 1 ] << 16 ) |
                       ( (unsigned int)pCdb->LBA[ 2 ] <<  8 ) |
                       ( (unsigned int)pCdb->LBA[ 3 ]       );
        lastLba    = (signed long long)pCmdCtx->pUnitCtx->media.lastLBA;

        if ( ( ( pCdb->f1_pmi == 0 ) && ( readCapaLba != 0 ) ) || 
             ( pCdb->f1_link ) || ( pCdb->f1_naca ) ) {
            /* Invalid Filed */
            pCmdCtx->totalTransLen = 0;
            pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
            setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_FIELD );
        }
        else if ( readCapaLba > lastLba ) {
            /* LBA out of Range */
            pCmdCtx->totalTransLen = 0;
            pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
            setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_LBA );
        } 
        else {
            if ( ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_READY          ) &&
                ( pCmdCtx->pUnitCtx->media.unitStatus  == GSTORAGE__UNIT_NOT_EJECT_START ) ){
                /* READ CAPACITY データを設定 */
                memcpy( pCmdCtx->pData, &pCmdCtx->pUnitCtx->DatReadCapa, sizeof( GSTORAGE_CMD__DatReadCapa ) );
                execLen = ( sizeof( GSTORAGE_CMD__DatReadCapa ) < bufLen ) ?
                    sizeof( GSTORAGE_CMD__DatReadCapa ) : bufLen;
                if ( bufLen < execLen ){
                    /* PhaseError設定 */
                    pCmdCtx->totalTransLen = 0;
                    pCmdCtx->status = GSTORAGE_CMD__STATUS_FATAL;
                    setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
                }
                else {
                    /* set output data */
                    pCmdCtx->totalTransLen = execLen;
                    pCmdCtx->status        = GSTORAGE_CMD__STATUS_PASS;
                    setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
                }
            }
            else {
                setMediaCommonError( pCmdCtx, 0 );
            }
        }
    }
    else {
        /* invalid command */
        pCmdCtx->totalTransLen = 0;
        pCmdCtx->status        = GSTORAGE_CMD__STATUS_FATAL;
        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_COMMAND );
    }

    return ret;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * readFormatCapa_ms
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：READ FORMAT CAPACITY コマンド処理【 Data IN 】
 * 入力  ：GSTORAGE_DRV__CmdContext のポインタ
 * 出力  ：GSTORAGE_DRV__CmdContext
 *         ( GSTORAGE_DRV__DeviceContext, GSTORAGE_DRV__UnitContext )
 * 返り値：GSTORAGE__SUCCESS/GSTORAGE__ERROR
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
GSTORAGE__Status readFormatCapa_ms( GSTORAGE_DRV__CmdContext *pCmdCtx )
{
    GSTORAGE__Status             ret = GSTORAGE__SUCCESS;
    GSTORAGE_CMD__CdbReadFmtCapa *pCdb;
    GSTORAGE_CMD__DatReadFmtCapa *pData;
    unsigned int blockNum;
    unsigned int bufLen = pCmdCtx->expectDatLen;
    unsigned int localLen;
    unsigned int maxLen;
    unsigned int execLen;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    /* drive reset ? */
    if ( pCmdCtx->pUnitCtx->driveReset == GSTORAGE__ON ){
        setDriveReset( pCmdCtx, 0 );
        return ret;
    }
    /* media changed ? */
    if ( pCmdCtx->pUnitCtx->media.changed == GSTORAGE__ON ){
        setMediaChanged( pCmdCtx, 0 );
        return ret;
    }

    pCdb  = ( GSTORAGE_CMD__CdbReadFmtCapa * )pCmdCtx->pCdb;
    pData = ( GSTORAGE_CMD__DatReadFmtCapa * )pCmdCtx->pData;
    memset( pData, 0x00, sizeof( GSTORAGE_CMD__DatReadFmtCapa ) );

    if ( ( pCmdCtx->dio == DATA_IN ) || ( pCmdCtx->dio == DATA_NON ) ){
        if ( ( pCdb->f1_link == 1 ) || ( pCdb->f1_naca == 1 ) ) {
            /* Invalid Filed */
            pCmdCtx->totalTransLen = 0;
            pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
            setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_FIELD );
        }
        else {
            /* LENGTH の算出 */
            localLen = ( (unsigned int)pCdb->AL[ 0 ] << 8 ) + 
                         (unsigned int)pCdb->AL[ 1 ];
            maxLen   = sizeof( GSTORAGE_CMD__DatReadFmtCapa );
            execLen  = ( localLen < maxLen ) ? localLen : maxLen;
            if ( bufLen < execLen ){
                /* PhaseError設定 */
                pCmdCtx->totalTransLen = 0;
                pCmdCtx->status = GSTORAGE_CMD__STATUS_FATAL;
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
            }
            else {
                if ( ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_READY          ) &&
                    ( pCmdCtx->pUnitCtx->media.unitStatus  == GSTORAGE__UNIT_NOT_EJECT_START ) ){
                    blockNum = GSTORAGE_TRANS_ENDIAN( pCmdCtx->pUnitCtx->media.blocks );
                    pData->f2_curDescript = 0x02;
                }
                else if ( ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_NOT_READY ) ||
                         ( pCmdCtx->pUnitCtx->media.unitStatus  == GSTORAGE__UNIT_EJECT      ) ){
                    /* blockNum は setInitInfo で取得した値を使用 */
                    blockNum = GSTORAGE_TRANS_ENDIAN( pCmdCtx->pUnitCtx->pDrive->blockNum );
                    pData->f2_curDescript = 0x03;
                }
                else if ( ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_UNKNOWN ) ||
                         ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_READY   ) ){
                    /* drive not ready */
                    setDriveNotReady( pCmdCtx, 0 );
                    return ret;
                }
                else {
                    /* media error */
                    pCmdCtx->totalTransLen = 0;
                    pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
                    setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
                    return ret;
                }
                pData->capaListLen = 0x10;
                memcpy( &( pData->curBlockNum ), &blockNum, 4 );
                pData->curBlockLen[ 0 ] = ( pCmdCtx->pUnitCtx->pDrive->blockLength & 0x000000FF );
                pData->curBlockLen[ 1 ] = ( pCmdCtx->pUnitCtx->pDrive->blockLength & 0x0000FF00 ) >>  8;
                pData->curBlockLen[ 2 ] = ( pCmdCtx->pUnitCtx->pDrive->blockLength & 0x00FF0000 ) >> 16;
                memcpy( &( pData->fmtBlockNum ), &blockNum, 4 );
                pData->fmtBlockLen[ 0 ] = ( pCmdCtx->pUnitCtx->pDrive->blockLength & 0x000000FF );
                pData->fmtBlockLen[ 1 ] = ( pCmdCtx->pUnitCtx->pDrive->blockLength & 0x0000FF00 ) >>  8;
                pData->fmtBlockLen[ 2 ] = ( pCmdCtx->pUnitCtx->pDrive->blockLength & 0x00FF0000 ) >> 16;

                /* set output data */
                pCmdCtx->totalTransLen = execLen;
                pCmdCtx->status        = GSTORAGE_CMD__STATUS_PASS;
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
            }
        }
    }
    else {
        /* invalid command */
        pCmdCtx->totalTransLen = 0;
        pCmdCtx->status        = GSTORAGE_CMD__STATUS_FATAL;
        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_COMMAND );
    }

    return ret;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * readFormatCapa_std
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：READ FORMAT CAPACITY コマンド処理【 Data IN 】
 * 入力  ：GSTORAGE_DRV__CmdContext のポインタ
 * 出力  ：GSTORAGE_DRV__CmdContext
 *         ( GSTORAGE_DRV__DeviceContext, GSTORAGE_DRV__UnitContext )
 * 返り値：GSTORAGE__SUCCESS/GSTORAGE__ERROR
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
GSTORAGE__Status readFormatCapa_std( GSTORAGE_DRV__CmdContext *pCmdCtx )
{
    GSTORAGE__Status             ret = GSTORAGE__SUCCESS;
    GSTORAGE_CMD__CdbReadFmtCapa *pCdb;
    GSTORAGE_CMD__DatReadFmtCapa *pData;
    unsigned int blockNum;
    unsigned int bufLen = pCmdCtx->expectDatLen;
    unsigned int localLen;
    unsigned int maxLen;
    unsigned int execLen;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    /* drive reset ? */
    if ( pCmdCtx->pUnitCtx->driveReset == GSTORAGE__ON ){
        setDriveReset( pCmdCtx, 0 );
        return ret;
    }
    /* media changed ? */
    if ( pCmdCtx->pUnitCtx->media.changed == GSTORAGE__ON ){
        setMediaChanged( pCmdCtx, 0 );
        return ret;
    }

    pCdb  = ( GSTORAGE_CMD__CdbReadFmtCapa * )pCmdCtx->pCdb;
    pData = ( GSTORAGE_CMD__DatReadFmtCapa * )pCmdCtx->pData;
    memset( pData, 0x00, sizeof( GSTORAGE_CMD__DatReadFmtCapa ) );

    if ( ( pCmdCtx->dio == DATA_IN ) || ( pCmdCtx->dio == DATA_NON ) ){
        if ( ( pCdb->f1_link == 1 ) || ( pCdb->f1_naca == 1 ) ) {
            /* Invalid Filed */
            pCmdCtx->totalTransLen = 0;
            pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
            setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_FIELD );
        }
        else {
            /* LENGTH の算出 */
            localLen = ( (unsigned int)pCdb->AL[ 0 ] << 8 ) + 
                         (unsigned int)pCdb->AL[ 1 ];
            maxLen   = sizeof( GSTORAGE_CMD__DatReadFmtCapa );
            execLen  = ( localLen < maxLen ) ? localLen : maxLen;
            if ( bufLen < execLen ){
                /* PhaseError設定 */
                pCmdCtx->totalTransLen = 0;
                pCmdCtx->status = GSTORAGE_CMD__STATUS_FATAL;
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
            }
            else {
                if ( ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_READY          ) &&
                    ( pCmdCtx->pUnitCtx->media.unitStatus  == GSTORAGE__UNIT_NOT_EJECT_START ) ){
                    blockNum = GSTORAGE_TRANS_ENDIAN( pCmdCtx->pUnitCtx->media.blocks );
                    pData->f2_curDescript = 0x02;
                }
                else if ( ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_NOT_READY ) ||
                         ( pCmdCtx->pUnitCtx->media.unitStatus  == GSTORAGE__UNIT_EJECT      ) ){
                    /* HDD/Hardの場合、FailでSense=NOT MEDIA */
                    pCmdCtx->totalTransLen = 0;
                    pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
                    setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_MEDIA );
                    return ret;
                }
                else if ( ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_UNKNOWN ) ||
                         ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_READY   ) ){
                    /* drive not ready */
                    setDriveNotReady( pCmdCtx, 0 );
                    return ret;
                }
                else {
                    /* media error */
                    pCmdCtx->totalTransLen = 0;
                    pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
                    setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
                    return ret;
                }
                pData->capaListLen = 0x10;
                memcpy( &( pData->curBlockNum ), &blockNum, 4 );
                pData->curBlockLen[ 0 ] = ( pCmdCtx->pUnitCtx->pDrive->blockLength & 0x000000FF );
                pData->curBlockLen[ 1 ] = ( pCmdCtx->pUnitCtx->pDrive->blockLength & 0x0000FF00 ) >>  8;
                pData->curBlockLen[ 2 ] = ( pCmdCtx->pUnitCtx->pDrive->blockLength & 0x00FF0000 ) >> 16;
                memcpy( &( pData->fmtBlockNum ), &blockNum, 4 );
                pData->fmtBlockLen[ 0 ] = ( pCmdCtx->pUnitCtx->pDrive->blockLength & 0x000000FF );
                pData->fmtBlockLen[ 1 ] = ( pCmdCtx->pUnitCtx->pDrive->blockLength & 0x0000FF00 ) >>  8;
                pData->fmtBlockLen[ 2 ] = ( pCmdCtx->pUnitCtx->pDrive->blockLength & 0x00FF0000 ) >> 16;

                /* set output data */
                pCmdCtx->totalTransLen = execLen;
                pCmdCtx->status        = GSTORAGE_CMD__STATUS_PASS;
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
            }
        }
    }
    else {
        /* invalid command */
        pCmdCtx->totalTransLen = 0;
        pCmdCtx->status        = GSTORAGE_CMD__STATUS_FATAL;
        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_COMMAND );
    }

    return ret;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * requestSense_std
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：REQUEST SENSE コマンド処理【 Data IN 】
 * 入力  ：GSTORAGE_DRV__CmdContext のポインタ
 * 出力  ：GSTORAGE_DRV__CmdContext
 *         ( GSTORAGE_DRV__DeviceContext, GSTORAGE_DRV__UnitContext )
 * 返り値：GSTORAGE__SUCCESS/GSTORAGE__ERROR
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
GSTORAGE__Status requestSense_std( GSTORAGE_DRV__CmdContext *pCmdCtx )
{
    GSTORAGE__Status              ret = GSTORAGE__SUCCESS;
    GSTORAGE_CMD__CdbRequestSense *pCdb;
    GSTORAGE_CMD__DatRequestSense *pData;
    unsigned int bufLen = pCmdCtx->expectDatLen;
    unsigned int execLen;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    pCdb  = ( GSTORAGE_CMD__CdbRequestSense * )pCmdCtx->pCdb;
    pData = ( GSTORAGE_CMD__DatRequestSense * )pCmdCtx->pData;
    memset( pData, 0x00, sizeof( GSTORAGE_CMD__DatRequestSense ) );

    if ( ( pCmdCtx->dio == DATA_IN ) || ( pCmdCtx->dio == DATA_NON ) ){
    
        if ( ( pCdb->f1_link == 1 ) || ( pCdb->f1_naca == 1 ) ) {
            /* Invalid Filed */
            pCmdCtx->totalTransLen = 0;
            pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
            setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_FIELD );
        }
        else {
            execLen  = ( pCdb->AL < GSTORAGE_LEN__REQUEST_SENSE_STD ) ?
                pCdb->AL : GSTORAGE_LEN__REQUEST_SENSE_STD;
            if ( bufLen < execLen ){
                /* PhaseError設定 */
                pCmdCtx->totalTransLen = 0;
                pCmdCtx->status = GSTORAGE_CMD__STATUS_FATAL;
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
            }
            else {
                /* REQUEST SENSE データを作成 */
                pData->f7_errorCode             = 0x70;
                pData->f4_senseKey              = pCmdCtx->pUnitCtx->sense.sense;
                pData->addSenseLen              = 0x0A;
                pData->addSenseCode             = pCmdCtx->pUnitCtx->sense.asc;
                pData->addSenseQualifier        = pCmdCtx->pUnitCtx->sense.ascq;

                /* Check Reset Occured */
                if ( getSense( &pCmdCtx->pUnitCtx->sense ) == GSTORAGE_CMD__SK_RESET_OCCURRED ){
                    pCmdCtx->pUnitCtx->driveReset = GSTORAGE__OFF;
                }
                else {
                    /* Check Media Changed */
                    if ( getSense( &pCmdCtx->pUnitCtx->sense ) == GSTORAGE_CMD__SK_MEDIA_CHANGED ){
                        pCmdCtx->pUnitCtx->media.changed = GSTORAGE__OFF;
                    }
                }

                /* set output data */
                pCmdCtx->totalTransLen = execLen;
                pCmdCtx->status        = GSTORAGE_CMD__STATUS_PASS;
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
            }
        }
    }
    else {
        /* invalid command */
        pCmdCtx->totalTransLen = 0;
        pCmdCtx->status        = GSTORAGE_CMD__STATUS_FATAL;
        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_COMMAND );
    }

    return ret;
}

/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * startStop_std
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：START STOP コマンド処理【 Data NON 】
 * 入力  ：GSTORAGE_DRV__CmdContext のポインタ
 * 出力  ：GSTORAGE_DRV__CmdContext
 *         ( GSTORAGE_DRV__DeviceContext, GSTORAGE_DRV__UnitContext )
 * 返り値：GSTORAGE__SUCCESS/GSTORAGE__ERROR
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
GSTORAGE__Status startStop_std( GSTORAGE_DRV__CmdContext *pCmdCtx )
{
    GSTORAGE_CMD__CdbStartStop *pCdb;
    GSTORAGE__Status           ret = GSTORAGE__SUCCESS;
    signed int result = 0;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    /* drive reset ? */
    if ( pCmdCtx->pUnitCtx->driveReset == GSTORAGE__ON ){
        setDriveReset( pCmdCtx, 0 );
        return ret;
    }
    /* media changed ? */
    if ( pCmdCtx->pUnitCtx->media.changed == GSTORAGE__ON ){
        setMediaChanged( pCmdCtx, 0 );
        return ret;
    }
    /* drive lock status ? */
    if ( pCmdCtx->pUnitCtx->currentLock == GSTORAGE__ON ){
        pCmdCtx->totalTransLen = 0;
        pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_MEDIUM_REMOVAL_PREVENTED );
        return ret;
    }

    pCdb = ( GSTORAGE_CMD__CdbStartStop * )pCmdCtx->pCdb;

    /* Invalid Field */
    if ( ( pCdb->f1_link == 1 ) || ( pCdb->f1_naca == 1 ) ) {
        pCmdCtx->totalTransLen = 0;
        pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_FIELD );
        return ret;
    }

    /* Power Condition */
    if ( pCdb->f4_power_condition != 0 ) {
        pCmdCtx->totalTransLen = 0;
        pCmdCtx->status        = GSTORAGE_CMD__STATUS_PASS;
        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
        return ret;
    }

    if ( pCmdCtx->pUnitCtx->pDrive->drvType == GSTORAGE__DRIVE_ID_MS
         || pCmdCtx->pUnitCtx->pDrive->drvType == GSTORAGE__DRIVE_ID_MSB ){
        result = startStop_ms( pCmdCtx );
    }
    else if ( ( pCmdCtx->pUnitCtx->pDrive->drvType == GSTORAGE__DRIVE_ID_HDD  ) ||
              ( pCmdCtx->pUnitCtx->pDrive->drvType == GSTORAGE__DRIVE_ID_CF   ) ||
              ( pCmdCtx->pUnitCtx->pDrive->drvType == GSTORAGE__DRIVE_ID_NAND ) ||
              ( pCmdCtx->pUnitCtx->pDrive->drvType == GSTORAGE__DRIVE_ID_SXS  ) ||
              ( pCmdCtx->pUnitCtx->pDrive->drvType == GSTORAGE__DRIVE_ID_SXS_ADPT  ) ||
              ( pCmdCtx->pUnitCtx->pDrive->drvType == GSTORAGE__DRIVE_ID_MMC  ) ){
        result = startStop_ata( pCmdCtx );
    }
    else {
        GS_ERR( GS_OUT__CMD_STD, "unsupported drive type [opcode=%d]", pCmdCtx->pCdb->opCode );
    }
    GS_DBG( GS_OUT__CMD_STD, "media.changed=%d, media.mediaStatus=%d, media.unitStatus=%d",
            pCmdCtx->pUnitCtx->media.changed, pCmdCtx->pUnitCtx->media.mediaStatus, pCmdCtx->pUnitCtx->media.unitStatus );

    return ret;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * startStop_ms
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：START STOP コマンド処理
 * 入力  ：GSTORAGE_DRV__CmdContext のポインタ
 * 出力  ：GSTORAGE_DRV__CmdContext
 *         ( GSTORAGE_DRV__DeviceContext, GSTORAGE_DRV__UnitContext )
 * 返り値：GSTORAGE__SUCCESS/GSTORAGE__ERROR
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
GSTORAGE__Status startStop_ms( GSTORAGE_DRV__CmdContext *pCmdCtx )
{
    GSTORAGE_CMD__CdbStartStop *pCdb;
    signed int result = 0;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    pCdb = ( GSTORAGE_CMD__CdbStartStop * )pCmdCtx->pCdb;

    /* 状態遷移設定 */
    if ( pCdb->f1_start == 0 ){
        if ( pCdb->f1_loEj == 0 ){
            if ( pCmdCtx->pUnitCtx->media.unitStatus == GSTORAGE__UNIT_NOT_EJECT_START ){
                if ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_READY ){
                    pCmdCtx->pUnitCtx->media.unitStatus = GSTORAGE__UNIT_NOT_EJECT_STOP;
                }
            }
        }
        else {
            if ( pCmdCtx->pUnitCtx->media.unitStatus != GSTORAGE__UNIT_EJECT ){
                pCmdCtx->pUnitCtx->media.unitStatus = GSTORAGE__UNIT_EJECT;
                if ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_READY ){
                    pCmdCtx->pUnitCtx->media.changed    = GSTORAGE__ON;
                }
            }
        }
    }
    else {
        if ( pCmdCtx->pUnitCtx->media.unitStatus == GSTORAGE__UNIT_NOT_EJECT_STOP ){
            if ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_READY ){
                pCmdCtx->pUnitCtx->media.unitStatus = GSTORAGE__UNIT_NOT_EJECT_START;
            }
        }
    }
    /* 応答設定 */
    if ( ( pCdb->f1_start == 0 ) && ( pCdb->f1_loEj == 1 ) ){
        pCmdCtx->totalTransLen = 0;
        pCmdCtx->status        = GSTORAGE_CMD__STATUS_PASS;
        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
    }
    else if ( pCmdCtx->pUnitCtx->media.unitStatus == GSTORAGE__UNIT_EJECT ){
        setMediaEjected( pCmdCtx, 0 );
    }
    else {
        switch( pCmdCtx->pUnitCtx->media.mediaStatus ){
          case GSTORAGE__MEDIA_UNKNOWN :
            setDriveNotReady( pCmdCtx, 0 );
            break;
          case GSTORAGE__MEDIA_NOT_READY :
            setMediaEjected( pCmdCtx, 0 );
            break;
          case GSTORAGE__MEDIA_READY :
            pCmdCtx->totalTransLen = 0;
            pCmdCtx->status        = GSTORAGE_CMD__STATUS_PASS;
            setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
            break;
          case GSTORAGE__MEDIA_ERROR :
            pCmdCtx->totalTransLen = 0;
            pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
            setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
            break;
        }
    }

    return result;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * startStop_ata
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：START STOP コマンド処理
 * 入力  ：GSTORAGE_DRV__CmdContext のポインタ
 * 出力  ：GSTORAGE_DRV__CmdContext
 *         ( GSTORAGE_DRV__DeviceContext, GSTORAGE_DRV__UnitContext )
 * 返り値：GSTORAGE__SUCCESS/GSTORAGE__ERROR
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
GSTORAGE__Status startStop_ata( GSTORAGE_DRV__CmdContext *pCmdCtx )
{
    GSTORAGE_CMD__CdbStartStop *pCdb;
    int          result = 0;
    int          result_flush = 0;
    struct file  *filp;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    pCdb = ( GSTORAGE_CMD__CdbStartStop * )pCmdCtx->pCdb;
    filp = pCmdCtx->pUnitCtx->fd;

    if ( pCdb->f1_immed == 1 ){
        /* This command is ineffective */
        pCmdCtx->totalTransLen = 0;
        pCmdCtx->status        = GSTORAGE_CMD__STATUS_PASS;
        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
        return 0;
    }

    /* Flush Cache */
    if ( pCmdCtx->pUnitCtx->pDrive->flushEnable ){
        if ( ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_READY          ) &&
            ( pCmdCtx->pUnitCtx->media.unitStatus  == GSTORAGE__UNIT_NOT_EJECT_START ) ){
            result_flush = flushCache_ata_cmn( pCmdCtx->pUnitCtx, filp );
            if ( result_flush ){
                GS_INF( GS_OUT__CMD_STD, "ioctl error (STOP)." );
            }
        }
    }
        
    /* 状態遷移設定 */
    if ( pCdb->f1_start == 0 ){
        result = standbyImmediate_ata_cmn( pCmdCtx->pUnitCtx, filp );
        if ( result ){
            GS_INF( GS_OUT__CMD_STD, "ioctl error (STOP)." );
        }
        if ( pCdb->f1_loEj == 0 ){
            if ( pCmdCtx->pUnitCtx->media.unitStatus == GSTORAGE__UNIT_NOT_EJECT_START ){
                if ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_READY ){
                    pCmdCtx->pUnitCtx->media.unitStatus = GSTORAGE__UNIT_NOT_EJECT_STOP;
                }
            }
        }
        else {
            if ( pCmdCtx->pUnitCtx->media.unitStatus != GSTORAGE__UNIT_EJECT ){
                pCmdCtx->pUnitCtx->media.unitStatus = GSTORAGE__UNIT_EJECT;
                if ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_READY ){
                    pCmdCtx->pUnitCtx->media.changed    = GSTORAGE__ON;
                }
            }
        }
    }
    else {
        result = idleImmediate_ata_cmn( pCmdCtx->pUnitCtx, filp );
        if ( result ){
            GS_INF( GS_OUT__CMD_STD, "ioctl error (START)." );
        }
        if ( pCmdCtx->pUnitCtx->media.unitStatus == GSTORAGE__UNIT_NOT_EJECT_STOP ){
            if ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_READY ){
                pCmdCtx->pUnitCtx->media.unitStatus = GSTORAGE__UNIT_NOT_EJECT_START;
            }
        }
    }
    /* 応答設定 */
    if ( ( pCdb->f1_start == 0 ) && ( pCdb->f1_loEj == 1 ) ){
        pCmdCtx->totalTransLen = 0;
        pCmdCtx->status        = GSTORAGE_CMD__STATUS_PASS;
        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
    }
    else if ( pCmdCtx->pUnitCtx->media.unitStatus == GSTORAGE__UNIT_EJECT ){
        setMediaEjected( pCmdCtx, 0 );
    }
    else {
        switch( pCmdCtx->pUnitCtx->media.mediaStatus ){
          case GSTORAGE__MEDIA_UNKNOWN :
            setDriveNotReady( pCmdCtx, 0 );
            break;
          case GSTORAGE__MEDIA_NOT_READY :
            setMediaEjected( pCmdCtx, 0 );
            break;
          case GSTORAGE__MEDIA_READY :
            pCmdCtx->totalTransLen = 0;
            if ( ( result == GSTORAGE__SUCCESS ) && ( result_flush == GSTORAGE__SUCCESS ) ){
                pCmdCtx->status = GSTORAGE_CMD__STATUS_PASS;
            }
            else {
                if ( result == GSTORAGE__SUCCESS ) {
                    result = result_flush;
                }
                pCmdCtx->status = GSTORAGE_CMD__STATUS_FAIL;
            }
            setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
            break;
          case GSTORAGE__MEDIA_ERROR :
            pCmdCtx->totalTransLen = 0;
            pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
            setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
            break;
        }
    }

    return result;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * synchronize_cache_std
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：SYNCHRONIZE CACHE10 コマンド処理【 Data NON 】
 * 入力  ：GSTORAGE_DRV__CmdContext のポインタ
 * 出力  ：GSTORAGE_DRV__CmdContext
 *         ( GSTORAGE_DRV__DeviceContext, GSTORAGE_DRV__UnitContext )
 * 返り値：GSTORAGE__SUCCESS/GSTORAGE__ERROR
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
GSTORAGE__Status synchronize_cache_std( GSTORAGE_DRV__CmdContext *pCmdCtx )
{
    GSTORAGE__Status ret = GSTORAGE__SUCCESS;
    GSTORAGE_CMD__CdbSynchronizeCache    *pCdb;
    struct file  *filp;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    pCdb = ( GSTORAGE_CMD__CdbSynchronizeCache * )pCmdCtx->pCdb;
    filp = pCmdCtx->pUnitCtx->fd;

    /* drive reset ? */
    if ( pCmdCtx->pUnitCtx->driveReset == GSTORAGE__ON ){
        setDriveReset( pCmdCtx, 0 );
        return ret;
    }
    /* media changed ? */
    if ( pCmdCtx->pUnitCtx->media.changed == GSTORAGE__ON ){
        setMediaChanged( pCmdCtx, 0);
        return ret;
    }
    /* media eject ? */
    if ( pCmdCtx->pUnitCtx->media.unitStatus == GSTORAGE__UNIT_EJECT ){
        setMediaEjected( pCmdCtx, 0 );
        return ret;
    }
    /* drive busy ? ( = media unknown ? ) */
    if ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_UNKNOWN ){
        setDriveNotReady( pCmdCtx, 0 );
        return ret;
    }

    /* Invalid Field ? */
    if ( ( pCdb->f1_link == 1 ) || ( pCdb->f1_naca == 1 ) ) {
        pCmdCtx->totalTransLen = 0;
        pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_FIELD );
        return ret;
    }

    /* This command is ineffective */
    pCmdCtx->totalTransLen = 0;
    pCmdCtx->status        = GSTORAGE_CMD__STATUS_PASS;
    setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
    return ret;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * testUnit_std
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：TEST UNIT READY コマンド処理【 Data NON 】
 * 入力  ：GSTORAGE_DRV__CmdContext のポインタ
 * 出力  ：GSTORAGE_DRV__CmdContext
 *         ( GSTORAGE_DRV__DeviceContext, GSTORAGE_DRV__UnitContext )
 * 返り値：GSTORAGE__SUCCESS/GSTORAGE__ERROR
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
GSTORAGE__Status testUnit_std( GSTORAGE_DRV__CmdContext *pCmdCtx )
{
    GSTORAGE_CMD__CdbTestUnit *pCdb;
    GSTORAGE__Status ret = GSTORAGE__SUCCESS;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    pCdb = ( GSTORAGE_CMD__CdbTestUnit * )pCmdCtx->pCdb;

    /* drive reset ? */
    if ( pCmdCtx->pUnitCtx->driveReset == GSTORAGE__ON ){
        setDriveReset( pCmdCtx, 0 );
        return ret;
    }
    /* media changed ? */
    if ( pCmdCtx->pUnitCtx->media.changed == GSTORAGE__ON ){
        setMediaChanged( pCmdCtx, 0);
        return ret;
    }
    /* media eject ? */
    if ( pCmdCtx->pUnitCtx->media.unitStatus == GSTORAGE__UNIT_EJECT ){
        setMediaEjected( pCmdCtx, 0 );
        return ret;
    }
    /* drive busy ? ( = media unknown ? ) */
    if ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_UNKNOWN ){
        setDriveNotReady( pCmdCtx, 0 );
        return ret;
    }

    /* Invalid Field? */
    if ( ( pCdb->f1_link == 1 ) || ( pCdb->f1_naca == 1 ) ) {
        pCmdCtx->totalTransLen = 0;
        pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_FIELD );
        return ret;
    }

    if ( ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_READY          ) &&
         ( pCmdCtx->pUnitCtx->media.unitStatus  == GSTORAGE__UNIT_NOT_EJECT_START ) ){
        /* set output data */
        pCmdCtx->totalTransLen = 0;
        pCmdCtx->status        = GSTORAGE_CMD__STATUS_PASS;
        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
    }
    else {
        setMediaCommonError( pCmdCtx, 0 );
    }

    return ret;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * verify_std
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：VERIFY コマンド処理【 Data NON 】
 * 入力  ：GSTORAGE_DRV__CmdContext のポインタ
 * 出力  ：GSTORAGE_DRV__CmdContext
 *         ( GSTORAGE_DRV__DeviceContext, GSTORAGE_DRV__UnitContext )
 * 返り値：GSTORAGE__SUCCESS/GSTORAGE__ERROR
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
GSTORAGE__Status verify_std( GSTORAGE_DRV__CmdContext *pCmdCtx )
{
    GSTORAGE__Status        ret = GSTORAGE__SUCCESS;
    GSTORAGE_CMD__CdbVerify *pCdb;
    signed long long lastLba;
    signed long long verifyLba;
    signed long long verifyLen;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    /* drive reset ? */
    if ( pCmdCtx->pUnitCtx->driveReset == GSTORAGE__ON ){
        setDriveReset( pCmdCtx, 0 );
        return ret;
    }
    /* media changed ? */
    if ( pCmdCtx->pUnitCtx->media.changed == GSTORAGE__ON ){
        setMediaChanged( pCmdCtx, 0);
        return ret;
    }
    /* media eject ? */
    if ( pCmdCtx->pUnitCtx->media.unitStatus == GSTORAGE__UNIT_EJECT ){
        setMediaEjected( pCmdCtx, 0 );
        return ret;
    }
    /* drive busy ? ( = media unknown ? ) */
    if ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_UNKNOWN ){
        setDriveNotReady( pCmdCtx, 0 );
        return ret;
    }

    pCdb = ( GSTORAGE_CMD__CdbVerify * )pCmdCtx->pCdb;

    if ( ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_READY          ) &&
         ( pCmdCtx->pUnitCtx->media.unitStatus  == GSTORAGE__UNIT_NOT_EJECT_START ) ){
        if ( ( pCdb->f1_byteChk == 0 ) && 
             ( pCdb->f1_dpo == 0 ) && 
             //( ( pCdb->f3_vrprotect == 0 ) || ( pCdb->f3_vrprotect > 4 ) ) && 
             ( pCdb->f1_link == 0 ) && ( pCdb->f1_naca == 0 ) ){
            lastLba   = (unsigned int)pCmdCtx->pUnitCtx->media.lastLBA;
            verifyLba = ( (unsigned int)( ( GSTORAGE_CMD__CdbVerify * )pCdb )->LBA[ 0 ] << 24 )
                      | ( (unsigned int)( ( GSTORAGE_CMD__CdbVerify * )pCdb )->LBA[ 1 ] << 16 )
                      | ( (unsigned int)( ( GSTORAGE_CMD__CdbVerify * )pCdb )->LBA[ 2 ] <<  8 )
                      | ( (unsigned int)( ( GSTORAGE_CMD__CdbVerify * )pCdb )->LBA[ 3 ] );
            verifyLen = ( (unsigned int)pCdb->verifyLen[ 0 ] << 8 )
                      | ( (unsigned int)pCdb->verifyLen[ 1 ] );
            if ( ( verifyLba + verifyLen <= lastLba + 1 ) &&
                 ( verifyLba             <= lastLba     ) ){
                /* set output data */
                pCmdCtx->totalTransLen = 0;
                pCmdCtx->status        = GSTORAGE_CMD__STATUS_PASS;
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
            }
            else {
                /* LBA エラー */
                pCmdCtx->totalTransLen = 0;
                pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_LBA );
            }
        }
        else {
            /* invalid field */
            pCmdCtx->totalTransLen = 0;
            pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
            setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_FIELD );
        }
    }
    else {
        setMediaCommonError( pCmdCtx, 0 );
    }

    return GSTORAGE__SUCCESS;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * write_std
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：WRITE10/WRITE12 コマンド処理【 Data OUT 】
 * 入力  ：GSTORAGE_DRV__CmdContext のポインタ
 * 出力  ：GSTORAGE_DRV__CmdContext
 *         ( GSTORAGE_DRV__DeviceContext, GSTORAGE_DRV__UnitContext )
 * 返り値：GSTORAGE__SUCCESS/GSTORAGE__ERROR
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
GSTORAGE__Status write_std( GSTORAGE_DRV__CmdContext *pCmdCtx )
{
    GSTORAGE__Status ret = GSTORAGE__SUCCESS;
    GSTORAGE__Cdb    *pCdb = pCmdCtx->pCdb;
    signed long long transLen;
    signed long long bufLen = (signed long long)pCmdCtx->expectDatLen;
    signed long long maxLen;
    signed long long execLen;
    signed long long lastLba;
    signed long long writeLba;
    signed long long writeSize;
    signed long long writeOffset;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    /* drive reset ? */
    if ( pCmdCtx->pUnitCtx->driveReset == GSTORAGE__ON ){
        setDriveReset( pCmdCtx, 0 );
        return ret;
    }
    /* media changed ? */
    if ( pCmdCtx->pUnitCtx->media.changed == GSTORAGE__ON ){
        setMediaChanged( pCmdCtx, 0);
        return ret;
    }
    /* media eject ? */
    if ( pCmdCtx->pUnitCtx->media.unitStatus == GSTORAGE__UNIT_EJECT ){
        setMediaEjected( pCmdCtx, 0 );
        return ret;
    }
    /* drive busy ? ( = media unknown ? ) */
    if ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_UNKNOWN ){
        setDriveNotReady( pCmdCtx, 0 );
        return ret;
    }

    if ( ( pCmdCtx->dio == DATA_OUT ) || ( pCmdCtx->dio == DATA_NON ) ){
    
        /* Invalid Field ? */
        if ( pCmdCtx->pCdb->opCode == GSTORAGE_CMD__OPCODE_WRITE10 ){
            /* WRITE(10) */
            if ( ( ( ( GSTORAGE_CMD__CdbWrite10 * )pCdb )->f1_fua_nv != 0 ) ||
                ( ( ( GSTORAGE_CMD__CdbWrite10 * )pCdb )->f1_fua != 0 ) ||
                ( ( ( GSTORAGE_CMD__CdbWrite10 * )pCdb )->f1_dpo != 0 ) ||
                //( ( ( ( GSTORAGE_CMD__CdbWrite10 * )pCdb )->f3_wrprotect >= 1 ) &&
                //  ( ( ( GSTORAGE_CMD__CdbWrite10 * )pCdb )->f3_wrprotect <= 4 ) ) ||
                ( ( ( GSTORAGE_CMD__CdbWrite10 * )pCdb )->f1_link != 0 ) ||
                ( ( ( GSTORAGE_CMD__CdbWrite10 * )pCdb )->f1_naca != 0 ) ) {
                pCmdCtx->totalTransLen = 0;
                pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_FIELD );
                return ret;
            }
        }
        else {
            /* WRITE(12) */
            if ( ( ( ( GSTORAGE_CMD__CdbWrite12 * )pCdb )->f1_fua_nv != 0 ) ||
                ( ( ( GSTORAGE_CMD__CdbWrite12 * )pCdb )->f1_fua != 0 ) ||
                ( ( ( GSTORAGE_CMD__CdbWrite12 * )pCdb )->f1_dpo != 0 ) ||
                //( ( ( ( GSTORAGE_CMD__CdbWrite12 * )pCdb )->f3_wrprotect >= 1 ) && 
                //  ( ( ( GSTORAGE_CMD__CdbWrite12 * )pCdb )->f3_wrprotect <= 4 ) ) ||
                ( ( ( GSTORAGE_CMD__CdbWrite12 * )pCdb )->f1_link != 0 ) ||
                ( ( ( GSTORAGE_CMD__CdbWrite12 * )pCdb )->f1_naca != 0 ) ) {
                pCmdCtx->totalTransLen = 0;
                pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_FIELD );
                return ret;
            }
        }    

        if ( pCmdCtx->pCdb->opCode == GSTORAGE_CMD__OPCODE_WRITE10 ){
            transLen = ( (unsigned int)( ( GSTORAGE_CMD__CdbWrite10 * )pCdb )->transLen[ 0 ] << 8 )
                     | ( (unsigned int)( ( GSTORAGE_CMD__CdbWrite10 * )pCdb )->transLen[ 1 ]      );
            writeLba = ( (unsigned int)( ( GSTORAGE_CMD__CdbWrite10 * )pCdb )->LBA[ 0 ] << 24 )
                     | ( (unsigned int)( ( GSTORAGE_CMD__CdbWrite10 * )pCdb )->LBA[ 1 ] << 16 )
                     | ( (unsigned int)( ( GSTORAGE_CMD__CdbWrite10 * )pCdb )->LBA[ 2 ] <<  8 )
                     | ( (unsigned int)( ( GSTORAGE_CMD__CdbWrite10 * )pCdb )->LBA[ 3 ]       );
        }
        else {
            transLen = ( (unsigned int)( ( GSTORAGE_CMD__CdbWrite12 * )pCdb )->transLen[ 0 ] << 24 )
                     | ( (unsigned int)( ( GSTORAGE_CMD__CdbWrite12 * )pCdb )->transLen[ 1 ] << 16 )
                     | ( (unsigned int)( ( GSTORAGE_CMD__CdbWrite12 * )pCdb )->transLen[ 2 ] <<  8 )
                     | ( (unsigned int)( ( GSTORAGE_CMD__CdbWrite12 * )pCdb )->transLen[ 3 ]       );
            writeLba = ( (unsigned int)( ( GSTORAGE_CMD__CdbWrite12 * )pCdb )->LBA[ 0 ] << 24 )
                     | ( (unsigned int)( ( GSTORAGE_CMD__CdbWrite12 * )pCdb )->LBA[ 1 ] << 16 )
                     | ( (unsigned int)( ( GSTORAGE_CMD__CdbWrite12 * )pCdb )->LBA[ 2 ] <<  8 )
                     | ( (unsigned int)( ( GSTORAGE_CMD__CdbWrite12 * )pCdb )->LBA[ 3 ]       );
        }
        lastLba    = (signed long long)pCmdCtx->pUnitCtx->media.lastLBA;
        maxLen      = (signed long long)pCmdCtx->pUnitCtx->media.blocks
                    * (signed long long)pCmdCtx->pUnitCtx->media.blockLength;
        writeSize   = transLen * (signed long long)pCmdCtx->pUnitCtx->media.blockLength;
        writeOffset = writeLba * (signed long long)pCmdCtx->pUnitCtx->media.blockLength;
        execLen     = ( writeSize < maxLen ) ? writeSize : maxLen;
        if ( bufLen < execLen ){
            /* PhaseError設定 */
            pCmdCtx->totalTransLen = 0;
            pCmdCtx->status = GSTORAGE_CMD__STATUS_FATAL;
            setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
        }
        else {
            if ( ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_READY          ) &&
                 ( pCmdCtx->pUnitCtx->media.unitStatus  == GSTORAGE__UNIT_NOT_EJECT_START ) ){
                if ( ( pCmdCtx->dio == DATA_OUT ) && ( transLen > 0 ) ) {
                    /* check Access LBA */
                    if ( writeLba + transLen <= lastLba + 1 ){
                        if ( ( pCmdCtx->pUnitCtx->pDrive->forceWP    == 0 ) &&
                             ( pCmdCtx->pUnitCtx->media.writeProtect == 0 ) ){
                            pCmdCtx->pUnitCtx->continueInfo.remainLen   = execLen;
                            pCmdCtx->pUnitCtx->continueInfo.address     = writeOffset;
                            pCmdCtx->pUnitCtx->continueInfo.transferLen = 0;
                            pCmdCtx->pUnitCtx->continueInfo.pProc       = write_std_continue;
                            GS_DBG( GS_OUT__CMD_RW, "bufLen=%llu, execLen=%llu, address=%llu, remainLen=%d, transferLen=%d",
                                    bufLen, execLen, pCmdCtx->pUnitCtx->continueInfo.address,
                                    pCmdCtx->pUnitCtx->continueInfo.remainLen, pCmdCtx->pUnitCtx->continueInfo.transferLen );
                            ret = write_std_continue( pCmdCtx );
                            /* 先読み情報のクリア（できれば範囲チェックも入れる） */
                            pCmdCtx->pUnitCtx->preRead.pAccessBuf = pCmdCtx->pUnitCtx->pSendReadBuf[0];
                            pCmdCtx->pUnitCtx->preRead.startAddr  = 0x0000000000000000;
                            pCmdCtx->pUnitCtx->preRead.nextAddr   = 0x0000000000000000;
                        }
                        else {
                            /* write protected error */
                            pCmdCtx->totalTransLen = 0;
                            pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
                            setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_WRITE_PROTECTED );
                        }
                    }
                    else {
                        /* LBA error */
                        pCmdCtx->totalTransLen = 0;
                        pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
                        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_LBA );
                    }
                }
                else {
                    pCmdCtx->totalTransLen = 0;
                    pCmdCtx->status        = GSTORAGE_CMD__STATUS_PASS;
                    setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
                }
            }
            else {
                setMediaCommonError( pCmdCtx, 0 );
            }
        }
    }
    else {
        /* invalid command */
        pCmdCtx->totalTransLen = 0;
        pCmdCtx->status        = GSTORAGE_CMD__STATUS_FATAL;
        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_COMMAND );
    }

    return ret;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * write_std_continue
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：Write コマンド処理
 * 入力  ：GSTORAGE_DRV__CmdContext のポインタ
 * 出力  ：GSTORAGE_DRV__CmdContext
 *         ( GSTORAGE_DRV__DeviceContext, GSTORAGE_DRV__UnitContext )
 * 返り値：GSTORAGE__SUCCESS/GSTORAGE__ERROR
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
GSTORAGE__Status write_std_continue( GSTORAGE_DRV__CmdContext *pCmdCtx )
{
    GSTORAGE__Status ret = GSTORAGE__SUCCESS;
    GSTORAGE__Status result;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    result = writeMedia( pCmdCtx );
    pCmdCtx->status = result;

    switch( result ){
      case GSTORAGE_CMD__STATUS_PASS:
        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
        break;
      case GSTORAGE_CMD__STATUS_CONTINUE:
        break;
      case GSTORAGE_CMD__STATUS_FAIL:
      case GSTORAGE_CMD__STATUS_FATAL:
      default:
        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_WRITE_SECTOR_ERROR );
        break;
    }

    return ret;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * writeMedia
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：Write 処理
 * 入力  ：GSTORAGE_DRV__CmdContext のポインタ
 * 出力  ：GSTORAGE_DRV__CmdContext
 *         ( GSTORAGE_DRV__DeviceContext, GSTORAGE_DRV__UnitContext )
 * 返り値：GSTORAGE__SUCCESS/GSTORAGE__ERROR
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
GSTORAGE__Status writeMedia( GSTORAGE_DRV__CmdContext *pCmdCtx )
{
    GSTORAGE__Status ret = GSTORAGE_CMD__STATUS_PASS;
    unsigned int result;
    unsigned int transLen;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    if ( pCmdCtx->pUnitCtx->continueInfo.remainLen > GSTORAGE__RECEIVE_BUF_SIZE ){
        transLen = GSTORAGE__RECEIVE_BUF_SIZE;
    }
    else {
        transLen = pCmdCtx->pUnitCtx->continueInfo.remainLen;
    }

    storage_drvcnt_setMediaAccess( pCmdCtx->pUnitCtx );
    pCmdCtx->pData = pCmdCtx->pUnitCtx->pReceiveBuf;
    result = storage_write_cmn( pCmdCtx->pUnitCtx, pCmdCtx->pUnitCtx->fd, pCmdCtx->pData, transLen,
                                pCmdCtx->pUnitCtx->continueInfo.address );

    if ( result == transLen ){
        pCmdCtx->pUnitCtx->continueInfo.remainLen   -= transLen;
        pCmdCtx->pUnitCtx->continueInfo.address     += transLen;
        pCmdCtx->pUnitCtx->continueInfo.transferLen += transLen;
        pCmdCtx->totalTransLen                      += transLen;
        if ( pCmdCtx->pUnitCtx->continueInfo.remainLen == 0 ){
            ret = GSTORAGE_CMD__STATUS_PASS;
        }
        else {
            ret = GSTORAGE_CMD__STATUS_CONTINUE;
        }
    }
    else {
        ret = GSTORAGE_CMD__STATUS_FAIL;
        GS_INF( GS_OUT__CMD_RW, "write error : result=%d", result );
    }

    return ret;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * unknownCmd_std
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：その他のコマンド処理
 * 入力  ：GSTORAGE_DRV__CmdContext のポインタ
 * 出力  ：GSTORAGE_DRV__CmdContext
 *         ( GSTORAGE_DRV__DeviceContext, GSTORAGE_DRV__UnitContext )
 * 返り値：GSTORAGE__SUCCESS/GSTORAGE__ERROR
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
GSTORAGE__Status unknownCmd_std( GSTORAGE_DRV__CmdContext *pCmdCtx )
{
    GSTORAGE__Status ret = GSTORAGE__SUCCESS;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    /* エラー出力の設定 */
    pCmdCtx->totalTransLen = 0;
    pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
    setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_INVALID_COMMAND );

    return ret;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * getTransLength
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：転送サイズ取得コマンド処理
 * 入力  ：
 * 出力  ：なし
 * 返り値：転送サイズ
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
unsigned int getTransLength( unsigned int expect, unsigned int allocate, unsigned int data )
{
    unsigned int len;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    if ( expect >= allocate ){
        len = ( allocate >= data ) ? data : allocate;
    }
    else {
        len = ( expect >= data ) ? data : expect;
    }

    return len;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * setMediaCommonError
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：メディア共通エラー処理(後)
 * 入力  ：Command Context のポインタ
 *         転送済サイズ
 * 出力  ：なし
 * 返り値：なし
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
void setMediaCommonError( GSTORAGE_DRV__CmdContext *pCmdCtx, unsigned int len )
{
    GS_DET( GS_OUT__CMD_STD, "func Req" );

    if ( ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_READY         ) &&
         ( pCmdCtx->pUnitCtx->media.unitStatus  == GSTORAGE__UNIT_NOT_EJECT_STOP ) ){
        /* drive not ready */
        setDriveNotReady( pCmdCtx, 0 );
    }
    else if ( pCmdCtx->pUnitCtx->media.mediaStatus == GSTORAGE__MEDIA_NOT_READY ){
        /* no media */
        setMediaEjected( pCmdCtx, 0 );
    }
    else {
        /* media error */
        pCmdCtx->totalTransLen = 0;
        pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
        setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
    }

    return;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * setDriveNotReady
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：ドライブ UNKNOWN 時処理
 * 入力  ：Command Context のポインタ
 * 出力  ：なし
 * 返り値：なし
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
void setDriveNotReady( GSTORAGE_DRV__CmdContext *pCmdCtx, unsigned int len )
{
    GS_DET( GS_OUT__CMD_STD, "func Req" );

    pCmdCtx->totalTransLen = len;
    pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
    setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_DRIVE_NOT_READY );

    return;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * setDriveReset
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：メディア入換発生時処理
 * 入力  ：Command Context のポインタ
 *         転送済サイズ
 * 出力  ：なし
 * 返り値：なし
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
void setDriveReset( GSTORAGE_DRV__CmdContext *pCmdCtx, unsigned int len )
{
    GS_DET( GS_OUT__CMD_STD, "func Req" );

    pCmdCtx->totalTransLen = len;
    pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
    setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_RESET_OCCURRED );

    return;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * setMediaChanged
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：メディア入換発生時処理
 * 入力  ：Command Context のポインタ
 *         転送済サイズ
 * 出力  ：なし
 * 返り値：なし
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
void setMediaChanged( GSTORAGE_DRV__CmdContext *pCmdCtx, unsigned int len )
{
    GS_DET( GS_OUT__CMD_STD, "func Req" );

    pCmdCtx->totalTransLen = len;
    pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
    setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_MEDIA_CHANGED );

    return;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * setMediaEjected
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：メディア Eject 中処理
 * 入力  ：Command Context のポインタ
 *         転送済サイズ
 * 出力  ：なし
 * 返り値：なし
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
void setMediaEjected( GSTORAGE_DRV__CmdContext *pCmdCtx, unsigned int len )
{
    GS_DET( GS_OUT__CMD_STD, "func Req" );

    pCmdCtx->totalTransLen = len;
    pCmdCtx->status        = GSTORAGE_CMD__STATUS_FAIL;
    setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_MEDIA );

    return;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * setSense
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：SENSE データ設定処理
 * 入力  ：SENSE
 *         ASC
 *         ASCQ
 * 出力  ：なし
 * 返り値：なし
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
void setSense( GSTORAGE_DRV__SenseCode *pSense, unsigned int senseKey )
{
    GS_DET( GS_OUT__CMD_STD, "func Req" );

    pSense->sense = ( senseKey & 0x000f0000 ) >> 16;
    pSense->asc   = ( senseKey & 0x0000ff00 ) >> 8;
    pSense->ascq  = ( senseKey & 0x000000ff );

    return;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * getSense
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * 概要  ：SENSE データ取得処理
 * 入力  ：なし
 * 出力  ：SENSE データ
 * 返り値：なし
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
unsigned int getSense( GSTORAGE_DRV__SenseCode *pSense )
{
    unsigned int senseKey;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    senseKey = ( ( (unsigned int)pSense->sense << 16 ) | 
               ( (unsigned int)pSense->asc << 8 ) | 
               ( pSense->ascq ) );

    return senseKey;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * storage_read_cmn
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
unsigned int storage_read_cmn( GSTORAGE_DRV__UnitContext *pUnitCtx,
                               struct file *fd, unsigned char *addr,
                               unsigned int count, signed long long offset )
{
    unsigned int result;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    if ( pUnitCtx->pDrive->drvType == GSTORAGE__DRIVE_ID_SXS ){
        result = storage_read_cdev( fd, addr, count, offset );
    }
    else {
        result = storage_read( fd, addr, count, offset );
    }
    return result;
}

/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * storage_read
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
unsigned int storage_read( struct file *fd, unsigned char *addr,
                           unsigned int count, signed long long offset )
{
#ifdef GSTORAGE_IO_BLOCK_DEV
    GS_DET( GS_OUT__CMD_STD, "func Req" );

    return bdio_read(fd, addr, count, offset);
#else
    unsigned int result;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    /* The cast to a user pointer is valid due to the set_fs() */
    result = kernel_read( fd, ( void __user * )addr, count, &offset );
    GS_DBG( GS_OUT__CMD_RW, "read result = %d", (unsigned int)result );

    return result;
#endif
}

/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * storage_read_cdev
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
unsigned int storage_read_cdev( struct file *fd, unsigned char *addr,
                           unsigned int count, signed long long offset )
{
    unsigned int result;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    /* The cast to a user pointer is valid due to the set_fs() */
    result = kernel_read( fd, ( void __user * )addr, count, &offset );
    GS_DBG( GS_OUT__CMD_RW, "read result = %d", (unsigned int)result );

    return result;
}

/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * storage_write_cmn
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
unsigned int storage_write_cmn( GSTORAGE_DRV__UnitContext *pUnitCtx,
                                struct file *fd, unsigned char *addr,
                                unsigned int count, signed long long offset )
{
    unsigned int result;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    if ( pUnitCtx->pDrive->drvType == GSTORAGE__DRIVE_ID_SXS ){
        result = storage_write_cdev( fd, addr, count, offset );
    }
    else {
        result = storage_write( fd, addr, count, offset );
    }
    return result;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * storage_write
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
unsigned int storage_write( struct file *fd, unsigned char *addr,
                            unsigned int count, signed long long offset )
{
#ifdef GSTORAGE_IO_BLOCK_DEV
    GS_DET( GS_OUT__CMD_STD, "func Req" );

    return bdio_write(fd, addr, count, offset);
#else
    unsigned int result;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    /* The cast to a user pointer is valid due to the set_fs() */
    result = kernel_write( fd, ( void __user * )addr, count, &offset );
    GS_DBG( GS_OUT__CMD_RW, "write result = %d", (unsigned int)result );

    return result;
#endif
}

/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * storage_write_cdev
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
unsigned int storage_write_cdev( struct file *fd, unsigned char *addr,
                            unsigned int count, signed long long offset )
{
    unsigned int result;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    /* The cast to a user pointer is valid due to the set_fs() */
    result = kernel_write( fd, ( void __user * )addr, count, &offset );
    GS_DBG( GS_OUT__CMD_RW, "write result = %d", (unsigned int)result );

    return result;
}


/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * storage_ata_standbyImmediate
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
void storage_ata_standbyImmediate( struct file *filp )
{
    int result = 0;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    result = standbyImmediate_ata( filp );
    if ( result ){
        GS_INF( GS_OUT__CMD_STD, "ioctl error." );
    }

    return;
}

/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * standbyImmediate_ata_cmn
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
int standbyImmediate_ata_cmn( GSTORAGE_DRV__UnitContext *pUnitCtx, struct file *filp )
{
    int  result = 0;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    if( pUnitCtx->pDrive->drvType == GSTORAGE__DRIVE_ID_NAND ) {

    } else if( pUnitCtx->pDrive->drvType == GSTORAGE__DRIVE_ID_MMC ) {

    } else if( pUnitCtx->pDrive->drvType == GSTORAGE__DRIVE_ID_SXS ) {

    } else if( pUnitCtx->pDrive->drvType == GSTORAGE__DRIVE_ID_SXS_ADPT ) {

    }
    else {
        result = standbyImmediate_ata( filp );
    }

    return result;
}
/* tentative */
#ifndef WIN_STANDBYNOW1
 #define WIN_STANDBYNOW1 ( 0xE0 )
#endif

/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * standbyImmediate_ata
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
int standbyImmediate_ata( struct file *filp )
{
    int          result = 0;
    char         cmd[4];

    GS_DET( GS_OUT__CMD_STD, "func Req" );
    
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 36)
    if ( filp && !IS_ERR( filp ) && filp->f_op && filp->f_op->unlocked_ioctl ){
#else
    if ( filp && !IS_ERR( filp ) && filp->f_op && filp->f_op->ioctl ){
#endif

        /* STANDBY IMMEDIATE : 0xE0 */
        cmd[0] = WIN_STANDBYNOW1;
        cmd[1] = 0;
        cmd[2] = 0;
        cmd[3] = 0;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 36)
        result = filp->f_op->unlocked_ioctl(  filp, HDIO_DRIVE_CMD, (unsigned long)&cmd[0] );
#else
        result = filp->f_op->ioctl( filp->f_dentry->d_inode , filp,
                                    HDIO_DRIVE_CMD, (unsigned long)&cmd[0] );
#endif

    }

    return result;
}

/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * storage_ata_idleImmediate
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
void storage_ata_idleImmediate( struct file *filp )
{
    int result = 0;
    
    GS_DET( GS_OUT__CMD_STD, "func Req" );

    result = idleImmediate_ata( filp );
    if ( result ){
        GS_INF( GS_OUT__CMD_STD, "ioctl error." );
    }

    return;
}

/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * idleImmediate_ata_cmn
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
int idleImmediate_ata_cmn( GSTORAGE_DRV__UnitContext *pUnitCtx, struct file *filp )
{
    int    result = 0;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    if( pUnitCtx->pDrive->drvType == GSTORAGE__DRIVE_ID_NAND ) {

    }
    else if( pUnitCtx->pDrive->drvType == GSTORAGE__DRIVE_ID_MMC ) {

    }
    else if( pUnitCtx->pDrive->drvType == GSTORAGE__DRIVE_ID_SXS ) {

    }

    else if( pUnitCtx->pDrive->drvType == GSTORAGE__DRIVE_ID_SXS_ADPT ) {

    }
    else {
        result = idleImmediate_ata( filp );
    }

    return result;
}

/* tentative */
#ifndef WIN_IDLEIMMEDIATE
 #define WIN_IDLEIMMEDIATE ( 0xE1 )
#endif

/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * idleImmediate_ata
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
int idleImmediate_ata( struct file *filp )
{
    int          result = 0;
    char         cmd[4];

    GS_DET( GS_OUT__CMD_STD, "func Req" );

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 36)
    if ( filp && !IS_ERR( filp ) && filp->f_op && filp->f_op->unlocked_ioctl ){
#else
    if ( filp && !IS_ERR( filp ) && filp->f_op && filp->f_op->ioctl ){
#endif

        /* STANDBY IMMEDIATE : 0xE0 */
        cmd[0] = WIN_IDLEIMMEDIATE;
        cmd[1] = 0;
        cmd[2] = 0;
        cmd[3] = 0;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 36)
        result = filp->f_op->unlocked_ioctl(  filp, HDIO_DRIVE_CMD, (unsigned long)&cmd[0] );
#else
        result = filp->f_op->ioctl( filp->f_dentry->d_inode , filp,
                                    HDIO_DRIVE_CMD, (unsigned long)&cmd[0] );
#endif

    }

    return result;
}

/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * storage_ata_flushCache
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
void storage_ata_flushCache( struct file *filp )
{
    int result = 0;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    result = flushCache_ata( filp );
    if ( result ){
        GS_INF( GS_OUT__CMD_STD, "ioctl error." );
    }

    return;
}

/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * flushCache_ata_cmn
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
int flushCache_ata_cmn( GSTORAGE_DRV__UnitContext *pUnitCtx, struct file *filp )
{
    int                        result = 0;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    if( pUnitCtx->pDrive->drvType == GSTORAGE__DRIVE_ID_NAND ) {

    }
    else if( pUnitCtx->pDrive->drvType == GSTORAGE__DRIVE_ID_MMC ) {

    }
    else if( pUnitCtx->pDrive->drvType == GSTORAGE__DRIVE_ID_SXS ) {

    }
    else if( pUnitCtx->pDrive->drvType == GSTORAGE__DRIVE_ID_SXS_ADPT ) {

    }
    else {
        result = flushCache_ata( filp );
    }

    return result;
}

/* tentative */
#ifndef WIN_FLUSH_CACHE
 #define WIN_FLUSH_CACHE ( 0xE7 )
#endif

/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * flushCache_ata
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
int flushCache_ata( struct file *filp )
{
    int           result = 0;
    char          cmd[4];

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    cmd[0] = WIN_FLUSH_CACHE;
    cmd[1] = 0x00;
    cmd[2] = 0x00;
    cmd[3] = 0x00;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 36)
        result = filp->f_op->unlocked_ioctl(  filp, HDIO_DRIVE_CMD, (unsigned long)&cmd[0] );
#else
        result = filp->f_op->ioctl( filp->f_dentry->d_inode , filp,
                                    HDIO_DRIVE_CMD, (unsigned long)&cmd[0] );
#endif

    return result;
}

/*━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*
 * checkExtCommand
 *━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━*/
GSTORAGE__Status checkExtCommand( GSTORAGE_DRV__CmdContext *pCmdCtx )
{
    unsigned int    ret;
    int             i;

    GS_DET( GS_OUT__CMD_STD, "func Req" );

    ret = GSTORAGE__NON_DETECT;

    if( pCmdCtx->dio == DATA_NON ) {
        for( i = 0; i < pCmdCtx->pUnitCtx->pDevCtx->extCmdNum; i++ ) {
            if( pCmdCtx->pCdb->opCode == pCmdCtx->pUnitCtx->pDevCtx->pExtCmd[ i ]->opCode ) {
                GS_DBG( GS_OUT__EXTCOMMAND, "ext command detect! lun = %d, OPcode = 0x%02X",
                        pCmdCtx->pCbw->cbwLun, pCmdCtx->pCdb->opCode );
                storage_detect_ExtCommand( pCmdCtx->pCbw->cbwLun, pCmdCtx->pCdb );
                /* set output data */
                /* PASSの設定 */
                pCmdCtx->totalTransLen = 0;
                pCmdCtx->status        = GSTORAGE_CMD__STATUS_PASS;
                setSense( &pCmdCtx->pUnitCtx->sense, GSTORAGE_CMD__SK_NO_SENSE );
                ret = GSTORAGE__DETECT;
                break;
            }
        }
    }

    return ret;
}

