// SPDX-License-Identifier: GPL-2.0-only
/*
 *  fs/proc/snsc_kstack.c
 *
 *  proc interface to list kernel stack usage
 *
 *  Copyright 2010 Sony Corporation
 *  Copyright 2022 Sony Group Corporation
 */

#include <linux/fs.h>
#include <linux/init.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/sched.h>
#include <linux/sched/clock.h>
#include <linux/sched/signal.h>
#include <linux/sched/task_stack.h>
#include <linux/nmi.h>
#include <linux/uaccess.h>
#include <linux/list.h>
#include <linux/semaphore.h>
#include <linux/slab.h>

#ifdef CONFIG_DEBUG_STACK_USAGE
extern int debug_stack_usage;
#endif
static int __read_mostly __kstack_enabled = 0;

struct kstack_log_data {
	struct list_head list;
	pid_t pid;
	pid_t tgid;
	char *comm;
	unsigned long free;
	unsigned long long time;
};
static LIST_HEAD(kstack_log_head);

static DEFINE_SEMAPHORE(kstack_log_mutex);

/*
 * Call this function to log the free stack size when a process exits.
 * This function is called from check_stack_usage() in kernel/exit.c
 */
void kstack_log(const struct task_struct *p, unsigned long free)
{
	struct kstack_log_data *d;
	int comm_len;

	if (!__kstack_enabled)
		return;

	d = kmalloc(sizeof(*d), GFP_KERNEL);
	if (d == NULL)
		return;
	comm_len = strlen(p->comm);
	d->comm = kmalloc(comm_len + 1, GFP_KERNEL);
	if (d->comm == NULL)
	{
		kfree(d);
		return;
	}

	strncpy(d->comm, p->comm, comm_len + 1);
	d->pid = p->pid;
	d->tgid = p->tgid;
	d->free = free;
	d->time = sched_clock();

	/* add to list */
	down(&kstack_log_mutex);
	list_add_tail(&d->list, &kstack_log_head);
	up(&kstack_log_mutex);
}

static void kstack_clear(void)
{
	struct kstack_log_data *d, *n;

	down(&kstack_log_mutex);
	list_for_each_entry_safe(d, n, &kstack_log_head, list) {
		kfree(d->comm);
		kfree(d);
	}
	INIT_LIST_HEAD(&kstack_log_head);
	up(&kstack_log_mutex);
}

static int kstack_proc_show(struct seq_file *m, void *v)
{
	struct kstack_log_data *d;
	struct task_struct *g, *p;

	if (__kstack_enabled)
		seq_printf(m, "kstack is on\n\n");
	else
		seq_printf(m, "kstack is off\n\n");


	seq_printf(m, "exited processes:\n");
	seq_printf(m, "                 TIME   PID  TGID UNUSED-STACK COMMAND\n");

	down(&kstack_log_mutex);
	list_for_each_entry(d, &kstack_log_head, list) {
		unsigned long long t = d->time;
		unsigned long nsec = do_div(t, 1000000000);

		seq_printf(m, "%11llu.%09lu %5d %5d %12lu %-32s\n", t, nsec, d->pid, d->tgid, d->free, d->comm);
	}
	up(&kstack_log_mutex);


	seq_printf(m, "\n");
	seq_printf(m, "live processes:\n");
	seq_printf(m, "  PID  TGID UNUSED-STACK COMMAND\n");

	/* copied from kernel/sched.c */
	read_lock(&tasklist_lock);
	do_each_thread(g, p) {
		unsigned long free;

		free = stack_not_used(p);
		seq_printf(m, "%5d %5d %12lu %-32s\n", p->pid, p->tgid, free, p->comm);
	} while_each_thread(g, p);
	read_unlock(&tasklist_lock);

	return 0;
}

static int kstack_set(char opt)
{
	switch (opt) {
	case '0':
	case 'n': case 'N':
		__kstack_enabled = 0;
		break;
	case '1':
	case 'y': case 'Y':
		__kstack_enabled = 1;
		debug_stack_usage = 1;
		break;
	case 'c': case 'C':
	case 'r': case 'R':
		kstack_clear();
		break;
	default:
		return 0;
	}
	return 1;
}

static ssize_t kstack_proc_write(struct file *file, const char __user *buf,
				 size_t count, loff_t *ppos)
{
	char c;

	if (!count)
		return 0;

	if (get_user(c, buf))
		return -EFAULT;
	if (!kstack_set(c))
		return -EFAULT;
	return count;
}

static int kstack_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, kstack_proc_show, NULL);
}

static const struct proc_ops kstack_proc_ops = {
	.proc_open	= kstack_proc_open,
	.proc_read	= seq_read,
	.proc_write	= kstack_proc_write,
	.proc_lseek	= seq_lseek,
	.proc_release	= single_release,
};

static int __init kstack_init(void)
{
	proc_create("kstack", 0644, NULL, &kstack_proc_ops);
	return 0;
}

static int __init kstack_setup(char *line)
{
	kstack_set(*line);
	return 1;
}

late_initcall(kstack_init);
__setup("kstack=", kstack_setup);
