// SPDX-License-Identifier: GPL-2.0
/*
 *  ssbootfs: attachment handling
 *
 *  Copyright 2021 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 */

#include <linux/mm.h>
#include <linux/fs.h>
#include <linux/namei.h>
#include <linux/mount.h>
#include <linux/list.h>
#include <linux/slab.h>
#include <linux/file.h>
#include <linux/mm.h>
#include <linux/pagemap.h>
#include <linux/fdtable.h>
#include <linux/task_work.h>

#include "../internal.h"
#include "../mount.h"
#include "ssbootfs.h"

int ssbootfs_backingfs_mntpath(struct ssbootfs_priv *priv, struct path *path)
{
	int ret;
	struct path sysfs, instance;

	ret = kern_path("/sys/fs/" SSBOOTFS_NAME, LOOKUP_DIRECTORY, &sysfs);
	if (ret) {
		ssbootfs_err("failed to lookup sysfs: %d, aborting\n", ret);
		goto out;
	}

	ret = vfs_path_lookup(sysfs.dentry, sysfs.mnt, priv->name, LOOKUP_DIRECTORY, &instance);
	if (ret) {
		ssbootfs_err("failed to lookup the instance dir: %d, aborting\n", ret);
		goto out_sysfs;
	}

	ret = vfs_path_lookup(instance.dentry, instance.mnt, SSBOOTFS_SYSFS_BACKINGFS,
			LOOKUP_DIRECTORY, &priv->backingfs);
	if (ret) {
		ssbootfs_err("failed to lookup the backing dir: %d, aborting\n", ret);
	}

	path_put(&instance);
out_sysfs:
	path_put(&sysfs);
out:
	return ret;
}

int ssbootfs_attach_unlocked(struct ssbootfs_priv *priv)
{
	int ret;
	struct super_block *sb;

	if (priv->attached) {
		ssbootfs_warn("trying to attach while already attached\n");
		ret = -EINVAL;
		goto out_err_lock;
	}

	ret = ssbootfs_backingfs_mntpath(priv, &priv->backingfs);
	if (ret)
		goto out_err_lock;

	if (!IS_ROOT(priv->backingfs.dentry)) {
		ssbootfs_err("backing fs path must be root of mount\n");
		goto out_err_path;
	}

	if (!may_umount(priv->backingfs.mnt)) {
		ssbootfs_err("backing fs must be safe to unmount\n");
		goto out_err_path;
	}

	sb = priv->backingfs.mnt->mnt_sb;
	if (!priv->has_attached) {
		/* copy the uuid from the super block so we can check it later */
		memcpy(&priv->backingfs_uuid, &sb->s_uuid, sizeof(priv->backingfs_uuid));

		/* duplicate the max file size from the real super block */
		priv->sb->s_maxbytes = sb->s_maxbytes;

		/* increase the stack depth from the backing fs */
		priv->sb->s_stack_depth = sb->s_stack_depth + 1;

		priv->has_attached = true;
	} else {
		if (memcmp(&priv->backingfs_uuid, &sb->s_uuid, sizeof(priv->backingfs_uuid)) != 0) {
			ssbootfs_err("backing fs uuid has changed, wrong block mounted? not attaching\n");
			goto out_err_path;
		}
	}

	priv->attached = true;

	ssbootfs_info("backing fs attached\n");

	goto success;

out_err_path:
	path_put(&priv->backingfs);
success:
out_err_lock:
	return ret;
}

int ssbootfs_attach(struct ssbootfs_priv *priv)
{
	int ret;

	ssbootfs_priv_lock(priv);
	ret = ssbootfs_attach_unlocked(priv);
	ssbootfs_priv_unlock(priv);
	return ret;
}

extern void flush_mntput_work(void);

int ssbootfs_disengage_sb(struct ssbootfs_priv *priv)
{
	int ret = 0;
	struct inode *inode;
	struct ssbootfs_inode *ssbootfs_inode;
	unsigned long inode_count = 0, file_count = 0;
	LIST_HEAD(to_disengage);

	if (!priv->backingfs.mnt) {
		ssbootfs_debug("disengaging but backingfs isn't mounted\n");
		return 0;
	}

	ssbootfs_priv_lock(priv);
	/* Collect all of the inodes we plan to disengage */
	spin_lock(&priv->sb->s_inode_list_lock);
	list_for_each_entry(inode, &priv->sb->s_inodes, i_sb_list) {
		ssbootfs_inode = SSBOOTFS_I_TO_SSBFSI(inode);
		if (ssbootfs_inode->vfs_inode.i_ino == 0)
			continue;
		inode_lock_nested(inode, I_MUTEX_PARENT);
		/* stop the inode from disappearing before the next loop */
		__iget(inode);
		inode_count++;
		list_add(&ssbootfs_inode->disengage, &to_disengage);
		inode_unlock(inode);
	}
	spin_unlock(&priv->sb->s_inode_list_lock);
	/* now disengage them */
	list_for_each_entry(ssbootfs_inode, &to_disengage, disengage) {
		file_count += ssbootfs_inode_disengage(priv, &ssbootfs_inode->vfs_inode);
		/* go free little inode... */
		iput(&ssbootfs_inode->vfs_inode);
	}
	ssbootfs_priv_unlock(priv);

	/*
	 * Usually all of real file closing will happen some time in the future
	 * and we won't know the real mount count here.. so flush the fput work
	 * wait until everything is finished.
	 */
	flush_delayed_fput();
	task_work_run();
	synchronize_rcu();

	/* TODO Only output when there is an issue */
	ssbootfs_info("%lu live inodes, %lu live files disengaged, %u mntcnt\n",
			inode_count, file_count, ssbootfs_debug_getrealmntcnt(priv));
	/* TODO detect unfinished delayed work etc and return an error */
	return ret;
}

static void ssbootfs_dump_realsbinfo(struct ssbootfs_priv *priv)
{
	struct super_block *sb = priv->backingfs.mnt->mnt_sb;
	struct inode *inode;
	unsigned int inode_count = 0;

	shrink_dcache_sb(sb);
	evict_inodes(sb);

	spin_lock(&sb->s_inode_list_lock);
	list_for_each_entry(inode, &priv->sb->s_inodes, i_sb_list) {
		if (inode->i_ino != 0) {
			ssbootfs_info("inode %lu on real superblock\n", inode->i_ino);
			inode_count++;
		}
	}
	spin_unlock(&sb->s_inode_list_lock);

	ssbootfs_info("%d inodes on real superblock\n", inode_count);
}

int ssbootfs_detach(struct ssbootfs_priv *priv, bool in_suspend)
{
	int ret;

	ssbootfs_priv_lock(priv);

	if (!priv->attached) {
		if (!in_suspend)
			ssbootfs_warn("trying to detach while already detached\n");
		ret = -EINVAL;
		goto out;
	}

	ssbootfs_info("umounting backingfs for %s\n", priv->name);

#ifdef CONFIG_SNSC_SSBOOT_FS_DEBUG_SANITY_CHECKS
	BUG_ON(!priv->backingfs.dentry);
	BUG_ON(!priv->backingfs.mnt);
#endif

	if (!may_umount(priv->backingfs.mnt)) {
		ssbootfs_err("kernel says backing fs is busy. Not detaching. mntcnt %u\n",
				ssbootfs_debug_getrealmntcnt(priv));
		ssbootfs_dump_realsbinfo(priv);
		ret = -EBUSY;
		goto out;
	}

	ret = path_umount(&priv->backingfs, 0);
	if (ret) {
		ssbootfs_err("failed to umount backing fs\n");
		goto out;
	}

	/* make sure the mount is really gone */
	flush_mntput_work();
	task_work_run();
	synchronize_rcu();

#ifdef CONFIG_SNSC_SSBOOT_FS_IN_KERNEL_MOUNT
	BUG_ON(vfs_mount_id(priv->backingfs.mnt) != -1);
#endif

	priv->attached = false;
	priv->backingfs.dentry = NULL;
	priv->backingfs.mnt = NULL;
	priv->in_suspend = in_suspend;

	ssbootfs_debug("backing fs is now detached!\n");

out:
	ssbootfs_priv_unlock(priv);
	return ret;
}
