// SPDX-License-Identifier: GPL-2.0
/*
 *  ssbootfs: backing filesystem glue
 *
 *  Copyright 2021 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 */

#include <linux/mm.h>
#include <linux/fs.h>
#include <linux/namei.h>
#include <linux/mount.h>
#include <linux/list.h>
#include <linux/slab.h>
#include <linux/file.h>
#include <linux/mm.h>
#include <linux/pagemap.h>
#include <linux/writeback.h>

#include "../internal.h"
#include "ssbootfs.h"

/* for a given fake dentry look up the real dentry */
int ssbootfs_backingfs_lookup(struct ssbootfs_priv *priv, struct dentry *dentry,
			      const char *name, struct dentry **res)
{
	int ret;
	struct path path;

#ifdef CONFIG_SNSC_SSBOOT_FS_DEBUG_SANITY_CHECKS
	BUG_ON(!priv);
	BUG_ON(!(priv->backingfs.mnt));
	BUG_ON(!dentry);
	BUG_ON(!name);
	BUG_ON(!res);
#endif

	SSBOOTFS_CHECK_ATTACHED(priv) {
		return -EIO;
	}

	ssbootfs_debug("looking for %s->%s\n", d_name(dentry), name);

	ret = vfs_path_lookup(dentry, priv->backingfs.mnt, name, 0, &path);

	if (ret) {
		ssbootfs_debug("lookup failed for (%s)/%s on backing fs, %d\n",
				d_name(dentry), name, ret);
		goto out;
	}

	/*
	 * We dget mapped here to avoid it disappearing when the path is put
	 * and so that a ref is held for the final result
	 */
	dget(path.dentry);
	path_put(&path);

	ssbootfs_debug("found %s on backing fs\n", name);
	*res = path.dentry;

out:
	return ret;
}

/* for a given fake file return a real file */
static int ssbootfs_backingfs_mapfile(struct file *file, struct path *realpath)
{
	struct ssbootfs_priv *priv = ssbootfs_priv_from_file(file);
	struct dentry *realdentry;
	int ret;

	ret = ssbootfs_mapping_map_dentry(file->f_path.dentry, &realdentry);
	if (ret)
		goto out;

	realpath->mnt = priv->backingfs.mnt;
	realpath->dentry = realdentry;

	path_get(realpath);

	dput(realdentry);

out:
	return ret;
}

struct file *ssbootfs_backingfs_open(struct file *file)
{
	struct path realpath;
	struct file *realfile = NULL;
	struct ssbootfs_file_priv *file_priv = file->private_data;
	int ret;

	ret = ssbootfs_backingfs_mapfile(file, &realpath);
	if (ret) {
		realfile = ERR_PTR(ret);
		goto out;
	}

#ifdef CONFIG_SNSC_SSBOOT_FS_DEBUG_SANITY_CHECKS
	BUG_ON(!realpath.mnt);
	BUG_ON(!realpath.dentry);
#endif

	/* Open the real file, pass on the flags and creds */
	realfile = dentry_open(&realpath, file->f_flags, file->f_cred);
	if (IS_ERR(realfile)) {
		ret = PTR_ERR(realfile);
		ssbootfs_debug("failed to open real file %s: %d\n", d_name(file->f_path.dentry), ret);
		goto out_mapped;
	}

	ssbootfs_sync_real_file_from_fake_file(file, realfile);

	file_priv->real_file = realfile;

	goto out_mapped;

out_mapped:
	path_put(&realpath);
out:
	return realfile;
}

void ssbootfs_backingfs_close(struct file *file)
{
	struct ssbootfs_file_priv *file_priv = file->private_data;

	if (likely(file_priv->real_file))
		fput(file_priv->real_file);
}

int ssbootfs_backingfs_create(struct user_namespace *ns, struct inode *dir, struct dentry *dentry,
		umode_t mode, bool want_excl, dev_t dev)
{
	int ret;
	unsigned int type;

	type = mode & S_IFMT;
	switch (type) {
	case S_IFREG:
		ret = vfs_create(ns, dir, dentry, mode, want_excl);
		if (ret) {
			goto out_failed;
		}
		break;
	case S_IFDIR:
		ret = vfs_mkdir(ns, dir, dentry, mode);
		if (ret)
			goto out_failed;
		break;
	case S_IFCHR:
	case S_IFBLK:
	case S_IFIFO:
	case S_IFSOCK:
		ret = vfs_mknod(ns, dir, dentry, mode, dev);
		break;
	default:
		ssbootfs_info("unsupported create: %x\n", (unsigned int) type);
		ret = -EINVAL;
		break;
	}

	return ret;

out_failed:
	ssbootfs_err("failed to create \"%s\" on backing fs: %d\n", d_name(dentry), ret);
	return ret;
}

ssize_t ssbootfs_backingfs_read_iter(struct kiocb *cb, struct iov_iter *iter)
{
	struct file *file = cb->ki_filp;
	struct ssbootfs_file_priv *file_priv = file->private_data;
	struct kiocb rcb;
	int ret;

#ifdef CONFIG_SNSC_SSBOOT_FS_DEBUG_SANITY_CHECKS
	BUG_ON(!file_priv);
	BUG_ON(!file_priv->real_file);
#endif

	kiocb_clone(&rcb, cb, file_priv->real_file);
	ret = vfs_iocb_iter_read(file_priv->real_file, &rcb, iter);
	cb->ki_pos = rcb.ki_pos;

	return ret;
}

ssize_t ssbootfs_backingfs_splice_read(struct file *in, loff_t *ppos,
				 struct pipe_inode_info *pipe, size_t len,
				 unsigned int flags)
{
	ssize_t ret = -EIO;
	struct ssbootfs_file_priv *file_priv = in->private_data;

#ifdef CONFIG_SNSC_SSBOOT_FS_DEBUG_SANITY_CHECKS
	BUG_ON(!file_priv);
	BUG_ON(!file_priv->real_file);
	BUG_ON(!file_priv->real_file->f_op->splice_read);
#endif

	ret = file_priv->real_file->f_op->splice_read(file_priv->real_file, ppos, pipe, len, flags);

	return ret;
}

ssize_t ssbootfs_backingfs_write_iter(struct kiocb *cb, struct iov_iter *iter)
{
	struct file *file = cb->ki_filp;
	struct ssbootfs_file_priv *file_priv = file->private_data;
	ssize_t ret;
	struct kiocb rcb;

#ifdef CONFIG_SNSC_SSBOOT_FS_DEBUG_SANITY_CHECKS
	BUG_ON(!file_priv);
	BUG_ON(!file_priv->real_file);
#endif

	kiocb_clone(&rcb, cb, file_priv->real_file);

	file_start_write(file_priv->real_file);
	ret = vfs_iocb_iter_write(file_priv->real_file, &rcb, iter);
	file_end_write(file_priv->real_file);

	cb->ki_pos = rcb.ki_pos;

	return ret;
}

ssize_t ssbootfs_backingfs_splice_write(struct pipe_inode_info *pipe, struct file *out,
		loff_t *ppos, size_t len, unsigned int flags)
{
	ssize_t ret = -EIO;
	struct ssbootfs_file_priv *file_priv = out->private_data;

#ifdef CONFIG_SNSC_SSBOOT_FS_DEBUG_SANITY_CHECKS
	BUG_ON(!file_priv);
	BUG_ON(!file_priv->real_file);
	BUG_ON(!file_priv->real_file->f_op->splice_write);
#endif

	ret = file_priv->real_file->f_op->splice_write(pipe, file_priv->real_file, ppos, len, flags);

	return ret;
}

int ssbootfs_backingfs_fsync(struct file *file, int datasync)
{
	struct ssbootfs_file_priv *file_priv = file->private_data;
	int ret = 0;

	if (file_priv->real_file)
		ret = vfs_fsync(file_priv->real_file, datasync);

	return ret;
}

int ssbootfs_backingfs_fsync_range(struct file *file, loff_t x, loff_t y, int datasync)
{
	struct ssbootfs_file_priv *file_priv = file->private_data;
	int ret = 0;

	if (file_priv->real_file)
		ret = vfs_fsync_range(file_priv->real_file, x, y, datasync);

	return ret;
}

int ssbootfs_backingfs_flush(struct file *file, fl_owner_t id)
{
	struct ssbootfs_file_priv *file_priv = file->private_data;
	int ret = 0;

	if (likely(file_priv->real_file))
		if (file_priv->real_file->f_op->flush)
			ret = file_priv->real_file->f_op->flush(file_priv->real_file, id);

	return ret;
}

/*
 * No-op for now. The code is currently in inode.c because of all of the
 * locking that is required for renames.
 */
int ssbootfs_backingfs_rename(struct inode *old_dir, struct dentry *old_dentry,
		struct inode *new_dir, struct dentry *new_dentry, unsigned int flags)
{
	return 0;
}

int ssbootfs_backingfs_file_readpage(struct file *file, struct page *page)
{
	int ret = -EIO;
	struct ssbootfs_file_priv *file_priv = file->private_data;
	void *buf;
	loff_t offset = page_offset(page);
	size_t filesz = i_size_read(file->f_inode);
	size_t len = min_t(size_t, filesz - offset, PAGE_SIZE);

#ifdef CONFIG_SNSC_SSBOOT_FS_DEBUG_SANITY_CHECKS
	BUG_ON(!file_priv);
	BUG_ON(!file_priv->real_file);
#endif

	/* map the page so we can fill it */
	buf = kmap(page);
	BUG_ON(!buf);

	/* Zero the whole page, we might only write part of it */
	memset(buf, 0, PAGE_SIZE);

	/*
	 * Trying to fill pages beyond the end of the actual file
	 * is apparently fine.
	 */
	if (offset < filesz) {
		/*
		 * The page is mapped into the kernel space so vfs_read doesn't
		 * work directly here.
		 */
		ret = kernel_read(file_priv->real_file, buf, len, &offset);
		if (ret >= 0){
			BUG_ON(ret != len);
			ret = 0;
		}
	} else
		ret = 0;

	flush_dcache_page(page);
	kunmap(page);

	return ret;
}

int ssbootfs_backingfs_file_writepage(struct file *file, struct page *page)
{
	int ret;
	struct ssbootfs_file_priv *file_priv = file->private_data;
	void *buf;
	const loff_t offset = page_offset(page);
	/* kernel_write trashes this value so we need a copy */
	loff_t tmpoffset = offset, size;
	size_t len = PAGE_SIZE;

	/*
	 * There doesn't seem to be a good source for what to do
	 * if the file size is less than a multiple of page size
	 * and we're writing to the last page in the file.
	 * xfstests is checking that writing to a partial page
	 * at the end of a file doesn't cause the file to grow
	 * to a multiple of page size. So lets take that as being
	 * correct for now.
	 */
	size = i_size_read(file->f_inode);
	if (offset + len > size)
		len = size - offset;

#ifdef CONFIG_SNSC_SSBOOT_FS_DEBUG_SANITY_CHECKS
	BUG_ON(!file_priv);
	BUG_ON(!file_priv->real_file);
#endif

	flush_dcache_page(page);

	buf = kmap(page);
	BUG_ON(!buf);

	/*
	 * the page is mapped into kernel space so vfs_write doesn't work
	 * directly here.
	 */
	ret = kernel_write(file_priv->real_file, buf, len, &tmpoffset);
	if (ret >= 0) {
		BUG_ON(ret != len);
		ret = 0;
	}

	kunmap(page);
	return ret;
}

loff_t ssbootfs_backingfs_llseek(struct file *file, loff_t offset, int whence)
{
	struct ssbootfs_file_priv *file_priv = file->private_data;

#ifdef CONFIG_SNSC_SSBOOT_FS_DEBUG_SANITY_CHECKS
	BUG_ON(!file_priv);
	BUG_ON(!file_priv->real_file);
#endif

	return generic_file_llseek(file_priv->real_file, offset, whence);
}
