// SPDX-License-Identifier: GPL-2.0
/*
 *  ssbootfs: dentry management
 *
 *  Copyright 2021 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 */
#include <linux/namei.h>

#include "ssbootfs.h"
#include "ssbootfs_util.h"

#ifndef CONFIG_SNSC_SSBOOT_FS_DEBUG_DISABLE_DENTRY_LINKAGE
static int ssbootfs_dentry_revalidate(struct dentry *dentry, unsigned int flags)
{
	struct inode *inode;
	struct ssbootfs_inode *ssbootfs_inode;
	struct dentry *mapped;

	if (flags & LOOKUP_RCU)
		return -ECHILD;

	inode = d_inode(dentry);
	if (inode) {
		ssbootfs_inode = SSBOOTFS_I_TO_SSBFSI(inode);
		if (SSBOOTFS_DENTRY_IS_CACHABLE(ssbootfs_inode) &&
				ssbootfs_inode->lookup_cache.needs_lookup) {

			if (ssbootfs_priv_from_dentry(dentry)->in_suspend) {
				ssbootfs_warn("attempting to validate dentry for %s while suspending.. assuming valid\n", d_name(dentry));
				goto out;
			}
			/*
			 * Really we should be returning 0 here and telling the kernel
			 * to handle working out if the dentry is valid or not. The issue is
			 * that if there is a mount on a dentry it's not easy to actually work that
			 * out and recreating the dentry leaks the mount with it because the link
			 * between dentrys and mounts is based on the address of the dentry.
			 * TL;DR; we do the fix up here instead if possible to avoid recreating.
			 */
			ssbootfs_debug("Fixing up dentry %s\n", d_name(dentry));
			if(ssbootfs_mapping_map_dentry(dentry, &mapped))
				return 0;
			ssbootfs_mapping_real_dentry_set(dentry, mapped);
			dput(mapped);
		}
	}

out:
	return 1;
}
#endif

#ifdef CONFIG_SNSC_SSBOOT_FS_IN_KERNEL_MOUNT
static int ssbootfs_dentry_manage(const struct path *path, bool rcu)
{
	struct ssbootfs_priv *priv = ssbootfs_priv_from_path(path);

	/*
	 * We might sleep here and we'll get told off if an rcu path
	 * walk is happening.
	 */
	if (rcu)
		return -ECHILD;

	if (priv->in_suspend) {
		ssbootfs_info("not allowing automount during suspend");
		return 0;
	}

	return ssbootfs_util_mount(priv);
}
#endif

static void ssbootfs_dentry_release(struct dentry *dentry)
{
	struct inode *inode = d_inode(dentry);

	if (inode) {
		ssbootfs_inode_lock(inode);
		ssbootfs_mapping_invalidate_inode(inode);
		ssbootfs_inode_unlock(inode);
	}
}

const struct dentry_operations ssbootfs_dentry_ops = {
#ifndef CONFIG_SNSC_SSBOOT_FS_DEBUG_DISABLE_DENTRY_LINKAGE
	.d_revalidate	= ssbootfs_dentry_revalidate,
#endif
#ifdef CONFIG_SNSC_SSBOOT_FS_IN_KERNEL_MOUNT
	.d_manage	= ssbootfs_dentry_manage,
#endif
	.d_release	= ssbootfs_dentry_release,
};
