// SPDX-License-Identifier: GPL-2.0
/*
 *  ssbootfs: virtual -> real file path mapping
 *
 *  Copyright 2021 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 */

#include <linux/namei.h>

#include "ssbootfs.h"

#define DEPTH_CACHE_ALWAYS_INVALID	0

/*
 * Work out the how many levels the target is above the root.
 * Slowest path that actually traverses down the complete path.
 */
static int ssbootfs_mapping_map_depth_slow(struct dentry *dentry)
{
	int depth = 0;
	struct dentry *d = dentry, *dd;

	dget(d);
	while (true) {
		if (IS_ROOT(d)) {
			dput(d);
			break;
		}
		dd = dget_parent(d);
		dput(d);
		d = dd;
		depth++;
	}
	return depth;
}

#ifndef CONFIG_SNSC_SSBOOT_FS_DEBUG_DISABLE_DEPTH_CACHE

/* fastest path, depth was already worked out */
static int ssbootfs_mapping_map_depth_very_fast(struct dentry *dentry)
{
	struct inode *inode = d_inode(dentry);
	struct ssbootfs_inode *ssbootfs_inode;

	int ret = -1;
	if (inode) {
		ssbootfs_inode = SSBOOTFS_I_TO_SSBFSI(inode);
		if (SSBOOTFS_DENTRY_IS_CACHABLE(ssbootfs_inode)) {
#ifdef CONFIG_SNSC_SSBOOT_FS_DEBUG_VERBOSE_STATS
			ssbootfs_inode->lookup_cache.depth_hits += 1;
			ssbootfs_debug("%u depth hits\n", ssbootfs_inode->lookup_cache.depth_hits);
#endif
			ret = ssbootfs_inode->lookup_cache.depth;
		}
	}
	return ret;
}

/* next fastest path, depth of the parent was already worked out */
static int ssbootfs_mapping_map_depth_fast(struct dentry *dentry)
{
	struct ssbootfs_priv *priv = ssbootfs_priv_from_dentry(dentry);
	struct dentry *parent = dget_parent(dentry);
	struct inode *parent_inode = d_inode(parent);
	struct ssbootfs_inode *ssbootfs_inode;
	int depth = -1;
#ifdef CONFIG_SNSC_SSBOOT_FS_DEBUG_VERBOSE_DEPTH_CACHE_VALIDATE
	int real_depth;
#endif

	if (parent_inode) {
		ssbootfs_inode = SSBOOTFS_I_TO_SSBFSI(parent_inode);
		if (ssbootfs_inode->lookup_cache.depth != -1 &&
		    atomic_read(&priv->token) == ssbootfs_inode->lookup_cache.depth_token) {
			depth = ssbootfs_inode->lookup_cache.depth + 1;
			ssbootfs_debug("have cached depth from parent (%s) of %s, depth is %d\n",
					d_name(parent), d_name(dentry), depth);

#ifdef CONFIG_SNSC_SSBOOT_FS_DEBUG_VERBOSE_DEPTH_CACHE_VALIDATE
			real_depth = ssbootfs_mapping_map_depth_slow(dentry);
			BUG_ON(depth != real_depth);
#endif
		}
	}

	dput(parent);
#if DEPTH_CACHE_ALWAYS_INVALID
	return -1;
#else
	return depth;
#endif
}

static void ssbootfs_mapping_map_depth_set(struct dentry *dentry, int depth)
{
	struct ssbootfs_priv *priv = ssbootfs_priv_from_dentry(dentry);
	struct inode *inode = d_inode(dentry);
	struct ssbootfs_inode *ssbootfs_inode;

	if (inode) {
		ssbootfs_inode = SSBOOTFS_I_TO_SSBFSI(inode);
		if (SSBOOTFS_DENTRY_IS_CACHABLE(ssbootfs_inode)) {
			ssbootfs_inode->lookup_cache.depth = depth;
			ssbootfs_inode->lookup_cache.depth_token = atomic_read(&priv->token);
		}
	}
}
#else
static int ssbootfs_mapping_map_depth_very_fast(struct dentry *dentry)
{
	return -1;
}

static int ssbootfs_mapping_map_depth_fast(struct dentry *dentry)
{
	return -1;
}



static void ssbootfs_mapping_map_depth_set(struct dentry *dentry, int depth)
{
}
#endif

#ifndef CONFIG_SNSC_SSBOOT_FS_DEBUG_DISABLE_DENTRY_LINKAGE
void ssbootfs_mapping_real_dentry_set(struct dentry *dentry, struct dentry *real_dentry)
{
	struct inode *inode = d_inode(dentry);
	struct ssbootfs_inode *ssbootfs_inode;

#ifdef CONFIG_SNSC_SSBOOT_FS_DEBUG_SANITY_CHECKS
	/* We should always have an inode really */
	WARN_ON(!inode);
#endif

	/*
	 * Instead of handling every case where we might have gotten
	 * NULL instead of a dentry before coming in here just handle
	 * it here by returning.
	 *
	 * This doesn't unset the cached dentry. It just means you
	 * don't have to null check the dentry in the caller.
	 */
	if (!real_dentry)
		return;

	if (!inode)
		return;

	ssbootfs_inode = SSBOOTFS_I_TO_SSBFSI(inode);
	ssbootfs_inode_lock(inode);
	if (SSBOOTFS_DENTRY_IS_CACHABLE(ssbootfs_inode)) {
		/* If the dentry is already cached just return */
		if (!ssbootfs_inode->lookup_cache.needs_lookup){
#ifdef CONFIG_SNSC_SSBOOT_FS_DEBUG_SANITY_CHECKS
			BUG_ON(real_dentry != ssbootfs_inode->lookup_cache.real_dentry);
#endif
			goto unlock;
		}

		ssbootfs_inode->lookup_cache.needs_lookup = false;
		ssbootfs_debug("caching dentry for %s %px\n", d_name(dentry), real_dentry);
#ifdef CONFIG_SNSC_SSBOOT_FS_DEBUG_SANITY_CHECKS
		BUG_ON(ssbootfs_inode->lookup_cache.real_dentry);
#endif
		dget(real_dentry);
		ssbootfs_inode->lookup_cache.real_dentry = real_dentry;
	}

unlock:
	ssbootfs_inode_unlock(inode);
}
#else
void ssbootfs_mapping_real_dentry_set(struct dentry *dentry, struct dentry *real_dentry)
{
}
#endif

#ifndef CONFIG_SNSC_SSBOOT_FS_DEBUG_DISABLE_DENTRY_LINKAGE
static struct dentry *ssbootfs_mapping_real_dentry_get(struct dentry *dentry)
{

	struct inode *inode = d_inode(dentry);
	struct ssbootfs_inode *ssbootfs_inode;
	struct dentry *result = NULL;

	if (!inode)
		return NULL;

	ssbootfs_inode = SSBOOTFS_I_TO_SSBFSI(inode);
	ssbootfs_inode_lock(inode);
	if (SSBOOTFS_DENTRY_IS_CACHABLE(ssbootfs_inode) && !ssbootfs_inode->lookup_cache.needs_lookup) {
		dget(ssbootfs_inode->lookup_cache.real_dentry);
		result = ssbootfs_inode->lookup_cache.real_dentry;
	}
	ssbootfs_inode_unlock(inode);

	return result;
}
#endif

/*
 * Invalidate the cached bits we have for an inode
 */
void ssbootfs_mapping_invalidate_inode(struct inode *inode)
{
	struct ssbootfs_inode *ssbootfs_inode = SSBOOTFS_I_TO_SSBFSI(inode);

	if (SSBOOTFS_DENTRY_IS_CACHABLE(ssbootfs_inode)) {
		ssbootfs_inode->lookup_cache.needs_lookup = true;

#ifndef CONFIG_SNSC_SSBOOT_FS_DEBUG_DISABLE_DENTRY_LINKAGE
		if (ssbootfs_inode->lookup_cache.real_dentry) {
			ssbootfs_debug("releasing real dentry for %s\n", d_name(ssbootfs_inode->lookup_cache.real_dentry));
			dput(ssbootfs_inode->lookup_cache.real_dentry);
			ssbootfs_inode->lookup_cache.real_dentry = NULL;
		}
#endif

#ifndef CONFIG_SNSC_SSBOOT_FS_DEBUG_DISABLE_DEPTH_CACHE
		ssbootfs_inode->lookup_cache.depth = -1;
#ifdef CONFIG_SNSC_SSBOOT_FS_DEBUG_VERBOSE_STATS
		ssbootfs_inode->lookup_cache.depth_hits = 0;
#endif
#endif
	} else {
		ssbootfs_debug("not invalidating inode, not a directory\n");
	}
}

/* we're looking up root, */
static int ssbootfs_mapping_map_dentry_very_fast(struct dentry *dentry, struct dentry **res)
{
	struct ssbootfs_priv *priv = ssbootfs_priv_from_dentry(dentry);

	/* we're working in the root directory so dget it and get out of here */
	ssbootfs_debug("doing very fast mapping lookup for %s\n", d_name(dentry));
	if (IS_ROOT(dentry)) {
		dget(priv->backingfs.dentry);
		*res = priv->backingfs.dentry;
		return 0;
	}
	return -1;
}

/*
 * This is the fast path to mapping a fake dentry to a real one.
 * This uses the real dentry that has been cached.
 */
static int ssbootfs_mapping_map_dentry_fast(struct dentry *dentry, struct dentry **res)
{
#ifndef CONFIG_SNSC_SSBOOT_FS_DEBUG_DISABLE_DENTRY_LINKAGE
	struct inode *inode = d_inode(dentry);
	struct ssbootfs_inode *ssbootfs_inode;
#endif
	int ret = -1;

#ifndef CONFIG_SNSC_SSBOOT_FS_DEBUG_DISABLE_DENTRY_LINKAGE
	ssbootfs_debug("trying fast lookup for %s\n", d_name(dentry));

	if (!inode)
		return -1;

	ssbootfs_inode = SSBOOTFS_I_TO_SSBFSI(inode);
	ssbootfs_inode_lock(inode);
	if (SSBOOTFS_DENTRY_IS_CACHABLE(ssbootfs_inode) && !ssbootfs_inode->lookup_cache.needs_lookup) {
		dget(ssbootfs_inode->lookup_cache.real_dentry);
		*res = ssbootfs_inode->lookup_cache.real_dentry;
		ssbootfs_debug("using cached dentry for %s %px %px\n", d_name(dentry), dentry, *res);
#ifdef CONFIG_SNSC_SSBOOT_FS_DEBUG_VERBOSE_STATS
		ssbootfs_inode->lookup_cache.dentry_hits += 1;
		ssbootfs_debug("%u dentry hits\n", ssbootfs_inode->lookup_cache.dentry_hits);
#endif
		ret = 0;
	}
	ssbootfs_inode_unlock(inode);
#endif

	return ret;
}

/*
 * This is the slow path to mapping a fake dentry to a real one.
 * This works by doing multiple reverse traversals on the fake tree
 * and one forward traversal on the real tree. Yes,.. it's that bad.
 *
 * This is currently O((depth(depth+1)/2) + depth) worst case and
 * O(1) best case I think.
 */
static int ssbootfs_mapping_map_dentry_slow(struct dentry *dentry, struct dentry **res)
{
	struct ssbootfs_priv *priv = ssbootfs_priv_from_dentry(dentry);
	struct dentry *real_root = priv->backingfs.dentry, *d = NULL, *dd = NULL,
			*current_path_element = NULL, *last_path_element = NULL;
#ifndef CONFIG_SNSC_SSBOOT_FS_DEBUG_DISABLE_DENTRY_LINKAGE
#ifndef CONFIG_SNSC_SSBOOT_FS_DEBUG_DISABLE_DENTRY_TELEPORT
	struct dentry *cached_dentry = NULL;
#endif
#endif
	unsigned int i, j, depth;
	int ret = 0;

#ifdef CONFIG_SNSC_SSBOOT_FS_DEBUG_SANITY_CHECKS
	BUG_ON(!dentry);
	BUG_ON(!res);
	BUG_ON(!priv->backingfs.dentry);
#endif

	ssbootfs_debug("doing slow mapping lookup for %s\n", d_name(dentry));

	depth = ssbootfs_mapping_map_depth_very_fast(dentry);
	if (depth == -1) {
		depth = ssbootfs_mapping_map_depth_fast(dentry);
		ssbootfs_mapping_map_depth_set(dentry, depth);
	}
	if (depth == -1) {
		depth = ssbootfs_mapping_map_depth_slow(dentry);
		ssbootfs_mapping_map_depth_set(dentry, depth);
	}

	for (i = 0; i < depth; i++) {
		d = dentry;
		dget(d);
		dd = NULL;

		/*
		 * d moves backwards down the path until it hits the current depth
		 * while it's current value is stored in dd so we can look up it's
		 * path element to get the next part in the real fs.
		 *
		 * The final d and dd need to be put after this loop.
		 */
		for (j = depth; j > i; j--) {
			/* the previous value of dd will now become invisible so put it */
			if (dd)
				dput(dd);
			dd = d;
			d = dget_parent(d);
#ifndef CONFIG_SNSC_SSBOOT_FS_DEBUG_DISABLE_DENTRY_LINKAGE
#ifndef CONFIG_SNSC_SSBOOT_FS_DEBUG_DISABLE_DENTRY_TELEPORT
			cached_dentry = ssbootfs_mapping_real_dentry_get(d);
			if (cached_dentry) {
				ssbootfs_debug("teleporting from %d to %d\n", i, j);
				i = j - 1;
				last_path_element = cached_dentry;
				break;
			}
#endif
#endif
		}

		/*
		 * This should walk current_path_element forward by one path element at a time
		 * after the initial iteration.
		 */
		ret = ssbootfs_backingfs_lookup(priv, i == 0 ? real_root : last_path_element,
				d_name(dd), &current_path_element);

		/* dd and d will have been dget'd by dget_parent so release them */
		dput(d);
		dput(dd);

		/* dput any lingering path element we've looked up */
		if (last_path_element)
			dput(last_path_element);

		if (ret)
			goto out;

		/* move mapped to the result for the next loop */
		last_path_element = current_path_element;
	}

	*res = current_path_element;

out:
	return ret;
}

/*
 * For a given fake dentry map it to a real backing fs dentry.
 * The result if there is one should be free'd with dput
 */
int ssbootfs_mapping_map_dentry(struct dentry *dentry, struct dentry **res)
{
	struct ssbootfs_priv *priv = ssbootfs_priv_from_dentry(dentry);
	int ret;

#ifdef CONFIG_SNSC_SSBOOT_FS_DEBUG_SANITY_CHECKS
	bool resultnamedifferent;
#endif

	down_read(&priv->tree_rw_sem);

#ifdef CONFIG_SNSC_SSBOOT_FS_DEBUG_SANITY_CHECKS
	BUG_ON(!dentry);
	BUG_ON(!res);
#endif

	ret = ssbootfs_mapping_map_dentry_very_fast(dentry, res);
	if (ret) {
		ret = ssbootfs_mapping_map_dentry_fast(dentry, res);
		if (ret) {
			ret = ssbootfs_mapping_map_dentry_slow(dentry, res);
		}
	}

/*
 * This is an extra check to catch race conditions.
 * xfstests was able to cause the original dentry and resulting dentry
 * to differ somewhere between doing the lookup and getting here.
 */
#ifdef CONFIG_SNSC_SSBOOT_FS_DEBUG_SANITY_CHECKS
	if (!ret) {
		resultnamedifferent = strcmp(d_name(dentry), d_name(*res)) != 0;
		if (resultnamedifferent) {
			ssbootfs_err("was looking for %s but result is %s\n", d_name(dentry), d_name(*res));
			BUG();
		}
	}
#endif

	up_read(&priv->tree_rw_sem);

	return ret;
}

/* Take a real parent directory and get a new negative dentry for item in the parent */
struct dentry *ssbootfs_map_target_dentry(struct dentry *mapped_parent, struct dentry *name_src)
{
	struct dentry *target = lookup_one_len(d_name(name_src), mapped_parent,
			name_src->d_name.len);
	return target;
}
