// SPDX-License-Identifier: GPL-2.0
/*
 *  ssbootfs: private header
 *
 *  Copyright 2021 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 */

#ifndef FS_SSBOOTFS_SSBOOTFS_H_
#define FS_SSBOOTFS_SSBOOTFS_H_

#include <linux/fs.h>
#include <linux/fs_stack.h>
#include <linux/kobject.h>
#include <linux/pipe_fs_i.h>
#include <linux/mount.h>
#include <linux/cred.h>
#include <linux/pid.h>
#include <linux/atomic.h>
#include <linux/sizes.h>

#define SSBOOTFS_NAME			"ssbootfs"
#define SSBOOTFS_LOG			SSBOOTFS_NAME": "
#define SSBOOTFS_SYSFS_BACKINGFS	"backingfs"

/*
 * copy back the new attributes into our inode
 * This is only really needed when the real inode is mutated.
 * atime is always read through the real inode.
 */
#define SSBOOTFS_SYNC_INODE(proxy, real) { \
	fsstack_copy_inode_size(proxy, real); \
	fsstack_copy_attr_all(proxy, real); \
}

/*
 * Turning this on makes ssbootfs very noisy and outputs
 * messages during critical paths, can cause systemd timeouts,
 * write amplification from journald trying to write all of the
 * messages out and complaining it can't causing even more
 * messages. This should only ever be turned on when debugging
 * ssbootfs mounted not as the rootfs.
 */
#ifdef CONFIG_SNSC_SSBOOT_FS_DEBUG_VERBOSE
	#define ssbootfs_debug(msg, ...) pr_debug(SSBOOTFS_LOG msg, ##__VA_ARGS__)
#else
	#define ssbootfs_debug(msg, ...)
#endif

/*
 * This prints out information that is useful for debugging how applications
 * are interacting with ssbootfs like what files were active when a snapshot
 * was taken. The level is debug so you either totally disable it at build
 * time or use dynamic debug to enable the messages at runtime.
 */
#define ssbootfs_debug_info(msg, ...) pr_debug(SSBOOTFS_LOG msg, ##__VA_ARGS__)

/* Something is happening, not an error, just an indication that something is happening */
#define ssbootfs_info(msg, ...) pr_info(SSBOOTFS_LOG msg, ##__VA_ARGS__)
/* Something unexpected but not fatal has happened */
#define ssbootfs_warn(msg, ...) pr_warn(SSBOOTFS_LOG msg, ##__VA_ARGS__)
/* Something bad has happened */
#define ssbootfs_err(msg, ...) pr_err(SSBOOTFS_LOG msg, ##__VA_ARGS__)

#define SSBOOTFS_CHECK_ATTACHED(priv) \
				if (!(priv)->attached) { \
					ssbootfs_err("%s called while backing fs is detached", __FUNCTION__); \
				} \
				if (!(priv)->attached)

struct ssbootfs_priv {
	/* lock for fields in this struct */
	struct mutex lock;

	/* marshalling for races between lookups and tree changes */
	struct rw_semaphore tree_rw_sem;

	const char *name;

	struct super_block *sb;

	/* Is the backing fs attached, has it ever been attached? */
	bool attached;
	bool has_attached;
	bool in_suspend;

#ifdef CONFIG_SNSC_SSBOOT_FS_IN_KERNEL_MOUNT
	const char *backingdev;
	const char *backingtype;
	char backingopts[SZ_512];
#endif
	struct path backingfs;

	/*
	 * If the backingfs has ever been attached this contains
	 * the uuid of the fs
	 */
	uuid_t backingfs_uuid;

	/* sysfs bits */
	struct kobject *mnt_root;
	struct kobject *backingfs_mntpoint;
	struct kobj_attribute attach_attr;
	struct kobj_attribute disengage_attr;
	struct kobj_attribute detach_attr;
	struct kobj_attribute status_attr;

	/* bdi bits */
	int bdi_id;

	/*
	 * this is a token you can use to track changes to the
	 * filesystem tree to ignore potentially inconsistent caches
	 * etc
	 */
	atomic_t token;
};

/* lookup cache */
struct ssbootfs_lookup_cache {
	bool needs_lookup;
#ifndef CONFIG_SNSC_SSBOOT_FS_DEBUG_DISABLE_DENTRY_LINKAGE
	struct dentry *real_dentry;
#endif

#ifndef CONFIG_SNSC_SSBOOT_FS_DEBUG_DISABLE_DEPTH_CACHE
	/* calculated depth */
	int depth;
	/* token for the depth */
	int depth_token;
#endif

#ifdef CONFIG_SNSC_SSBOOT_FS_DEBUG_VERBOSE_STATS
	/* stats */
#ifndef CONFIG_SNSC_SSBOOT_FS_DEBUG_DISABLE_DEPTH_CACHE
	unsigned int depth_hits;
	unsigned int dentry_hits;
#endif
#endif
};

/* extended inode */
struct ssbootfs_inode {
	struct inode vfs_inode;

	struct mutex lock;

	/* tracking of open files for this inode */
	struct list_head files;

	/* look up cache, valid for directories */
	struct ssbootfs_lookup_cache lookup_cache;

	/* for disengage */
	struct list_head disengage;
};

/* file priv */

/* File status flags */
#define SSBOOTFS_FLAG_VALID	BIT(0)
#define SSBOOTFS_FLAG_MMAPPED	BIT(1)
#define SSBOOTFS_FLAG_MMAPPED_W	BIT(2)

/*
 * The context which is holding the lock within ssbootfs_file_priv.
 * This is for situations where while inside of a callback we trigger
 * calling another callback and the second callback needs to skip taking
 * the lock as trying to do so would cause a dead lock.
 * i.e. If we are being called from vfs_write(), allow mmap read page to
 * run if the data being written is from a mmap'd region of the file
 * without taking the lock relying on that fact that the write callback
 * has the lock and is waiting for read page to complete.
 */
#define SSBOOTFS_LOCK_CONTEXT_WRITE	BIT(0)
#define SSBOOTFS_LOCK_CONTEXT_READ	BIT(1)
#define SSBOOTFS_LOCK_CONTEXT_RELEASE	BIT(2)
#define SSBOOTFS_LOCK_CONTEXT_FSYNC	BIT(3)
#define SSBOOTFS_LOCK_CONTEXT_FLUSH	BIT(4)
#define SSBOOTFS_LOCK_CONTEXT_DISENGAGE	BIT(5)
#define SSBOOTFS_LOCK_CONTEXT_FADVISE	BIT(6)
#define SSBOOTFS_LOCK_CONTEXT_OTHER	BIT(7)

/*
 * We are being called from release(), fsync() or flush() and the lock is held
 * so write the dirty pages back without taking the lock.
 */
#define SSBOOTFS_LOCK_CONTEXT_MMAPWSAFE_MASK (SSBOOTFS_LOCK_CONTEXT_RELEASE  | \
					      SSBOOTFS_LOCK_CONTEXT_FSYNC    | \
					      SSBOOTFS_LOCK_CONTEXT_FLUSH    | \
					      SSBOOTFS_LOCK_CONTEXT_DISENGAGE | \
						  SSBOOTFS_LOCK_CONTEXT_FADVISE)

struct ssbootfs_file_priv {
	/* A mutex to protect everything except lock_context. */
	struct mutex lock;
	/*
	 * If the lock is being held this contains where
	 * so you can proceed without taking the lock if that is safe.
	 * The context must be updated while holding the lock.
	 * The context where the lock is held must ensure that anything
	 * nested that is going to ignore the lock finishes completely
	 * before it releases the lock.
	 */
	atomic_t lock_context;

	u32 flags;
	struct list_head i_files;
	struct file *fake_file;
	struct file *real_file;
};

/* Convert a garden variety inode back into an ssbootfs inode */
#define SSBOOTFS_I_TO_SSBFSI(inode) container_of(inode, struct ssbootfs_inode, vfs_inode)

/* priv helpers */
static inline void ssbootfs_priv_lock(struct ssbootfs_priv *priv)
{
	mutex_lock(&priv->lock);
}

static inline void ssbootfs_priv_unlock(struct ssbootfs_priv *priv)
{
	mutex_unlock(&priv->lock);
}

/*
 * Use priv as a makeshift barrier.
 * When the real filesystem is being ripped out priv will be locked
 * for the whole process so it will not be possible to return from
 * here while that is happening.
 *
 * You *must* pass the barrier before taking either the inode or the
 * file lock.
 */
static inline void ssbootfs_barrier(struct ssbootfs_priv *priv)
{
	ssbootfs_priv_lock(priv);
	ssbootfs_priv_unlock(priv);
}

/* dentry mapping */
int ssbootfs_mapping_map_dentry(struct dentry *dentry, struct dentry **res);
struct dentry *ssbootfs_map_target_dentry(struct dentry *mapped_parent, struct dentry *name_src);
void ssbootfs_mapping_invalidate_inode(struct inode *inode);
void ssbootfs_mapping_real_dentry_set(struct dentry *dentry, struct dentry *real_dentry);

/* backing filesystem helpers */
int ssbootfs_backingfs_mntpath(struct ssbootfs_priv *priv, struct path *path);

int ssbootfs_backingfs_lookup(struct ssbootfs_priv *priv, struct dentry *dentry,
								const char *name, struct dentry **res);

ssize_t ssbootfs_backingfs_read_iter(struct kiocb *cb, struct iov_iter *iter);
ssize_t ssbootfs_backingfs_splice_read(struct file *in, loff_t *ppos,
				 struct pipe_inode_info *pipe, size_t len,
				 unsigned int flags);
ssize_t ssbootfs_backingfs_write_iter(struct kiocb *cb, struct iov_iter *iter);
ssize_t ssbootfs_backingfs_splice_write(struct pipe_inode_info *pipe, struct file *out,
							loff_t *ppos, size_t len, unsigned int flags);

int ssbootfs_backingfs_create(struct user_namespace *ns, struct inode *dir, struct dentry *dentry,
		umode_t mode, bool want_excl, dev_t dev);
int ssbootfs_backingfs_rename(struct inode *old_dir, struct dentry *old_dentry,
		struct inode *new_dir, struct dentry *new_dentry, unsigned int flags);
struct file *ssbootfs_backingfs_open(struct file *file);
void ssbootfs_backingfs_close(struct file *file);
int ssbootfs_backingfs_fsync(struct file *file, int datasync);
int ssbootfs_backingfs_fsync_range(struct file *f, loff_t x, loff_t y, int datasync);
int ssbootfs_backingfs_flush(struct file *file, fl_owner_t id);

int ssbootfs_backingfs_file_readpage(struct file *file, struct page *page);
int ssbootfs_backingfs_file_writepage(struct file *file, struct page *page);

loff_t ssbootfs_backingfs_llseek(struct file *file, loff_t offset, int whence);

/* inode stuff */
extern const struct inode_operations ssbootfs_inode_ops;
struct inode *ssbootfs_inode_new_fake_inode(struct super_block *sb, u64 ino, u32 generation, umode_t mode);
int ssbootfs_disengage_priv_locked(struct ssbootfs_priv *priv);
unsigned long ssbootfs_inode_disengage(struct ssbootfs_priv *priv, struct inode *inode);
static inline void ssbootfs_inode_lock(struct inode *inode)
{
	struct ssbootfs_inode *ssbootfs_inode = SSBOOTFS_I_TO_SSBFSI(inode);

	mutex_lock(&ssbootfs_inode->lock);
}

static inline void ssbootfs_inode_unlock(struct inode *inode)
{
	struct ssbootfs_inode *ssbootfs_inode = SSBOOTFS_I_TO_SSBFSI(inode);

	mutex_unlock(&ssbootfs_inode->lock);
}

/*
 * Test if the left and right inodes are the *same*.
 * This relies on the inodes having stable numbers and the inode
 * generation id being correctly updated when a number is reused.
 */
static inline bool ssbootfs_inode_issame(struct inode *left, struct inode *right)
{
	return ((left->i_ino == right->i_ino) &&
	    (left->i_generation == right->i_generation));
}

/* dentry stuff */
extern const struct dentry_operations ssbootfs_dentry_ops;

/* file stuff */
extern const struct file_operations ssbootfs_dir_fops;
extern const struct file_operations ssbootfs_file_fops;
extern const struct file_operations ssbootfs_symlink_fops;
extern const struct address_space_operations ssbootfs_file_aops;
int ssbootfs_file_disengage(struct ssbootfs_priv *priv, struct file *file);

/* file helpers */
static inline int ssbootfs_sync_real_file_from_fake_file(struct file *fakefile, struct file *realfile)
{
	/* This doesn't seem to be needed */
	/*fsstack_copy_inode_size(realfile->f_inode, fakefile->f_inode);*/
	return vfs_setpos(realfile, fakefile->f_pos, fakefile->f_pos);
}

/* dentry helpers */
static inline void lock_dentry(struct dentry *dentry)
{
	dget(dentry);
	inode_lock_nested(d_inode(dentry), I_MUTEX_PARENT);
}

static inline void unlock_dentry(struct dentry *dir)
{
	inode_unlock(d_inode(dir));
	dput(dir);
}

/* file priv */
struct ssbootfs_file_priv *ssbootfs_file_priv_alloc(void);
void ssbootfs_file_priv_free(struct ssbootfs_file_priv *file_priv);

/* misc helpers */
static inline struct inode *ssbootfs_inode_from_file(struct file *file)
{
	struct dentry *dentry = file->f_path.dentry;

	return dentry->d_inode;
}

static inline struct super_block *ssbootfs_sb_from_file(struct file *file)
{
	return ssbootfs_inode_from_file(file)->i_sb;
}

static inline struct super_block *ssbootfs_sb_from_inode(struct inode *inode)
{
	return inode->i_sb;
}

static inline struct ssbootfs_priv *ssbootfs_priv_from_file(struct file *file)
{
	return (struct ssbootfs_priv *) ssbootfs_sb_from_file(file)->s_fs_info;
}

static inline struct ssbootfs_priv *ssbootfs_priv_from_inode(struct inode *inode)
{
	return (struct ssbootfs_priv *) ssbootfs_sb_from_inode(inode)->s_fs_info;
}

static inline struct ssbootfs_priv *ssbootfs_priv_from_dentry(struct dentry *dentry)
{
	struct ssbootfs_priv *priv = NULL;

	if (dentry->d_sb) {
		priv = dentry->d_sb->s_fs_info;
	}
	return priv;
}

static inline struct ssbootfs_priv *ssbootfs_priv_from_path(const struct path *path)
{
	return path->mnt->mnt_sb->s_fs_info;
}

int ssbootfs_attach_unlocked(struct ssbootfs_priv *priv);
int ssbootfs_attach(struct ssbootfs_priv *priv);
int ssbootfs_disengage_sb(struct ssbootfs_priv *priv);
int ssbootfs_detach(struct ssbootfs_priv *priv, bool in_suspend);

/*
 * For filesystems that support hardlinks a single inode
 * can appear in multiple places and have multiple dentrys that
 * point to it. The dentry caching creates a 1:1 mapping between
 * inode and dentry so that won't work.
 * Directories can only be linked to a single location so it's
 * safe to map a directory inode to a single dentry.
 */
#define SSBOOTFS_DENTRY_IS_CACHABLE(fi) (S_ISDIR(fi->vfs_inode.i_mode))

unsigned int ssbootfs_debug_getrealmntcnt(struct ssbootfs_priv *priv);

/*
 * Some tricky access patterns do things like open
 * a path as one user and then switch the user to another
 * that would not normally have access to the path.
 *
 * Because some of the kernel interfaces don't give us
 * the existing context to work from we need to do a lookup
 * or something again but now we're a user that can't
 * complete that look up.
 *
 * This allows switching to root for vfs operations to work
 * around that. You must restore the creds when you're done!
 */

struct ssbootfs_creds {
	const struct cred *original;
	struct cred *override;
};

static inline int ssbootfs_switch_to_root(struct ssbootfs_creds *creds)
{
	struct cred *tmp_cred = prepare_creds();

	if (!tmp_cred)
		return -ENOMEM;

	tmp_cred->fsuid.val = 0;
	tmp_cred->fsgid.val = 0;

	creds->original = override_creds(tmp_cred);
	creds->override = tmp_cred;

	return 0;
}

static inline void ssbootfs_switch_from_root(struct ssbootfs_creds *creds)
{
	revert_creds(creds->original);
	put_cred(creds->override);
}

#define d_name(d) ((d)->d_name.name)
#define f_name(f) (d_name((f)->f_path.dentry))

static inline pid_t ssbootfs_file_pid(struct file *file)
{
	return pid_vnr(file->f_owner.pid);
}

#ifdef CONFIG_SNSC_SSBOOT_FS_IN_KERNEL_MOUNT
int ssbootfs_util_mount(struct ssbootfs_priv *priv);
#define vfs_mount_id(m) (container_of(m, struct mount, mnt))->mnt_id
#endif

#endif /* FS_SSBOOTFS_SSBOOTFSFS_H_ */
