// SPDX-License-Identifier: GPL-2.0
/*
 *  ssbootfs: sysfs interface
 *
 *  Copyright 2021 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 */

#include <linux/kobject.h>

#include "ssbootfs.h"
#include "ssbootfs_util.h"
#include "sysfs.h"

static struct kobject *ssbootfs_root;

#define ATTR_TO_PRIV(attr, field) container_of(attr, struct ssbootfs_priv, field)

static ssize_t detachall_store(struct kobject *kobj, struct kobj_attribute *attr,
								const char *buf, size_t count)
{
	ssbootfs_util_disengage_all();
	ssbootfs_util_detach_all();
	return count;
}

static struct kobj_attribute detachall_attribute = __ATTR_WO(detachall);

static ssize_t attachall_store(struct kobject *kobj, struct kobj_attribute *attr,
								const char *buf, size_t count)
{
	ssbootfs_util_reattach_all();
	return count;
}

static struct kobj_attribute attachall_attribute = __ATTR_WO(attachall);

static ssize_t attach_store(struct kobject *kobj, struct kobj_attribute *attr,
							const char *buf, size_t count)
{
	struct ssbootfs_priv *priv = ATTR_TO_PRIV(attr, attach_attr);

	ssbootfs_attach(priv);
	return count;
}

static struct kobj_attribute attach_attribute = __ATTR_WO(attach);

static ssize_t disengage_store(struct kobject *kobj, struct kobj_attribute *attr,
								const char *buf, size_t count)
{
	struct ssbootfs_priv *priv = ATTR_TO_PRIV(attr, disengage_attr);
	int ret;

	ret = ssbootfs_disengage_sb(priv);
	if (ret)
		return ret;
	return count;
}

static struct kobj_attribute disengage_attribute = __ATTR_WO(disengage);

static ssize_t detach_store(struct kobject *kobj, struct kobj_attribute *attr,
							const char *buf, size_t count)
{
	struct ssbootfs_priv *priv = ATTR_TO_PRIV(attr, detach_attr);
	int ret;

	/*
	 * detach might have triggered sys to get reattached
	 * this should probably warn that this is happening
	 */
	ret = ssbootfs_disengage_sb(priv);
	if (ret)
		return ret;

	ret = ssbootfs_detach(priv, false);
	if (ret)
		return ret;
	return count;
}

static struct kobj_attribute detach_attribute = __ATTR_WO(detach);

static ssize_t status_show(struct kobject *kobj, struct kobj_attribute *attr, char *buf)
{
	struct ssbootfs_priv *priv = ATTR_TO_PRIV(attr, status_attr);
	int ret;
	int status = 0;

	ssbootfs_priv_lock(priv);

	if (priv->attached)
		status = 1;

	ssbootfs_priv_unlock(priv);

	ret = sprintf(buf, "%d\n", status);

	return ret;
}

static struct kobj_attribute status_attribute = __ATTR_RO(status);

int ssbootfs_sysfs_init_mount(struct ssbootfs_priv *priv)
{
	int ret;

	priv->mnt_root = kobject_create_and_add(priv->name, ssbootfs_root);
	if (!priv->mnt_root) {
		ssbootfs_err("Failed to create sysfs root for mount, there is probably another mount with the same name");
		return -ENOMEM;
	}

	memcpy(&priv->attach_attr, &attach_attribute, sizeof(attach_attribute));
	memcpy(&priv->disengage_attr, &disengage_attribute, sizeof(disengage_attribute));
	memcpy(&priv->detach_attr, &detach_attribute, sizeof(detach_attribute));
	memcpy(&priv->status_attr, &status_attribute, sizeof(status_attribute));

	sysfs_attr_init(&priv->attach_attr.attr);
	sysfs_attr_init(&priv->disengage_attr.attr);
	sysfs_attr_init(&priv->detach_attr.attr);
	sysfs_attr_init(&priv->status_attr.attr);

	ret = sysfs_create_file(priv->mnt_root, &priv->attach_attr.attr);
	if (ret)
		goto err_attach;

	ret = sysfs_create_file(priv->mnt_root, &priv->disengage_attr.attr);
	if (ret)
		goto err_disengage;

	ret = sysfs_create_file(priv->mnt_root, &priv->detach_attr.attr);
	if (ret)
		goto err_detach;

	ret = sysfs_create_file(priv->mnt_root, &priv->status_attr.attr);
	if (ret)
		goto err_status;

	priv->backingfs_mntpoint = kobject_create_and_add(SSBOOTFS_SYSFS_BACKINGFS, priv->mnt_root);
	if (!priv->backingfs_mntpoint)
		goto err_mntpoint;

	return 0;

err_mntpoint:
	sysfs_remove_file(priv->mnt_root, &priv->status_attr.attr);
err_status:
	sysfs_remove_file(priv->mnt_root, &priv->detach_attr.attr);
err_detach:
	sysfs_remove_file(priv->mnt_root, &priv->disengage_attr.attr);
err_disengage:
	sysfs_remove_file(priv->mnt_root, &priv->attach_attr.attr);
err_attach:
	kobject_put(priv->mnt_root);
	return ret;
}

void ssbootfs_sysfs_destroy_mount(struct ssbootfs_priv *priv)
{
	sysfs_remove_file(priv->mnt_root, &priv->attach_attr.attr);
	sysfs_remove_file(priv->mnt_root, &priv->disengage_attr.attr);
	sysfs_remove_file(priv->mnt_root, &priv->detach_attr.attr);
	sysfs_remove_file(priv->mnt_root, &priv->status_attr.attr);
	kobject_put(priv->backingfs_mntpoint);
	kobject_put(priv->mnt_root);
}

int __init ssbootfs_sysfs_init(void)
{
	int ret;

	ssbootfs_root = kobject_create_and_add(SSBOOTFS_NAME, fs_kobj);
	if (!ssbootfs_root)
		return -ENOMEM;

	ret = sysfs_create_file(ssbootfs_root, &detachall_attribute.attr);
	if (ret)
		goto out_put_root;

	ret = sysfs_create_file(ssbootfs_root, &attachall_attribute.attr);
	if (ret)
		goto out_remove_detachall;

	return 0;

out_remove_detachall:
	sysfs_remove_file(ssbootfs_root, &detachall_attribute.attr);
out_put_root:
	kobject_put(ssbootfs_root);
	return ret;
}

void ssbootfs_sysfs_exit(void)
{
	kobject_put(ssbootfs_root);
	ssbootfs_root = NULL;
}
