// SPDX-License-Identifier: GPL-2.0
// Copyright 2022 Sony Corporation, SOCIONEXT INC.
// Portions Copyright (C) 2024 Synopsys, Inc.  Used with permission. All rights reserved.

#ifndef _DMAC_LLD_H
#define _DMAC_LLD_H

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*
 * LLD_ER
 */
typedef int32_t LLD_ER;
#define LLD_OK			(0)	/* Normal end or Success */
#define LLD_DUPLICATE_OPEN_ERR	(-1)	/* Specified ch is already opened */
#define LLD_NOT_OPEN_ERR	(-2)	/* Specified ch is not opened */
#define LLD_HW_BUSY_ERR		(-3)	/* Busy state, under DMA transfer */
#define LLD_INVALID_CALLBACK_ERR (-4)	/* Callback function is invalid */
#define LLD_DATA_ERR		(-9)	/* Parameter error */

/*
 * LLD_CH
 */
typedef int32_t LLD_CH;
/* Channels in DMAC id = <0> */
#define LLD_CH0			(0)	/* CH0  */
#define LLD_CH1			(1)	/* CH1  */
#define LLD_CH2			(2)	/* CH2  */
#define LLD_CH3			(3)	/* CH3  */
#define LLD_CH4			(4)	/* CH4  */
#define LLD_CH5			(5)	/* CH5  */
#define LLD_CH6			(6)	/* CH6  */
#define LLD_CH7			(7)	/* CH7  */
/* Channels in DMAC id = <1> */
#define LLD_CH8			(8)	/* CH8  */
#define LLD_CH9			(9)	/* CH9  */
#define LLD_CH10		(10)	/* CH10 */
#define LLD_CH11		(11)	/* CH11 */
#define LLD_CH12		(12)	/* CH12 */
#define LLD_CH13		(13)	/* CH13 */
#define LLD_CH14		(14)	/* CH14 */
#define LLD_CH15		(15)	/* CH15 */
/* Channels in DMAC id = <2> */
#define LLD_CH16		(16)	/* CH16 */
#define LLD_CH17		(17)	/* CH17 */
#define LLD_CH18		(18)	/* CH18 */
#define LLD_CH19		(19)	/* CH19 */
#define LLD_CH20		(20)	/* CH20 */
#define LLD_CH21		(21)	/* CH21 */
#define LLD_CH22		(22)	/* CH22 */
#define LLD_CH23		(23)	/* CH23 */
/* Channels in DMAC id = <3> */
#define LLD_CH24		(24)	/* CH24 */
#define LLD_CH25		(25)	/* CH25 */
#define LLD_CH26		(26)	/* CH26 */
#define LLD_CH27		(27)	/* CH27 */
#define LLD_CH28		(28)	/* CH28 */
#define LLD_CH29		(29)	/* CH29 */
#define LLD_CH30		(30)	/* CH30 */
#define LLD_CH31		(31)	/* CH31 */

/*
 * AXI master
 */
#define LLD_DMAC_AXI_MASTER1	(0)	/* AXI Master 1 */
#define LLD_DMAC_AXI_MASTER2	(1)	/* AXI Master 2 */

/*
 * Address Increment
 */
#define LLD_DMAC_ADR_INC	(0)	/* Address is Increment */
#define LLD_DMAC_ADR_FIX	(1)	/* Address is Fixed */

/*
 * Transfer Width
 */
#define LLD_DMAC_WIDTH8		(0)	/*   8 bit */
#define LLD_DMAC_WIDTH16	(1)	/*  16 bit */
#define LLD_DMAC_WIDTH32	(2)	/*  32 bit */
#define LLD_DMAC_WIDTH64	(3)	/*  64 bit */
#define LLD_DMAC_WIDTH128	(4)	/* 128 bit */
#define LLD_DMAC_WIDTH256	(5)	/* 256 bit */
#define LLD_DMAC_WIDTH512	(6)	/* 512 bit */

/*
 * Burst Transaction Length
 */
#define LLD_DMAC_BSIZE1		(0)	/*    1 item  */
#define LLD_DMAC_BSIZE4		(1)	/*    4 items */
#define LLD_DMAC_BSIZE8		(2)	/*    8 items */
#define LLD_DMAC_BSIZE16	(3)	/*   16 items */
#define LLD_DMAC_BSIZE32	(4)	/*   32 items */
#define LLD_DMAC_BSIZE64	(5)	/*   64 items */
#define LLD_DMAC_BSIZE128	(6)	/*  128 items */
#define LLD_DMAC_BSIZE256	(7)	/*  256 items */
#define LLD_DMAC_BSIZE512	(8)	/*  512 items */
#define LLD_DMAC_BSIZE1024	(9)	/* 1024 items */

/*
 * LLD_FACT
 */
typedef uint32_t LLD_FACT;
#define LLD_DMAC_TRAN_END	(0)	/* Transfer complete */
#define LLD_DMAC_TRAN_ERR	(1)	/* Transfer end with error */

/*
 * LLD_FUNC_FLAG
 */
typedef uint32_t LLD_FUNC_FLAG;
#define LLD_FUNC_NOTHREAD	(0)	/* execute LLD_FUNC at interrupt context */
#define LLD_FUNC_THREADED	(1)	/* execute LLD_FUNC at thread context */

/*
 * Flow Control
 */
#define LLD_DMAC_MM_DMAC	(0)	/* memory-to-memory     (CTL:DMAC) */
#define LLD_DMAC_MP_DMAC	(1)	/* memory-to-peripheral (CTL:DMAC) */
#define LLD_DMAC_PM_DMAC	(2)	/* peripheral-to-memory (CTL:DMAC) */
#define LLD_DMAC_PM_PERI	(4)	/* peripheral-to-memory (CTL:SRC) */
#define LLD_DMAC_MP_PERI	(6)	/* memory-to-peripheral (CTL:DST) */

/*
 * Peripheral ID
 */
#define LLD_DMAC_REQ_SPI0_TX	(0)	/* SPI0 : TX */
#define LLD_DMAC_REQ_SPI0_RX	(1)	/* SPI0 : RX */
#define LLD_DMAC_REQ_SPI1_TX	(2)	/* SPI1 : TX */
#define LLD_DMAC_REQ_SPI1_RX	(3)	/* SPI1 : RX */
#define LLD_DMAC_REQ_SPI2_TX	(4)	/* SPI2 : TX */
#define LLD_DMAC_REQ_SPI2_RX	(5)	/* SPI2 : RX */
#define LLD_DMAC_REQ_SPI3_TX	(6)	/* SPI3 : TX */
#define LLD_DMAC_REQ_SPI3_RX	(7)	/* SPI3 : RX */
#define LLD_DMAC_REQ_UART2_TX	(8)	/* UART2 : TX */
#define LLD_DMAC_REQ_UART2_RX	(9)	/* UART2 : RX */
#define LLD_DMAC_REQ_UART3_TX	(10)	/* UART3 : TX */
#define LLD_DMAC_REQ_UART3_RX	(11)	/* UART3 : RX */

/**
 * Maximum values
 */
#define LLD_CH_MAX		(LLD_CH31)		/* LLD_CH */
#define LLD_DMAC_AXI_MASTER_MAX	(LLD_DMAC_AXI_MASTER2)	/* AXI master */
#define LLD_DMAC_ADR_MAX	(LLD_DMAC_ADR_FIX)	/* Address Increment */
#define LLD_DMAC_WIDTH_MAX	(LLD_DMAC_WIDTH128)	/* Transfer Width */
#define LLD_DMAC_BSIZE_MAX	(LLD_DMAC_BSIZE1024)	/* Burst Transaction Length */
#define LLD_FUNC_MAX		(LLD_FUNC_THREADED)	/* LLD_FUNC_FLAG */
#define LLD_DMAC_FC_MAX		(LLD_DMAC_MP_PERI)	/* Flow Control */
#define LLD_DMAC_PERI_MAX	(LLD_DMAC_REQ_UART3_RX)	/* Peripheral ID */
#define LLD_DMAC_PRIORITY_MAX	(0x7)			/* Priority */
#define LLD_DMAC_AXI_QOS_MAX	(0xF)			/* AXI QoS */
#define LLD_DMAC_OUTSTAND_MAX	(0xF)			/* Outstanding */
#define LLD_DMAC_TRANS_CNT_MAX	(0x3FFFFF)		/* Transfer count */

/**
 * DMA status
 */
#define LLD_DMAC_END		(0)	/* Transfer has finished */
#define LLD_DMAC_EXEC		(3)	/* Transfer is running */

/**
 * DMA registers
 */
/* CH_CFG */
#define CH_CFG_H_SRC_OSR_LMT_POS	(23)
#define CH_CFG_H_DST_OSR_LMT_POS	(27)
#define CH_CFG_DST_MULTBLK_TYPE_POS	(CH_CFG_L_DST_MULTBLK_TYPE_POS)
#define CH_CFG_SRC_MULTBLK_TYPE_POS	(CH_CFG_L_SRC_MULTBLK_TYPE_POS)

/* CH_CTL */
#define CH_CTL_H_AWLEN_EN_POS		(15)
#define CH_CTL_H_ARLEN_EN_POS		(6)
#define CH_CTL_H_AW_PROT_POS		(3)
#define CH_CTL_H_AR_PROT_POS		(0)
#define CH_CTL_L_LASTWRITE_EN_POS	(30)
#define CH_CTL_L_AW_CACHE_POS		(26)
#define CH_CTL_L_AR_CACHE_POS		(22)

#define DWAXIDMAC_AXI_CACHE_DEVICE_NON_BUF			(0x0)
#define DWAXIDMAC_AXI_NON_CACHE_AND_NON_BUF			(0x2)
#define DWAXIDMAC_AXI_CACHE_WRITEBACK_AND_WRITE_ALLOCATE	(0xf)

/*
 * Callback Function Type
 */
typedef void(*LLD_FUNC)(LLD_CH ch, LLD_FACT fact, void *cookies);

/*
 * Linked List Pointer
 */
typedef struct {
	uint64_t	lms:1;		/* AXI Master to load the next LLI */
	uint64_t	__1:5;		/* reserved */
	uint64_t	lli_adr:58;	/* Physical address of Linked List Item structure */
} LLD_DMAC_IND_INFO;

/*
 * Transfer control
 */
typedef struct {
	uint64_t	f1_smch:1;	/* AXI master (src) */
	uint64_t	__1:1;		/* reserved */
	uint64_t	f1_dmch:1;	/* AXI master (dst) */
	uint64_t	__2:1;		/* reserved */
	uint64_t	f1_si:1;	/* Address Increment (src) */
	uint64_t	__3:1;		/* reserved */
	uint64_t	f1_di:1;	/* Address Increment (dst) */
	uint64_t	__4:1;		/* reserved */
	uint64_t	f3_swidth:3;	/* Transfer Width (src) */
	uint64_t	f3_dwidth:3;	/* Transfer Width (dst) */
	uint64_t	f4_sbsize:4;	/* Burst Transaction Length (src) */
	uint64_t	f4_dbsize:4;	/* Burst Transaction Length (dst) */
	uint64_t	f4_arcache:4;	/* AXI ar_cache signal */
	uint64_t	f4_awcache:4;	/* AXI aw_cache signal */
	uint64_t	f1_lastwren:1;	/* non posted last write enable */
	uint64_t	__5:1;		/* reserved */
	uint64_t	f3_arprot:3;	/* AXI ar_prot signal */
	uint64_t	f3_awprot:3;	/* AXI aw_prot signal */
	uint64_t	f1_arlenen:1;	/* Source Burst length Enable */
	uint64_t	f8_arlen:8;	/* Source Burst length */
	uint64_t	f1_awlenen:1;	/* Destination Burst length Enable */
	uint64_t	f8_awlen:8;	/* Destination Burst length */
	uint64_t	f1_srcstaten:1;	/* Source Status Enable (fixed to 0) */
	uint64_t	f1_dststaten:1;	/* Destination Status Enable (fixed to 0) */
	uint64_t	f1_IOCBLKTFR:1;	/* Interrupt On completion of BLOCK Transfer */
	uint64_t	__6:3;		/* reserved */
	uint64_t	f1_lli_last:1;	/* last LLI */
	uint64_t	f1_lli_valid:1;	/* LLI valid */
} LLD_DMAC_CTR;

/*
 * Linked List Item
 */
typedef struct {
	uint64_t	sadr;		/* Physical address (src) */
	uint64_t	dadr;		/* Physical address (dst) */
	uint32_t	tsz;		/* Transfer count - 1 */
	uint32_t	__1;		/* reserved */
	LLD_DMAC_IND_INFO	s_lli;	/* Next Linked List Pointer */
	LLD_DMAC_CTR	ul_ctl;		/* Control data */
	uint32_t	sstat;		/* SSTAT */
	uint32_t	dstat;		/* DSTAT */
	uint64_t	lstat;		/* LLP_STATUS */
	uint64_t	__2;		/* reserved */
} LLD_DMAC_LINKLIST;

/*
 * Function setting information
 */
typedef struct {
	uint8_t		flowctl;	/* Flow control */
	uint8_t		srcid;		/* Peripheral ID (src) */
	uint8_t		dstid;		/* Peripheral ID (dst) */
} LLD_DMAC_CH_INFO;

/*
 * Configuration of DMAC channel
 */
typedef struct {
	uint8_t		prio;		/* Priority between DMAC channels */
	uint8_t		aw_qos;		/* AXI AW QoS */
	uint8_t		ar_qos;		/* AXI AR QoS */
	uint8_t		src_outstand;	/* outstanding limit-1 (src) */
	uint8_t		dst_outstand;	/* outstanding limit-1 (dst) */
} LLD_DMAC_CH_CONF;

/*
 * Transfer information
 */
typedef struct {
	uint64_t	sadr;		/* Physical address (src) */
	uint64_t	dadr;		/* Physical address (dst) */
	uint32_t	tsz;		/* Transfer count -1 (0 ? 0x3FFFFF) */
	uint8_t		f1_si:1;	/* Address increment (src) */
	uint8_t		f1_di:1;	/* Address increment (dst) */
	uint8_t		__1:6;		/* reserved */
	uint8_t		f4_sbsize:4;	/* Burst transaction length (src) */
	uint8_t		f4_dbsize:4;	/* Burst transaction length (dst) */
	uint8_t		f4_swidth:4;	/* Data transfer width (src) */
	uint8_t		f4_dwidth:4;	/* Data transfer width (dst) */
	uint8_t		f1_sauto:1;	/* Auto data transfer width setting enable (src) */
	uint8_t		f1_dauto:1;	/* Auto data transfer width setting enable (dst) */
	uint8_t		__2:6;		/* reserved */
} LLD_DMAC_TRAN_INFO;

LLD_ER lld_dmac_open(const LLD_CH ch);
LLD_ER lld_dmac_close(const LLD_CH ch);
LLD_ER lld_dmac_set_func(const LLD_CH ch, const LLD_DMAC_CH_INFO *info,
			 const LLD_FUNC int_func, void *cookies,
			 const LLD_FUNC_FLAG flag);
LLD_ER lld_dmac_set_config(const LLD_CH ch, const LLD_DMAC_CH_CONF *conf);
LLD_ER lld_dmac_start(const LLD_CH ch, const LLD_DMAC_TRAN_INFO *tran_info);
LLD_ER lld_dmac_sc_start(const LLD_CH ch, const LLD_DMAC_IND_INFO lli);
LLD_ER lld_dmac_stop(const LLD_CH ch);
int lld_dmac_check_status(const LLD_CH ch);
LLD_ER lld_dmac_get_trans_status(const LLD_CH ch, uint64_t *dadr, uint32_t *tsize);
LLD_ER lld_dmac_get_int_status(const LLD_CH ch, uint64_t *intstat);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* _DMAC_LLD_H */
