/*
 * dmac_lld_misc.h
 *
 * LLD DMAC driver support routines
 *
 * Copyright 2022 Sony Corporation
 *
 */
#ifndef __DMAC_LLD_MISC_H
#define __DMAC_LLD_MISC_H

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#if defined(__KERNEL__) || !defined(TARGET_APP_USERSPACE)
# include <mach/noncache.h>
#endif

#include <mach/dw_axi_dmac.h>
#define AXI_MODIFIABLE	0x2

#ifndef DW_DMAC_LL_ALIGN
#define DW_DMAC_LL_SHIFT	6 /* Linked List: 2^6 bytes align */
#define DW_DMAC_LL_ALIGN	(1UL << DW_DMAC_LL_SHIFT)
#endif

static inline int lld_dmac_check_lli(void *ll)
{
	if (!ll)
		return -1;
	if (((uintptr_t)ll & (DW_DMAC_LL_ALIGN - 1)) != 0)
		return -1;
	return 0;
}

static inline int __lld_dmac_unaligned(uint64_t addr, uint32_t shift)
{
	return (addr & ((1UL << shift) - 1));
}

static inline uint64_t __lld_dmac_align(uint64_t addr)
{
	return addr & ~(MACH_DW_DMAC_WIDTH - 1);
}

/*
 * NAME
 *	lld_dmac_lli_num
 *		- Get a number of lined list items required
 *
 * SYNOPSIS
 *	#include <dmac_lld_misc.h>
 *	int lld_dmac_lli_num(uint64_t src, uint32_t len);
 * INPUT
 *	src:	source address
 *	len:	length in bytes
 * RETURN VALUE
 *	number of linked list item required.(value=1 or 2)
 */
static inline int lld_dmac_lli_num(uint64_t src, uint32_t len)
{
	uint64_t s, e;

	s = __lld_dmac_align(src);
	e = __lld_dmac_align(src + len - 1);
	if (s == e) {
		return 1;
	}

	/* end of src is aligned to transfer width ? */
	if (!__lld_dmac_unaligned(src + len, MACH_DW_DMAC_WIDTH_SHIFT)) {
		return 1;
	}
	return 2;
}

/*
 * NAME
 *	lld_dmac_make_lli_fill
 *		- Fill a linked list item
 *		- s_lli is filled only in kernel space.
 *
 * SYNOPSIS
 *	#include <dmac_lld_misc.h>
 *	void lld_dmac_make_lli_fill(LLD_DMAC_LINKLIST *ll, int n);
 * INPUT
 *	ll:	Linked List pointer(virtual address)
 *	n:	number of elements in ll
 * RETURN VALUE
 *      0:	SUCCESS
 *	-EINVAL: ll is NULL or unaligned.
 */
static inline int lld_dmac_make_lli_fill(LLD_DMAC_LINKLIST *ll, int n)
{
	if (lld_dmac_check_lli(ll) < 0)
		return -EINVAL;
	if (n < 1)
		return -EINVAL;

#if defined(__KERNEL__) || !defined(TARGET_APP_USERSPACE)
	if (n > 1) {
		ll->s_lli.lli_adr = VA_TO_PHYS((uintptr_t)(ll + 1)) >> DW_DMAC_LL_SHIFT;
	} else {
		ll->s_lli.lli_adr = 0;
	}
#endif
	ll->ul_ctl.f1_smch	= MACH_DW_DMAC_SRC_MASTER;
	ll->ul_ctl.f1_dmch	= MACH_DW_DMAC_DST_MASTER;
	ll->ul_ctl.f1_si	= LLD_DMAC_ADR_INC;
	ll->ul_ctl.f1_di	= LLD_DMAC_ADR_INC;
	ll->ul_ctl.f3_swidth	= MACH_LLD_DMAC_WIDTH;
	ll->ul_ctl.f3_dwidth	= MACH_LLD_DMAC_WIDTH;
	ll->ul_ctl.f4_sbsize	= MACH_LLD_DMAC_BSIZE;
	ll->ul_ctl.f4_dbsize	= MACH_LLD_DMAC_BSIZE;
	ll->ul_ctl.f4_arcache	= AXI_MODIFIABLE;
	ll->ul_ctl.f4_awcache	= AXI_MODIFIABLE;
	ll->ul_ctl.f1_lli_valid = 1;
	return 0;
}

static inline uint32_t lld_dmac_swidth(uint64_t src, uint64_t end, uint32_t len, uint32_t *block_ts)
{
	uint32_t tsz, swidth;

	if (end & 0x1) {
		tsz = len;
		swidth = LLD_DMAC_WIDTH8;
	} else if (end & 0x2) {
		tsz = (len + 1) >> 1;
		swidth = LLD_DMAC_WIDTH16;

	} else if (end & 0x4) {
		tsz = (len + 3) >> 2;
		swidth = LLD_DMAC_WIDTH32;
	} else if (end & 0x8) {
		tsz = (len + 7) >> 3;
		swidth = LLD_DMAC_WIDTH64;
	} else {
		tsz = (len + 15) >> 4;
		swidth = LLD_DMAC_WIDTH128;
	}
#if 1 /* workaround */
	if (__lld_dmac_unaligned(src, swidth) && (1 == tsz)) {
		if (len & 0x8) {
			swidth = LLD_DMAC_WIDTH64;
			tsz = (len + 7) >> 3;
		} else if (len & 0x4) {
			swidth = LLD_DMAC_WIDTH32;
			tsz = (len + 3) >> 2;
		} else if (len & 0x2) {
			swidth = LLD_DMAC_WIDTH16;
			tsz = (len + 1) >> 1;
		} else {
			swidth = LLD_DMAC_WIDTH8;
			tsz = len;
		}
	}
#endif
	*block_ts = tsz;
	return swidth;
}

static inline void lld_dmac_make_lli_swidth(LLD_DMAC_LINKLIST *ll, uint64_t end, uint32_t len)
{
	uint32_t tsz;

	ll->ul_ctl.f3_swidth = lld_dmac_swidth(ll->sadr, end, len, &tsz);
	ll->tsz = tsz - 1;
}

/*
 * NAME
 *	lld_dmac_make_lli
 *		- Build linked list items
 *
 * SYNOPSIS
 *	#include <dmac_lld_misc.h>
 *	int lld_dmac_make_lli(LLD_DMAC_LINKLIST *ll, int n, int fill, uint64_t src, uint64_t dst, uint32_t len);
 * INPUT
 *	ll:	Linked List pointer(virtual address)
 *	n:	number of elements in ll
 *	fill:	Fill or not
 *	src:	source address
 *	dst:	destination address
 *	len:	length in bytes
 * RETURN VALUE
 *	number of linked list items consumed.
 *	-EINVAL: The pointer is NULL or unaligned.
 */
static inline int lld_dmac_make_lli(LLD_DMAC_LINKLIST *ll, int n, int fill,
				    uint64_t src, uint64_t dst, uint32_t len)
{
	uint64_t src_end = src + len;
	uint64_t s, e;
	uint32_t sz;

	if (lld_dmac_check_lli(ll) < 0)
		return -EINVAL;
	if (n < 1)
		return -EINVAL;

	/* end of src is aligned to transfer width ? */
	if (!__lld_dmac_unaligned(src_end, MACH_DW_DMAC_WIDTH_SHIFT)) {
		if (fill) {
			memset(ll, 0, sizeof (LLD_DMAC_LINKLIST));
			if (lld_dmac_make_lli_fill(&ll[0], n) != 0)
				return -EINVAL;
		} else {
			ll[0].ul_ctl.f1_lli_last = 0;
			ll[0].ul_ctl.f1_IOCBLKTFR = 0;
			ll[0].ul_ctl.f1_lastwren = 0;
		}
		ll[0].sadr = src;
		ll[0].dadr = dst;
		lld_dmac_make_lli_swidth(&ll[0], 0UL, len);
		return 1;
	}

	/* unaligned */
	s = __lld_dmac_align(src);
	e = __lld_dmac_align(src + len - 1);
	if (s == e) { /* single word */
		if (fill) {
			memset(ll, 0, sizeof (LLD_DMAC_LINKLIST));
			if (lld_dmac_make_lli_fill(&ll[0], n) != 0)
				return -EINVAL;
		} else {
			ll[0].ul_ctl.f1_lli_last = 0;
			ll[0].ul_ctl.f1_IOCBLKTFR = 0;
			ll[0].ul_ctl.f1_lastwren = 0;
		}
		ll[0].sadr = src;
		ll[0].dadr = dst;
		lld_dmac_make_lli_swidth(&ll[0], src_end, len);
		return 1;
	}

	/* split into two items */
	if (n < 2)
		return -EINVAL;
	if (fill) {
		memset(ll, 0, 2 * sizeof (LLD_DMAC_LINKLIST));
		if (lld_dmac_make_lli_fill(&ll[0], n) != 0)
			return -EINVAL;
		if (lld_dmac_make_lli_fill(&ll[1], n - 1) != 0)
			return -EINVAL;
	} else {
		ll[0].ul_ctl.f1_lli_last = 0;
		ll[0].ul_ctl.f1_IOCBLKTFR = 0;
		ll[0].ul_ctl.f1_lastwren = 0;
		ll[1].ul_ctl.f1_lli_last = 0;
		ll[1].ul_ctl.f1_IOCBLKTFR = 0;
		ll[1].ul_ctl.f1_lastwren = 0;
	}
	e = __lld_dmac_align(src_end);
	sz = e - src;
	ll[0].sadr = src;
	ll[0].dadr = dst;
	lld_dmac_make_lli_swidth(&ll[0], 0UL, sz);
	ll[1].sadr = e;
	ll[1].dadr = dst + sz;
	lld_dmac_make_lli_swidth(&ll[1], src_end, len - sz);
	return 2;
}

/*
 * NAME
 *	lld_dmac_make_lli_last
 *		- setup Last LLI
 *
 * SYNOPSIS
 *	#include <dmac_lld_misc.h>
 *	void lld_dmac_make_lli_last(LLD_DMAC_LINKLIST *ll);
 * INPUT
 *	Linked List pointer
 * RETURN VALUE
 *      0: SUCCESS
 *	-EINVAL: The pointer is NULL or unaligned.
 */
static inline int lld_dmac_make_lli_last(LLD_DMAC_LINKLIST *ll)
{
	if (lld_dmac_check_lli(ll) < 0)
		return -EINVAL;

	ll->ul_ctl.f1_lli_last = 1;
	ll->ul_ctl.f1_IOCBLKTFR = 1;
	ll->ul_ctl.f1_lastwren = 1;
	return 0;
}

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* __DMAC_LLD_MISC_H */
