/*
 * arch/arm/common/wbi_raw.c
 *
 * WBI non-compress sequencer
 *
 * Copyright 2019 Sony Imaging Products & Solutions Inc.
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 *
 */
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/snsc_boot_time.h>
#include <linux/wdt.h>
#include <linux/warmboot.h>
#include <linux/wbi.h>

#include <asm/cacheflush.h>
#include <asm/io.h>
#include <mach/noncache.h>
#include <mach/moduleparam.h>

#ifdef CONFIG_WBI_LZ77_SW_COMPRESS
#include "wbi_lz77.h"
#endif /* CONFIG_WBI_LZ77_SW_COMPRESS */

#define SECTOR_SHIFT 9
#define SECTOR_SIZE (1 << SECTOR_SHIFT)
/*------------------------ API ----------------------------*/
static char *wbuf;
static int cur_sector = 0;

#ifdef CONFIG_WBI_LZ77_SW_COMPRESS
static int wbi_lz77_buffer_size = 0;
static unsigned char *wbi_lz77_buffer = NULL;
#endif /* CONFIG_WBI_LZ77_SW_COMPRESS */

int wbi_send_data(void *b, ulong len)
{
	int n, ret;

	while (len > 0) {
		n = len;
		if (n < SECTOR_SIZE) {
			memset(wbuf + n, 0, SECTOR_SIZE - n);
		} else if (n > SECTOR_SIZE) {
			n = SECTOR_SIZE;
		}
		memcpy(wbuf, b, n);
		ret = wb_default_device->write_sector(wb_default_device,
						      wbuf, cur_sector, 1);
		if (ret < 0) {
			goto err;
		}
		cur_sector++;
		b += n;
		len -= n;
	}
	return 0;

err:
	wbi_print("ERROR:%s: cannot write(%p,%d) = %d\n", __FUNCTION__, wbuf, cur_sector, ret);
	return ret;
}

#if 0
void wbi_align_sector(void)
{
	wbi_wbuf_align();
}
#endif

int wbi_flush(void)
{
	return 0;
}

static ulong sector_align(ulong x)
{
	x += SECTOR_SIZE - 1;
	x &= ~(ulong)(SECTOR_SIZE - 1);
	return x;
}

#define MAX_SECTOR 0xffff

static int wbi_send_data_in_sectors(void *b, ulong len, ulong *wlen)
{
	int n, nsect, ret;

	if (wlen) {
		*wlen = 0;
	}
	while (len >= SECTOR_SIZE) {
		nsect = (int)len / SECTOR_SIZE;
		if (nsect > MAX_SECTOR) {
			nsect = MAX_SECTOR;
		}
		n = nsect << SECTOR_SHIFT;
		ret = wb_default_device->write_sector(wb_default_device,
						      b, cur_sector, nsect);
		if (ret < 0) {
			goto err;
		}
		cur_sector += nsect;
		b += n;
		len -= n;
		if (wlen) {
			*wlen += n;
		}

		wbi_watchdog();
	}
	return 0;

err:
	wbi_print("ERROR:%s: cannot write(%p,%d,%d) = %d\n", __FUNCTION__, b, cur_sector, nsect, ret);
	return ret;
}

static int wbi_send_section_nocompress(struct wb_section *p)
{
	/*
	 * Note:
	 *   During write sections, DO NOT USE kernel API.
	 *   (ex. printk, BOOT_TIME)
	 *   Because there are side effects on heap. (ex. lock variable)
	 */
	void *va;
	ulong len;
	int ret;

	/* write rlen & checksum */
	p->rlen = (p->flag & WBI_SH_ZERO) ? 0: p->olen;
	INIT_CKSUM(p->meta_cksum);
	wbi_calc_cksum(&p->meta_cksum, (void *)p,
				offsetof(struct wb_section, meta_cksum));
	if (p->flag & WBI_SH_ZERO) {
		goto end;
	}

	/* write data */
	va = (void *)PHYS_TO_VA(p->addr);
	len = sector_align(p->olen);
	ret = wbi_send_data_in_sectors(va, len, NULL);
	if (ret < 0) {
		wbi_print("ERROR:%s: cannot send(va=%p,len=%lu) = %d\n",
				__func__, va, len, ret);
		return ret;
	}

end:
	return 0;
}

#ifdef CONFIG_WBI_LZ77_SW_COMPRESS
static int wbi_send_section_compress(struct wb_section *p)
{
	/*
	 * Note:
	 *   During write sections, DO NOT USE kernel API.
	 *   (ex. printk, BOOT_TIME)
	 *   Because there are side effects on heap. (ex. lock variable)
	 */
	unsigned char *va;
	ulong rest, pos, len, wlen;
	int ret;

	/* check original data length  */
	if (p->olen > WBI_LZ77_MAX_SLEN) {
		wbi_print("ERROR:%s: original data length (%lu) > WBI_LZ77_MAX_SLEN (%d)\n",
			__func__, p->olen, WBI_LZ77_MAX_SLEN);
		return -1;
	}

	va = (unsigned char *)PHYS_TO_VA(p->addr);
	rest = p->olen;
	pos = 0;
	len = 0;
	while (rest > 0) {
		/* deflate */
		ret = wbi_lz77_deflate(&va, &rest, wbi_lz77_buffer + pos, wbi_lz77_buffer_size - (int)pos);
		if (ret <= 0) {
			wbi_print("ERROR:%s: cannot deflate(%p,%lu,%p,%d) = %d\n",
				__func__, va, rest, wbi_lz77_buffer + pos, wbi_lz77_buffer_size - (int)pos, ret);
			return ((ret == 0) ? -1 : ret);
		}
		pos += ret;
		len += ret;
		wbi_watchdog();

		/* write data */
		wlen = 0;
		ret = wbi_send_data_in_sectors(wbi_lz77_buffer, pos, &wlen);
		if (ret < 0) {
			wbi_print("ERROR:%s: cannot send(va=%p,len=%lu) = %d\n",
				__func__, wbi_lz77_buffer, pos, ret);
			return ret;
		}

		/* remove written data */
		if (wlen > 0) {
			if (wlen < pos) {
				/* the source and destination areas do not overlap */
				memcpy(wbi_lz77_buffer, wbi_lz77_buffer + wlen, pos - wlen);
			}
			pos -= wlen;
		}
	}

	/* write data to last sector */
	if (pos > 0) {
		ret = wbi_send_data(wbi_lz77_buffer, pos);
		if (ret != 0) {
			wbi_print("ERROR:%s: cannot send(%p,%lu) = %d\n",
				__func__, wbi_lz77_buffer, pos, ret);
			return ret;
		}
		pos = 0;
		wbi_watchdog();
	}

	/* write rlen & checksum */
	p->rlen = len;
	INIT_CKSUM(p->meta_cksum);
	wbi_calc_cksum(&p->meta_cksum, (void *)p,
				offsetof(struct wb_section, meta_cksum));

	return 0;
}
#endif /* CONFIG_WBI_LZ77_SW_COMPRESS */

int wbi_send_sections(struct wb_header *header, struct wb_section *s, ulong n)
{
	int i, ret;
	struct wb_section *p;
	int (*send_section)(struct wb_section *);
	ulong r_data_size;

	send_section =
#ifdef CONFIG_WBI_LZ77_SW_COMPRESS
			(header->flag & WBI_COMPRESS) ?
			wbi_send_section_compress :
#endif /* CONFIG_WBI_LZ77_SW_COMPRESS */
			wbi_send_section_nocompress;

	/* send sections data */
	r_data_size = 0;
	for (i = 0, p = s; i < n; i++, p++) {
		ret = send_section(p);
		if (ret < 0) {
			wbi_print("ERROR:%s: [%d/%lu] send_section failed(%p) = %d\n",
				__func__, i, n, p, ret);
			return ret;
		}
		r_data_size += sector_align(p->rlen);
	}

	/* summary */
	for (i = 0, p = s; i < n; i++, p++) {
		wbi_debug_print("%3d %08lx:%08lx:%08lx %08lx<->%08lx\n",
			  i, p->addr, p->cksum, p->flag,
			  p->olen, p->rlen);
		wbi_watchdog();
	}

	/* write meta data */
	ret = wbi_send_data(s, sizeof(*s) * n);
	if (ret) {
		goto err2;
	}

	/* overwrite r_data_size */
	header->r_data_size = r_data_size;

	return 0;

 err2:
	wbi_print("%s: write meta data error: %d\n", __FUNCTION__, ret);
	return ret;
}

int wbi_rewrite_header(struct wb_header *header)
{
	int ret;

	/* ASSUME: header size <= sector_size */
	memset(wbuf, 0, SECTOR_SIZE);
	memcpy(wbuf, header, sizeof (*header));

	ret = wb_default_device->write_sector(wb_default_device, wbuf, 0, 1);
	wbi_watchdog();
	return ret;
}

int wbi_read_header(struct wb_header *header)
{
	int ret;

	ret = wb_default_device->read_sector(wb_default_device, wbuf, 0, 1);
	wbi_watchdog();

	/* ASSUME: header size <= sector_size */
	memcpy((void *)header, wbuf, sizeof (*header));
	return ret;
}

void wbi_stat(void)
{
	wbi_debug_print("%d sectors\n", cur_sector);
}

int wbi_setup(ulong work, ulong *wlenp, ulong dst, ulong *dlenp)
{
	cur_sector = 0;
	wbuf = kmalloc(SECTOR_SIZE, GFP_KERNEL);
	if (!wbuf) {
		wbi_print("%s:Can not alloc wbuf\n", __func__);
		return -1;
	}
#ifdef CONFIG_WBI_LZ77_SW_COMPRESS
	if (dlenp) {
		wbi_lz77_buffer_size = SECTOR_SIZE + WBI_LZ77_MIN_DST_LEN;
		if (wbi_lz77_buffer_size < *dlenp) {
			wbi_lz77_buffer_size = *dlenp;
		}
		wbi_lz77_buffer_size = PAGE_ALIGN(wbi_lz77_buffer_size);
		wbi_lz77_buffer = kmalloc(wbi_lz77_buffer_size, GFP_KERNEL);
		if (!wbi_lz77_buffer) {
			wbi_print("%s:Can not alloc wbi_lz77buf (size=%d)\n", __func__, wbi_lz77_buffer_size);
			return -1;
		}
	}
#endif /* CONFIG_WBI_LZ77_SW_COMPRESS */
	return 0;
}

void wbi_finish(void)
{
	if (wbuf) {
		kfree(wbuf);
		wbuf = NULL;
	}
#ifdef CONFIG_WBI_LZ77_SW_COMPRESS
	if (wbi_lz77_buffer) {
		kfree(wbi_lz77_buffer);
		wbi_lz77_buffer = NULL;
	}
#endif /* CONFIG_WBI_LZ77_SW_COMPRESS */
}
