/*
 * kernel/tmonitor.c
 *
 * Copyright 2018 Sony Corporation.
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */
#include <linux/module.h>
#include <linux/sched.h>
#include <linux/sched/debug.h>
#include <linux/delay.h>
#include <linux/kallsyms.h>
#include <linux/seq_file.h>
#include <linux/proc_fs.h>
#include <linux/fs.h>
#include <linux/vmalloc.h>
#include <linux/ctype.h>
#include <linux/dma-map-ops.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#ifdef CONFIG_SNSC_BOOT_TIME
#include <linux/snsc_boot_time.h>
#endif
#include <linux/tmonitor.h>
#include <linux/suspend.h>
#include <linux/kthread.h>

#ifdef CONFIG_EJ_SCHED_DEBUG
#include <mach/kslog.h>
#endif /* CONFIG_EJ_SCHED_DEBUG */

/* get timestamp */
unsigned long long __weak tmonitor_ts(void)
{
	return sched_clock();
}

/* timestamp to usecs */
unsigned long long __weak tmonitor_ts_to_us(unsigned long long ts)
{
	return (ts / 1000); /* ts is in nsec. */
}

int trace_cpu = 0;
int tmon_irqdebug = 0;
int tmon_trace_hidden_irq = 0;

static ulong buf_addr;
static ulong buf_size;
static ulong dfl_cpu;
#ifndef param_check_ulongH
# define ulongH ulong
#endif
module_param_named(addr, buf_addr, ulongH, S_IRUSR|S_IWUSR);
module_param_named(size, buf_size, ulongH, S_IRUSR|S_IWUSR);
module_param_named(mask, dfl_cpu, ulong, S_IRUSR|S_IWUSR);
module_param_named(irqdebug, tmon_irqdebug, int, S_IRUSR|S_IWUSR);
module_param_named(hidden_irq, tmon_trace_hidden_irq, int, S_IRUSR|S_IWUSR);
/*
 * mode
 * 0 - ringbuffer mode
 * 1 - snapshot mode
 */
static int mode;
module_param_named(mode, mode, int, S_IRUSR|S_IWUSR);

int tmon_boostlog;
module_param_named(boost, tmon_boostlog, int, S_IRUSR|S_IWUSR);

/*
 * auto start mode
 * 0 - Do not start automatically.
 * 1 - boot_time_add() string match
 */
static int tmon_autostart = 0;
module_param_named(autostart, tmon_autostart, int, S_IRUSR|S_IWUSR);
static char *tmon_autostart_str;
module_param_named(autostart_str, tmon_autostart_str, charp, S_IRUSR|S_IWUSR);

/*
 * auto stop mode (bitmap)
 * 0 - Do not stop automatically.
 * 1 - boot_time_add() string match
 * 2 - N ticks elapsed after last boot_time_add().
 * 4 - boot_time_add() at an interval of N ticks or longer.
 */
int tmon_autostop = 0;
module_param_named(autostop, tmon_autostop, int, S_IRUSR|S_IWUSR);
static int tmon_autostop_retrigger = 0;
module_param_named(autostop_retrigger, tmon_autostop_retrigger, int, S_IRUSR|S_IWUSR);
static char *tmon_autostop_str;
module_param_named(autostop_str, tmon_autostop_str, charp, S_IRUSR|S_IWUSR);
static ulong tmon_autostop_ticks;
module_param_named(autostop_ticks, tmon_autostop_ticks, ulong, S_IRUSR|S_IWUSR);
static bool tmon_autostop_segv;
module_param_named(autostop_segv, tmon_autostop_segv, bool, S_IRUSR|S_IWUSR);
static bool tmon_autostop_tasklist;
module_param_named(autostop_tasklist, tmon_autostop_tasklist, bool, S_IRUSR|S_IWUSR);
static bool tmon_autostop_sus;
module_param_named(autostop_sus, tmon_autostop_sus, bool, S_IRUSR|S_IWUSR);

static int tmon_autodump = 0;
module_param_named(autodump, tmon_autodump, int, S_IRUSR|S_IWUSR);
static char *tmon_autodump_path;
module_param_named(autodump_path, tmon_autodump_path, charp, S_IRUSR|S_IWUSR);

static uint tmon_dumper_delay = 0;
module_param_named(dumper_delay, tmon_dumper_delay, uint, S_IRUSR|S_IWUSR);

static int tmon_autostopped = 0;

static u32 max_entry;
#define MAX_ENTRY max_entry

struct trace_entry {
	cycles_t time;
	unsigned char type; /* [7:4]=type, [3:0]=CPU */
	unsigned char pri;
	unsigned short state;
	unsigned short ppid; /* prev pid, irq#, user tag */
	unsigned short npid;
	long data;
	char prev[TASK_COMM_LEN-1]; /* task name, irq handler name */
	char padding;
};

/* type member */
#define TYPE_CPUMASK	0x0f
#define TYPE_MASK	0xf0
#define ENTRY_TYPE(x)	((x)->type & TYPE_MASK)
#define  TYPE_DISPATCH	0x00
#define  TYPE_IRQ	0x10
#define  TYPE_USER	0xf0
#define ENTRY_CPU(x)	((x)->type & TYPE_CPUMASK)
/* user tag */
#define USRTAG_OFFSET	offsetof(struct trace_entry, ppid)
#define USRTAG(x)	((char *)(x) + USRTAG_OFFSET)
#define USRTAG_MAXLEN	(offsetof(struct trace_entry, padding) - USRTAG_OFFSET)

struct trace_data {
	int index;
	unsigned char overflow;
#define TMON_HEADER_VERSION (1)
	unsigned char version;
	unsigned short reserved;
#define TMON_TS_REF_VAL (18000) /* a number that tends to be an integer when divided by any integer */
	int ts_ref_val;
	int us_ref_val;
	struct trace_entry entries[1];
};

#define MAX_HASH 0x100
struct name_hash {
	unsigned long addr;
	unsigned long offset;
	char name[32];
};

struct dump_cookie {
	struct name_hash hash[MAX_HASH];
	char namebuf[KSYM_SYMBOL_LEN];
};

static struct trace_data *trace_datas = NULL;
static DEFINE_RAW_SPINLOCK(lock);
static int maxidx;
static int old_trace_cpu;

void tmonitor_clear(void)
{
	struct trace_data *pdata = trace_datas;
	if (pdata) {
		pdata->index = 0;
		pdata->overflow = 0;
	}
	smp_mb();
}
EXPORT_SYMBOL(tmonitor_clear);

void
add_trace_entry(int cpu, struct task_struct *prev, struct task_struct *next, long data)
{
	struct trace_data *pdata = trace_datas;
	struct trace_entry *entry;
	unsigned long flags;

	if (system_state != SYSTEM_RUNNING)
		return;

	if (!pdata) {
		return;
	}

	if (!(trace_cpu & (1 << cpu))) {
		return;
	}

	raw_spin_lock_irqsave(&lock, flags);
	if (pdata->index < MAX_ENTRY) {
		entry = pdata->entries + pdata->index ++;
	}
	else {
		if (mode) {
			raw_spin_unlock_irqrestore(&lock, flags);
			return;
		}

		pdata->overflow = 1;
		pdata->index = 0;
		entry = pdata->entries + pdata->index ++;
	}
	raw_spin_unlock_irqrestore(&lock, flags);

	entry->time = tmonitor_ts();
	entry->type = cpu & TYPE_CPUMASK;
	if (!prev) {
		entry->type |= TYPE_IRQ;
		entry->ppid = (uintptr_t)next;
		entry->data = data;
		return;
	}
	entry->type |= TYPE_DISPATCH;
	entry->pri = prev->rt_priority;
	entry->state = prev->__state;
	entry->ppid = prev->pid;
	entry->npid = next->pid;
	entry->data = data;
	memcpy(entry->prev, prev->comm, sizeof(entry->prev));
	entry->padding = 0;
}

static void init_cookie(struct dump_cookie *cookie)
{
	struct name_hash *p = cookie->hash;
	int n;

	n = MAX_HASH;
	while (--n >= 0) {
		p++->addr = 0;
	}
}

static int
dump_trace_entry(struct seq_file *m, int i)
{
	struct dump_cookie *cookie = m->private;
	struct trace_data *pdata = trace_datas;
	struct trace_entry *entry = pdata->entries + (i % MAX_ENTRY);
	cycles_t time = tmonitor_ts_to_us(entry->time);
	u32 sec = time / 1000000;
	u32 us = time % 1000000;

#define CRLF "\r\n"
	switch (ENTRY_TYPE(entry)) {
	case TYPE_USER:
		seq_printf(m, "[ %5d.%06d ] -profile user %s cpu:%u" CRLF,
			   sec, us, USRTAG(entry), ENTRY_CPU(entry));
		return 0;
	case TYPE_IRQ:
		seq_printf(m, "[ %5d.%06d ] -profile user irq:%u,%llu cpu:%u" CRLF,
			   sec, us, entry->ppid, tmonitor_ts_to_us(entry->data), ENTRY_CPU(entry));
		return 0;
	default:
	{
		char *modname;
		const char *fname = NULL;
		unsigned long offset, size;
		unsigned long addr = entry->data;
		struct name_hash *hash = &cookie->hash[addr % MAX_HASH];

		if (addr && hash->addr == addr) {
			fname = hash->name;
			offset = hash->offset;
		}
		else if (addr) {
			fname = kallsyms_lookup(entry->data, &size, &offset, &modname, cookie->namebuf);
			if (fname) {
				memset(hash->name, 0, sizeof(hash->name));
				strncpy(hash->name, fname, sizeof(hash->name)-1);
				hash->addr = addr;
				hash->offset = offset;
			}
		}

		seq_printf(m, "[ %5d.%06d ] -sched next:%03u < prev:%03u state:%x wchan:%s%s%lx(%lx) task:%s cpu:%u prio:%u" CRLF,
			   sec, us, entry->npid, entry->ppid, entry->state,
			   fname?:"", fname?"+":"", fname?offset:addr, addr,
			   entry->prev,
			   ENTRY_CPU(entry), entry->pri);
	}
	}

	return 0;
}

static void set_trace_cpu(int cpus)
{
	if (!buf_addr || !buf_size)
		return;

	if (!trace_cpu) {
		struct trace_data *pdata = (struct trace_data *)phys_to_virt(buf_addr);
		trace_datas = NULL;
		smp_mb();
		if (buf_size < sizeof(struct trace_data))
			return;
		max_entry = (buf_size - offsetof(struct trace_data, entries)) / sizeof(struct trace_entry);
		pdata->index = 0;
		pdata->version = TMON_HEADER_VERSION;
		pdata->reserved = 0;
		pdata->overflow = 0;
		pdata->ts_ref_val = TMON_TS_REF_VAL;
		pdata->us_ref_val = tmonitor_ts_to_us(TMON_TS_REF_VAL);
		smp_mb();
		trace_datas = pdata;
		smp_mb();
		trace_cpu = cpus;
		arch_sync_dma_for_device((phys_addr_t)buf_addr, sizeof(struct trace_data), DMA_TO_DEVICE);
		arch_sync_dma_for_cpu((phys_addr_t)buf_addr, sizeof(struct trace_data), DMA_TO_DEVICE);
	}
	else {
		trace_cpu = cpus;
	}
}

void tmonitor_start(int cpus)
{
	set_trace_cpu(cpus);
}
EXPORT_SYMBOL(tmonitor_start);

void tmonitor_stop(void)
{
	trace_cpu = 0;
}
EXPORT_SYMBOL(tmonitor_stop);

void tmonitor_restart(void)
{
	int use_cpu = trace_cpu;
	tmonitor_stop();
	tmonitor_start(use_cpu);
}
EXPORT_SYMBOL(tmonitor_restart);

void tmonitor_write(const char *buffer) {
	int count;
	struct trace_data *pdata = trace_datas;
	struct trace_entry *entry;
	unsigned long flags;

	if ((unsigned long)buffer < 0x10) {
		unsigned long cpus = (unsigned long)buffer;
		set_trace_cpu(cpus);
		tmon_autostopped = 0;
		return;
	}

	if (!pdata || !trace_cpu)
		return;

	raw_spin_lock_irqsave(&lock, flags);
	if (pdata->index < MAX_ENTRY) {
		entry = pdata->entries + pdata->index ++;
	}
	else {
		if (mode) {
			raw_spin_unlock_irqrestore(&lock, flags);
			return;
		}

		pdata->overflow = 1;
		pdata->index = 0;
		entry = pdata->entries + pdata->index ++;
	}
	raw_spin_unlock_irqrestore(&lock, flags);
	entry->type = (raw_smp_processor_id() & TYPE_CPUMASK)|TYPE_USER;
	entry->state = 0;
	entry->time = tmonitor_ts();
	memset(USRTAG(entry), 0, USRTAG_MAXLEN);
	count = strlen(buffer);
	if (count > USRTAG_MAXLEN)
		count = USRTAG_MAXLEN;
	memcpy(USRTAG(entry), buffer, count);
	entry->padding = 0;
}
EXPORT_SYMBOL(tmonitor_write);

static ssize_t write_tmonitor(struct file *file, const char __user *buffer, size_t count, loff_t *ppos)
{
	int i = 0;
	int cnt;
	char ubuf[TASK_COMM_LEN];

	if (!buf_addr || !buf_size)
		goto out;

	if (count > sizeof(ubuf) - 1)
		count = sizeof(ubuf) - 1;
	if (copy_from_user(ubuf, buffer, count)) {
		return -EFAULT;
	}
	ubuf[count] = 0;

	/* atoi() */
	for ( cnt = 0; cnt < count; cnt++ ) {
		if (isdigit(ubuf[cnt]))
			i = i*10 + ubuf[cnt] - '0';
		else {
			i = -1;
			break;
		}
	}

	if ( i < 0 || i >= (1 << NR_CPUS) ) {
		if (!trace_cpu)
			goto out;
		tmonitor_write(ubuf);
	}
	else {
		u8 cpus = (u8)i;
		set_trace_cpu(cpus);
		tmon_autostopped = 0;
#ifdef CONFIG_EJ_SCHED_DEBUG
		if (cpus) { /* start */
			kslog_init();
		} else { /* stop */
			kslog_show();
		}
#endif /* CONFIG_EJ_SCHED_DEBUG */
	}

out:
	return count;
}


static void tmonitor_autostart(char *str)
{
	if (!tmon_autostart || tmon_autostopped)
		return;
	if (!str || !tmon_autostart_str)
		return;
	if (strncmp(str, tmon_autostart_str, strlen(tmon_autostart_str)))
		return;
#ifdef CONFIG_EJ_SCHED_DEBUG
	kslog_init();
#endif /* CONFIG_EJ_SCHED_DEBUG */
	set_trace_cpu(TMON_TRACE_ALL);
}

static void __tmonitor_dump(const char *path, int flags)
{
	struct file *sfp, *tfp;
	char buf[128];
	ssize_t sz;

	tfp = filp_open(path, flags, 0444);
	if (IS_ERR(tfp)) {
		printk(KERN_ERR "  open %s failed.\n", path);
		goto out2;
	}

	sfp = filp_open("/proc/tmonitor", O_RDONLY, 0444);
	if (IS_ERR(sfp)) {
		printk(KERN_ERR "  open /proc/tmontor failed.\n");
		goto out1;
	}

	while ((sz = kernel_read(sfp, buf, sizeof(buf), &sfp->f_pos)) > 0) {
		kernel_write(tfp, buf, sz, &tfp->f_pos);
	}
	vfs_fsync(tfp, 1);

	filp_close(sfp, NULL);
out1:
	filp_close(tfp, NULL);
out2:
	;
}

static void tmonitor_autodump(const char *path)
{
	int flags;

	flags = (tmon_autodump & TMON_AUTODUMP_OVERWRITE) ?
		O_WRONLY | O_CREAT | O_TRUNC :
		O_WRONLY | O_CREAT | O_EXCL;

	__tmonitor_dump(path, flags);
}

void tmonitor_dump(const char *path)
{
	__tmonitor_dump(path, O_WRONLY | O_CREAT | O_TRUNC);
}
EXPORT_SYMBOL(tmonitor_dump);

#ifdef CONFIG_SNSC_BOOT_TIME
static unsigned long boot_time_last = INITIAL_JIFFIES;
static struct task_struct *boot_time_last_task;

static unsigned long boot_time_elapse(void)
{
	return jiffies - boot_time_last;
}
#endif /* CONFIG_SNSC_BOOT_TIME */

static void tmonitor_autostop_timer(void)
{
#ifdef CONFIG_SNSC_BOOT_TIME
	if (time_after(boot_time_elapse(), tmon_autostop_ticks)) {
		tmon_autostopped = 1;
		tmonitor_stop();
		if (tmon_autostop_sus) {
#ifdef CONFIG_ARCH_HAS_PM_SUSPEND_STOP
			pm_suspend_stop(1);
#endif
		}
#ifdef CONFIG_EJ_SCHED_DEBUG
		kslog_show();
#endif /* CONFIG_EJ_SCHED_DEBUG */
		if (tmon_autostop_tasklist) {
			show_state();
		}
		printk(KERN_ERR "tmon:LONG_ELAPSE\n");
		if ((tmon_autodump & TMON_AUTODUMP_DELAY) && tmon_autodump_path) {
			printk(KERN_ERR "tmon:dump(timer) to %s\n", tmon_autodump_path);
			tmonitor_autodump(tmon_autodump_path);
			printk(KERN_ERR "tmon:done.\n");
		}
		if (tmon_autostop_segv && boot_time_last_task) {
			send_sig(SIGSEGV, boot_time_last_task, 1);
		}
	}
#endif /* CONFIG_SNSC_BOOT_TIME */
}

static void tmonitor_autostop_str(char *str)
{
	if (!str || !tmon_autostop_str)
		return;
	if (strncmp(str, tmon_autostop_str, strlen(tmon_autostop_str)))
		return;
	tmon_autostopped = 1;
	tmonitor_stop();
	if ((tmon_autodump & TMON_AUTODUMP_STR) && tmon_autodump_path) {
		printk(KERN_ERR "tmon:dump(str) to %s\n", tmon_autodump_path);
		tmonitor_autodump(tmon_autodump_path);
		printk(KERN_ERR "tmon:done.\n");
	}
	if (tmon_autostop_retrigger & TMON_AUTOSTOP_STR) {
		tmon_autostopped = 0;
	}
	if (tmon_autostopped  &&  tmon_autostop_sus) {
#ifdef CONFIG_ARCH_HAS_PM_SUSPEND_STOP
		pm_suspend_stop(1);
#endif
	}

	/* printk(KERN_ERR "tmon:STRING_MATCH\n"); */
}

static void tmonitor_autostop(char *str)
{
	if (!trace_cpu || !tmon_autostop || tmon_autostopped)
		return;

	if (tmon_autostop & TMON_AUTOSTOP_STR) {
		tmonitor_autostop_str(str);
	}
	if (tmon_autostop & TMON_AUTOSTOP_DELAY2) {
		tmonitor_autostop_timer();
	}
}

#ifdef CONFIG_SNSC_BOOT_TIME
void tmonitor_boot_time_notify(char *comment)
{
	if (comment) {
		tmonitor_autostart(comment);
		tmonitor_write(comment);
	}
	tmonitor_autostop(comment);
	boot_time_last = jiffies;
	boot_time_last_task = current;
}
#endif /* CONFIG_SNSC_BOOT_TIME */

void tmonitor_timer_notify(void)
{
	if (!trace_cpu || tmon_autostopped)
		return;
	tmonitor_autostop_timer();
}


static void *
tmonitor_seq_start(struct seq_file *m, loff_t *pos)
{
	int off = (int)*pos;
	struct trace_data *pdata = trace_datas;

	m->private = vmalloc(sizeof(struct dump_cookie));
	if (!m->private)
		return NULL;
	init_cookie((struct dump_cookie *)m->private);

	if (!pdata)
		return NULL;

	if (!off) {
		if (!(pdata->index | pdata->overflow))
			return NULL;

		/* stop tracing */
		old_trace_cpu = trace_cpu;
		trace_cpu = 0;
		msleep(10);
		maxidx = pdata->index;
		if (pdata->overflow)
			maxidx += MAX_ENTRY;
	}

	if (pdata->overflow)
		off += pdata->index;

	if (off >= maxidx)
		return NULL;

	return (void *)(uintptr_t)(off + 1);
}

static void
tmonitor_seq_stop(struct seq_file *m, void *v)
{
	vfree(m->private);
}

static void *
tmonitor_seq_next(struct seq_file *m, void *v, loff_t *pos)
{
	int off = (uintptr_t)v;
	struct trace_data *pdata = trace_datas;

	(*pos)++;

	off ++;

	if (off <= maxidx) {
		return (void *)(uintptr_t)(off);
	}
	else {
		/* resume tracing */
		maxidx = 0;
		if (pdata) {
			pdata->index = 0;
			pdata->overflow = 0;
		}
		smp_mb();
		trace_cpu = old_trace_cpu;
		old_trace_cpu = 0;
		return NULL;
	}
}

static int
tmonitor_seq_show(struct seq_file *m, void *v)
{
	int off = (uintptr_t)v;
	dump_trace_entry(m, off-1);
	return 0;
}

static struct seq_operations tmonitor_seqop = {
	.start = tmonitor_seq_start,
	.next  = tmonitor_seq_next,
	.stop  = tmonitor_seq_stop,
	.show  = tmonitor_seq_show
};

static int open_tmonitor(struct inode *inode, struct file *file)
{
	return seq_open(file, &tmonitor_seqop);
}

static struct proc_ops tmonitor_operations = {
	.proc_open	= open_tmonitor,
	.proc_read_iter	= seq_read_iter,
	.proc_write	= write_tmonitor,
	.proc_lseek	= seq_lseek,
	.proc_release	= seq_release,
};

static int tmon_dump = 0;

static int tmon_dumper(void *data)
{
	while (!kthread_should_stop()) {
		set_current_state(TASK_INTERRUPTIBLE);
		if (tmon_dump && tmon_autodump_path) {
			__set_current_state(TASK_RUNNING);
			if (tmon_dumper_delay > 0) {
				msleep(tmon_dumper_delay);
			}
			tmonitor_stop();
			tmonitor_dump(tmon_autodump_path);
			BUG();
		}
		schedule();
	}
	return 0;
}

static struct task_struct *tdump_task;

void tmonitor_dumper_wakeup(void)
{
	if (tdump_task && tmon_autodump_path) {
		tmon_dump = 1;
		wake_up_process(tdump_task);
	}
}
EXPORT_SYMBOL(tmonitor_dumper_wakeup);

static int __init tmonitor_init(void)
{
	proc_create("tmonitor", S_IRUSR | S_IWUSR, NULL, &tmonitor_operations);
	if (tmon_autodump_path) {
		tdump_task = kthread_run(tmon_dumper, NULL, "tmon_dumper");
		if (IS_ERR(tdump_task)) {
			printk(KERN_ERR "%s:ERROR:create tmon_dumper\n",
			       __func__);
			tdump_task = NULL;
		}
	}

	set_trace_cpu(dfl_cpu);
	return 0;
}

module_init(tmonitor_init);
