/*
 * lib/heap_profile.c
 *
 * heap profiler: /proc/profile_heap
 *
 * Copyright 2005-2007, 2010, 2019, 2024 Sony Group Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 *
 */

#include <linux/mm.h>
#include <linux/types.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/kallsyms.h>
#include <linux/sched.h>
#include <linux/vmalloc.h>
#include <linux/slab.h>
#include <linux/seq_file.h>
#include <asm/uaccess.h>
#include <linux/sched/clock.h>

static int records = 0;
module_param(records, int, S_IRUSR);

#define  RECORDS records

static DEFINE_SEMAPHORE(heap_record_index_sem);
typedef struct {
	unsigned long long time; /* sched_clock */
	unsigned long type;      /* measurement point
				    0:try_to_freepages,
				    1: __vm_enough_memory
				 */
	unsigned long nr_used_pages;
	unsigned int nr_free_pages;
	unsigned long nr_cache_pages;
	unsigned long req_pages;
	int pid; 		/* current pid */
} profile_heap_record_t;

static profile_heap_record_t *profile_heap_records;
static profile_heap_record_t profile_heap_used_max_record = {0,};
static profile_heap_record_t profile_heap_needed_max_record = {0,};
static unsigned long used_max = 0;
static unsigned long needed_max = 0;
static unsigned long heap_record_index = 0;
static unsigned int is_buffer_full;

#include <linux/pagemap.h>
#include <linux/vmstat.h> 	/* for global_zone_page_state
				   nr_cache_pages = cached + bufferram
						    + total_swapcache_pages
				*/

void profile_heap_record(unsigned long type, unsigned long req_pages)
{
	unsigned long used = totalram_pages();

	if (!profile_heap_records)
		return;

	if( down_trylock(&heap_record_index_sem) ) {
		printk(KERN_DEBUG "Heap profile down_trylock() failed.\n");
		return;
	}

#ifdef CONFIG_SNSC_DEBUG_PROFILE_HEAP_RINGBUFF
	if(heap_record_index >= RECORDS) {
		heap_record_index -= RECORDS;
		is_buffer_full = 1;
	}
#else
	if(heap_record_index >= RECORDS) {
		is_buffer_full = 1;
		goto out;
	}
#endif

	profile_heap_records[heap_record_index].time = sched_clock();
	profile_heap_records[heap_record_index].type = type;
	profile_heap_records[heap_record_index].nr_free_pages = global_zone_page_state(NR_FREE_PAGES);
	used -= profile_heap_records[heap_record_index].nr_free_pages;
	profile_heap_records[heap_record_index].nr_cache_pages = global_node_page_state(NR_FILE_PAGES);
	used -= profile_heap_records[heap_record_index].nr_cache_pages;
	profile_heap_records[heap_record_index].pid = current->pid;
	profile_heap_records[heap_record_index].req_pages = req_pages;
	profile_heap_records[heap_record_index].nr_used_pages = used;

	if ( used_max <= used ) {
		used_max = used;
		profile_heap_used_max_record = profile_heap_records[heap_record_index];
	}
	used += req_pages;
	if ( needed_max <= used ) {
		needed_max = used;
		profile_heap_needed_max_record = profile_heap_records[heap_record_index];
	}
	heap_record_index++;
#ifndef CONFIG_SNSC_DEBUG_PROFILE_HEAP_RINGBUFF
out:
#endif
	up(&heap_record_index_sem);
	return ;
}
EXPORT_SYMBOL(profile_heap_record);

#define PROFILE_HEAP_HEADINGS "\ntime,used,cache,free,needed,req,pid,type\n"
#define MAX_NEEDED_STR "Max needed record\n"
#define MAX_USED_STR "\nMax used record\n"

static void profile_heap_output_file(struct seq_file *sfile,
				     profile_heap_record_t *record)
{
	unsigned long nanosec_rem;
	unsigned long long t;

	t = record->time;
	nanosec_rem = do_div(t, 1000000000);

	seq_printf(sfile, "%8lu.%06lu,%8lu," "%8lu"  ",%8u" ",%8lu"   ",%8lu" ",%6d"  ",%2lu\n",
			  /* time used cache free used+req req pid type */
			  (unsigned long)t,
			  nanosec_rem/1000,
			  record->nr_used_pages  << (PAGE_SHIFT - 10),
			  record->nr_cache_pages << (PAGE_SHIFT - 10),
			  record->nr_free_pages << (PAGE_SHIFT - 10),
			  (record->req_pages + record->nr_used_pages) << (PAGE_SHIFT - 10),
			  record->req_pages << (PAGE_SHIFT - 10),
			  record->pid,
			  record->type);
}

static unsigned long get_buffer_count(void);
static profile_heap_record_t *get_record(unsigned long line)
{
	unsigned long start_index = 0;
	unsigned long index;

	if (!profile_heap_records)
		return NULL;
	if (line >= get_buffer_count())
		return NULL;
#ifdef CONFIG_SNSC_DEBUG_PROFILE_HEAP_RINGBUFF
	if (is_buffer_full) {
		start_index = heap_record_index;
	}
#endif
	index = (start_index + line) % RECORDS;
	return &profile_heap_records[index];
}

char *profile_heap_to_string(unsigned long line, char *buf, int size)
{
	unsigned long nanosec_rem;
	unsigned long long t;
	profile_heap_record_t *record;

	record = get_record(line);
	if (!record)
		return NULL;

	t = record->time;
	nanosec_rem = do_div(t, 1000000000);
	scnprintf(buf, size, "%8lu.%06lu,%8lu," "%8lu"  ",%8u" ",%8lu"   ",%8lu" ",%6d"  ",%2lu\n",
		  (unsigned long)t,
		  nanosec_rem/1000,
		  record->nr_used_pages  << (PAGE_SHIFT - 10),
		  record->nr_cache_pages << (PAGE_SHIFT - 10),
		  record->nr_free_pages << (PAGE_SHIFT - 10),
		  (record->req_pages + record->nr_used_pages) << (PAGE_SHIFT - 10),
		  record->req_pages << (PAGE_SHIFT - 10),
		  record->pid,
		  record->type);
	return buf;
}

static ssize_t profile_heap_proc_write(struct file *file,
				       const char __user *buffer,
				       size_t count, loff_t *ppos)
{
	heap_record_index = 0;	/* Reset index in order to record again */
				/* in Non-RingBuff Mode. */
	used_max = 0;
	needed_max = 0;
	is_buffer_full = 0;

	return count;
}

static unsigned long get_buffer_count(void)
{
	return is_buffer_full ? RECORDS : heap_record_index;
}

static void *profile_heap_proc_start(struct seq_file *sfile, loff_t *pos)
{
	loff_t off = *pos;

	if (off >= get_buffer_count())
		return NULL;

	return pos;
}

static void *profile_heap_proc_next(struct seq_file *sfile, void *v,
				    loff_t *pos)
{
	loff_t off = ++*pos;

	if (off >= get_buffer_count())
		return NULL;

	return pos;
}

static int profile_heap_proc_show(struct seq_file *sfile, void *v)
{

	loff_t off = *(loff_t *)v;

#ifdef CONFIG_SNSC_DEBUG_PROFILE_HEAP_RINGBUFF
	unsigned long start_index = is_buffer_full ? heap_record_index : 0;
	unsigned long index = (start_index + off) % RECORDS;
#else
	unsigned long index = off;
#endif

	if (off == 0) {
		seq_puts(sfile, MAX_NEEDED_STR);
		profile_heap_output_file(sfile,
					 &profile_heap_needed_max_record);

		seq_puts(sfile, MAX_USED_STR);
		profile_heap_output_file(sfile, &profile_heap_used_max_record);

		seq_puts(sfile, PROFILE_HEAP_HEADINGS);
	}

	if (off < get_buffer_count())
		profile_heap_output_file(sfile,  &profile_heap_records[index]);

	return 0;
}

static void profile_heap_proc_stop(struct seq_file *sfile, void *v)
{
	/* Do nothing */
}

static const struct seq_operations profile_heap_proc_sops = {
	.start = profile_heap_proc_start,
	.next = profile_heap_proc_next,
	.stop = profile_heap_proc_stop,
	.show = profile_heap_proc_show
};

static int profile_heap_proc_open(struct inode *inode, struct file *file)
{
	if (down_interruptible(&heap_record_index_sem))
		return -EINTR;

	return seq_open(file, &profile_heap_proc_sops);
}

static int profile_heap_proc_release(struct inode *inode, struct file *file)
{
	int ret;

	ret = seq_release(inode, file);
	up(&heap_record_index_sem);

	return ret;
}

static const struct proc_ops proc_heap_operations = {
	.proc_open = profile_heap_proc_open,
	.proc_read = seq_read,
	.proc_write	= profile_heap_proc_write,
	.proc_lseek = seq_lseek,
	.proc_release = profile_heap_proc_release
};

static int __init debug_heap_proc_profile(void)
{
	struct proc_dir_entry *ent;

	if (0 >= records)
		return 0;
	profile_heap_records = (profile_heap_record_t *)vmalloc(sizeof (profile_heap_record_t) * records);
	if (!profile_heap_records) {
		printk(KERN_ERR "ERROR: can not allocate profile_heap (%d records)\n", records);
		return -ENOMEM;
	}

	ent = proc_create("profile_heap", S_IFREG|S_IRUGO|S_IWUSR, NULL, &proc_heap_operations);
	if (!ent) {
		printk(KERN_ERR "create profile_heap proc entry failed\n");
		vfree(profile_heap_records);
		return -ENOMEM;
	}
	return 0;
}
late_initcall(debug_heap_proc_profile);

