/*
 *  Boot time analysis
 *
 *  Copyright 2001-2009 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.
 */

#include <linux/kernel.h>
#include <linux/proc_fs.h>
#include <linux/module.h>
#include <linux/seq_file.h>
#include <linux/dma-mapping.h>
#include <linux/snsc_boot_time.h>
#include <linux/slab.h>

#include <linux/memblock.h>

#if defined(CONFIG_SNSC_SSBOOT) && !defined(CONFIG_EJ)
#include <linux/ssboot.h>
#endif

#ifdef CONFIG_SNSC_BOOT_TIME_OF
#include <linux/of.h>
#include <linux/of_address.h>
#elif defined(CONFIG_SNSC_NBLARGS)
#include <linux/snsc_nblargs.h>
#endif
#include <linux/sched.h>
#include <linux/sched/clock.h>
#include <asm/io.h>
#include <asm/string.h>
#include <linux/uaccess.h>
#include <asm/uaccess.h>
#include <asm/pgalloc.h>

#ifdef CONFIG_THREAD_MONITOR
# include <linux/tmonitor.h>
#endif

#undef DEBUG_BOOT_TIME

#if !defined(CONFIG_SNSC_BOOT_TIME_VERSION_1) && \
    !defined(CONFIG_SNSC_BOOT_TIME_OF)
#if (CONFIG_SNSC_BOOT_TIME_MAX_COMMENT & 3)
#error CONFIG_BOOT_TIME_MAX_COMMENT should be 4 bytes aligned value.
#endif
#endif

#ifndef CONFIG_SNSC_BOOT_TIME_OF
#define BOOT_TIME_BASE		CONFIG_SNSC_DEFAULT_BOOT_TIME_BASE
#define BOOT_TIME_SIZE		CONFIG_SNSC_DEFAULT_BOOT_TIME_SIZE
#endif

#define BOOT_TIME_MAGIC_BASE	0x4E554355

#ifndef CONFIG_SNSC_BOOT_TIME_OF
#if defined CONFIG_SNSC_BOOT_TIME_VERSION_1
static u32 boot_time_version = 1;
#elif defined CONFIG_SNSC_BOOT_TIME_VERSION_2
static u32 boot_time_version = 2;
#endif
#else
static u32 boot_time_version;
#endif

#define BOOT_TIME_VERSION	boot_time_version
#define BOOT_TIME_MAGIC		(BOOT_TIME_MAGIC_BASE + BOOT_TIME_VERSION)

#define BOOT_TIME_CLEAR_KEY	"CLEAR"

#ifdef CONFIG_SNSC_BOOT_TIME_USE_NBLARGS
#define BOOT_TIME_NBLARGS_KEY	"boottime"
#endif

#define MAX_COMMENT (boot_time_version == 1 ? 24 : boot_time_max_size)

struct boot_time_entry {
	u32	count_lo;
	u32	count_hi;
	char	comment[];
};

struct boot_time {
	u32	magic;
	u32	offHead;
	u32	offNext;
	u32	offMax;
	union {
		struct {
			u32	max_comment;
			u32     numWritten;
			struct boot_time_entry first_entry_v2;
		};

		struct boot_time_entry first_entry_v1;
	};
};

static u32 boottime_bufsize;
static u32 boottime_nr_entry;

#ifdef CONFIG_SNSC_BOOT_TIME_IN_USEC
static bool boottime_in_usec = true;
#else
static bool boottime_in_usec = false;
#endif
module_param_named(us, boottime_in_usec, bool, S_IRUSR|S_IWUSR);

#ifndef CONFIG_SNSC_BOOT_TIME_USE_NBLARGS
#define SNSC_BOOT_TIME_UNDEF 0xffffffffUL
static unsigned long boottime_addr = SNSC_BOOT_TIME_UNDEF;
static unsigned long boottime_size = SNSC_BOOT_TIME_UNDEF;
#endif /* !CONFIG_SNSC_BOOT_TIME_USE_NBLARGS */

static struct boot_time *pBootTime = NULL;
static int boot_time_max_size;
static bool boot_time_ring_buffer;

static DEFINE_RAW_SPINLOCK(boot_time_lock);

static atomic_t boot_time_offset = ATOMIC_INIT(0);
static DEFINE_SEMAPHORE(proc_mutex);

/*
 * Offset used with sched_clock to show continuity for boottime
 * entries from u-boot to kernel in below cases:
 * 1. Booting from snapshot image
 *    offset = rd->epoch_ns - cyc_ns, where:
 *     rd->epoch_ns - total time elapsed at the time snapshot is taken
 *     cyc_ns - Timer cycles in ns at the time snapshot is taken
 *    boot_time_entry = sched_clock() - offset
 *
 * 2. Normal boot
 *    offset = cyc_ns, where
 *     cyc_ns - Timer cycles in ns during initialzation of sched_clock
 *    boot_time_entry = sched_clock() + offset
 */
#ifndef CONFIG_EJ
unsigned long long boot_time_clock_offset;
#endif

#define BOOT_TIME_HEADER_SIZE_V1 \
	((unsigned long)&(pBootTime->first_entry_v1) - (unsigned long)pBootTime)

#define BOOT_TIME_HEADER_SIZE_V2 \
	((unsigned long)&(pBootTime->first_entry_v2) - (unsigned long)pBootTime)

#define BOOT_TIME_HEADER_SIZE ({ \
		int __size; \
		if (boot_time_version == 2) \
			__size = BOOT_TIME_HEADER_SIZE_V2; \
		else \
			__size = BOOT_TIME_HEADER_SIZE_V1; \
		__size; \
	})

#define BOOT_TIME_ENTRY_SIZE	(sizeof(struct boot_time_entry) + MAX_COMMENT)
#define PENTRY(off)		((struct boot_time_entry *)((unsigned long)pBootTime + off))

#define INC_OFFSET_WITHOUT_RINGBUF(off)                        \
	do { (off) += BOOT_TIME_ENTRY_SIZE; } while(0)

#define INC_OFFSET_WITH_RINGBUF(head, off, max)                \
	do {                                                   \
		if (((off) + BOOT_TIME_ENTRY_SIZE) >= (max)) { \
			(off) = (head);                        \
		} else {                                       \
			(off) += BOOT_TIME_ENTRY_SIZE;         \
		}                                              \
	} while(0)

#define INC_OFFSET(head, off, max)                             \
	do { \
		if (boot_time_ring_buffer) \
			INC_OFFSET_WITH_RINGBUF(head, off, max); \
		else \
			INC_OFFSET_WITHOUT_RINGBUF(off); \
	} while (0)

#define WORD_ALIGNED(addr)	(((unsigned long)(addr) & (4 - 1)) == 0)

static int
inc_offset_and_return_current_offset(int head, int max)
{
	unsigned long flags;
	int current_offset;

	if (!boot_time_ring_buffer) {
		current_offset = atomic_add_return(BOOT_TIME_ENTRY_SIZE,
						   &boot_time_offset);
		current_offset -= BOOT_TIME_ENTRY_SIZE;
		return current_offset;
	}

	raw_spin_lock_irqsave(&boot_time_lock, flags);
	current_offset = atomic_read(&boot_time_offset);
	if ((current_offset + BOOT_TIME_ENTRY_SIZE) >= max) {
		atomic_set(&boot_time_offset, pBootTime->offHead);
	} else {
		atomic_add_return(BOOT_TIME_ENTRY_SIZE, &boot_time_offset);
	}
	raw_spin_unlock_irqrestore(&boot_time_lock, flags);
	return current_offset;
}

__attribute__((weak)) unsigned long long notrace
boot_time_cpu_clock(int cpu)
{
	return cpu_clock(cpu);
}

static void
boot_time_count_set(struct boot_time_entry *pEntry,
			unsigned long long t)
{
	pEntry->count_lo = (u32)t;
	pEntry->count_hi = (u32)(t >> 32);
}

static unsigned long long get_boot_time_cpu_clock(void)
{
	unsigned long long t;

	t = boot_time_cpu_clock(raw_smp_processor_id());
	if (!t) {
		t = (unsigned long long)(jiffies - INITIAL_JIFFIES) *
			(1000000000 / HZ);
	}

#ifndef CONFIG_EJ
#ifdef CONFIG_SNSC_SSBOOT
	if (ssboot_is_resumed()) {
		t = t - boot_time_clock_offset;
		return t;
	}
#endif

	t = t + boot_time_clock_offset;
#endif

	return t;
}

/* add new measurement point */
void
boot_time_add(char *comment)
{
	int offset, len = 0;
	int nr_entry;
	unsigned long long t;

	struct boot_time_entry *p;

#ifdef CONFIG_THREAD_MONITOR
	tmonitor_boot_time_notify(comment);
#endif

	if (pBootTime == NULL) {
		return;
	}

	if (!boot_time_ring_buffer &&
	    pBootTime->offNext >= pBootTime->offMax) {
		return;
	}

	t = get_boot_time_cpu_clock();
	/*
	 * First get the next offset and then count the current offset
	 */
	offset = inc_offset_and_return_current_offset(pBootTime->offHead,
						      pBootTime->offMax);

	if (boot_time_version != 1) {
		nr_entry = (pBootTime->offMax - pBootTime->offHead) /
			   BOOT_TIME_ENTRY_SIZE;

		if (pBootTime->numWritten < nr_entry) {
			pBootTime->numWritten++;
		}
	}

	if (!boot_time_ring_buffer &&
	    offset >= pBootTime->offMax) {
		return;
	}

	p = PENTRY(offset);
	boot_time_count_set(p, t);
	if (comment) {
#ifdef CONFIG_SMP
		unsigned int cpu = get_cpu();
		put_cpu();
		len = snprintf(p->comment, MAX_COMMENT, "%u: ", cpu);
#endif
		strncpy(&(p->comment[len]), comment, MAX_COMMENT - len);
	} else {
		p->comment[0] = '\0';
	}

	pBootTime->offNext = atomic_read(&boot_time_offset);
	if (!boot_time_ring_buffer &&
	    pBootTime->offNext > pBootTime->offMax) {
		pBootTime->offNext = pBootTime->offMax;
	}
	return ;
}

/* (*pos)
 *   Upper 32bit is the offset when seq_start.
 *   Lower 32bit is the position.
 */
# define BOOTTIME_GET_POS(x)	(lower_32_bits(x))
# define BOOTTIME_GET_START(x)	(upper_32_bits(x))
# define BOOTTIME_SET_START(x)	((u64)(x) << 32)

static void *
boot_time_seq_start(struct seq_file *m, loff_t *pos)
{
	u_int64_t off;
	loff_t n = *pos;
	int nr_entry;
	u32 start;
	s32 diff;

	if (down_interruptible(&proc_mutex))
		return NULL;

	if (pBootTime == NULL) {
		return NULL;
	}

#ifdef DEBUG_BOOT_TIME
	printk(KERN_ERR "seq_start: 0x%llx, ", n);
#endif
	if (!boot_time_ring_buffer) {
		off = pBootTime->offHead;
		if (off == pBootTime->offNext)
			return NULL;
	} else {
		nr_entry = (pBootTime->offMax - pBootTime->offHead) /
			   BOOT_TIME_ENTRY_SIZE;

		if (pBootTime->numWritten >= nr_entry) {
			off = pBootTime->offNext;
		} else {
			off = pBootTime->offHead;
		}
		diff = 0;
		if (!n) {
			/* save the offset into upper 32bit. */
			*pos = BOOTTIME_SET_START(off);
		} else {
			start = BOOTTIME_GET_START(n);
			n = BOOTTIME_GET_POS(n);
			if ((diff = off - start) < 0) {
				diff += boottime_bufsize;
			}
			diff /= BOOT_TIME_ENTRY_SIZE;
			if (n < diff) {
				/* catch up (leap) */
				n = diff;
				*pos = BOOTTIME_SET_START(start) | diff;
			}
		}
		if (n > pBootTime->numWritten || pBootTime->numWritten == 0)
			return NULL;

		/* adjust (relative to current off) */
		n -= diff;
	}

	while (n--) {
		INC_OFFSET(pBootTime->offHead, off, pBootTime->offMax);
		if (off == pBootTime->offNext)
			return NULL;
	}
#ifdef DEBUG_BOOT_TIME
	printk(KERN_ERR "off=0x%x\n", off);
#endif
	return (void *)(uintptr_t)off;
}

static void
boot_time_seq_stop(struct seq_file *m, void *v)
{
#ifdef DEBUG_BOOT_TIME
	printk(KERN_ERR "seq_stop: off=%p\n", v);
#endif
	up(&proc_mutex);
}

static void *
boot_time_seq_next(struct seq_file *m, void *v, loff_t *pos)
{
	u_int64_t off = (uintptr_t)v;

	(*pos)++;
	INC_OFFSET(pBootTime->offHead, off, pBootTime->offMax);
	if (boot_time_ring_buffer &&
	    BOOTTIME_GET_POS(*pos) > boottime_nr_entry) {
		/* to avoid infinity loop */
		return NULL;
	}

	return (off == pBootTime->offNext) ? NULL : (void *)(uintptr_t)off;
}

static int
boot_time_seq_show(struct seq_file *m, void *v)
{
	u_int64_t off = (uintptr_t)v;
	u_int32_t start;
	int nr_entry;
	struct boot_time_entry *p;
	int i;
	int space = 0;
	u64 dividend;

	if (!boot_time_ring_buffer)
		start = pBootTime->offHead;
	else {
		nr_entry =  (pBootTime->offMax - pBootTime->offHead) /
			BOOT_TIME_ENTRY_SIZE;

		if (pBootTime->numWritten >= nr_entry) {
			start = pBootTime->offNext;
		} else {
			start = pBootTime->offHead;
		}
	}

	if (off == start) {
		seq_printf(m, "boot time (%p)\n", pBootTime);
	}

	p = PENTRY((uintptr_t)off);
	for(i =0; i < MAX_COMMENT; i++) {
		if (space || p->comment[i] == '\0' || p->comment[i] == '\n') {
			seq_putc(m, ' ');
			space = 1;
		} else {
			seq_printf(m, "%c", p->comment[i]);
		}
	}

	dividend = ((u64)p->count_hi << 32) | p->count_lo;
	if (boottime_in_usec) {
	do_div(dividend, 1000);	/* sched_clock() returns nano-sec. */
	seq_printf(m, " : %8Lu [us]\n", dividend);
	} else {
	do_div(dividend, 1000000);	/* sched_clock() returns nano-sec. */
	seq_printf(m, " : %5u [ms]\n", (u32)dividend);
	}

	return 0;
}

static void
do_cache_sync(const void *addr, size_t size, int rw)
{
	dma_addr_t handle;
	struct device *dev;
	dev = kzalloc(sizeof(struct device), GFP_KERNEL);
	if (!dev){
		printk(KERN_ERR "Failed to allocate memory\n");
		return;
	}
	dev->dma_mask = kzalloc(sizeof(*dev->dma_mask), GFP_KERNEL);
	if (!dev->dma_mask){
		printk(KERN_ERR "Failed to allocate memory\n");
		kfree(dev);
		return;
	}
	*dev->dma_mask = DMA_BIT_MASK(64);
	handle = dma_map_single(dev, (void*)addr, size, rw);
	if (dma_mapping_error(dev, handle)) {
		printk(KERN_ERR "dma_map_single failed for addr=%px\n", (void *)addr);
		goto out;
	}
	dma_unmap_single(dev, handle, size, rw);

out:
	kfree(dev->dma_mask);
	kfree(dev);
}

/* clear boot-time region */
static void
boot_time_clear(void)
{
	if (pBootTime == NULL) {
		return;
	}

	pBootTime->offNext = pBootTime->offHead;
 	atomic_set(&boot_time_offset, pBootTime->offNext);
	if (boot_time_version != 1)
		pBootTime->numWritten = 0;

	do_cache_sync(pBootTime,
		      (unsigned long)pBootTime->offMax,
		      DMA_TO_DEVICE);
}

static ssize_t
proc_boot_time_write(struct file *file, const char *buffer, size_t count, loff_t *ppos)
{
	unsigned long count0 = count;
	int len1;
	char *comment;

	comment = kmalloc(MAX_COMMENT + 1, GFP_KERNEL);
	if (comment == NULL) {
		printk(KERN_INFO
		    "Unable to allocate memory for boot_time proc buffer\n");
		return -ENOMEM;
	}
	if (down_interruptible(&proc_mutex)) {
		kfree(comment);
		return -EINTR;
	}
	if (count0 > MAX_COMMENT) {
		len1 = MAX_COMMENT;
	} else {
		len1 = count0;
	}
	count0 -= len1;
	if (copy_from_user(comment, buffer, len1)) {
		kfree(comment);
		return -EFAULT;
	}
	comment[len1] = '\0';

	if (strncmp(comment, BOOT_TIME_CLEAR_KEY, strlen(BOOT_TIME_CLEAR_KEY)) == 0)
		boot_time_clear();
	else
		boot_time_add(comment);

	up(&proc_mutex);
	kfree(comment);

	return count;
}

static struct seq_operations proc_boot_time_seqop = {
	.start = boot_time_seq_start,
	.next  = boot_time_seq_next,
	.stop  = boot_time_seq_stop,
	.show  = boot_time_seq_show
};

static int proc_boot_time_open(struct inode *inode, struct file *file)
{
	return seq_open(file, &proc_boot_time_seqop);
}

static struct proc_ops proc_boot_time_operations = {
	.proc_open    = proc_boot_time_open,
	.proc_read    = seq_read,
	.proc_write   = proc_boot_time_write,
	.proc_lseek  = seq_lseek,
	.proc_release = seq_release,
};


#ifdef CONFIG_SNSC_BOOT_TIME_OF
/*
 * Parse the boot time memory configuration from device tree.
 * If the required properties are not found, the boot time feature
 * will be disabled. The device tree should contain the required
 * properties(either manually inserted or inserted by bootloader)
 * if the boottime functionality is desired.
 */
static int __init
boot_time_memory_init(u32 *size)
{
	int ret;
	struct resource mem;
	struct device_node *rmem, *bt;

	*size = 0;

	rmem = of_find_node_by_path("/reserved-memory");
	if (IS_ERR_OR_NULL(rmem)) {
		pr_err("boottime: can not open /reserved-memory\n");
		return IS_ERR(rmem) ? PTR_ERR(rmem) : -ENOENT;
	}

	bt = of_get_child_by_name(rmem, "boottime");
	if (IS_ERR_OR_NULL(rmem)) {
		pr_err("boottime: can not open boottime node\n");
		return IS_ERR(rmem) ? PTR_ERR(rmem) : -ENOENT;
	}

	if (!of_device_is_available(bt)) {
		pr_err("boottime: no available node\n");
		return -ENODEV;
	}

	ret = of_property_read_u32(bt, "version", &boot_time_version);
	if (ret < 0) {
		pr_err("boottime: can not get version\n");
		return ret;
	}

	if (boot_time_version == 1)
		boot_time_max_size = 24;
	else {
		ret = of_property_read_u32(bt, "max-comment",
				&boot_time_max_size);
		if (ret < 0) {
			pr_err("boottime: can not get max-comment\n");
			return ret;
		}

		if (boot_time_max_size & 3) {
			pr_err("boottime: max-comment doesn't align with 4\n");
			return -EINVAL;
		}
	}

	/* Get reserved memory for boottime */
	ret = of_address_to_resource(bt, 0, &mem);
	if (ret) {
		pr_err("boottime: can not get boot time region\n");
		return ret;
	}

	pBootTime = (struct boot_time *)__va(mem.start);
	*size = mem.end - mem.start;
	boot_time_ring_buffer = of_property_read_bool(bt,"ring-buffer");

	return 0;
}
#else
static int
reserve_default_boottime_region(void)
{
	phys_addr_t addr;

	addr = memblock_phys_alloc_range(BOOT_TIME_SIZE, PAGE_SIZE, BOOT_TIME_BASE,
			        BOOT_TIME_BASE + BOOT_TIME_SIZE);
	if (addr != BOOT_TIME_BASE) {
  		printk(KERN_ERR "cannot reserve memory at 0x%08x size 0x%x "
  		       "it is already reserved\n",
			(unsigned int)BOOT_TIME_BASE, (unsigned int)BOOT_TIME_SIZE);
  		return -ENOMEM;
  	}
	return 0;
}
#endif /* CONFIG_SNSC_BOOT_TIME_OF */

void boot_time_takeover(void)
{
	/* initialize the atomic offset */
	atomic_set(&boot_time_offset, (u32)pBootTime->offNext);
}

static void
check_boot_time_region(void)
{
	if ((pBootTime->magic != BOOT_TIME_MAGIC)
	    || ((u_int32_t)pBootTime->offHead > (u_int32_t)pBootTime->offMax)
	    || ((u_int32_t)pBootTime->offNext > (u_int32_t)pBootTime->offMax)
	    || (!WORD_ALIGNED(pBootTime->offHead))
	    || (!WORD_ALIGNED(pBootTime->offNext))) {
		/* initialize boot time region */
		pBootTime->magic = BOOT_TIME_MAGIC;
		pBootTime->offHead = BOOT_TIME_HEADER_SIZE;
		pBootTime->offNext = pBootTime->offHead;
		if (boot_time_version != 1) {
			pBootTime->max_comment = boot_time_max_size;
			pBootTime->numWritten = 0;
		}
	}

	boot_time_takeover();
}

/*
 *  Initialize boottime
 */
void __init
boot_time_init(void)
{
#ifdef CONFIG_SNSC_BOOT_TIME_USE_NBLARGS
	struct nblargs_entry na;
#endif
	static int first = 1;
	u32 size, nr_entry;
	int err = 0;

	if (!first)
		return;
	first = 0;

#ifdef CONFIG_SNSC_BOOT_TIME_RINGBUFFER
	boot_time_ring_buffer = true;
#else
	boot_time_ring_buffer = false;
#endif

#if defined(CONFIG_SNSC_BOOT_TIME_OF)
	err = boot_time_memory_init(&size);
	if (err < 0) {
		pBootTime = 0;
		return;
	}
#elif defined(CONFIG_SNSC_BOOT_TIME_USE_NBLARGS)
	if (nblargs_get_key(BOOT_TIME_NBLARGS_KEY, &na) < 0) {
		printk(KERN_INFO "NBLArgs key \"" BOOT_TIME_NBLARGS_KEY
		       "\" not found, using default(0x%08x)\n",
		       (unsigned int)BOOT_TIME_BASE);

		err = reserve_default_boottime_region();
	       	if (err) {
 			pBootTime = NULL;
 			return;
 		}
		pBootTime = (struct boot_time *)__va(BOOT_TIME_BASE);
		size = BOOT_TIME_SIZE;
	} else {
		printk(KERN_INFO "NBLArgs key \"" BOOT_TIME_NBLARGS_KEY
		       "\" found(0x%08x)\n", na.addr);

		/* specified boottime region is already reserved by
		   nblargs_init() */
		pBootTime = (struct boot_time *)nbl_to_va(na.addr);
		size = na.size;
	}
#else
	if (SNSC_BOOT_TIME_UNDEF != boottime_size) {
		pBootTime = (struct boot_time *)__va(boottime_addr);
		size = boottime_size;
	} else {
	err = reserve_default_boottime_region();
 	if (err) {
 		pBootTime = NULL;
 		return;
 	}
	pBootTime = (struct boot_time *)__va(BOOT_TIME_BASE);
	size = BOOT_TIME_SIZE;
	}
#endif

#ifndef CONFIG_SNSC_BOOT_TIME_OF
	if (boot_time_version != 1) {
		if (pBootTime->magic == BOOT_TIME_MAGIC) {
			boot_time_max_size = pBootTime->max_comment;
		} else {
			boot_time_max_size = CONFIG_SNSC_BOOT_TIME_MAX_COMMENT;
		}
	}
#endif

	/* force offMax align to entry start address */
	nr_entry = (u32)((size - BOOT_TIME_HEADER_SIZE) /
			BOOT_TIME_ENTRY_SIZE);
	pBootTime->offMax = (u32)(BOOT_TIME_HEADER_SIZE +
				  nr_entry * BOOT_TIME_ENTRY_SIZE);
	boottime_bufsize = pBootTime->offMax - pBootTime->offHead;
	boottime_nr_entry = nr_entry;
#ifdef DEBUG_BOOT_TIME
	printk(KERN_ERR "boottime_bufsize=%u\n", boottime_bufsize);
	printk(KERN_ERR "boottime_nr_entry=%u\n", boottime_nr_entry);
#endif

	if (size < (u32)(BOOT_TIME_HEADER_SIZE + BOOT_TIME_ENTRY_SIZE)) {
		printk(KERN_INFO
		       "too small boot time save area(0x%x)\n", size);
		pBootTime = NULL;
		return;
	}

	/* check boot time region and setup the atomic offset */
	check_boot_time_region();
}

/* initialize /proc for boot time */
static int
boot_time_proc_init(void)
{
	struct proc_dir_entry *entry;

	entry = proc_create("snsc_boot_time", 0666, NULL, &proc_boot_time_operations);

	return 0;
}
late_initcall(boot_time_proc_init);

#ifdef CONFIG_PM
/*
 * sync boot_time_offset with the offset updated by boot loader.
 */
void boot_time_resume(void)
{
	if (pBootTime != NULL)
		check_boot_time_region();
}
#endif

#ifndef CONFIG_SNSC_BOOT_TIME_USE_NBLARGS
#include <asm/setup.h>

static int __init parse_boottime(char *p)
{
	char *endp;

	boottime_size = memparse(p, &endp);
	if (*endp == '@' || *endp == '$' || *endp == '#') {
		boottime_addr = memparse(endp + 1, NULL);
	}
	return 0;
}
early_param("boottime", parse_boottime);
#endif /* !CONFIG_SNSC_BOOT_TIME_USE_NBLARGS */

EXPORT_SYMBOL(boot_time_add);
