/*
 * mm/oom_callback.c
 *
 * for printing information during OOM
 *
 * Copyright 2012,2023 Sony Corporation
 *
 * This code is based on fs/proc/meminfo.c
 */

#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/hugetlb.h>
#include <linux/mman.h>
#include <linux/swap.h>
#ifdef CONFIG_CMA
#include <linux/cma.h>
#endif
#include <asm/page.h>
#include <linux/oom_callback.h>

#define K(x) ((x) << (PAGE_SHIFT-10))

static void show_kb(const char *name, unsigned long kbytes)
{
	printk(KERN_EMERG "%s%8lu kB\n", name, kbytes);
}

static void show_val_kb(const char *name, unsigned long num_pages)
{
	show_kb(name, K(num_pages));
}

static void dump_meminfo(void)
{
	struct sysinfo i;
	unsigned long committed;
	long cached;
	long available;
	unsigned long pages[NR_LRU_LISTS];
	unsigned long sreclaimable, sunreclaim;
	int lru;

	si_meminfo(&i);
	si_swapinfo(&i);
	committed = vm_memory_committed();

	cached = global_node_page_state(NR_FILE_PAGES) -
		total_swapcache_pages() - i.bufferram;
	if (cached < 0)
		cached = 0;

	for (lru = LRU_BASE; lru < NR_LRU_LISTS; lru++)
		pages[lru] = global_node_page_state(NR_LRU_BASE + lru);

	available = si_mem_available();
	sreclaimable = global_node_page_state_pages(NR_SLAB_RECLAIMABLE_B);
	sunreclaim = global_node_page_state_pages(NR_SLAB_UNRECLAIMABLE_B);

	show_val_kb("MemTotal:       ", i.totalram);
	show_val_kb("MemFree:        ", i.freeram);
	show_val_kb("MemAvailable:   ", available);
	show_val_kb("Buffers:        ", i.bufferram);
	show_val_kb("Cached:         ", cached);
	show_val_kb("SwapCached:     ", total_swapcache_pages());
	show_val_kb("Active:         ", pages[LRU_ACTIVE_ANON]+pages[LRU_ACTIVE_FILE]);
	show_val_kb("Inactive:       ", pages[LRU_INACTIVE_ANON]+pages[LRU_INACTIVE_FILE]);
	show_val_kb("Active(anon):   ", pages[LRU_ACTIVE_ANON]);
	show_val_kb("Inactive(anon): ", pages[LRU_INACTIVE_ANON]);
	show_val_kb("Active(file):   ", pages[LRU_ACTIVE_FILE]);
	show_val_kb("Inactive(file): ", pages[LRU_INACTIVE_FILE]);
	show_val_kb("Unevictable:    ", pages[LRU_UNEVICTABLE]);
	show_val_kb("Mlocked:        ", global_zone_page_state(NR_MLOCK));
#ifdef CONFIG_HIGHMEM
	show_val_kb("HighTotal:      ", i.totalhigh);
	show_val_kb("HighFree:       ", i.freehigh);
	show_val_kb("LowTotal:       ", i.totalram - i.totalhigh);
	show_val_kb("LowFree:        ", i.freeram - i.freehigh);
#endif
#ifndef CONFIG_MMU
	show_val_kb("MmapCopy:       ", (unsigned long)atomic_long_read(&mmap_pages_allocated));
#endif
	show_val_kb("SwapTotal:      ", i.totalswap);
	show_val_kb("SwapFree:       ", i.freeswap);
	show_val_kb("Dirty:          ", global_node_page_state(NR_FILE_DIRTY));
	show_val_kb("Writeback:      ", global_node_page_state(NR_WRITEBACK));
	show_val_kb("AnonPages:      ", global_node_page_state(NR_ANON_MAPPED));
	show_val_kb("Mapped:         ", global_node_page_state(NR_FILE_MAPPED));
	show_val_kb("Shmem:          ", i.sharedram);
	show_val_kb("KReclaimable:   ", sreclaimable + global_node_page_state(NR_KERNEL_MISC_RECLAIMABLE));
	show_val_kb("Slab:           ", sreclaimable + sunreclaim);
	show_val_kb("SReclaimable:   ", sreclaimable);
	show_val_kb("SUnreclaim:     ", sunreclaim);
	show_kb    ("KernelStack:    ", global_node_page_state(NR_KERNEL_STACK_KB));
#ifdef CONFIG_SHADOW_CALL_STACK
	show_kb    ("ShadowCallStack:", global_node_page_state(NR_KERNEL_SCS_KB));
#endif
	show_val_kb("PageTables:     ", global_node_page_state(NR_PAGETABLE));
	show_val_kb("NFS_Unstable:   ", 0);
	show_val_kb("Bounce:         ", global_zone_page_state(NR_BOUNCE));
	show_val_kb("WritebackTmp:   ", global_node_page_state(NR_WRITEBACK_TEMP));
	show_val_kb("CommitLimit:    ", vm_commit_limit());
	show_val_kb("Committed_AS:   ", committed);

#ifdef CONFIG_TRANSPARENT_HUGEPAGE
	show_val_kb("AnonHugePages:  ", global_node_page_state(NR_ANON_THPS));
	show_val_kb("ShmemHugePages: ", global_node_page_state(NR_SHMEM_THPS));
	show_val_kb("ShmemPmdMapped: ", global_node_page_state(NR_SHMEM_PMDMAPPED));
	show_val_kb("FileHugePages:  ", global_node_page_state(NR_FILE_THPS));
	show_val_kb("FilePmdMapped:  ", global_node_page_state(NR_FILE_PMDMAPPED));
#endif
#ifdef CONFIG_CMA
	show_val_kb("CmaTotal:       ", totalcma_pages);
	show_val_kb("CmaFree:        ", global_zone_page_state(NR_FREE_CMA_PAGES));
#endif
#if 0
	hugetlb_report_meminfo();
#endif
}

int __weak before_oom_kill_task(struct task_struct *target)
{
	printk(KERN_EMERG "Out Of Memory - pid(%d) to be killed\n", target->pid);
	dump_meminfo();
	return SIGSEGV;
}

void __weak after_oom_kill_task(struct task_struct *p)
{
	return;
}
