/*
 *  exception-arm64.c  - arm64 specific part of Exception Monitor
 *  Based on exception-arm.c
 *
 */

/* With non GPL files, use following license */
/*
 * Copyright 2005,2008,2009,2016 Sony Corporation.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/* Otherwise with GPL files, use following license */
/*
 *  Copyright 2005,2008,2009,2016 Sony Corporation.
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/ctype.h>
#include <linux/elf.h>
#include <linux/interrupt.h>
#include <linux/vmalloc.h>
#include <linux/version.h>
#include <asm/uaccess.h>
#include <linux/pci.h>
#include <linux/console.h>
#include <linux/fs.h>
#include <linux/sched.h>
#include <linux/sched/signal.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/kallsyms.h>
#include <asm/stacktrace.h>
#include <asm/traps.h>
#include "exception.h"
#ifdef CONFIG_SNSC_ALT_BACKTRACE
#include <linux/snsc_backtrace.h>
#endif
#ifdef CONFIG_SNSC_EM_DEMANGLE
#include <demangle.h>
#endif

#define TASK_STATE_TO_CHAR_STR "RSDTtZXxKWP"

//#define DEBUG
#ifdef DEBUG
#define dbg(fmt, argv...) em_dump_write(fmt, ##argv)
#else
#define dbg(fmt, argv...) do{}while(0)
#endif

#define ARRAY_NUM(x) (sizeof(x)/sizeof((x)[0]))
#define ALIGN4(x) (((x) + 0x3) & 0xfffffffc)
#define FILE int

extern struct pt_regs *em_regs;

/*
 * Show task states
 */
struct stacktrace_state {
	unsigned int depth;
};

struct em_task_state_data {
	struct stackframe frame;
	struct stacktrace_state state;
};

#ifdef CONFIG_SNSC_EM_INITDUMP_PROC_NOTIFY
#define BT_MAX 100

struct stackframe_entry {
	struct stackframe entry;
	unsigned int depth;
};

struct em_initdump_backtrace_data {
	struct stackframe frame;
	struct stackframe_entry ent;
};

static void em_dump_write_buffer(const char *format, ...)
{
	char buf[WRITE_BUF_SZ];
	va_list args;

	va_start(args, format);
	vsnprintf(buf, WRITE_BUF_SZ, format, args);
	va_end(args);
	buf[WRITE_BUF_SZ - 1] = '\0';

	em_dump_string_to_buffer(buf, strlen(buf));
}

/*
 * This function is from a following file.
 * arch/arm64/kernel/traps.c (dump_mem())
 */
static void em_dump_exception_stack(unsigned long bottom, unsigned long top)
{
	unsigned long first;
	int i;

	em_dump_write_buffer("Exception stack(0x%016lx to 0x%016lx)\n", bottom, top);

	for (first = bottom & ~31; first < top; first += 32) {
		unsigned long p;
		char str[(sizeof(" 1234567812345678") - 1) * 4 + 1];

		memset(str, ' ', sizeof(str));
		str[sizeof(str) - 1] = '\0';

		for (p = first, i = 0; i < (32/8) && p < top; i++, p += 8) {
			if (p >= bottom && p < top) {
				unsigned long val;
				val = *(unsigned long *)p;
				scnprintf(str + i * 17, sizeof(str) - i * 17, " %016lx", val);
			}
		}
		em_dump_write_buffer("%04lx:%s\n", first & 0xffff, str);
	}
}

/*
 * This function is from a following file.
 * arch/arm64/kernel/trap.c (dump_backtrace_entry())
 */
static void em_dump_backtrace_entry(unsigned long where,
				    unsigned long from, unsigned long frame)
{
#ifdef CONFIG_KALLSYMS
	char sym1[KSYM_SYMBOL_LEN], sym2[KSYM_SYMBOL_LEN];
	sprint_symbol(sym1, where);
	sprint_symbol(sym2, from);

	em_dump_write_buffer("[<%08lx>] (%s) from [<%08lx>] (%s)\n", where, sym1, from, sym2);
#else
	em_dump_write_buffer("Function entered at [<%08lx>] from [<%08lx>]\n", where, from);
#endif

	if (in_entry_text(from) && (((struct pt_regs *)frame)->pstate & PSR_MODE_EL1h))
		em_dump_exception_stack(frame, frame + sizeof(struct pt_regs));
}

static bool report_trace(void *d, unsigned long pc)
{
	struct em_initdump_backtrace_data *p = d;
	unsigned long stack;

	stack = p->frame.fp - offsetof(struct pt_regs, stackframe);

	/* skip the first one */
	if (p->ent.depth != BT_MAX)
		em_dump_backtrace_entry(p->ent.entry.pc, p->frame.pc, stack);

	p->ent.depth--;
	p->ent.entry.fp = p->frame.fp;
	p->ent.entry.pc = p->frame.pc;

	return p->ent.depth != 0;
}

/* Simillar to dump_backtrace() in arch/arm64/kernel/traps.c */
static void em_initdump_bt(struct pt_regs *_regs)
{
	struct em_initdump_backtrace_data btd;

	if (!is_initdumping)
		return;

	if (_regs) {
		/* kstack_regs */
		start_backtrace(&btd.frame,
				_regs->regs[29],
				_regs->pc);
	} else {
		/* kstack */
		start_backtrace(&btd.frame,
				(unsigned long)__builtin_frame_address(0),
				(unsigned long)em_initdump_bt);
	}

	btd.ent.depth = BT_MAX;

	walk_stackframe(current, &btd.frame, report_trace, &btd);

	return;
}

static void em_initdump_usr_regs(struct pt_regs *usr_regs)
{
	int i, top_reg;
	u64 lr, sp;

	if (!is_initdumping)
		return;

	em_dump_write("\n[user register dump]\n");

	if (compat_user_mode(usr_regs)) {
		lr = usr_regs->compat_lr;
		sp = usr_regs->compat_sp;
		top_reg = 12;
	} else {
		lr = usr_regs->regs[30];
		sp = usr_regs->sp;
		top_reg = 29;
	}

	show_regs_print_info(KERN_DEFAULT);
	em_dump_write("pc : [<0x%016llx>] lr : [<0x%016llx>] pstate: 0x%016llx\n",
		usr_regs->pc, lr, usr_regs->pstate);
	em_dump_write("sp : 0x%016llx\n", sp);
	for (i = top_reg; i >= 0; i--) {
		em_dump_write("x%-2d: 0x%016llx ", i, usr_regs->regs[i]);
		if (i % 2 == 0)
			em_dump_write("\n");
	}
	em_dump_write("\n");
}
#else
static void em_initdump_bt(struct pt_regs *_regs) {}
static void em_initdump_usr_regs(struct pt_regs *usr_regs) {}
#endif

/*
 * for demangle
 */
#ifdef CONFIG_SNSC_EM_DEMANGLE
/* static int demangle_flag = DMGL_PARAMS | DMGL_ANSI | DMGL_VERBOSE | DMGL_TYPES; */
static int demangle_flag = DMGL_PARAMS | DMGL_ANSI | DMGL_VERBOSE;
#endif

static inline ulong em_get_user(ulong *p)
{
	ulong v;

	if (__get_user(v, p))
		v = 0;

	return v;
}

static inline ulong em_put_user(ulong v, ulong *p)
{
	return __put_user(v, p);
}

static inline ulong arch_stack_pointer(ulong *frame)
{
	/* Since two new members valid for EL1 exceptions were added in struct pt_regs
	 * in 5.10 kernel, fp-2 does not point to sp for kernel mode exceptions. fp-4 has
	 * to be used to point to sp.
	 */
	if(user_mode(em_regs))
		return em_get_user(frame - 2);
	else
		return *(frame - 4);
}

static inline ulong arch_caller_address(ulong *frame)
{
	return em_get_user(frame - 1);
}

static inline ulong *arch_prev_frame(ulong *frame)
{
	if (user_mode(em_regs))
		return (ulong *)em_get_user(frame - 3);
	else
		return (ulong *)(frame - 3);
}

void em_dump_regs(int argc, char **argv)
{
	char *mode;
	char mode_list[][5] = {"EL0t","EL1t","EL1h","EL2t","EL2h"
			       ,"EL3t","EL3h","???"};
	int i, top_reg;
	u64 lr, sp;

	if (compat_user_mode(em_regs)) {
		lr = em_regs->compat_lr;
		sp = em_regs->compat_sp;
		top_reg = 12;
	} else {
		lr = em_regs->regs[30];
		sp = em_regs->sp;
		top_reg = 29;
	}

	em_dump_write("\n[register dump]\n");
	show_regs_print_info(KERN_DEFAULT);
	em_dump_write("pc : [<0x%016llx>] lr : [<0x%016llx>] pstate: 0x%016llx\n",
	       em_regs->pc, lr, em_regs->pstate);
	em_dump_write("sp : 0x%016llx\n", sp);
	for (i = top_reg; i >= 0; i--) {
		em_dump_write("x%-2d: 0x%016llx ", i, em_regs->regs[i]);
		if (i % 2 == 0)
			em_dump_write("\n");
	}
	em_dump_write("\n");

	switch (em_regs->pstate & PSR_MODE_MASK) {
	case PSR_MODE_EL0t: mode = mode_list[0]; break;
	case PSR_MODE_EL1t: mode = mode_list[1]; break;
	case PSR_MODE_EL1h: mode = mode_list[2]; break;
	case PSR_MODE_EL2t: mode = mode_list[3]; break;
	case PSR_MODE_EL2h: mode = mode_list[4]; break;
	case PSR_MODE_EL3t: mode = mode_list[5]; break;
	case PSR_MODE_EL3h: mode = mode_list[6]; break;
	default: mode = mode_list[7];break;
	}
	em_dump_write("pstate: 0x%016lx: Condition Flags: %c%c%c%c, "
		      "IRQ: O%s, FIQ: O%s, \n Execution state: %s, "
		      " Debug mask: %smasked System Error: %smasked Mode: %s\n",
		      em_regs->pstate,
		      (em_regs->pstate & PSR_N_BIT) ? 'N' : 'n',
		      (em_regs->pstate & PSR_Z_BIT) ? 'Z' : 'z',
		      (em_regs->pstate & PSR_C_BIT) ? 'C' : 'c',
		      (em_regs->pstate & PSR_V_BIT) ? 'V' : 'v',
		      (em_regs->pstate & PSR_I_BIT) ? "FF" : "N",
		      (em_regs->pstate & PSR_F_BIT) ? "FF" : "N",
		      (em_regs->pstate & PSR_MODE32_BIT) ? "ARM" : "A64",
		      (em_regs->pstate & PSR_D_BIT) ? "" : "Un",
		      (em_regs->pstate & PSR_A_BIT) ? "" : "Un",
		      mode);

	return;
}

#ifdef CONFIG_SNSC_EM_DUMP_VFP_REGISTER
/* Print the the current VFP state from the provided structures. */
void em_dump_vfp(int argc, char **argv)
{
	struct user_fpsimd_state *hwstate;
	struct task_struct *tsk = current;
	int i;

	/* Ensure that the saved uregs is up-to-date. */
	fpsimd_restore_current_state();
	hwstate = &tsk->thread.uw.fpsimd_state;

	em_dump_write("\n[FPSIMD register dump]\n");

#ifdef CONFIG_SMP
	em_dump_write("cpu:%d\n", &tsk->thread.fpsimd_cpu);
#endif
	for (i = 32; i >= 0; i--) {
		em_dump_write("V%-2d: 0x%032llx ", i, hwstate->vregs[i]);
		if (i % 2 == 0)
			em_dump_write("\n");
	}

	em_dump_write("fpsr:0x%08x:\n\tCondition Flags: %c%c%c%c%s\n"
		      "\tException bits: IDC:O%s, IXC:O%s, UFC:O%s, OFC:O%s, DZC:O%s, IOC:O%s\n",
		      hwstate->fpcr,
		      (hwstate->fpsr & PSR_N_BIT) ? 'N' : 'n',
		      (hwstate->fpsr & PSR_Z_BIT) ? 'Z' : 'z',
		      (hwstate->fpsr & PSR_C_BIT) ? 'C' : 'c',
		      (hwstate->fpsr & PSR_V_BIT) ? 'V' : 'v',
		      (hwstate->fpsr & PSR_AA32_Q_BIT) ? "Q" : "q",
		      (hwstate->fpsr & (1 << 7)) ? "FF" : "N",
		      (hwstate->fpsr & (1 << 4)) ? "FF" : "N",
		      (hwstate->fpsr & (1 << 3)) ? "FF" : "N",
		      (hwstate->fpsr & (1 << 2)) ? "FF" : "N",
		      (hwstate->fpsr & (1 << 1)) ? "FF" : "N",
		      (hwstate->fpsr & (1 << 0)) ? "FF" : "N");

	em_dump_write("fpcr:0x%08x:\n\tRMode:%d FZ:%sabled DN:O%s AHP:O%s\n",
			hwstate->fpcr, hwstate->fpcr & (0x3 << 22),
			hwstate->fpcr & (0x1 << 24) ? "En" : "Dis",
			hwstate->fpcr & (0x1 << 25) ? "FF" : "N",
			hwstate->fpcr & (0x1 << 26) ? "FF" : "N");
}
#endif

void em_dump_regs_detail(int argc, char **argv)
{
	em_dump_regs(1, NULL);
	return;
}

static void em_dump_till_end_of_page(unsigned long *sp)
{
	unsigned long *tail = sp;
	unsigned long stackdata;
	int i = 0;
	char buf[33];

	tail = (unsigned long *)(((unsigned long)sp + PAGE_SIZE - 1) & PAGE_MASK);

#define MIN_STACK_LEN 2048
	if (((unsigned long)tail - (unsigned long)sp) < MIN_STACK_LEN)
		tail = (unsigned long *)((unsigned long) sp + MIN_STACK_LEN);

	buf[32] = 0;
	while (sp < tail) {

		if ((i % 4) == 0) {
			em_dump_write("0x%016lx : ", (unsigned long)sp);
		}
		if (user_mode(em_regs)) {
			if (__get_user(stackdata, sp++)) {
				em_dump_write(" (Bad stack address)\n");
				break;
			}
		} else {
			if (!kern_addr_valid((unsigned long)sp)) {
				em_dump_write(" (Bad stack address)\n");
				break;
			}
			stackdata = *sp++;
		}
		em_dump_write(" 0x%016lx", (unsigned long)stackdata);
		buf[(i % 4) * 8]     = em_convert_char(stackdata);
		buf[(i % 4) * 8 + 1] = em_convert_char(stackdata >> 8);
		buf[(i % 4) * 8 + 2] = em_convert_char(stackdata >> 16);
		buf[(i % 4) * 8 + 3] = em_convert_char(stackdata >> 24);
		buf[(i % 4) * 8 + 4] = em_convert_char(stackdata >> 32);
		buf[(i % 4) * 8 + 5] = em_convert_char(stackdata >> 40);
		buf[(i % 4) * 8 + 6] = em_convert_char(stackdata >> 48);
		buf[(i % 4) * 8 + 7] = em_convert_char(stackdata >> 56);

		if ((i % 4) == 3) {
			em_dump_write(" : %s\n", buf);
		}

		++i;
	}
}

void em_dump_stack(int argc, char **argv)
{
	unsigned long *sp = (unsigned long *)(em_regs->sp & ~0x03);
	unsigned long *fp = (unsigned long *)(em_regs->regs[29]& ~0x03);
	unsigned long *tail;
	unsigned long backchain;
	unsigned long stackdata;
	int frame = 1;

	tail = sp + PAGE_SIZE / 4;

	em_dump_write("\n[stack dump]\n");

	backchain = arch_stack_pointer(fp);
	while (sp < tail) {
		if (backchain == (unsigned long)sp) {
			em_dump_write("|");
			fp = arch_prev_frame(fp);
			if (!fp)
				break;

			backchain = arch_stack_pointer(fp);
			if (!backchain)
				break;
		} else {
			em_dump_write(" ");
		}

		if (backchain < (unsigned long)sp) {
			break;
		}

		if (user_mode(em_regs)) {
			if (__get_user(stackdata, sp)) {
				em_dump_write("\n (bad stack address)\n");
				break;
			}
		} else {
			if (!kern_addr_valid((unsigned long)sp)) {
				em_dump_write("\n (bad stack address)\n");
				break;
			}
			stackdata = *sp;
		}

		if (((unsigned long)tail-(unsigned long)sp) % 0x10 == 0) {
			if (frame) {
				em_dump_write("\n0x%px:|", sp);
				frame = 0;
			} else {
				em_dump_write("\n0x%px: ", sp);
			}
		}

		em_dump_write("0x%016lx", stackdata);

		sp++;
	}

	em_dump_write("\n");

	em_dump_write("\n #################em_dump_till_end_of_page###########\n");
	em_dump_till_end_of_page(sp);
	em_dump_write("\n");
}

static struct fsr_info {
	const char *name;
} fsr_info[] = {
	{"ttbr address size fault"	},
	{"level 1 address size fault"	},
	{"level 2 address size fault"	},
	{"level 3 address size fault"	},
	{"level 0 translation fault"	},
	{"level 1 translation fault"	},
	{"level 2 translation fault"	},
	{"level 3 translation fault"	},
	{"unknown 8"			},
	{"level 1 access flag fault"	},
	{"level 2 access flag fault"	},
	{"level 3 access flag fault"	},
	{"unknown 12"			},
	{"level 1 permission fault"	},
	{"level 2 permission fault"	},
	{"level 3 permission fault"	},
	{"synchronous external abort"	},
	{"synchronous tag check fault"	},
	{"unknown 18"			},
	{"unknown 19"			},
	{"level 0 (translation table walk)" },
	{"level 1 (translation table walk)" },
	{"level 2 (translation table walk)" },
	{"level 3 (translation table walk)" },
	{"synchronous parity or ECC error" },
	{"unknown 25"			},
	{"unknown 26"			},
	{"unknown 27"			},
	{"level 0 synchronous parity error (translation table walk)" },
	{"level 1 synchronous parity error (translation table walk)" },
	{"level 2 synchronous parity error (translation table walk)" },
	{"level 3 synchronous parity error (translation table walk)" },
	{"unknown 32"			},
	{"alignment fault"		},
	{"unknown 34"			},
	{"unknown 35"			},
	{"unknown 36"			},
	{"unknown 37"			},
	{"unknown 38"			},
	{"unknown 39"			},
	{"unknown 40"			},
	{"unknown 41"			},
	{"unknown 42"			},
	{"unknown 43"			},
	{"unknown 44"			},
	{"unknown 45"			},
	{"unknown 46"			},
	{"unknown 47"			},
	{"TLB conflict abort"		},
	{"Unsupported atomic hardware update fault" },
	{"unknown 50"			},
	{"unknown 51"			},
	{"implementation fault (lockdown abort)" },
	{"implementation fault (unsupported exclusive)"	},
	{"unknown 54"			},
	{"unknown 55"			},
	{"unknown 56"			},
	{"unknown 57"			},
	{"unknown 58"			},
	{"unknown 59"			},
	{"unknown 60"			},
	{"section domain fault"		},
	{"page domain fault"		},
	{"unknown 63"			},
};

void em_show_syndrome(void)
{
	unsigned long fsr, far;
	struct fsr_info *inf;
	struct task_struct *tsk = current;

	em_dump_write("\n\n[exception syndrome]\n");
	fsr = tsk->thread.fault_code;
	far = tsk->thread.fault_address;
	inf = fsr_info + (fsr & 63);
	em_dump_write("tp_value:	0x%-20lx pc:\t\t0x%px\n",
			tsk->thread.uw.tp_value, em_regs->pc);
	em_dump_write("fault_code:	0x%-20lx fault_address:	0x%px"
			"\nfault:\t\t%s\n", fsr, far, inf->name);
}

#ifdef CONFIG_SNSC_ALT_BACKTRACE
static void em_print_symbol(const char *str)
{
#ifdef CONFIG_SNSC_EM_DEMANGLE
	char *demangle = cplus_demangle_v3(str, demangle_flag);
	if (demangle) {
		em_dump_write("%s", demangle);
		kfree(demangle);
		return;
	}
#endif
	em_dump_write("%s", str);
}

/* variables needed during disasseble of instruction  */
unsigned long func_start, exception_addr;

int em_bt_ustack_callback(struct bt_arch_callback_arg *cbarg, void *user)
{
	static bool copy_addr = 0;
	em_dump_write("[0x%016lx] ", cbarg->ba_addr);
	if (bt_status_is_error(cbarg->ba_status)) {
		em_dump_write("stop backtracing: %s\n", cbarg->ba_str);

		/* if PC is invalid/zero, unwind/disassemble may tried with the valid LR.
		   CPSR bit need to be adjusted as per the LR register
		   if return addr is thumb2, last bit of LR is always 1
		   So set the T bit in cpsr regsiter. */
		/*
		 * In AARCH64 mode
		 * The nRW bit indicates the execution state of the CPU
		 * 1 is 32bit mode (AARCH32 and T32 modes)
		 * 0 is 64bit mode (AARCH64)
		 */
		if ((copy_addr == 0) && (compat_user_mode(em_regs))) {
			/*
			 * In AARCH32 mode the behaviour is exactly similar to ARMv7
			 * CPSR 5th bit indicates the
			 * 1 is T32 mode
			 * 0 is ARM mode
			 * Set the bit accordingly in the pstate.
			 */
			if (em_regs->compat_lr & 1)
				em_regs->pstate |= PSR_AA32_T_BIT;
			else
				em_regs->pstate &= ~PSR_AA32_T_BIT;
		}
		return 0;
	}

	/* copying the func start addr and pc, needed during disassebly
	   disassembly need func start address as a starting point to avoid arm/thumb2 confusion */
	if (copy_addr == 0) {
		unsigned int insn_size = 4;
		exception_addr = (cbarg->ba_addr/insn_size) * insn_size;
		func_start = (cbarg->ba_sym_start/insn_size) * insn_size;
		copy_addr = 1;
	}

	if (cbarg->ba_str[0]) {
		em_print_symbol(cbarg->ba_str);
	}
	/* condition for missing addsymlist information */
	else if (cbarg->ba_sym_start == 0) {
		em_dump_write("stripped (%s +%#lx) (%s hash:00000000)\n",
			      bt_file_name(cbarg->ba_file),
			      cbarg->ba_addr - cbarg->ba_adjust,
			      bt_file_name(cbarg->ba_file));
		return 0;
	}
	else {
		em_dump_write("0x%016lx", cbarg->ba_sym_start);
	}
	if (bt_hash_valid(cbarg)) {
		/* by symlist section */
		const unsigned char *hash = (unsigned char *)cbarg->ba_hash;
		em_dump_write("+%#lx (%s hash:%02x%02x%02x%02x adj:%ld)\n",
			      cbarg->ba_addr - cbarg->ba_sym_start,
			      bt_file_name(cbarg->ba_file),
			      hash[0], hash[1], hash[2], hash[3],
			      cbarg->ba_adjust);
	}
	else {
		/* by symtab section */
		em_dump_write("+%#lx/%#lx (%s +%#lx)\n",
			      cbarg->ba_addr - cbarg->ba_sym_start,
			      cbarg->ba_sym_size,
			      bt_file_name(cbarg->ba_file),
                  cbarg->ba_addr - cbarg->ba_adjust);
	}
	return 0;
}

static void em_dump_callstack_ustack(const char *mode)
{
	/* cant pass em_regs here, it contains kernel register values during kernel exception */
	struct pt_regs *usr_regs = task_pt_regs(current);
	em_initdump_usr_regs(usr_regs);
	em_dump_write("\n[call stack (ustack)]\n");
	bt_ustack(mode, !not_interrupt, usr_regs, em_bt_ustack_callback, NULL);
	em_dump_write("\n");
}

int em_bt_kstack_callback(struct bt_arch_callback_arg *cbarg, void *user)
{
	em_dump_write("[0x%08lx] ", cbarg->ba_addr);
	if (!cbarg->ba_str) {
		em_dump_write("0x%08lx\n", cbarg->ba_addr);
		return 0;
	}
	em_dump_write("%s+%#lx/%#lx", cbarg->ba_str,
		      cbarg->ba_addr - cbarg->ba_sym_start,
		      cbarg->ba_sym_size);
	if (cbarg->ba_modname)
		em_dump_write(" [%s]\n", cbarg->ba_modname);
	else
		em_dump_write("\n");
	return 0;
}

static void em_dump_callstack_kstack(const char *mode)
{
	em_dump_write("\n[call stack (kstack)]\n");
	bt_kstack_current(mode, em_bt_kstack_callback, NULL);
	em_initdump_bt(NULL);
	em_dump_write("\n");
}

static void em_dump_callstack_kstack_regs(const char *mode)
{
	em_dump_write("\n[call stack (kstack_regs)]\n");
	bt_kstack_regs(current, em_regs, em_bt_kstack_callback, NULL, 1);
	em_initdump_bt(em_regs);
	em_dump_write("\n");
}
#endif

static int em_is_param_char(char c)
{
	return isalnum(c) || c == '_';
}

static const char *em_param_match(const char *param, const char *name)
{
	const char *from = strstr(param, name);
	const char *to;

	if (from == NULL)
		return NULL;
	if (from > param && em_is_param_char(from[-1])) /* suffix match */
		return NULL;
	to = from + strlen(name);
	if (em_is_param_char(*to))
		return NULL; /* prefix match */
	return to;
}

static int em_callstack_mode(const char *mode)
{
	int count = 0;
	em_dump_write("\n[em_callstack_mode]\n mode : %s\n\n", mode);
#ifdef CONFIG_SNSC_ALT_BACKTRACE
	if (em_param_match(mode, "kstack")) {
		em_dump_callstack_kstack(mode);
		count++;
	}
	if (!not_interrupt && em_param_match(mode, "kstack_regs")) {
		em_dump_callstack_kstack_regs(mode);
		count++;
	}
	if (current->mm && em_param_match(mode, "ustack")) {
		em_dump_callstack_ustack(mode);
		count++;
	}
#endif
	return count;
}

void em_dump_callstack(int argc, char **argv)
{
	int count;

	if (!argv || argc <= 1)
		count = em_callstack_mode(em_callstack_param);
	else
		count = em_callstack_mode(argv[1]);
	if (count == 0)
		em_dump_write("\n[call stack]\nno callstack selected\n\n");
}

static bool report_traces(void *d, unsigned long pc)
{
	struct em_task_state_data *std = d;

	if (std->state.depth) {
		em_dump_write("  pc: %px (%pS), fp %px\n",
				std->frame.pc, std->frame.pc,
				std->frame.fp);
		std->state.depth--;
		return true;
	}
	em_dump_write("  ...\n");

	return std->state.depth != 0;
}

static void em_sched_show_task(struct pt_regs *regs, struct task_struct *p)
{
	unsigned task_state;
	static const char stat_nam[] = TASK_STATE_TO_CHAR_STR;
	struct em_task_state_data std;
	unsigned int state;


	pr_debug("%s(regs = %p tsk = %p)\n", __func__, regs, p);

	/* show task state */
	state = READ_ONCE(p->__state);
	task_state = state ? __ffs(state) + 1 : 0;
	em_dump_write("%-13.13s %c", p->comm,
		task_state < sizeof(stat_nam) - 1 ? stat_nam[task_state] : '?');
	if (task_state == TASK_RUNNING)
		em_dump_write(" running  ");
	else
		em_dump_write(" %08lx ", thread_saved_pc(p));
	em_dump_write("tgid %5d ", task_tgid_nr(p));
	em_dump_write("pid %5d ", task_pid_nr(p));
	em_dump_write("parent %6d ", task_pid_nr(p->real_parent));
	em_dump_write("flags 0x%08lx ",
				(unsigned long)task_thread_info(p)->flags);
	em_dump_write("cpu %d\n",task_cpu(p));

	/* show backtrace */
	if (regs) {
		/* PC might be corrupted, use LR in that case. */
		start_backtrace(&std.frame,
				regs->regs[29],
				(kernel_text_address(regs->pc)
				? regs->pc : regs->regs[30]));
	} else if (p == current) {
		start_backtrace(&std.frame,
				(unsigned long)__builtin_frame_address(0),
				(unsigned long)em_sched_show_task);
	} else {
		/* task blocked in __switch_to */
		start_backtrace(&std.frame,
				thread_saved_fp(p),
				thread_saved_pc(p));
	}

	std.state.depth = 100;
	walk_stackframe(p, &std.frame, report_traces, &std);
}

void em_task_log(int argc, char **argv)
{
	struct task_struct *g, *p;

	/* show each task's state */
	do_each_thread(g, p) {
		em_sched_show_task(NULL, p);
	} while_each_thread(g, p);
}
