/*
 * Copyright 2018 Sony Imaging Products & Solutions Inc.
 * Copyright 2018 Sony Corporation.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

/*
 * Divider clock implementation
 */

#include <linux/clk-provider.h>
#include <linux/regmap.h>
#include <linux/slab.h>

#include "cxd-clk-define.h"
#include "cxd-clk-div.h"

#define to_cxd_clk_div(_hw) container_of(_hw, struct cxd_clk_div, hw)

static unsigned long cxd_clk_div_recalc_rate(struct clk_hw *hw, unsigned long parent_rate)
{
        struct cxd_clk_div *div = to_cxd_clk_div(hw);
        u32 reg_val;

        regmap_read(div->regmap, div->reg, &reg_val);
        reg_val = reg_val >> div->shift;
        reg_val &= div->mask;

        return divider_recalc_rate(hw, parent_rate, reg_val, NULL, div->flags);
}

static long cxd_clk_div_round_rate(struct clk_hw *hw, unsigned long rate,
                unsigned long *prate)
{
        struct cxd_clk_div *div = to_cxd_clk_div(hw);

        return divider_round_rate(hw, rate, prate, NULL,
                        div->width, div->flags);
}

static int cxd_clk_div_set_rate(struct clk_hw *hw, unsigned long rate,
                unsigned long parent_rate)
{
        struct cxd_clk_div *div = to_cxd_clk_div(hw);
        unsigned long flags;
        u32 val;

        val = divider_get_val(rate, parent_rate, NULL, div->width, div->flags);

        if (div->lock)
	{
                spin_lock_irqsave(div->lock, flags);

        regmap_update_bits(div->regmap, div->reg, (div->mask << div->shift), (val << div->shift));

                spin_unlock_irqrestore(div->lock, flags);
	} else {
        	regmap_update_bits(div->regmap, div->reg, (div->mask << div->shift), (val << div->shift));
	}

        return 0;
}

static struct clk_ops cxd_clk_div_ops = {
        .recalc_rate = cxd_clk_div_recalc_rate,
        .round_rate = cxd_clk_div_round_rate,
        .set_rate = cxd_clk_div_set_rate,
};

struct clk *cxd_clk_register_div(struct device *dev, const char *name,
                const char *parent_name, unsigned long flags, struct regmap *regmap,
                u32 reg_ofs, u8 bit_ofs, u8 bit_width, u8 clk_gate_flags, spinlock_t *lock)
{
        struct cxd_clk_div *div;
        struct clk *clk;
        struct clk_init_data init;

        div = kzalloc(sizeof(struct cxd_clk_div), GFP_KERNEL);
        if (!div) {
                err_print("memory allocate fail\n");
                return ERR_PTR(-ENOMEM);
        }

        /* struct cxd_clk_gate assignments */
        div->regmap = regmap;
        div->reg = reg_ofs;
        div->mask = BIT(bit_width) - 1;
        div->shift = bit_ofs;
        div->width = bit_width;
        div->flags = flags;
        div->lock = lock;

        init.name = name;
        init.ops = &cxd_clk_div_ops;
        init.flags = flags;
        init.parent_names = parent_name ? &parent_name : NULL;
        init.num_parents =  parent_name ? 1 : 0;

        div->hw.init = &init;

        clk = clk_register(dev, &div->hw);
        if (IS_ERR(clk)) {
                err_print("register div clock error\n");
                kfree(div);
        }

        return clk;
}

