/*
 * Copyright 2008 Sony Corporation
 * Copyright 2018 Sony Imaging Products and Solutions Incorporated.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef CONFIG_OSAL_UDIF
#error "CONFIG_OSAL_UDIF" not set.
#endif

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/moduleparam.h>
#include <linux/wait.h>
#include <linux/errno.h>
#include <linux/usb/gcore/usb_event.h>
#include <linux/usb/gcore/usb_otgcore.h>

#include <linux/udif/cdev.h>
#include <mach/udif/devno.h>
#include <linux/file.h>
#include <linux/udif/module.h>
#include <linux/udif/mutex.h>

#include <linux/usb/specific_gadget/usb_ioctl.h>
#include <linux/usb/specific_gadget/usb_ioctl_current.h>

#include "usb_current_conf.h"

/* for Debug */
//#define USB_CURRENT_DEBUG

#ifdef USB_CURRENT_DEBUG
#define PDEBUG(fmt,args...)     printk(KERN_INFO MY_NAME ": " fmt, ## args)
#else
#define PDEBUG(fmt,args...)     do { } while (0)
#endif

#define PERR(fmt,args...)       printk(KERN_ERR MY_NAME ": " fmt, ## args)

//#define FORCED_TEST_MODE

enum {
    STATE_OFF = 0,
    STATE_LIMIT,
    STATE_UNLIMIT,
    STATE_NBROF,
};

enum {
    INPUT_INIT = 0,
    INPUT_NOTIFY_START_HOST,
    INPUT_NOTIFY_STOP_HOST,
    INPUT_NOTIFY_B_DISCONNECT,
    INPUT_NOTIFY_CID_A_TO_B,
    INPUT_IOCTRL_LIMIT,
    INPUT_IOCTRL_UNLIMIT,
    INPUT_NBROF,
};

#ifdef USB_CURRENT_DEBUG
static char* state_name[STATE_NBROF] = {
    "STATE_OFF",
    "STATE_LIMIT",
    "STATE_UNLIMIT",
};

static char* input_name[INPUT_NBROF] = {
    "INPUT_INIT",
    "INPUT_NOTIFY_START_HOST",
    "INPUT_NOTIFY_STOP_HOST",
    "INPUT_NOTIFY_B_DISCONNECT",
    "INPUT_NOTIFY_CID_A_TO_B",
    "INPUT_IOCTRL_LIMIT",
    "INPUT_IOCTRL_UNLIMIT",
};
#endif

static int  usb_current_bind(struct usb_otg_driver*, struct usb_otgcore*);
static int  usb_current_unbind(struct usb_otg_driver*, struct usb_otgcore*);
static void usb_current_notify(struct usb_otg_driver*, int);
static int  usb_current_query_oc(struct usb_otg_driver*);

static int  main_ctrl(int input);
static int  get_state(void);
static void state_init(void);
static int  state_ctrl(int input);
static int  set_to_limit(void);
static int  set_to_unlimit(void);

static UDIF_ERR usb_current_fops_open ( UDIF_FILE *file );
static UDIF_ERR usb_current_fops_close( UDIF_FILE *file );
static UDIF_ERR usb_current_fops_ioctl( UDIF_FILE *file, UDIF_IOCTL *ictl );
/*-------------------------------------------------------------------------*/
static struct usb_otg_driver current_driver = {
    .function	= MY_DESC,
    .bind		= usb_current_bind,
    .unbind		= usb_current_unbind,
    .notify	= usb_current_notify,
    .notify_with_param = NULL,
    .query_over_current	= usb_current_query_oc,
};

static struct UDIF_CDEV_OPS usb_current_fops = {
    .open       = usb_current_fops_open,
    .close      = usb_current_fops_close,
    .ioctl      = usb_current_fops_ioctl,
    .read       = NULL,
    .write      = NULL,
    .lseek      = NULL,
    .mmap       = NULL,
};

static UDIF_CDEV cdev_usb_current;

/* Driver callbacks prototype */
static UDIF_ERR usb_current_init(UDIF_VP data);
static UDIF_ERR usb_current_exit(UDIF_VP data);

static UDIF_DRIVER_OPS current_udif_ops = {
    .init       = usb_current_init,
    .exit       = usb_current_exit,
    .probe      = NULL,
    .remove     = NULL,
    .suspend    = NULL,
    .resume     = NULL,
    .shutdown   = NULL
};
/* memo 
   UDIF_IDS  -->  foCXiWX^j𒼐ڐ䂷ꍇ
                  pӂꂽIDŐ錾B

   UDIF_DEPS() --> unifiedƂ֌WۂB
   UDIF_VP     --> Callback̈ɓnprivate data
*/

UDIF_MODULE(usb_current         ,   /* .name    : */
            "usbcur"            ,   /* .string  : NULL8ȓ */
            "3.0"               ,   /* .version : NULL4ȓ */
            current_udif_ops   ,   /* .opsNULLs */
            NULL                ,   /* .ids UDIF_IDS*^ ۃhCoNULL */
            NULL                ,   /* .deps    : unifiedȂꍇ͕sv */
            NULL                    /* .data : Private data */
            );


/*-------------------------------------------------------------------------*/
static int g_state;
static UDIF_MUTEX   g_lock;

/*-------------------------------------------------------------------------*/
static int  usb_current_bind(struct usb_otg_driver* me, struct usb_otgcore* otgcore)
{
	PDEBUG("Call: %s\n", __func__);
	
	return 0;
}

static int  usb_current_unbind(struct usb_otg_driver* me, struct usb_otgcore* otgcore)
{
	PDEBUG("Call: %s\n", __func__);
	
	return 0;
}

static void usb_current_notify(struct usb_otg_driver* me, int event)
{
	PDEBUG("Call: %s\n", __func__);
	
	switch(event){
	  case USB_OTGCORE_START_HOST:
	    PDEBUG("%s event received: USB_OTGCORE_START_HOST\n", __func__);
	    main_ctrl(INPUT_NOTIFY_START_HOST);
	    break;
	    
	  case USB_OTGCORE_STOP_HOST:
	    PDEBUG("%s event received: USB_OTGCORE_STOP_HOST\n", __func__);
	    main_ctrl(INPUT_NOTIFY_STOP_HOST);
	    break;
	    
	  case USB_OTGCORE_B_DISCONNECT:
	    PDEBUG("%s event received: USB_OTGCORE_B_DISCONNECT\n", __func__);
	    main_ctrl(INPUT_NOTIFY_B_DISCONNECT);
	    break;
	    
	  case USB_OTGCORE_CID_A_TO_B:
	    PDEBUG("%s event received: USB_OTGCORE_CID_A_TO_B\n", __func__);
	    main_ctrl(INPUT_NOTIFY_CID_A_TO_B);
	    break;
	    
	  default:
	  	PDEBUG("%s event(%d) is Unknown!!\n", __func__, event);
	}
}

static int  usb_current_query_oc(struct usb_otg_driver* me)
{
	int result;
	
	PDEBUG("Call: %s\n", __func__);
	
	if(get_state() != STATE_UNLIMIT){
		/* UNLIMITԂł͂Ȃ̂OTG CoreOverCurrentʒm͉\ */
		result = 0;
	}else{
		/* UNLIMITԂȂ̂OTG CoreOverCurrentʒm͔s֎~ */
		result = 1;
	}
	
	return result;
}

/*
  ԃRg[[
  
  []
   Eint input: 
  [return]
  Eint : 
      0:        I
      -EINVAL:  ͂`
      -EFAULT:  Xe[gƓ͂s
  []
  ECurrent Limits
  */
static int main_ctrl(int input)
{
	int old_state, now_state, err;

	
	// Mutex lock
    udif_mutex_lock(&g_lock);
	
	// input͑Ostate擾
	old_state = get_state();
	
	// 󂯎inputXe[g}Vɓ͂stateJڂ
	err = state_ctrl(input);
	if(err != 0){
		PERR("%s input error\n", __func__);
		goto SUB_RET;
	}
	
	// Jڂstate擾
	now_state = get_state();
	
	// JڑOstateɉGPIO𐧌
	if(old_state == STATE_LIMIT && now_state == STATE_UNLIMIT){
		PDEBUG("LIMIT -> UNLIMIT\n");
		set_to_unlimit();
	}else if(old_state == STATE_UNLIMIT && now_state == STATE_LIMIT){
		PDEBUG("UNLIMIT -> LIMIT\n");
		set_to_limit();
	}else if(old_state == STATE_UNLIMIT && now_state == STATE_OFF){
		PDEBUG("UNLIMIT -> LIMIT\n");
		set_to_limit();
	}

SUB_RET:
	
	// mutex unlock
	udif_mutex_unlock(&g_lock);
	
	return err;
}

/*
  Ԏ擾
  
  []
   Ȃ
  [return]
  Eint
      ݂̏
  []
  EXe[g}V̏Ԃ擾
  */
static int get_state(void)
{
	return g_state;
}

/*
  Xe[g}V
  
  []
   Ȃ
  [return]
   Ȃ
  []
  EXe[g}V̏Ԃ
  */
static void state_init(void)
{
	g_state = STATE_OFF;
}

/*
  Xe[g}V
  
  []
  Eint input:  
  [return]
  Eint
      -EINVAL:  ͂`
      -EFAULT:   Xe[gƓ͂s
  []
  E͂ɉēԂω
  */
static int state_ctrl(int input)
{
	int err = 0;
	
	if(input < INPUT_NBROF){
		PDEBUG("%s input : %s\n", __func__, input_name[input]);
	}
	if(g_state < STATE_NBROF){
		PDEBUG("%s old state : %s\n", __func__, state_name[g_state]);
	}
	
	switch(g_state){
	  case STATE_OFF:
	    switch(input){
		  case INPUT_INIT:					/* ω */
		    break;
		  case INPUT_NOTIFY_START_HOST:		/* LIMITԂ */
		    g_state = STATE_LIMIT;
		    break;
		  case INPUT_NOTIFY_STOP_HOST:		/* ω */
		    break;
		  case INPUT_NOTIFY_B_DISCONNECT:	/* ω */
		    break;
		  case INPUT_NOTIFY_CID_A_TO_B:		/* ω */
		    break;
		  case INPUT_IOCTRL_LIMIT:			/* ω */
		    break;
		  case INPUT_IOCTRL_UNLIMIT:		/* ω ُ */
		    err = -EFAULT;
		    break;
		  default:
	        PERR("%s state is Error\n", __func__);
	        break;
		}
	    break;
	    
	  case STATE_LIMIT:
	    switch(input){
		  case INPUT_INIT:					/* OFFԂ */
		    g_state = STATE_OFF;
		    break;
		  case INPUT_NOTIFY_START_HOST:		/* ω */
		    break;
		  case INPUT_NOTIFY_STOP_HOST:		/* OFFԂ */
		    g_state = STATE_OFF;
		    break;
		  case INPUT_NOTIFY_B_DISCONNECT:	/* ω */
		    break;
		  case INPUT_NOTIFY_CID_A_TO_B:		/* ω */
		    break;
		  case INPUT_IOCTRL_LIMIT:			/* ω */
		    break;
		  case INPUT_IOCTRL_UNLIMIT:		/* UNLIMITԂ */
		    g_state = STATE_UNLIMIT;
		    break;
		  default:
  	        PERR("%s state is Error\n", __func__);
	        break;
		}
	    break;
	    
	  case STATE_UNLIMIT:
	    switch(input){
		  case INPUT_INIT:					/* OFFԂ */
		    g_state = STATE_OFF;
		    break;
		  case INPUT_NOTIFY_START_HOST:		/* ω */
		    break;
		  case INPUT_NOTIFY_STOP_HOST:		/* OFFԂ */
		    g_state = STATE_OFF;
		    break;
		  case INPUT_NOTIFY_B_DISCONNECT:	/* LIMITԂ */
		    g_state = STATE_LIMIT;
		    break;
		  case INPUT_NOTIFY_CID_A_TO_B:		/* LIMITԂ */
		    g_state = STATE_LIMIT;
		    break;
		  case INPUT_IOCTRL_LIMIT:			/* LIMITԂ */
		    g_state = STATE_LIMIT;
		    break;
		  case INPUT_IOCTRL_UNLIMIT:		/* ω */
		    break;
		  default:
  	        PERR("%s state is Error\n", __func__);
	        break;
		}
	    break;
	    
	  default:
	    PERR("%s state is Error\n", __func__);
	    err = -EINVAL;
	    break;
	}
	
	if(g_state < STATE_NBROF){
		PDEBUG("%s new state : %s\n", __func__, state_name[g_state]);
	}
	
	return err;
}

static int set_to_limit(void)
{
	int err;
    err = 0;    
	return err;
}

static int set_to_unlimit(void)
{
	int err;

    err = 0;
	return err;
}

/*-------------------------------------------------------------------------*/
/* static int usb_current_fops_open(struct inode *inode, struct file *file) */
UDIF_ERR usb_current_fops_open ( UDIF_FILE *file )
{
	int err = 0;
	
	PDEBUG("Call: %s\n", __func__);
	
	return err;
}

/* static int usb_current_fops_release(struct inode *inode, struct file *file) */
UDIF_ERR usb_current_fops_close( UDIF_FILE *file )
{
	int err = 0;
	
	PDEBUG("Call: %s\n", __func__);
	
	return err;
}

/* static int usb_current_fops_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg) */
UDIF_ERR usb_current_fops_ioctl( UDIF_FILE *file, UDIF_IOCTL *ictl )
{
	int err = 0;
    UDIF_UINT cmd = ictl->cmd;
	
	PDEBUG("Call: %s\n", __func__);
	
	switch(cmd){
	case USB_IOC_CUR_LIMIT :
		PDEBUG("USB_IOC_CUR_LIMIT\n");
#ifndef FORCED_TEST_MODE
        main_ctrl(INPUT_IOCTRL_LIMIT);
#else
        set_to_limit();
#endif
		break;
	case USB_IOC_CUR_UNLIMIT :
		PDEBUG("USB_IOC_CUR_UNLIMIT\n");
#ifndef FORCED_TEST_MODE
        main_ctrl(INPUT_IOCTRL_UNLIMIT);
#else
        set_to_unlimit();
#endif
		break;
		
	default:
		PERR("cmd not found\n");
		err = -ENOTTY;
		break;
	}
	
	return err;
}

/*-------------------------------------------------------------------------*/
/* static int __init start_module(void) */
UDIF_ERR UDIF_INIT usb_current_init(UDIF_VP data)
{
	int err_otg = 0;
    UDIF_ERR err = 0;

    udif_cdev_init(
        &cdev_usb_current, /* 錾\̕ϐ */
        udif_device_node(UDIF_NODE_USB_CUR),
        &usb_current_fops,
        NULL
    );    
	
	PDEBUG("Call: %s\n", __func__);

    // Mutex 
    udif_mutex_init (&g_lock);
	
	// Xe[g}V
	state_init();
	
	// regist to OTG Core
	err_otg = usb_otgcore_register_driver(&current_driver);
	if(err_otg){
		err = UDIF_ERR_NOMEM;
		goto SUB_RET;
	}
    // Character Device register
    err = udif_cdev_register(&cdev_usb_current);
    if( 0 != err ){
        goto SUB_RET;
    }

    err = 0;
SUB_RET:
	return err;
}
/* static void __exit stop_module(void) */
UDIF_ERR UDIF_EXIT usb_current_exit(UDIF_VP data)
{
	int err_otg = 0;
    UDIF_ERR err = 0;
	
	PDEBUG("Call: %s\n", __func__);
    
    err = 0;
    
    // Character Device unregist
    err = udif_cdev_unregister(&cdev_usb_current);
    if( 0 != err ){
        PERR("ERROR : udif_cdev_unregister = %d", err); 
    }
    
	// unregist from OTG Core
	err_otg = usb_otgcore_unregister_driver(&current_driver);
	if(err_otg){
		PERR("usb_otgcore_unregister_driver error = %d", err_otg);
        err = err_otg;
	}
    return err;
}
