/**
 *	@file	usb_extcmd_main.c
 *	@brief	USB EXTCMD(USB Extension Command) - main
 *	
 *		Copyright 2008,2011 Sony Corporation
 * Copyright 2018, 2019 Sony Imaging Products and Solutions Incorporated.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/moduleparam.h>
#include <linux/init.h>
#include <linux/version.h>

#ifdef CONFIG_OSAL_UDIF
#include <linux/udif/cdev.h>
#include <mach/udif/devno.h>
#endif

#define FNAME "usb_extcmd_main.c"
#include "usb_extcmd_pvt.h"


/*
 *	@brief	Licence
 */
MODULE_AUTHOR("Sony Corporation");	///< author
MODULE_DESCRIPTION(MY_DESC);		///< description
MODULE_VERSION(MY_VER);				///< version
MODULE_LICENSE("GPL");		///< license


/**
 *	@brief	File operations (/dev/usb/extcmd)
 */
static struct UDIF_CDEV_OPS extcmd_fops = {
    .open  = extcmd_open,
    .close = extcmd_release,
    .ioctl = extcmd_ioctl,
};


#ifdef CONFIG_OSAL_UDIF
/**
 *	@brief	driver information for cdev
 */
struct UDIF_CDEV extcmd_dev;
#else
/**
 *	@brief	driver information for sonymisc
 */
static struct sonymisc_device extcmd_dev = {
    .class	= DEV_CLASS_CHR,
    .minor	= MY_MINOR,
    .name	= MY_NAME,
    .fops	= &extcmd_fops,
    .gd		= NULL,
};
#endif



/*
 *	@brief	Driver load
 */
static int __init start_module(void)
{
#ifdef CONFIG_OSAL_UDIF
    UDIF_DEVNODE *node;
    UDIF_ERR res;
#endif
	int err;

	// driver register
#ifdef CONFIG_OSAL_UDIF
	node = udif_device_node(UDIF_NODE_USB_EXTCMD);
	udif_cdev_init(&extcmd_dev, node, &extcmd_fops, NULL);
	res = udif_cdev_register(&extcmd_dev);
	err = (int)res;
	if (UDIF_ERR_OK != res) {
		printk(KERN_ERR "%s extcmd udif_cdev_register err=%d\n", __FUNCTION__, res);
		return err;
	}
#else
	err = sonymisc_register(&extcmd_dev);
	if(err){
		return err;
	}
#endif
	EXTCMD_API();

	// create extcmd instance
	err = extcmd_create_this();
	if(err){
#ifdef CONFIG_OSAL_UDIF
		udif_cdev_unregister(&extcmd_dev);
#else
	    sonymisc_deregister(&extcmd_dev);
#endif
		return err;
	}
	// create command list
	err = list_init();
	if(err){
#ifdef CONFIG_OSAL_UDIF
		udif_cdev_unregister(&extcmd_dev);
#else
	    sonymisc_deregister(&extcmd_dev);
#endif
		extcmd_delete_this();
		return err;
	}
	EXTCMD_LOCK_INI();

	return 0;
}

/*
 *	@brief	Driver unload
 */
static void __exit stop_module(void)
{
	EXTCMD_API();

	// delete command list
	list_finalize();
	// delete extcmd instance
	extcmd_delete_this();
	// check memory leak
	extcmd_mem_check();

	// driver deregister
#ifdef CONFIG_OSAL_UDIF
    // CharacterDevice̓o^폜
	udif_cdev_unregister(&extcmd_dev);
#else
    sonymisc_deregister(&extcmd_dev);
#endif
}


// module_init/module_exit
module_init(start_module);
module_exit(stop_module);

