/*
 *  i2c-guc.c - The GUC I2C interface.
 */
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/errno.h>
#include <linux/i2c.h>
#include <linux/interrupt.h>
#include <linux/mutex.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <linux/io.h>
#include "i2c-guc.h"
#include <linux/clk.h>
#include <linux/err.h>
#include <linux/sched.h>
#include <linux/pm_runtime.h>

#if 1
#define dprintk        printk
#else
#define dprintk(msg...) do { } while(0)
#endif


static int bus_i2c_reset_fifo(struct gucasic_i2c *bus)
{
	writel(1, &bus->i2c_reset_fifo);

	if ((readl(&bus->i2c_tx_cnt) != 0) ||
	    (readl(&bus->i2c_rx_cnt) != 0) ||
	    (readl(&bus->i2c_tx_valid_cnt) != 0) ||
	    (readl(&bus->i2c_rx_valid_cnt) != 0))
		return -EACCES;
	else
		return 0;
}

static int bus_i2c_set_bus_speed(void *base, int clk, u32 clk_freq)
{
	struct gucasic_i2c *i2c_regs = (struct gucasic_i2c *)base;

	unsigned int t_scl_hcnt,t_scl_lcnt, stop_setup, start_hold, data_hold, restart_setup, spike, i2c_speed, k_clk;


	i2c_speed = clk_freq/1000;//400 KHz
	//printk("clk_freq = %d, guc i2c_speed = %d  KHz, clk = %d\n", clk_freq, i2c_speed, clk);
	k_clk = clk/1000;

	//use 400KHz --> 1T = 2.5us, low count = 1.6us, high count = 0.9us
	t_scl_lcnt = (64 * k_clk)/(100 * i2c_speed);//t_scl_lcnt = 80;
	t_scl_hcnt = (36 * k_clk)/(100 * i2c_speed);//t_scl_hcnt = 45;
	start_hold = (44 * k_clk)/(100 * i2c_speed);//start_hold = 55;
	stop_setup = (56 * k_clk)/(100 * i2c_speed);//stop_setup = 70;
	restart_setup = (52 * k_clk)/(100 * i2c_speed);//restart_setup = 65;
	data_hold = (20 * k_clk)/(100 * i2c_speed);;//data_hold = 1;//data_hold = 1;
	spike = 1;//spike = 1;


	//dprintk("t_scl_hcnt = %d, start_hold = %d, stop_setup=%d, , data_hold=%d\n",t_scl_hcnt, start_hold, stop_setup, data_hold);
	writel((t_scl_lcnt & 0xffff), &i2c_regs->i2c_scl_lo);
	writel((t_scl_hcnt & 0xffff), &i2c_regs->i2c_scl_hi);
	writel((restart_setup & 0xffff), &i2c_regs->i2c_restart_setup);
	writel((start_hold & 0xffff), &i2c_regs->i2c_start_hold);
	writel((stop_setup & 0xffff), &i2c_regs->i2c_stop_setup);
	writel((data_hold & 0xffff), &i2c_regs->i2c_data_hold);
	writel((spike & 0xff), &i2c_regs->i2c_spike);

	return 0;
}


void bus_gucasic_i2c_init(void *base, int clk, u32 clk_freq)
{
	struct gucasic_i2c *i2c_regs = (struct gucasic_i2c *)base;

	//Set FIFO
	writel(6, &i2c_regs->i2c_tx_full);
	writel(2, &i2c_regs->i2c_tx_empty);
	writel(1, &i2c_regs->i2c_rx_full);
	//writel(2, &i2c_regs->i2c_rx_empty);

	bus_i2c_set_bus_speed( base , clk, clk_freq);
}


static void __i2c_guc_enable(struct guc_i2c_hs *dev, bool enable)
{
	int timeout = 100;
	struct gucasic_i2c *i2c_regs = (struct gucasic_i2c *)dev->base;

	do {
		writel(enable, &i2c_regs->i2c_en);
		if ((readl(&i2c_regs->i2c_en) & 1) == enable)
			return;

		/*
		 * Wait 10 times the signaling period of the highest I2C
		 * transfer supported by the driver (for 400KHz this is
		 * 25us) as described in the DesignWare I2C databook.
		 */
		usleep_range(25, 250);
	} while (timeout--);

	dev_warn(dev->dev, "timeout in %sabling adapter\n",
		 enable ? "en" : "dis");
}

void i2c_guc_disable_int(struct guc_i2c_hs *dev)
{
	struct gucasic_i2c *i2c_regs = (struct gucasic_i2c *)dev->base;

	writel(0xffffffff, &i2c_regs->i2c_intr_mask);
}

static void i2c_guc_xfer_init(struct guc_i2c_hs *dev)
{
	struct gucasic_i2c *i2c_regs = (struct gucasic_i2c *)dev->base;

	/* Disable the adapter */
	__i2c_guc_enable(dev, false);
	writel(DMA_DISABLE, &i2c_regs->i2c_dma_en);

	/* enforce disabled interrupts (due to HW issues) */
	i2c_guc_disable_int(dev);

	/* Clear and enable interrupts */
	writel(0xffffffff, &i2c_regs->i2c_intr_clr); //Clear interrupts
	writel(~(IC_INTR_DEFAULT_MASK), &i2c_regs->i2c_intr_mask);//enable interrupts
}

static int i2c_sendbyte(struct guc_i2c_hs *dev, __u8 *data)
{
	struct gucasic_i2c *i2c_regs = (struct gucasic_i2c *)dev->base;

	writel(data[0], &i2c_regs->i2c_tx_data);
	return 0;
}

static void i2c_guc_xfer_msg(struct guc_i2c_hs *dev)
{
	struct i2c_msg *msgs = dev->msgs;
	u32 intr_mask;
	int tx_limit, rx_limit;
	u32 addr = msgs[dev->msg_write_idx].addr;
	u32 buf_len = dev->tx_buf_len, i;
	u8 *buf = dev->tx_buf;
	bool need_restart = false;
	struct gucasic_i2c *i2c_regs = (struct gucasic_i2c *)dev->base;

	intr_mask = readl(&i2c_regs->i2c_intr_mask);

	for (; dev->msg_write_idx < dev->msgs_num; dev->msg_write_idx++)
	{
		/*
		 * if target address has changed, we need to
		 * reprogram the target address in the i2c
		 * adapter when we are done with this transfer
		 */
		if (msgs[dev->msg_write_idx].addr != addr)
		{
			dev_err(dev->dev,
				"%s: invalid target address\n", __func__);
			dev->msg_err = -EINVAL;
			break;
		}

		if (msgs[dev->msg_write_idx].len == 0)
		{
			dev_err(dev->dev,
				"%s: invalid message length\n", __func__);
			dev->msg_err = -EINVAL;
			break;
		}

		if (!(dev->status & STATUS_WRITE_IN_PROGRESS))
		{
			/* new i2c_msg */
			buf = msgs[dev->msg_write_idx].buf;
			buf_len = msgs[dev->msg_write_idx].len;
			writel(buf_len, &i2c_regs->i2c_data_cnt);//guc

			/* If both IC_EMPTYFIFO_HOLD_MASTER_EN and
			 * IC_RESTART_EN are set, we must manually
			 * set restart bit between messages.
			 */
			if (dev->msg_write_idx > 0)
				need_restart = true;
		}

		tx_limit = dev->tx_fifo_depth - readl(&i2c_regs->i2c_tx_cnt);
		rx_limit = dev->rx_fifo_depth - readl(&i2c_regs->i2c_rx_cnt);

		i = 0;
		while (buf_len > 0 && tx_limit > 0 && rx_limit > 0)
		{
			u32 cmd = 0;

			/*
			 * If IC_EMPTYFIFO_HOLD_MASTER_EN is set we must
			 * manually set the stop bit. However, it cannot be
			 * detected from the registers so we set it always
			 * when writing/reading the last byte.
			 */
			if (dev->msg_write_idx == dev->msgs_num - 1 && buf_len == 1)
				cmd |= BIT(2); //STOP

			if (need_restart)
			{
				cmd |= BIT(1);//RESTART
				need_restart = false;
			}

			if (msgs[dev->msg_write_idx].flags & I2C_M_RD)
			{
				/* avoid rx buffer overrun */
				if (rx_limit - dev->rx_outstanding <= 0)
					break;

				writel(0x1, &i2c_regs->i2c_dir);//read:0x1
				writel(cmd, &i2c_regs->i2c_cmd);
				rx_limit--;
				dev->rx_outstanding++;
			}
			else
			{
				//cmd |= BIT(0);//START
				writel(0x0, &i2c_regs->i2c_dir); //write : 0
				writel(cmd, &i2c_regs->i2c_cmd);
				i2c_sendbyte(dev, &buf[i]);
			}
			tx_limit--; buf_len--;i++;
		}


		if (buf_len > 0)
		{
			dev->tx_buf = &buf[i];
			dev->tx_buf_len = buf_len;
			/* more bytes to be written */
			dev->status |= STATUS_WRITE_IN_PROGRESS;
			break;
		}
		 else
			dev->status &= ~STATUS_WRITE_IN_PROGRESS;
	}

	/*
	 * If i2c_msg index search is completed, we don't need TX_EMPTY
	 * interrupt any more.
	 */
	if (dev->msg_write_idx == dev->msgs_num)
		intr_mask |= I2C_TX_FIFO_EMPTY_INT;//intr_mask &= ~I2C_TX_FIFO_EMPTY_INT;

	if (dev->msg_err)
		intr_mask = 0xffffffff;

	writel(intr_mask, &i2c_regs->i2c_intr_mask);
}


static void i2c_guc_xfer_msg_start(struct guc_i2c_hs *dev)
{
	struct i2c_msg *msgs = dev->msgs;
	struct gucasic_i2c *i2c_regs = (struct gucasic_i2c *)dev->base;
	u32 cmd = 0, buf_len, intr_mask, i, j;
	u8 *buf;

	intr_mask = readl(&i2c_regs->i2c_intr_mask);

	buf = msgs[0].buf;
	buf_len = msgs[0].len;
	cmd = I2C_START_CMD;
	writel(buf_len, &i2c_regs->i2c_data_cnt);//guc
	writel(msgs[0].addr, &i2c_regs->i2c_tar); //slave address
	//writel(0x0, &i2c_regs->i2c_dir); //write : 0

	/*device id detection*/
	/* write slave address */
	if((msgs[0].flags & I2C_M_RD)  && (msgs[0].len == 1) )
	{
		cmd |= I2C_STOP_CMD;
		writel(0x1, &i2c_regs->i2c_dir); //read : 1
		intr_mask &= ~I2C_SLAVE_NACK_INT; //unmask I2C_SLAVE_NACK_INT
		writel(intr_mask, &i2c_regs->i2c_intr_mask); //enable I2C_SLAVE_NACK_INT/TX_FIFO_EMPTY_INT
	}
	else if(!(msgs[0].flags & I2C_M_RD))//write
	{
		//pre send to TX_FIFO
		i = i2c_tx_fifo_size;
		j = 0;
		while((buf_len > 0) && (i > 0))
		{
			i2c_sendbyte(dev, &buf[j]);
			buf_len--;
			i--; j++;
		}

		if(dev->msgs_num == 1) //only write
		{
			writel(0x0, &i2c_regs->i2c_dir); //write : 0
			cmd |= I2C_STOP_CMD;
			dev->tx_buf = &buf[j];
			if(buf_len > 0)
			{
				//printk("buf_len = %d\n", buf_len);
				dev->tx_buf_len = buf_len-j;
				intr_mask &= ~I2C_TX_FIFO_EMPTY_INT; //unmask TX_FIFO_EMPTY_INT
				writel(intr_mask, &i2c_regs->i2c_intr_mask); //enable TX_FIFO_EMPTY_INT
			}
			else
			{
				dev->tx_buf_len = 0 ;
				intr_mask |= I2C_TX_FIFO_EMPTY_INT; //unmask TX_FIFO_EMPTY_INT
				writel(intr_mask, &i2c_regs->i2c_intr_mask); //disable TX_FIFO_EMPTY_INT
			}
		}
		else if(dev->msgs_num > 1) //need repeat start
		{
			writel(0x0, &i2c_regs->i2c_dir); //write : 0
			if (msgs[1].flags & I2C_M_RD) //next msg is read
			{
				//dev->msg_read_idx = 1;
				dev->read_flag = READ_FLAG;
				dev->rx_buf_len = msgs[1].len;
			}
			else //next msg is write
			{
				dev->read_flag = WRITE_FLAG;
				dev->tx_buf = msgs[1].buf;
				dev->tx_buf_len = msgs[1].len;
			}
			intr_mask &= ~I2C_START_INT;
			writel(intr_mask, &i2c_regs->i2c_intr_mask);//open START
		}
	}


	writel(cmd, &i2c_regs->i2c_cmd);
	/* Enable the adapter */
	__i2c_guc_enable(dev, true);
}


static int i2c_guc_hs_xfer(struct i2c_adapter *adap, struct i2c_msg msgs[],
			  int num)
{
	struct guc_i2c_hs *dev = i2c_get_adapdata(adap);
	struct gucasic_i2c	*bus;
	int ret;

	ret = 0;

	bus = (struct gucasic_i2c*)(dev->base);

	mutex_lock(&dev->lock);
	pm_runtime_get_sync(dev->dev);

	reinit_completion(&dev->cmd_complete);
	dev->msgs = msgs;
	dev->msgs_num = num;
	dev->cmd_err = 0;
	dev->msg_write_idx = 0;
	dev->msg_read_idx = 0;
	dev->msg_err = 0;
	dev->status = STATUS_IDLE;
	dev->rx_outstanding = 0;
	dev->tx_buf_len = 0 ;
	dev->rx_buf_len = 0 ;

	if (dev->acquire_lock) {
		ret = dev->acquire_lock(dev);
		if (ret) {
			dev_err(dev->dev, "couldn't acquire bus ownership\n");
			goto done_nolock;
		}
	}

	//dprintk("guc_xfer : slave addr=0x%x, flag=%d, len=%d, num=%d\r\n", msgs[0].addr, msgs[0].flags, msgs[0].len,num);

	//I2C need initial before transfer/receive data
	bus_gucasic_i2c_init( bus , dev->clk, dev->clk_freq);

	//reset FIFO
	if(bus_i2c_reset_fifo(bus) < 0)
	{
		dev_err(dev->dev, "I2C reset FIFO fail !\n");
	}

	/* start the transfers */
	i2c_guc_xfer_init(dev);

	i2c_guc_xfer_msg_start(dev);

	/* wait for tx to complete */
	if (!wait_for_completion_timeout(&dev->cmd_complete, HZ)) {
		dev_err(dev->dev, "controller timed out\n");
		/* bus_gucasic_i2c_init implicitly disables the adapter */
		bus_gucasic_i2c_init( bus , dev->clk, dev->clk_freq);
		ret = -ETIMEDOUT;
		goto done;
	}

	/*
	 * We must disable the adapter before unlocking the &dev->lock mutex
	 * below. Otherwise the hardware might continue generating interrupts
	 * which in turn causes a race condition with the following transfer.
	 * Needs some more investigation if the additional interrupts are
	 * a hardware bug or this driver doesn't handle them correctly yet.
	 */
	__i2c_guc_enable(dev, false);

	if (dev->msg_err) {
		ret = dev->msg_err;
		goto done;
	}

	/* no error */
	if (likely(!dev->cmd_err)) {
		ret = num;
		goto done;
	}

	ret = -EIO;

done:
	if (dev->release_lock)
		dev->release_lock(dev);

done_nolock:
	pm_runtime_mark_last_busy(dev->dev);
	pm_runtime_put_autosuspend(dev->dev);
	mutex_unlock(&dev->lock);

	return ret;

}

static u32 i2c_guc_hs_func(struct i2c_adapter *adap)
{
	return I2C_FUNC_I2C | I2C_FUNC_SMBUS_EMUL |
	       I2C_FUNC_SMBUS_BYTE | I2C_FUNC_SMBUS_BYTE_DATA;
}
/*!
 * Stores the pointers for the i2c algorithm functions. The algorithm functions
 * is used by the i2c bus driver to talk to the i2c bus
 */
static struct i2c_algorithm Guci2c_hs_algorithm = {
	.master_xfer = i2c_guc_hs_xfer,
	.functionality = i2c_guc_hs_func
};

static u32 i2c_guc_read_clear_intrbits(struct guc_i2c_hs *dev)
{
	u32 stat;
	struct gucasic_i2c *i2c_regs = (struct gucasic_i2c *)dev->base;

	stat = readl(&i2c_regs->i2c_intr);

	if (stat & I2C_RX_EARLY_STOP_INT)
		writel(I2C_RX_EARLY_STOP_INT, &i2c_regs->i2c_intr_clr);
	if (stat & I2C_TX_EARLY_STOP_INT)
		writel(I2C_TX_EARLY_STOP_INT, &i2c_regs->i2c_intr_clr);
	if (stat & I2C_RX_FIFO_UNDERFLOW_INT)
		writel(I2C_RX_FIFO_UNDERFLOW_INT, &i2c_regs->i2c_intr_clr);

	if (stat & I2C_TX_FIFO_OVERFLOW_INT)
		writel(I2C_TX_FIFO_OVERFLOW_INT, &i2c_regs->i2c_intr_clr);
	/*if (stat & I2C_SLAVE_NACK_INT)
		writel(I2C_SLAVE_NACK_INT, &i2c_regs->i2c_intr_clr);*/
	/*if (stat & I2C_STOP_INT)
		writel(I2C_STOP_INT, &i2c_regs->i2c_intr_clr);*/
	if (stat & I2C_REPEATED_START_INT)
		writel(I2C_REPEATED_START_INT, &i2c_regs->i2c_intr_clr);
	/*if (stat & I2C_START_INT)
		writel(I2C_START_INT, &i2c_regs->i2c_intr_clr);*/

	return stat;
}

static void i2c_guc_read(struct guc_i2c_hs *dev)
{
	struct i2c_msg *msgs = dev->msgs;
	int rx_valid;
	struct gucasic_i2c *i2c_regs = (struct gucasic_i2c *)dev->base;
//printk(">>>>>>i2c_guc_read()idx = %d, num=%d\n", dev->msg_read_idx, dev->msgs_num);
	for (; dev->msg_read_idx < dev->msgs_num; dev->msg_read_idx++)
	{
		u32 len;
		u8 *buf;

		if (!(msgs[dev->msg_read_idx].flags & I2C_M_RD))
			continue;

		if (!(dev->status & STATUS_READ_IN_PROGRESS))
		{
			len = msgs[dev->msg_read_idx].len;
			buf = msgs[dev->msg_read_idx].buf;
		}
		else
		{
			len = dev->rx_buf_len;
			buf = dev->rx_buf;
		}

		rx_valid = readl(&i2c_regs->i2c_rx_cnt);

		for (; len > 0 && rx_valid > 0; len--, rx_valid--)
		{
			*buf++ =readl(&i2c_regs->i2c_rx_data);
			dev->rx_outstanding--;
		}

		if (len > 0)
		{
			dev->status |= STATUS_READ_IN_PROGRESS;
			dev->rx_buf_len = len;
			dev->rx_buf = buf;
			return;
		}
		else
			dev->status &= ~STATUS_READ_IN_PROGRESS;
	}
}


static irqreturn_t i2c_guc_isr(int this_irq, void *dev_id)
{
	struct guc_i2c_hs *dev = dev_id;
	u32 stat, enabled, stat_mask;
	struct gucasic_i2c *i2c_regs = (struct gucasic_i2c *)dev->base;


	enabled = readl(&i2c_regs->i2c_en);

	stat = readl(&i2c_regs->i2c_raw_intr);

	//printk("=====>i2c_guc_isr, stat=0x%x, enabled = %d\n", stat, enabled);

	stat = i2c_guc_read_clear_intrbits(dev);
	stat_mask = readl(&i2c_regs->i2c_intr_mask);

	if (stat & I2C_RX_FIFO_ALMOST_FULL_INT)//if (stat & I2C_RX_FIFO_FULL_INT)
	{
		i2c_guc_read(dev);
	}

	if (stat & I2C_START_INT) //for repeat start use
	{
		writel(I2C_START_INT, &i2c_regs->i2c_intr_clr); //clear START interrupt

		if(dev->read_flag == READ_FLAG) //for repeat start READ
		{
			dev->msg_read_idx = 1;
			writel(dev->rx_buf_len, &i2c_regs->i2c_data_cnt);
			writel(0x1, &i2c_regs->i2c_dir);//read:0x1
		}
		else  //for repeat start WRITE
		{
			dev->msg_write_idx = 1;
			writel(dev->tx_buf_len, &i2c_regs->i2c_data_cnt);
			writel(0x0, &i2c_regs->i2c_dir);//write:0x0
			//Open TX empty interrupt, for into i2c_guc_xfer_msg()
			stat_mask &= ~I2C_TX_FIFO_EMPTY_INT;
			writel(stat_mask, &i2c_regs->i2c_intr_mask);
		}
		writel(I2C_RE_START_CMD|I2C_STOP_CMD, &i2c_regs->i2c_cmd);
	}


	if (stat & I2C_TX_FIFO_EMPTY_INT)
	{
		i2c_guc_xfer_msg(dev);
	}

	if(stat & I2C_SLAVE_NACK_INT)
	{
		writel(I2C_SLAVE_NACK_INT, &i2c_regs->i2c_intr_clr); //clear interrupt
		dev->msg_err = -EINVAL;
		complete(&dev->cmd_complete);
	}

	if ((stat & ( I2C_STOP_INT)) || dev->msg_err)
	{
		writel(I2C_STOP_INT, &i2c_regs->i2c_intr_clr); //clear interrupt
		complete(&dev->cmd_complete);
	}

	return IRQ_HANDLED;
}


static int i2c_guc_probe(struct platform_device *pdev)
{
	int ret = 0;
	struct guc_i2c_hs *dev;
	struct i2c_adapter *adap;
	struct resource *mem;
	int id = pdev->id;
	int irq, r;

	printk("%s:%d \n", __func__, __LINE__);

	irq = platform_get_irq(pdev, 0);
	if (irq < 0)
		return irq;

	dev = devm_kzalloc(&pdev->dev,sizeof(struct guc_i2c_hs), GFP_KERNEL);
	if (!dev) {
		return -ENOMEM;
	}

	mem = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	dev->base = devm_ioremap_resource(&pdev->dev, mem);
	//dprintk("==========>guc i2c dev->base = 0x%llx\n", dev->base);
	if (IS_ERR(dev->base))
		return PTR_ERR(dev->base);

	dev->dev = &pdev->dev;
	dev->irq = irq;
	platform_set_drvdata(pdev, dev);

	/* fast mode by default because of legacy reasons */
	dev->clk_freq = 400000;
	if (pdev->dev.of_node)
	{
		//printk("guc i2c use of_node, dev->clk_freq = %d\n", dev->clk_freq);
		of_property_read_u32(pdev->dev.of_node, "clock-frequency",
				     &dev->clk_freq);

		/* Only standard mode at 100kHz and fast mode at 400kHz
		 * are supported.
		 */
		if (dev->clk_freq != 100000 && dev->clk_freq != 400000)
		{
			dev_err(&pdev->dev, "Only 100kHz and 400kHz supported");
			return -EINVAL;
		}
	}

	strlcpy((dev->adapter).name, pdev->name, 48);

	(dev->adapter).nr = id;
	(dev->adapter).algo = &Guci2c_hs_algorithm;
	(dev->adapter).timeout = 1;
	dev->tx_fifo_depth = 8;
	dev->rx_fifo_depth = 8;

	dev->ipg_clk = devm_clk_get(&pdev->dev, NULL);
	if (IS_ERR(dev->ipg_clk))
		return PTR_ERR(dev->ipg_clk);
	else
	{
		dev->clk = clk_get_rate(dev->ipg_clk);
		dprintk("===========>guc i2c, clk = %d\n", dev->clk);
		clk_prepare_enable(dev->ipg_clk);
	}

	bus_gucasic_i2c_init( dev->base , dev->clk, dev->clk_freq);

	adap = &dev->adapter;
	adap->owner = THIS_MODULE;
	adap->class = I2C_CLASS_DEPRECATED;
	adap->dev.of_node = pdev->dev.of_node;

	i2c_set_adapdata(&(dev->adapter), dev);

	ret = i2c_add_numbered_adapter(&(dev->adapter));
	snprintf(dev->adapter.name, sizeof(dev->adapter.name),
		 "GUC I2C adapter");

	//for lock initial
	init_completion(&dev->cmd_complete);
	mutex_init(&dev->lock);


	//irq
	i2c_guc_disable_int(dev);
	r = devm_request_irq(dev->dev, dev->irq, i2c_guc_isr, IRQF_SHARED,
			     dev_name(dev->dev), dev);
	if (r) {
		dev_err(dev->dev, "failure requesting irq %i: %d\n",
			dev->irq, r);
		return r;
	}

	return 0;
}

static int i2c_guc_remove(struct platform_device *pdev)
{
	struct guc_i2c_hs *i2c_hs = platform_get_drvdata(pdev);
	i2c_del_adapter(&(i2c_hs->adapter));

	platform_set_drvdata(pdev, NULL);
	iounmap(i2c_hs->base);
	kfree(i2c_hs);

	return 0;
}


#ifdef CONFIG_OF
static const struct of_device_id guc_i2c_of_match[] = {
	{ .compatible = "guc,guc-i2c", },
	{},
};
MODULE_DEVICE_TABLE(of, guc_i2c_of_match);
#endif

static int i2c_guc_suspend(struct platform_device *pdev, pm_message_t state)
{
	return 0;
}

static int i2c_guc_resume(struct platform_device *pdev)
{
	return 0;
}
static struct platform_driver i2c_guc_driver = {
	.probe		= i2c_guc_probe,
	.remove		= i2c_guc_remove,
	.suspend	= i2c_guc_suspend,
	.resume		= i2c_guc_resume,
	.driver		= {
		.name	= "guc-i2c",
		.of_match_table = of_match_ptr(guc_i2c_of_match),
		.owner	= THIS_MODULE,
	},
};

static int __init i2c_guc_init(void)
{
	return platform_driver_register(&i2c_guc_driver);
}

static void __exit i2c_guc_exit(void)
{
	platform_driver_unregister(&i2c_guc_driver);
}

subsys_initcall(i2c_guc_init);
module_exit(i2c_guc_exit);

MODULE_DESCRIPTION("I2C driver");
MODULE_AUTHOR("Chester Wang ");
MODULE_LICENSE("GPL");
MODULE_ALIAS("platform:guc-i2c");
