/*
 * Synopsys DesignWare PCIe host controller driver
 *
 * Copyright (C) 2013 Samsung Electronics Co., Ltd.
 *		http://www.samsung.com
 *
 * Author: Jingoo Han <jg1.han@samsung.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/irqdomain.h>
#include <linux/of_address.h>
#include <linux/of_pci.h>
#include <linux/pci_regs.h>
#include <linux/platform_device.h>
#include <linux/delay.h>

#include "pcie-designware.h"

static struct pci_ops dw_pcie_ops;

static int dw_pcie_rd_own_conf(struct pcie_port *pp, int where, int size,
			       u32 *val)
{
	struct dw_pcie *pci;
	int ret;

	if (pp->ops->rd_own_conf)
		return pp->ops->rd_own_conf(pp, where, size, val);

	pci = to_dw_pcie_from_pp(pp);
	ret = dw_pcie_read(pci->dbi_base + where, size, val);
	PCIE_DW_HOST_TRACE_CFG(pci->dev->of_node->name,TRACE_RC, TRACE_READ, 0,(u32)where, *val);
	return ret;
}

static int dw_pcie_wr_own_conf(struct pcie_port *pp, int where, int size,
			       u32 val)
{
	struct dw_pcie *pci;

	if (pp->ops->wr_own_conf)
		return pp->ops->wr_own_conf(pp, where, size, val);

	pci = to_dw_pcie_from_pp(pp);
	PCIE_DW_HOST_TRACE_CFG(pci->dev->of_node->name,TRACE_RC, TRACE_WRITE, 0,(u32)where, val);
	return dw_pcie_write(pci->dbi_base + where, size, val);
}

static void dwc_irq_ack(struct irq_data *d)
{
	struct msi_desc *msi = irq_data_get_msi_desc(d);
	struct pcie_port *pp = msi_desc_to_pci_sysdata(msi);
	int pos = d->hwirq % 32;
	int i = d->hwirq / 32;

	dw_pcie_wr_own_conf(pp, PCIE_MSI_INTR0_STATUS + i * 12, 4, BIT(pos));
}

static struct irq_chip dw_msi_irq_chip = {
	.name = "PCI-MSI",
	.irq_ack = dwc_irq_ack,
	.irq_enable = pci_msi_unmask_irq,
	.irq_disable = pci_msi_mask_irq,
	.irq_mask = pci_msi_mask_irq,
	.irq_unmask = pci_msi_unmask_irq,
};

/* MSI int handler */
irqreturn_t dw_handle_msi_irq(struct pcie_port *pp)
{
    u32 val;
    unsigned long ulval;
    int i, pos, irq;
    irqreturn_t ret = IRQ_NONE;
    struct dw_pcie *pci = to_dw_pcie_from_pp(pp);
    u32 stat0;

    stat0 = dw_pcie_readl_phy(pci, INTERRUPT0_STATUS_OFF);
    dw_pcie_clear_Interrupt0(pci);
    dw_pcie_clear_Interrupt1(pci);
    for (i = 0; i < MAX_MSI_CTRLS; i++) {
        dw_pcie_rd_own_conf(pp, PCIE_MSI_INTR0_STATUS + i * 12, 4,
                &val);
        if (!val)
            continue;

        ret = IRQ_HANDLED;
        pos = 0;
        ulval = (unsigned long)val;
        while ((pos = find_next_bit( &ulval, 32,pos)) != 32) {
            irq = irq_find_mapping(pp->irq_domain, i * 32 + pos);
            generic_handle_irq(irq);
            pos++;
        }
    }
    /* Legacy INTx */
    if (pp->intx_domain) {
	    unsigned long intx;
	    int bit;

	    intx = (stat0 >> INTX_SHIFT) & INTX_MASK;
	    if (intx) {
		    ret = IRQ_HANDLED;
		    for_each_set_bit(bit, &intx, INTX_BITS) {
			    irq = irq_find_mapping(pp->intx_domain, bit);
			    generic_handle_irq(irq);
		    }
	    }
    }

	return ret;
}

void dw_pcie_msi_init(struct pcie_port *pp)
{
	u64 msi_target;

	pp->msi_data = __get_free_pages(GFP_KERNEL, 0);
	msi_target = virt_to_phys((void *)pp->msi_data);

	/* program the msi_data */
	dw_pcie_wr_own_conf(pp, PCIE_MSI_ADDR_LO, 4,
			    (u32)(msi_target & 0xffffffff));
	dw_pcie_wr_own_conf(pp, PCIE_MSI_ADDR_HI, 4,
			    (u32)(msi_target >> 32 & 0xffffffff));
}

static void dw_pcie_msi_clear_irq(struct pcie_port *pp, int irq)
{
	unsigned int res, bit, val;

	res = (irq / 32) * 12;
	bit = irq % 32;
	dw_pcie_rd_own_conf(pp, PCIE_MSI_INTR0_ENABLE + res, 4, &val);
	val &= ~(1 << bit);
	dw_pcie_wr_own_conf(pp, PCIE_MSI_INTR0_ENABLE + res, 4, val);
}

static void clear_irq_range(struct pcie_port *pp, unsigned int irq_base,
			    unsigned int nvec, unsigned int pos)
{
	unsigned int i;

	for (i = 0; i < nvec; i++) {
		irq_set_msi_desc_off(irq_base, i, NULL);
		/* Disable corresponding interrupt on MSI controller */
		if (pp->ops->msi_clear_irq)
			pp->ops->msi_clear_irq(pp, pos + i);
		else
			dw_pcie_msi_clear_irq(pp, pos + i);
	}

	bitmap_release_region(pp->msi_irq_in_use, pos, order_base_2(nvec));
}

static void dw_pcie_msi_set_irq(struct pcie_port *pp, int irq)
{
	unsigned int res, bit, val;

	res = (irq / 32) * 12;
	bit = irq % 32;
	dw_pcie_rd_own_conf(pp, PCIE_MSI_INTR0_ENABLE + res, 4, &val);
	val |= 1 << bit;
	dw_pcie_wr_own_conf(pp, PCIE_MSI_INTR0_ENABLE + res, 4, val);
}

static int assign_irq(int no_irqs, struct msi_desc *desc, int *pos)
{
	int irq, pos0, i;
	struct pcie_port *pp;

	pp = (struct pcie_port *)msi_desc_to_pci_sysdata(desc);
	pos0 = bitmap_find_free_region(pp->msi_irq_in_use, MAX_MSI_IRQS,
				       order_base_2(no_irqs));
	if (pos0 < 0)
		goto no_valid_irq;

	irq = irq_find_mapping(pp->irq_domain, pos0);
	if (!irq)
		goto no_valid_irq;

	/*
	 * irq_create_mapping (called from dw_pcie_host_init) pre-allocates
	 * descs so there is no need to allocate descs here. We can therefore
	 * assume that if irq_find_mapping above returns non-zero, then the
	 * descs are also successfully allocated.
	 */

	for (i = 0; i < no_irqs; i++) {
		if (irq_set_msi_desc_off(irq, i, desc) != 0) {
			clear_irq_range(pp, irq, i, pos0);
			goto no_valid_irq;
		}
		/*Enable corresponding interrupt in MSI interrupt controller */
		if (pp->ops->msi_set_irq)
			pp->ops->msi_set_irq(pp, pos0 + i);
		else
			dw_pcie_msi_set_irq(pp, pos0 + i);
	}

	*pos = pos0;
	desc->nvec_used = no_irqs;
	desc->msi_attrib.multiple = order_base_2(no_irqs);

	return irq;

no_valid_irq:
	*pos = pos0;
	return -ENOSPC;
}

static void dw_msi_setup_msg(struct pcie_port *pp, unsigned int irq, u32 pos)
{
	struct msi_msg msg;
	u64 msi_target;

	if (pp->ops->get_msi_addr)
		msi_target = pp->ops->get_msi_addr(pp);
	else
		msi_target = virt_to_phys((void *)pp->msi_data);

	msg.address_lo = (u32)(msi_target & 0xffffffff);
	msg.address_hi = (u32)(msi_target >> 32 & 0xffffffff);

	if (pp->ops->get_msi_data)
		msg.data = pp->ops->get_msi_data(pp, pos);
	else
		msg.data = pos;

	pci_write_msi_msg(irq, &msg);
}

static int dw_msi_setup_irq(struct msi_controller *chip, struct pci_dev *pdev,
			    struct msi_desc *desc)
{
	int irq, pos;
	struct pcie_port *pp = pdev->bus->sysdata;

	if (desc->msi_attrib.is_msix)
		return -EINVAL;

	irq = assign_irq(1, desc, &pos);
	if (irq < 0)
		return irq;

	dw_msi_setup_msg(pp, irq, pos);

	return 0;
}

static int dw_msi_setup_irqs(struct msi_controller *chip, struct pci_dev *pdev,
			     int nvec, int type)
{
#ifdef CONFIG_PCI_MSI
	int irq, pos;
	struct msi_desc *desc;
	struct pcie_port *pp = pdev->bus->sysdata;

	/* MSI-X interrupts are not supported */
	if (type == PCI_CAP_ID_MSIX)
		return -EINVAL;

	WARN_ON(!list_is_singular(&pdev->dev.msi_list));
	desc = list_entry(pdev->dev.msi_list.next, struct msi_desc, list);

	irq = assign_irq(nvec, desc, &pos);
	if (irq < 0)
		return irq;

	dw_msi_setup_msg(pp, irq, pos);

	return 0;
#else
	return -EINVAL;
#endif
}

static void dw_msi_teardown_irq(struct msi_controller *chip, unsigned int irq)
{
    struct irq_data *data = irq_get_irq_data(irq);
    struct msi_desc *msi;
    struct pcie_port *pp;

    if(data != NULL)
    {
        msi = irq_data_get_msi_desc(data);
        pp = (struct pcie_port *)msi_desc_to_pci_sysdata(msi);
        clear_irq_range(pp, irq, 1, data->hwirq);
    }
}

static struct msi_controller dw_pcie_msi_chip = {
	.setup_irq = dw_msi_setup_irq,
	.setup_irqs = dw_msi_setup_irqs,
	.teardown_irq = dw_msi_teardown_irq,
};

static int dw_pcie_msi_map(struct irq_domain *domain, unsigned int irq,
			   irq_hw_number_t hwirq)
{
	irq_set_chip_and_handler(irq, &dw_msi_irq_chip, handle_edge_irq);
	irq_set_chip_data(irq, domain->host_data);

	return 0;
}

static const struct irq_domain_ops msi_domain_ops = {
	.map = dw_pcie_msi_map,
};

/*----------------- Legacy INTx -------------------*/
static void dw_enable_intx(struct irq_data *data)
{
	struct pcie_port *pp = irq_data_get_irq_chip_data(data);
	struct dw_pcie *pci = to_dw_pcie_from_pp(pp);
	dw_pcie_set_Interrupt0_en(pci, 1U << (data->hwirq + INTX_SHIFT), 1);
}

static void dw_disable_intx(struct irq_data *data)
{
	struct pcie_port *pp = irq_data_get_irq_chip_data(data);
	struct dw_pcie *pci = to_dw_pcie_from_pp(pp);
	dw_pcie_set_Interrupt0_en(pci, 1U << (data->hwirq + INTX_SHIFT), 0);
}

static void noop(struct irq_data *data){}
static struct irq_chip dw_intx_chip = {
	.name		= "PCI-INT",
	.irq_ack	= noop,
	.irq_enable	= dw_enable_intx,
	.irq_disable	= dw_disable_intx,
	.irq_mask	= noop,
	.irq_unmask	= noop,
	.flags		= IRQCHIP_SKIP_SET_WAKE,
};

static int dw_pcie_intx_map(struct irq_domain *domain, unsigned int irq,
			    irq_hw_number_t hwirq)
{
	irq_set_chip_and_handler(irq, &dw_intx_chip, handle_simple_irq);
	irq_set_chip_data(irq, domain->host_data);
	return 0;
}

static const struct irq_domain_ops intx_domain_ops = {
	.map = dw_pcie_intx_map,
};

static int dw_create_intx_domain(struct pcie_port *pp, struct device_node *np)
{
	int i;

	pp->intx_domain = irq_domain_add_linear(np, INTX_BITS,
						&intx_domain_ops, pp);
	if (!pp->intx_domain)
		return -1;

	for (i = 0; i < INTX_BITS; i++) {
		irq_create_mapping(pp->intx_domain, i);
	}
	return 0;
}

static void dw_remove_intx_domain(struct pcie_port *pp)
{
	int i;
	unsigned int irq;

	if (!pp->intx_domain)
		return;
	for (i = 0; i < INTX_BITS; i++) {
		irq = irq_find_mapping(pp->intx_domain, i);
		if (irq > 0)
			irq_dispose_mapping(irq);
	}
	irq_domain_remove(pp->intx_domain);
}
/*-------------------------------------------------*/

static void dw_pcie_free_resource_list(void *data)
{
	struct list_head *res = (struct list_head *)data;
	if (res)
		pci_free_resource_list(res);
}

int dw_pcie_host_init(struct pcie_port *pp)
{
	struct dw_pcie *pci = to_dw_pcie_from_pp(pp);
	struct device *dev = pci->dev;
	struct device_node *np = dev->of_node;
	struct device_node *intc;
	struct platform_device *pdev = to_platform_device(dev);
	struct pci_bus *bus, *child;
	struct resource *cfg_res;
	int i, ret;
	struct list_head *res;
	LIST_HEAD(ref_res);
	struct resource_entry *win, *tmp;

	res = devm_kzalloc(dev, sizeof(*res), GFP_KERNEL);
	if (!res) {
		ret = -ENOMEM;
		goto error;
	}
	INIT_LIST_HEAD(res);
	ret = devm_add_action(dev, dw_pcie_free_resource_list, res);
	if (ret)
		goto error;

	cfg_res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "config");
	if (cfg_res) {
		pp->cfg0_size = resource_size(cfg_res) / 2;
		pp->cfg1_size = resource_size(cfg_res) / 2;
		pp->cfg0_base = cfg_res->start;
		pp->cfg1_base = cfg_res->start + pp->cfg0_size;
	} else if (!pp->va_cfg0_base) {
		dev_err(dev, "missing *config* reg space\n");
	}

	ret = of_pci_get_host_bridge_resources(np, 0, 0xff, res, &pp->io_base);
	if (ret)
		return ret;

	ret = devm_request_pci_bus_resources(dev, res);
	if (ret)
		goto error;

	/* Get the I/O and memory ranges from DT */
	resource_list_for_each_entry_safe(win, tmp, res) {
		switch (resource_type(win->res)) {
		case IORESOURCE_IO:
			ret = pci_remap_iospace(win->res, pp->io_base);
			if (ret) {
				dev_warn(dev, "error %d: failed to map resource %pR\n",
					 ret, win->res);
				resource_list_destroy_entry(win);
			} else {
				pp->io = win->res;
				pp->io->name = "I/O";
				pp->io_size = resource_size(pp->io);
				pp->io_bus_addr = pp->io->start - win->offset;
			}
			break;
		case IORESOURCE_MEM:
			pp->mem = win->res;
			pp->mem->name = "MEM";
			pp->mem_size = resource_size(pp->mem);
			pp->mem_bus_addr = pp->mem->start - win->offset;
			break;
		case 0:
			pp->cfg = win->res;
			pp->cfg0_size = resource_size(pp->cfg) / 2;
			pp->cfg1_size = resource_size(pp->cfg) / 2;
			pp->cfg0_base = pp->cfg->start;
			pp->cfg1_base = pp->cfg->start + pp->cfg0_size;
			break;
		case IORESOURCE_BUS:
			pp->busn = win->res;
			break;
		}
	}

	if (!pci->dbi_base) {
		pci->dbi_base = devm_ioremap(dev,
						pp->cfg->start,
						resource_size(pp->cfg));
		if (!pci->dbi_base) {
			dev_err(dev, "error with ioremap\n");
			ret = -ENOMEM;
			goto error;
		}
	}

	pp->mem_base = pp->mem->start;

	if (!pp->va_cfg0_base) {
		pp->va_cfg0_base = devm_ioremap(dev,
					pp->cfg0_base, pp->cfg0_size);
		if (!pp->va_cfg0_base) {
			dev_err(dev, "error with ioremap in function\n");
			ret = -ENOMEM;
			goto error;
		}
	}

	if (!pp->va_cfg1_base) {
		pp->va_cfg1_base = devm_ioremap(dev,
						pp->cfg1_base,
						pp->cfg1_size);
		if (!pp->va_cfg1_base) {
			dev_err(dev, "error with ioremap\n");
			ret = -ENOMEM;
			goto error;
		}
	}

	ret = of_property_read_u32(np, "num-viewport", &pci->num_viewport);
	if (ret)
		pci->num_viewport = 2;

	if (IS_ENABLED(CONFIG_PCI_MSI)) {
		if (!pp->ops->msi_host_init) {
			pp->irq_domain = irq_domain_add_linear(dev->of_node,
						MAX_MSI_IRQS, &msi_domain_ops,
						&dw_pcie_msi_chip);
			if (!pp->irq_domain) {
				dev_err(dev, "irq domain init failed\n");
				ret = -ENXIO;
				goto error;
			}

			for (i = 0; i < MAX_MSI_IRQS; i++)
				irq_create_mapping(pp->irq_domain, i);
		} else {
			ret = pp->ops->msi_host_init(pp, &dw_pcie_msi_chip);
			if (ret < 0)
				goto error;
		}
	}

	/* INTx */
	intc = of_get_child_by_name(np, "interrupt-controller");
	if (intc) {
		if (dw_create_intx_domain(pp, intc) < 0) {
			dev_err(dev, "%s:INTx domain init failed\n", np->name);
			ret = -ENXIO;
			goto error;
		}
	}

	if (pp->ops->host_init) {
		ret = pp->ops->host_init(pp);
		if (ret)
			goto error;
	}

	resource_list_for_each_entry_safe(win, tmp, res) {
		struct resource_entry *win_ = resource_list_create_entry(win->res, 0);
		if (!win_) {
			ret = -ENOMEM;
			goto error;
		}
		win_->offset = win->offset;
		resource_list_add_tail(win_, &ref_res);
	}

	pp->root_bus_nr = pp->busn->start;
	if (IS_ENABLED(CONFIG_PCI_MSI)) {
		bus = pci_scan_root_bus_msi(dev, pp->root_bus_nr,
					    &dw_pcie_ops, pp, &ref_res,
					    &dw_pcie_msi_chip);
		dw_pcie_msi_chip.dev = dev;
	} else
		bus = pci_scan_root_bus(dev, pp->root_bus_nr, &dw_pcie_ops,
					pp, &ref_res);
	if (!bus) {
		ret = -ENOMEM;
		goto error;
	}
	else
	{
                   pp->root_bus = bus;
	}

	if (pp->ops->scan_bus)
		pp->ops->scan_bus(pp);

	pci_bus_size_bridges(bus);
	pci_bus_assign_resources(bus);

	list_for_each_entry(child, &bus->children, node)
		pcie_bus_configure_settings(child);

	pci_bus_add_devices(bus);
#if 0
	#ifdef CONFIG_PM
	dw_pcie_enable_l1_pm_substate(pci);
	#endif
#endif

	return 0;

error:
	pci_free_resource_list(&ref_res);
	if (res)
		pci_free_resource_list(res);
	return ret;
}

static int dw_pcie_rd_other_conf(struct pcie_port *pp, struct pci_bus *bus,
				 u32 devfn, int where, int size, u32 *val)
{
	int ret, type;
	u32 busdev, cfg_size;
	u64 cpu_addr;
	void __iomem *va_cfg_base;
	struct dw_pcie *pci = to_dw_pcie_from_pp(pp);

	if (!pp->prsnt) {
		*val = ~0U;
		return -1;
	}
	if (pp->ops->rd_other_conf)
		return pp->ops->rd_other_conf(pp, bus, devfn, where, size, val);

	busdev = PCIE_ATU_BUS(bus->number) | PCIE_ATU_DEV(PCI_SLOT(devfn)) |
		 PCIE_ATU_FUNC(PCI_FUNC(devfn));

	if (bus->parent->number == pp->root_bus_nr) {
		type = PCIE_ATU_TYPE_CFG0;
		cpu_addr = pp->cfg0_base;
		cfg_size = pp->cfg0_size;
		va_cfg_base = pp->va_cfg0_base;
	} else {
		type = PCIE_ATU_TYPE_CFG1;
		cpu_addr = pp->cfg1_base;
		cfg_size = pp->cfg1_size;
		va_cfg_base = pp->va_cfg1_base;
	}

	dw_pcie_prog_outbound_atu(pci, PCIE_ATU_REGION_INDEX1,
				  type, cpu_addr,
				  busdev, cfg_size);
	ret = dw_pcie_read(va_cfg_base + where, size, val);
	if (pci->num_viewport <= 2  &&  pp->io_size)
		dw_pcie_prog_outbound_atu(pci, PCIE_ATU_REGION_INDEX1,
					  PCIE_ATU_TYPE_IO, pp->io_base,
					  pp->io_bus_addr, pp->io_size);

	PCIE_DW_HOST_TRACE_CFG(pci->dev->of_node->name,TRACE_EP, TRACE_READ,busdev>>16, (u32)where, *val);
	return ret;
}

static int dw_pcie_wr_other_conf(struct pcie_port *pp, struct pci_bus *bus,
				 u32 devfn, int where, int size, u32 val)
{
    int ret, type;
    u32 busdev, cfg_size;
    u64 cpu_addr;
    void __iomem *va_cfg_base;
    struct dw_pcie *pci = to_dw_pcie_from_pp(pp);

    if (!pp->prsnt) {
	return -1;
    }
    if (pp->ops->wr_other_conf)
        return pp->ops->wr_other_conf(pp, bus, devfn, where, size, val);

    busdev = PCIE_ATU_BUS(bus->number) | PCIE_ATU_DEV(PCI_SLOT(devfn)) |
        PCIE_ATU_FUNC(PCI_FUNC(devfn));

    if (bus->parent->number == pp->root_bus_nr) {
        type = PCIE_ATU_TYPE_CFG0;
        cpu_addr = pp->cfg0_base;
        cfg_size = pp->cfg0_size;
        va_cfg_base = pp->va_cfg0_base;
    } else {
        type = PCIE_ATU_TYPE_CFG1;
        cpu_addr = pp->cfg1_base;
        cfg_size = pp->cfg1_size;
        va_cfg_base = pp->va_cfg1_base;
    }

    dw_pcie_prog_outbound_atu(pci, PCIE_ATU_REGION_INDEX1,
            type, cpu_addr,
            busdev, cfg_size);
    ret = dw_pcie_write(va_cfg_base + where, size, val);
    if (pci->num_viewport <= 2  &&  pp->io_size)
        dw_pcie_prog_outbound_atu(pci, PCIE_ATU_REGION_INDEX1,
                PCIE_ATU_TYPE_IO, pp->io_base,
                pp->io_bus_addr, pp->io_size);

    PCIE_DW_HOST_TRACE_CFG(pci->dev->of_node->name, TRACE_EP, TRACE_WRITE,busdev>>16, (u32)where, val);
    return ret;
}

static int dw_pcie_valid_device(struct pcie_port *pp, struct pci_bus *bus,
				int dev)
{
	struct dw_pcie *pci = to_dw_pcie_from_pp(pp);

	/* If there is no link, then there is no device */
	if (bus->number != pp->root_bus_nr) {
		if (!dw_pcie_link_up(pci))
			return 0;
	}

	/* access only one slot on each root port */
	if (bus->number == pp->root_bus_nr && dev > 0)
		return 0;

	return 1;
}

static int dw_pcie_rd_conf(struct pci_bus *bus, u32 devfn, int where,
			   int size, u32 *val)
{
	struct pcie_port *pp = bus->sysdata;

	if (!dw_pcie_valid_device(pp, bus, PCI_SLOT(devfn))) {
		*val = 0xffffffff;
		return PCIBIOS_DEVICE_NOT_FOUND;
	}

	if (bus->number == pp->root_bus_nr)
		return dw_pcie_rd_own_conf(pp, where, size, val);

	return dw_pcie_rd_other_conf(pp, bus, devfn, where, size, val);
}

static int dw_pcie_wr_conf(struct pci_bus *bus, u32 devfn,
			   int where, int size, u32 val)
{
	struct pcie_port *pp = bus->sysdata;

	if (!dw_pcie_valid_device(pp, bus, PCI_SLOT(devfn)))
		return PCIBIOS_DEVICE_NOT_FOUND;

	if (bus->number == pp->root_bus_nr)
		return dw_pcie_wr_own_conf(pp, where, size, val);

	return dw_pcie_wr_other_conf(pp, bus, devfn, where, size, val);
}

static struct pci_ops dw_pcie_ops = {
	.read = dw_pcie_rd_conf,
	.write = dw_pcie_wr_conf,
};

int dw_pcie_bus_write_config(struct pci_bus *bus, unsigned int devfn,
			     int pos, int size, unsigned int val)
{
	struct pcie_port *pp = bus->sysdata;

	if (!dw_pcie_valid_device(pp, bus, PCI_SLOT(devfn)))
		return PCIBIOS_DEVICE_NOT_FOUND;

	if (bus->number == pp->root_bus_nr)
		return dw_pcie_wr_own_conf(pp, pos, size, val);

	if (bus->parent->number != pp->root_bus_nr) {
		/* not support */
		return PCIBIOS_FUNC_NOT_SUPPORTED;
	}
	return dw_pcie_write_relaxed(pp->va_cfg0_base + pos, size, val);
}

static u8 dw_pcie_iatu_unroll_enabled(struct dw_pcie *pci)
{
	u32 val;

	val = dw_pcie_readl_dbi(pci, PCIE_ATU_VIEWPORT);
	if (val == 0xffffffff)
		return 1;

	return 0;
}

#define INT0MASK (AER_RC_ERR_INT|AER_RC_ERR_MSI_LAT|PME_INT|PME_MSI_LAT|EDMA_INT)

void dw_pcie_setup_rc(struct pcie_port *pp)
{
    u32 val;
    struct dw_pcie *pci = to_dw_pcie_from_pp(pp);

    /*enable sideband interrupts*/
    dw_pcie_set_Interrupt0_en(pci, INT0MASK, 1);
    dw_pcie_set_Interrupt1_en(pci,MSI_CTRL_INT|RADM_FATAL_ERR|RADM_NONFATAL_ERR|RADM_CORRECTBLE_ERR, 1);

    /*set SERR_EN*/
    dw_pcie_writeb_dbi(pci, PCI_BRIDGE_CONTROL, dw_pcie_readb_dbi(pci, PCI_BRIDGE_CONTROL)|PCI_BRIDGE_CTL_SERR);

    dw_pcie_setup(pci);

    /* setup RC BARs */
    dw_pcie_dbi_ro_wr_en(pci);
    dw_pcie_writel_dbi(pci, DW_SHADOW_OFF+PCI_BASE_ADDRESS_0, 0xffffffffU);
    dw_pcie_writel_dbi(pci, DW_SHADOW_OFF+PCI_BASE_ADDRESS_1, 0x7fffffffU);
    dw_pcie_writel_dbi(pci, PCI_BASE_ADDRESS_0, 0x00000004);
    dw_pcie_writel_dbi(pci, PCI_BASE_ADDRESS_1, 0x00000000);
    dw_pcie_dbi_ro_wr_dis(pci);

    /* setup interrupt pins */
    dw_pcie_dbi_ro_wr_en(pci);
    val = dw_pcie_readl_dbi(pci, PCI_INTERRUPT_LINE);
    val &= 0xffff00ff;
    val |= 0x00000100;
    dw_pcie_writel_dbi(pci, PCI_INTERRUPT_LINE, val);
    dw_pcie_dbi_ro_wr_dis(pci);

    /* setup bus numbers */
    val = dw_pcie_readl_dbi(pci, PCI_PRIMARY_BUS);
    val &= 0xff000000;
    val |= 0x00ff0100;
    dw_pcie_writel_dbi(pci, PCI_PRIMARY_BUS, val);

    /* setup command register */
    val = dw_pcie_readl_dbi(pci, PCI_COMMAND);
    val &= 0xffff0000;
    val |= PCI_COMMAND_IO | PCI_COMMAND_MEMORY |
        PCI_COMMAND_MASTER | PCI_COMMAND_SERR;
    dw_pcie_writel_dbi(pci, PCI_COMMAND, val);

    /*
     * If the platform provides ->rd_other_conf, it means the platform
     * uses its own address translation component rather than ATU, so
     * we should not program the ATU here.
     */
    if (!pp->ops->rd_other_conf) {
        /* get iATU unroll support */
        pci->iatu_unroll_enabled = dw_pcie_iatu_unroll_enabled(pci);
        dev_dbg(pci->dev, "iATU unroll: %s\n",
                pci->iatu_unroll_enabled ? "enabled" : "disabled");
	if (pp->handover) {
		int i;
		/* disable all iATU */
		for (i = 0; i < CX_ATU_NUM_INBOUND; i++) {
			dw_pcie_disable_atu(pci, i, DW_PCIE_REGION_INBOUND);
		}
		for (i = 0; i < CX_ATU_NUM_OUTBOUND; i++) {
			dw_pcie_disable_atu(pci, i, DW_PCIE_REGION_OUTBOUND);
		}
	}

        dw_pcie_prog_outbound_atu(pci, PCIE_ATU_REGION_INDEX0,
                PCIE_ATU_TYPE_MEM, pp->mem_base,
                pp->mem_bus_addr, pp->mem_size);
        if (pci->num_viewport > 2  &&  pp->io_size)
            dw_pcie_prog_outbound_atu(pci, PCIE_ATU_REGION_INDEX2,
                    PCIE_ATU_TYPE_IO, pp->io_base,
                    pp->io_bus_addr, pp->io_size);
    }

    dw_pcie_wr_own_conf(pp, PCI_BASE_ADDRESS_0, 4, 0);

    /* Enable write permission for the DBI read-only register */
    dw_pcie_dbi_ro_wr_en(pci);
    /* program correct class for RC */
    dw_pcie_wr_own_conf(pp, PCI_CLASS_DEVICE, 2, PCI_CLASS_BRIDGE_PCI);
    /* Better disable write permission right after the update */
    dw_pcie_dbi_ro_wr_dis(pci);
    if (!pp->handover) {
    /*clear and set 'Directed Speed Change'*/
    dw_pcie_rd_own_conf(pp, PCIE_LINK_WIDTH_SPEED_CONTROL, 4, &val);
    dw_pcie_wr_own_conf(pp, PCIE_LINK_WIDTH_SPEED_CONTROL, 4, val & (~PORT_LOGIC_SPEED_CHANGE));
    dw_pcie_wr_own_conf(pp, PCIE_LINK_WIDTH_SPEED_CONTROL, 4, val |PORT_LOGIC_SPEED_CHANGE );
    }
    if (pp->handover) {
	    /* disable global slave error response mapping */
            val = dw_pcie_readl_dbi(pci,AMBA_ERROR_RESPONSE_DEFAULT_OFF);
	    val &= ~AMBA_ERROR_RESPONSE_GLOBAL;
	    dw_pcie_writel_dbi(pci, AMBA_ERROR_RESPONSE_DEFAULT_OFF, val);
    }
#if 0
    /*enable global slave error response mapping*/
    dw_pcie_writel_dbi(pci, AMBA_ERROR_RESPONSE_DEFAULT_OFF, \
            dw_pcie_readl_dbi(pci,AMBA_ERROR_RESPONSE_DEFAULT_OFF) |AMBA_ERROR_RESPONSE_GLOBAL);
#endif
    /*initial low power*/
    dw_pcie_writel_dbi(pci, AUX_CLK_FREQ_OFF, AUX_CLK_FREQ_1MHz);

}

void dw_pcie_set_PERST(struct dw_pcie *pci, int enable)
{
	u32 val;
	val = dw_pcie_readl_phy(pci, GENERAL_SIGNAL_CTRL_OFF);
	val &= (~(0x1<<0));
	//low active
	if (!enable)
		val |= (1 << 0);
	dw_pcie_writel_phy(pci, GENERAL_SIGNAL_CTRL_OFF, val);
}

void dw_pcie_RC_Assert_Reset(struct dw_pcie *pci)
{
	//Assert reset
	dw_pcie_set_PERST(pci, 1);
	udelay(1);
	dw_pcie_set_PERST(pci, 0);
}

static int dw_pcie_disable_msi(struct dw_pcie *pci)
{
	unsigned int i, irq;

	/* mask the MSI interrupt */
	dw_pcie_wr_own_conf(&pci->pp, PCIE_MSI_INTR0_ENABLE , 4, 0);

	/* disable all MSI vectors */
          //TBD

	for (i = 0; i < MAX_MSI_IRQS; i++) {
		irq = irq_find_mapping(pci->pp.irq_domain, i);
		if (irq > 0)
			irq_dispose_mapping(irq);
	}

	irq_domain_remove(pci->pp.irq_domain);

	return 0;
}

int dw_pcie_Remove_RC(struct dw_pcie *pci)
{
#if 0
	struct pci_dev *dev, *temp;
#endif
	struct pci_bus *parent;
	int ret = 0;
#if 0
	u8 bctl = 0;
#endif

          if(pci->pp.root_bus == NULL)
          	{
               return ret;
	}

	parent = pci->pp.root_bus;
#if 0
	dw_pcie_set_Interrupt0_en(pci, INT0MASK, 0);
	dw_pcie_set_Interrupt1_en(pci,MSI_CTRL_INT, 0);
	list_for_each_entry_safe_reverse(dev, temp, &parent->devices,bus_list) {
		pci_dev_get(dev);
		if (dev->hdr_type == PCI_HEADER_TYPE_BRIDGE ) {
			pci_read_config_byte(dev, PCI_BRIDGE_CONTROL, &bctl);
			if (bctl & PCI_BRIDGE_CTL_VGA) {
				pci_dev_put(dev);
				ret = -EINVAL;
				break;
			}
		}

		//printk("Remove device %s\n", pci_name(dev));
		pci_stop_and_remove_bus_device(dev);
		pci_dev_put(dev);
	}
#else
	pci_stop_root_bus(parent);
	pci_remove_root_bus(parent);
	dw_pcie_set_Interrupt0_en(pci, INT0MASK, 0);
	dw_pcie_set_Interrupt1_en(pci,MSI_CTRL_INT, 0);
#endif
           dw_pcie_disable_msi(pci);
	   dw_remove_intx_domain(&pci->pp);
           dw_pcie_set_LTSSM_en(pci, 0);
           dw_pcie_set_PERST(pci, 1);

	if(pci->pp.io != NULL)
		pci_unmap_iospace(pci->pp.io);

	return ret;
}

int dw_pcie_lanes_dynamic_change(struct dw_pcie *pci, u32 target_lanes)
{
    int ret;
    u32 val;
    u32 lanes;
    struct device *dev = pci->dev;
    struct device_node *np = dev->of_node;

    ret = of_property_read_u32(np, "num-lanes", &lanes);
    if (ret)
        return ret;

    if((target_lanes > lanes) ||(!target_lanes) ||((target_lanes %2) && (target_lanes !=1)) ||\
            ((dw_pcie_readl_phy(pci,PM_LTSSM_STATE)&SMLH_LTSSM_STATE_MASK) != SMLH_LTSSM_STATE_L0))
        return -EPERM;

    val = dw_pcie_readl_dbi(pci, MULTI_LANE_CONTROL);
    val = (val&(~TARGET_LINK_WIDTH_MASK))|target_lanes;
    dw_pcie_writel_dbi(pci, MULTI_LANE_CONTROL, val);
    val = dw_pcie_readl_dbi(pci, MULTI_LANE_CONTROL)|DIRECT_LINK_WIDTH_CHANGE;
    dw_pcie_writel_dbi(pci, MULTI_LANE_CONTROL, val);

    return 0;
}

#if 0
#ifdef CONFIG_PM
#define PCI_EXT_CAP_L1_PM_SUBSTATE 0x1E
#define PCI_EXT_CAP_L1_PM_SUBSTATE_CAP_REG 0x04
#define PCI_EXT_CAP_L1_PM_SUBSTATE_CTRL1_REG 0x08
#define PCI_PM_L1P2_ENABLE   (0x01U<<0)
#define PCI_PM_L1P1_ENABLE   (0x01U<<1)
static int _enable_l1_pm_substate(struct pci_dev * dev, void * userdata)
{
    int pos;
    u16 u16cap, u16ctrl1;

    pci_dev_get(dev);

    pos = pci_find_ext_capability(dev, PCI_EXT_CAP_L1_PM_SUBSTATE);
    if(!pos)
    {
        return -EIO;
    }

    pci_read_config_word(dev, pos + PCI_EXT_CAP_L1_PM_SUBSTATE_CAP_REG, &u16cap);
    pci_read_config_word(dev, pos + PCI_EXT_CAP_L1_PM_SUBSTATE_CTRL1_REG, &u16ctrl1);
    if(u16cap & PCI_PM_L1P2_ENABLE)
    {
        u16ctrl1 |= PCI_PM_L1P2_ENABLE;
    }
    else if(u16cap & PCI_PM_L1P1_ENABLE)
    {
        u16ctrl1 |= PCI_PM_L1P1_ENABLE;
    }
    pci_write_config_word(dev, pos + PCI_EXT_CAP_L1_PM_SUBSTATE_CTRL1_REG, u16ctrl1);

    pci_dev_put(dev);

    return 0;
}

int dw_pcie_enable_l1_pm_substate(struct dw_pcie *pci)
{
    if(pci->pp.root_bus == NULL)
    {
        return -ENODEV;
    }

    pci_walk_bus(pci->pp.root_bus , _enable_l1_pm_substate, NULL);

    return 0;
}
#endif
#endif
