/*
 * drivers/udif/mach-cxd900xx/error.c
 *
 * ERROR Indicator
 *
 * Copyright 2015,2016,2017,2018 Sony Corporation
 * Copyright 2019 Sony Imaging Products & Solutions Inc
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <mach/moduleparam.h>
#include <linux/init.h>
#include <linux/cpu.h>
#include <linux/delay.h>
#include <linux/kthread.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/snsc_boot_time.h>
#include <linux/em_export.h>
#include <linux/udif/timer.h>
#include <asm/cacheflush.h>
#include <mach/errind.h>
#include <mach/platform.h>
#include <mach/regs-gpio.h>
#include <mach/irqs.h>

#define ERRIND_USE_THREAD
#define ERRIND_USE_CONSOLE_CALLBACK

#define PORT_UNDEF 0xffffffffU
#define N_ERRIND_LED 2
static unsigned int errind_led[N_ERRIND_LED] = {
	PORT_UNDEF,
	PORT_UNDEF,
};
module_param_named(led,   errind_led[0], port, S_IRUSR|S_IWUSR);
module_param_named(led_2, errind_led[1], port, S_IRUSR|S_IWUSR);

static unsigned int abt_o_port = PORT_UNDEF;
module_param_named(abt_o, abt_o_port, port, S_IRUSR|S_IWUSR);
#define N_ABT_IN 1
static unsigned int abt_i_port[N_ABT_IN] = {
	PORT_UNDEF,
};
module_param_named(abt_i0, abt_i_port[0], port, S_IRUSR|S_IWUSR);

/*
 * ALG_BOOT port
 */
static unsigned int alg_i_port = PORT_UNDEF;
module_param_named(alg_i, alg_i_port, port, S_IRUSR|S_IWUSR);
#define ALG_IRQ_NAME "alg_boot"
#define ALG_IRQ_FLAG (IRQF_NO_THREAD)

/* ERRIND mode */
static int errind_led_pro = 0;
module_param_named(led_pro, errind_led_pro, int , S_IRUSR|S_IWUSR);

static int remote_mode = 1;
module_param_named(remote, remote_mode, int , S_IRUSR|S_IWUSR);

struct errind_ctrl {
	unsigned int intro;
	unsigned int on;
	unsigned int off;
	unsigned int interval;
	unsigned int ptnlen;
	char ptn[8];
};
static const struct errind_ctrl errind_user = { 1500, 200, 200,   0, 3,"337" };
static const struct errind_ctrl errind_oom  = { 1500,1000, 500,1500, 1,"3" };
static const struct errind_ctrl errind_oops = { 1500,  50,   0,   0, 1,"1" };
static const struct errind_ctrl errind_rmt  = { 1500, 200, 200,   0, 3,"235" };
static const struct errind_ctrl errind_emg  = { 1500, 125,   0,   0, 1,"1" };
static const struct errind_ctrl *errind_ctrls[] = {
	&errind_user,
	&errind_oom,
	&errind_oops,
	&errind_rmt,
	&errind_emg,
};

static int errind_event = ERRIND_NONE;
static const struct errind_ctrl *errind_ctl = NULL;

/* LED driver */
static void set_led(int on)
{
	int i;

	for (i = 0; i < N_ERRIND_LED; i++) {
		unsigned int port, bit;

		if (PORT_UNDEF == errind_led[i])
			continue;

		port = errind_led[i] & 0xff;
		bit  = (errind_led[i] >> 8) & 0xff;

		if (on)
			writel_relaxed(BIT(bit), VA_GPIO(port)+GPIO_WDATA+GPIO_CLR);
		else
			writel_relaxed(BIT(bit), VA_GPIO(port)+GPIO_WDATA+GPIO_SET);
	}
}



#ifdef ERRIND_USE_CONSOLE_CALLBACK
/* subroutine return value */
#define STATE_DONE 1
#define STATE_CONT 0

/*------------------ delay subroutine --------------------*/
static struct {
	enum {
		DELAY_INIT = 0,
		DELAY_WAIT,
		DELAY_DONE
	} state;
	unsigned long long t;
} st_delay = { .state = DELAY_INIT, };

static int errind_delay(unsigned int ms)
{
	int ret = STATE_CONT;
	unsigned long long now;
	long long dt;

	if (!ms) {
		return STATE_DONE;
	}
	now = udif_read_freerun();
	switch (st_delay.state) {
	case DELAY_INIT:
		st_delay.t = now + udif_usecs_to_cycles(ms * 1000ULL);
		st_delay.state = DELAY_WAIT;
		break;
	case DELAY_WAIT:
		dt = now - st_delay.t;
		if (dt < 0)
			break;
		st_delay.state = DELAY_DONE;
		break;
	case DELAY_DONE:
	default:
		st_delay.state = DELAY_INIT;
		ret = STATE_DONE;
		break;
	}
	return ret;
}
/*--------------------------------------------------------*/

struct state_main {
	enum {
		ST_INIT = 0,
		ST_INTRO_OFF,
		ST_INFINI_LOOP,
		ST_PTN_LOOP,
		ST_CHAR_LOOP,
		ST_OFF,
		ST_CHAR_LOOP_END,
		ST_PTN_LOOP_END
	} state;
	unsigned int pos;
	int cnt;
};
static struct state_main st_main = { .state = ST_INIT, };

static void errind_work(void)
{
	const struct errind_ctrl *p;
	unsigned int pos;

	smp_rmb();
	if (!errind_event)
		return;

	p = errind_ctl;
	switch (st_main.state) {
	case ST_INIT:
		set_led(1);
		if (errind_delay(p->intro))
			st_main.state = ST_INTRO_OFF;
		break;
	case ST_INTRO_OFF:
		set_led(0);
		if (errind_delay(p->on + p->off))
			st_main.state = ST_INFINI_LOOP;
		break;
	case ST_INFINI_LOOP:
		st_main.pos = 0;
		st_main.state = ST_PTN_LOOP;
		break;
	case ST_PTN_LOOP: /*=========== ptn[] loop ==============*/
		pos = st_main.pos;
		if (pos >= p->ptnlen) {
			st_main.state = ST_PTN_LOOP_END;
			break;
		}
		st_main.cnt = p->ptn[pos] - '0';
		st_main.pos = pos + 1;
		st_main.state = ST_CHAR_LOOP;
		break;
	case ST_CHAR_LOOP: /*----- On/Off ptn[pos] times -----*/
		if (st_main.cnt <= 0) {
			st_main.state = ST_CHAR_LOOP_END;
			break;
		}
		set_led(1);
		if (errind_delay(p->on))
			st_main.state = ST_OFF;
		break;
	case ST_OFF:
		set_led(0);
		if (errind_delay(p->off)) {
			st_main.cnt--;
			st_main.state = ST_CHAR_LOOP;
		}
		break;
	case ST_CHAR_LOOP_END: /*-----------------------------*/
		if (errind_delay(p->on + p->off))
			st_main.state = ST_PTN_LOOP;
		break;
	case ST_PTN_LOOP_END:  /*================================*/
		if (errind_delay(p->interval))
			st_main.state = ST_INFINI_LOOP;
		break;
	default:
		st_main.state = ST_INIT;
		break;
	}
}
#endif /* ERRIND_USE_CONSOLE_CALLBACK */

static void errind_sequencer(int code, void (*fn)(unsigned int))
{
	const struct errind_ctrl *ctl;
	int pos, cnt;

	if (code < 1  ||  ERRIND_MAX < code) {
		printk(KERN_ERR "errind_sequncer:ERROR:code=%d\n", code);
		return;
	}
	ctl = errind_ctrls[code - 1];

	set_led(1);
	(*fn)(ctl->intro);
	set_led(0);
	(*fn)(ctl->on + ctl->off);
	while (1) {
		for (pos = 0; pos < ctl->ptnlen; pos++) {
			cnt = ctl->ptn[pos] - '0';
			while (cnt > 0) {
				set_led(1);
				(*fn)(ctl->on);
				set_led(0);
				(*fn)(ctl->off);
				cnt--;
			}
			(*fn)(ctl->on + ctl->off);
		}
		(*fn)(ctl->interval);
	}
	/* never reach */
}

#ifdef ERRIND_USE_THREAD
static struct task_struct *errind_task = NULL;
static unsigned int errind_task_event = ERRIND_NONE;

static int errind_thread(void *data)
{
	struct sched_param param = { .sched_priority = MAX_RT_PRIO - 1 };

	sched_setscheduler(current, SCHED_FIFO, &param);
	set_current_state(TASK_INTERRUPTIBLE);
	while (!kthread_should_stop()) {
		schedule();
		if (kthread_should_stop())
			break;
		if (errind_task_event) {
			errind_sequencer(errind_task_event, msleep);
		}
		set_current_state(TASK_INTERRUPTIBLE);
	}
	__set_current_state(TASK_RUNNING);
	return 0;
}

void errind_start(int code)
{
	if (!errind_task)
		return;
	errind_task_event = code;
	wake_up_process(errind_task);
}
#endif /* ERRIND_USE_THREAD */

/*----------------------- remote -----------------------*/
static void errind_to_remote(int code)
{
	unsigned int port, bit;

	if (PORT_UNDEF == abt_o_port)
		return;
	port = abt_o_port & 0xff;
	bit = (abt_o_port >> 8) & 0xff;
	writel_relaxed(BIT(bit), VA_GPIO(port)+GPIO_WDATA+GPIO_CLR);
}

static void errind_mdelay(unsigned int msec)
{
	mdelay(msec);
}

#define REMOTE_IRQ_NAME "remote"
#define REMOTE_IRQ_FLAG 0

static irqreturn_t remote_intr(int irq, void *dev_id)
{
	char buf[64];
	int code;

	scnprintf(buf, sizeof buf, "REMOTE Exception detected:irq%d\n", irq);
	BOOT_TIME_ADD1(buf);
	flush_dcache_all();
	printk(KERN_ERR "%s\n", buf);

	/* single shot */
	disable_irq_nosync(irq);

	code = ERRIND_REMOTE;
	if (errind_led_pro) {
		code = ERRIND_EMG;
	}

	switch (remote_mode) {
	case 1: /* thread mode */
#ifdef ERRIND_USE_THREAD
		errind_start(code);
#endif /* ERRIND_USE_THREAD */
		break;
	case 2: /* stop all task */
		smp_send_stop();
		errind_sequencer(code, errind_mdelay);
		break;
	case 3: /* Reboot */
		em_reboot(1);
		break;
	default:
		break;
	}
	return IRQ_HANDLED;
}

static void errind_handler(struct pt_regs *regs, unsigned int errcode)
{
	int code;
#ifdef ERRIND_USE_CONSOLE_CALLBACK
	extern void (*console_idle_callback)(void);
#endif /* ERRIND_USE_CONSOLE_CALLBACK */


	/* analyze */
	code = ERRIND_OOPS;
	if (errind_led_pro) {
		code = ERRIND_EMG;
	} else if (user_mode(regs)) {
		code = ERRIND_USER;
		if (test_tsk_thread_flag(current, TIF_MEMDIE)) {
			code = ERRIND_OOM;
		}
	}

	/* local indicator */
	errind_event = code;
	errind_ctl = errind_ctrls[code - 1];
#ifdef ERRIND_USE_CONSOLE_CALLBACK
	console_idle_callback = errind_work;
#endif /* ERRIND_USE_CONSOLE_CALLBACK */
	smp_wmb();

	/* remote indicator */
	errind_to_remote(code);
}

static void __init errind_remote_init(void)
{
	int i;

	for (i = 0; i < N_ABT_IN; i++) {
		unsigned int port, bit;
		int irq;

		if (PORT_UNDEF == abt_i_port[i])
			continue;
		port = abt_i_port[i] & 0xff;
		bit = (abt_i_port[i] >> 8) & 0xff;
		irq = gpiopin_to_irq(port, bit);
		if (irq < 0) {
			printk(KERN_ERR "%s: invalid abt_i%d port:(%d,%d)\n",
			       __func__, i, port, bit);
			return;
		}
		if (request_threaded_irq(irq, NULL, remote_intr,
					 REMOTE_IRQ_FLAG|IRQF_ONESHOT,
					 REMOTE_IRQ_NAME, NULL)) {
			printk(KERN_ERR "%s:%d: request_irq %d failed.\n",
			       __func__, i, irq);
		}
	}
}

static void __exit errind_remote_exit(void)
{
}

static irqreturn_t alg_intr(int irq, void *dev_id)
{
	char buf[64];

	scnprintf(buf, sizeof buf, "ALG_BOOT Notify\n");
	BOOT_TIME_ADD1(buf);

	if (smp_send_stop_nosync()) {
		udelay(100);
	}
	flush_dcache_all();

	printk(KERN_ERR "%s\n", buf);
	/* single shot */
	disable_irq_nosync(irq);

	/* ToDO: SelfRefresh */

	return IRQ_HANDLED;
}

static void __init errind_alg_init(void)
{
	unsigned int port, bit;
	int irq;

	if (PORT_UNDEF == alg_i_port)
		return;
	port = alg_i_port & 0xff;
	bit = (alg_i_port >> 8) & 0xff;
	irq = gpiopin_to_irq(port, bit);
	if (irq < 0) {
		printk(KERN_ERR "%s: invalid alg_i port:(%d,%d)\n",
		       __func__, port, bit);
		return;
	}
	if (request_irq(irq, alg_intr, ALG_IRQ_FLAG, ALG_IRQ_NAME, NULL)) {
		printk(KERN_ERR "%s: request_irq %d failed.\n",
		       __func__, irq);
		return;
	}
}

static int __init errind_init(void)
{
#ifdef ERRIND_USE_THREAD
	struct task_struct *th;

	th = kthread_create(errind_thread, NULL, "errind");
	if (IS_ERR(th)) {
		printk(KERN_ERR "errind:ERROR:kthread_create failed.\n");
	} else {
		wake_up_process(th);
		errind_task = th;
	}
#endif /* ERRIND_USE_THREAD */

	errind_remote_init();
	em_hook = errind_handler;
	smp_wmb();

	errind_alg_init();
	return 0;
}

static void __exit errind_exit(void)
{
#ifdef ERRIND_USE_THREAD
	struct task_struct *th;
#endif /* ERRIND_USE_THREAD */

	errind_remote_exit();
#ifdef ERRIND_USE_THREAD
	if ((th = errind_task) != NULL) {
		errind_task = NULL;
		kthread_stop(th);
	}
#endif /* ERRIND_USE_THREAD */

	em_hook = NULL;
	smp_wmb();
}

module_init(errind_init);
module_exit(errind_exit);
