/*
 * arch/arm/common/wbi_raw.c
 *
 * WBI non-compress sequencer
 *
 * Copyright 2019 Sony Imaging Products & Solutions Inc.
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 *
 */
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/snsc_boot_time.h>
#include <linux/wdt.h>
#include <linux/warmboot.h>
#include <linux/wbi.h>

#include <asm/cacheflush.h>
#include <asm/io.h>
#include <mach/noncache.h>
#include <mach/moduleparam.h>

#define SECTOR_SHIFT 9
#define SECTOR_SIZE (1 << SECTOR_SHIFT)

/*------------------------ API ----------------------------*/
static char *wbuf;
static int cur_sector = 0;

int wbi_send_data(void *b, ulong len)
{
	int n, ret;

	while (len > 0) {
		n = len;
		if (n < SECTOR_SIZE) {
			memset(wbuf + n, 0, SECTOR_SIZE - n);
		} else if (n > SECTOR_SIZE) {
			n = SECTOR_SIZE;
		}
		memcpy(wbuf, b, n);
		ret = wb_default_device->write_sector(wb_default_device,
						      wbuf, cur_sector, 1);
		if (ret < 0) {
			goto err;
		}
		cur_sector++;
		b += n;
		len -= n;
	}
	return 0;

err:
	wb_printk("ERROR:%s: cannot write(%p,%d) = %d\n", __FUNCTION__, wbuf, cur_sector, ret);
	return ret;
}

#if 0
void wbi_align_sector(void)
{
	wbi_wbuf_align();
}
#endif

int wbi_flush(void)
{
	return 0;
}

#define MAX_SECTOR 0xffff

int wbi_send_sections(struct wb_header *header, struct wb_section *s, ulong n)
{
	int i, ret;
	struct wb_section *p;

	/*
	 * Note:
	 *   During write sections, DO NOT USE kernel API.
	 *   (ex. printk, BOOT_TIME)
	 *   Because there are side effects on heap. (ex. lock variable)
	 */
	/* write sections */
	for (i = 0, p = s; i < n; i++, p++) {
		int rest, nsect;
		void *va;

		p->rlen = (p->flag & WBI_SH_ZERO) ? 0: p->olen;
		INIT_CKSUM(p->meta_cksum);
		wbi_calc_cksum(&p->meta_cksum, (void *)p,
			       offsetof(struct wb_section, meta_cksum));
		if (p->flag & WBI_SH_ZERO)
			continue;
		rest = (p->olen + SECTOR_SIZE - 1) >> SECTOR_SHIFT;
		va = (void *)PHYS_TO_VA(p->addr);
		while (rest > 0) {
			nsect = rest;
			if (nsect > MAX_SECTOR)
				nsect = MAX_SECTOR;
			ret = wb_default_device->write_sector(wb_default_device,
							      va, cur_sector,
							      nsect);
			if (ret < 0) {
				wb_printk("ERROR:%s: cannot write(va=%p,sector=%d,count=%d) = %d\n",
					  __func__, va, cur_sector, nsect, ret);
				return ret;
			}
			cur_sector += nsect;
			va += nsect << SECTOR_SHIFT;
			rest -= nsect;
			wbi_watchdog();
		}
	}
	/* summary */
	for (i = 0, p = s; i < n; i++, p++) {
		wb_printk("WBI: %3d %08lx:%08lx:%08lx %08lx<->%08lx\n",
			  i, p->addr, p->cksum, p->flag,
			  p->olen, p->rlen);
		wbi_watchdog();
	}

	/* write meta data */
	ret = wbi_send_data(s, sizeof(*s) * n);
	if (ret) {
		goto err2;
	}

	return 0;

 err2:
	wb_printk("%s: write meta data error: %d\n", __FUNCTION__, ret);
	return ret;
}

int wbi_rewrite_header(struct wb_header *header)
{
	int ret;

	/* ASSUME: header size <= sector_size */
	memset(wbuf, 0, SECTOR_SIZE);
	memcpy(wbuf, header, sizeof (*header));

	ret = wb_default_device->write_sector(wb_default_device, wbuf, 0, 1);
	wbi_watchdog();
	return ret;
}

int wbi_read_header(struct wb_header *header)
{
	int ret;

	ret = wb_default_device->read_sector(wb_default_device, wbuf, 0, 1);
	wbi_watchdog();

	/* ASSUME: header size <= sector_size */
	memcpy((void *)header, wbuf, sizeof (*header));
	return ret;
}

void wbi_stat(void)
{
	wb_printk("%d sectors\n", cur_sector);
}

int wbi_setup(ulong work, ulong *wlenp, ulong dst, ulong *dlenp)
{
	cur_sector = 0;
	wbuf = kmalloc(SECTOR_SIZE, GFP_KERNEL);
	if (!wbuf) {
		wb_printk("%s:Can not alloc wbuf\n", __func__);
		return -1;
	}
	return 0;
}

void wbi_finish(void)
{
	if (wbuf) {
		kfree(wbuf);
		wbuf = NULL;
	}
}
