/*
 *  proc.c
 *
 *  Copyright 2018 Sony Corporation.
 *  Copyright 2019,2020 Sony Imaging Products & Solutions Inc
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */
#include "pcie_dma.h"

#ifdef CONFIG_PROC_FS
#include <linux/proc_fs.h>
#include <linux/seq_file.h>

#define PROC_PCIE_FMT "driver/pcie/%d/dma"
#if 0
#define PROC_PCIE_DIR "driver/pcie"
static struct proc_dir_entry *proc_pcie;
#endif

/* 0..7 == DMAC CH0..7 */
#define to_chan(p)	((int)(uintptr_t)(p) - 1)
/* 1..8 == DMAC CH0..7 */
#define chan_to(n)	((void *)((uintptr_t)((n) + 1)))

static void *pcidma_seq_start(struct seq_file *m, loff_t *pos)
{
	int ch = *pos;

	if (ch >= PCIDMA_CH_MAX)
		return NULL;
	return chan_to(ch);
}

static void *pcidma_seq_next(struct seq_file *m, void *p, loff_t *pos)
{
	int ch = to_chan(p);

	(*pos)++;
	ch++;
	if (ch >= PCIDMA_CH_MAX)
		return NULL;
	return chan_to(ch);
}

static void pcidma_seq_stop(struct seq_file *m, void *arg)
{
}

static char *pcidma_show_state(struct dma_info_t *chan)
{
	if (!chan->state)
		return "----";
	else if (chan->cur)
		return "busy";
	return "idle";
}

static int pcidma_seq_show(struct seq_file *m, void *arg)
{
	int ch = (int)(uintptr_t)arg - 1;  /* 0..7 == DMAC CH0..7 */
	struct pci_info_t *p = (struct pci_info_t *)m->private;
	struct dma_info_t *chan = &p->dma_info[ch];
	unsigned long flags;
	char *state;
	struct dma_stat_t stat;
#ifdef ER9001359142
	int ready;
#endif

	seq_printf(m, "%d ", chan->id);
	LOCK(chan);
	state = pcidma_show_state(chan);
#ifdef ER9001359142
	ready = chan->ready;
#endif
	stat = chan->stat;
	UNLOCK(chan);
	seq_printf(m, "%s ", state);
#ifdef ER9001359142
	seq_printf(m, "%s ", (ready)?"rdy":"---");
#endif
	seq_printf(m, "pending=%d ", stat.pending_cnt);
	seq_printf(m, "%s ", stat.ll_mode ? "LL" : "NL");
	seq_printf(m, "cnt=%ld,%ld,%ld ", stat.abort_cnt, stat.go_cnt, stat.done_cnt);
	seq_printf(m, "\n");
	return 0;
}

static struct seq_operations pcidma_seq_ops = {
	.start	= pcidma_seq_start,
	.next	= pcidma_seq_next,
	.stop	= pcidma_seq_stop,
	.show	= pcidma_seq_show,
};

static int pcidma_proc_open(struct inode *inode, struct file *file)
{
	int ret;

	ret = seq_open(file, &pcidma_seq_ops);
	if (!ret) {
		struct seq_file *m = file->private_data;
		m->private = PDE_DATA(inode);
	}
	return ret;
}

static ssize_t pcidma_proc_write(struct file *file, const char __user *buffer,
				 size_t count, loff_t *ppos)
{
	struct pci_info_t *p = PDE_DATA(file_inode(file));

	log_stop();
	logdump(p);
	log_init(&p->log);
	return count;
}

static struct file_operations pcidma_proc_ops = {
	.owner		= THIS_MODULE,
	.open		= pcidma_proc_open,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= seq_release,
	.write		= pcidma_proc_write,
};

void pcidma_proc_create(struct pci_info_t *p)
{
	char path[80];

	scnprintf(path, sizeof path, PROC_PCIE_FMT, p->id);
	p->proc = proc_create_data(path, 0, NULL, &pcidma_proc_ops, p);
}

void pcidma_proc_remove(struct pci_info_t *p)
{
	if (p->proc) {
		proc_remove(p->proc);
		p->proc = NULL;
	}
}

#if 0
void UDIF_INIT pcidma_proc_dir_create(void)
{
	proc_pcie = proc_mkdir(PROC_PCIE_DIR, NULL);
}

void UDIF_EXIT pcidma_proc_dir_remove(void)
{
	proc_remove(proc_pcie);
}
#endif
#endif /* CONFIG_PROC_FS */
