/*
 * drivers/char/mmap.c
 *   mmap device for DIRECT_IO
 *
 * Copyright 2019 Sony Imaging Products & Solutions Inc
 *
 * This code is based on drivers/char/mem.c.
 */
/*
 *  linux/drivers/char/mem.c
 *
 *  Copyright (C) 1991, 1992  Linus Torvalds
 *
 *  Added devfs support.
 *    Jan-11-1998, C. Scott Ananian <cananian@alumni.princeton.edu>
 *  Shared /dev/zero mmapping support, Feb 2000, Kanoj Sarcar <kanoj@sgi.com>
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/mman.h>
#include <linux/miscdevice.h>
#include <linux/uaccess.h>
#include <linux/udif/cache.h>
#include <mach/noncache.h>

static struct page *dmmap_find_special_page(struct vm_area_struct *vma, unsigned long addr)
{
	pgd_t *pgd;
	pud_t *pud;
	pmd_t *pmd;
	pte_t *pte;
	unsigned long pfn;

	pgd = pgd_offset(vma->vm_mm, addr);
	if (pgd_none(*pgd))
		return NULL;
	pud = pud_offset(pgd, addr);
	if (pud_none(*pud))
		return NULL;
	pmd = pmd_offset(pud, addr);
	if (pmd_none(*pmd))
		return NULL;
	pte= pte_offset_map(pmd, addr);
	if (pte_none(*pte))
		return NULL;
	pfn = pte_pfn(*pte);
	return (pfn_to_page(pfn));
}

static const struct vm_operations_struct dmmap_vm_ops = {
	.find_special_page = dmmap_find_special_page,
};

static int dmmap_mmap(struct file *file, struct vm_area_struct *vma)
{
	if (file->f_flags & O_DSYNC)
		vma->vm_page_prot = pgprot_writecombine(vma->vm_page_prot);

	vma->vm_ops = &dmmap_vm_ops;

	if (remap_pfn_range(vma,
			    vma->vm_start,
			    vma->vm_pgoff,
			    vma->vm_end - vma->vm_start,
			    vma->vm_page_prot)) {
		return -EAGAIN;
	}
#ifdef CONFIG_EJ_WORKAROUND_MMAP_RESERVED_PAGE
	vma->vm_flags &= ~(VM_IO|VM_PFNMAP); /* for DIRECT_IO */
#endif
	return 0;
}

static long dmmap_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	u64 param[2]; /* phys, size */
	uintptr_t va;

	if ('C' != cmd && 'I' != cmd)
		return -ENOTTY;

	if (copy_from_user(param, (void __user *)arg, sizeof param))
		return -EFAULT;
	if (!param[1])
		return 0;
	if ((va = PHYS_TO_VA(param[0])) == PA2NC_ERR)
		return -EINVAL;
	udif_cache_ctrl((UDIF_VA)va, param[1], ('C'==cmd)?UDIF_CACHE_FLUSH:UDIF_CACHE_INVALIDATE);
	return 0;
}

static struct file_operations dmmap_fops = {
	.owner		= THIS_MODULE,
	.mmap		= dmmap_mmap,
	.unlocked_ioctl	= dmmap_ioctl,
};

static struct miscdevice dmmap_miscdev = {
	.minor = MISC_DYNAMIC_MINOR,
	.name  = "mmap",
	.fops  = &dmmap_fops,
};

static int __init dmmap_init(void)
{
	int ret;

	ret = misc_register(&dmmap_miscdev);
	if (ret) {
		printk(KERN_ERR "%s: misc_register failed. ret=%d\n",
		       __func__, ret);
		return -ENODEV;
	}

	return 0;
}

static void __exit dmmap_cleanup(void)
{
	misc_deregister(&dmmap_miscdev);
}

module_init(dmmap_init);
module_exit(dmmap_cleanup);

MODULE_LICENSE("GPL");
