/*
 *  cdev.c
 *
 *  Copyright 2018 Sony Corporation.
 *  Copyright 2019,2020 Sony Imaging Products & Solutions Inc
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */
#include <linux/udif/cdev.h>
#include <mach/udif/devno.h>
#include <linux/udif/uaccess.h>
#include <linux/udif/wait.h>
#include "pcie_dma.h"

struct cb_info {
	UDIF_WAIT wait;
	UDIF_UINT cond;
	int err;
};

static void pcidma_ioctl_cb(int err, void *cookies)
{
	struct cb_info *cbinfo = (struct cb_info *)cookies;

	cbinfo->err = err;
	cbinfo->cond = 1;
	udif_wake_up(&cbinfo->wait);
}

static UDIF_ERR pcidma_ioctl(UDIF_FILE *file, UDIF_IOCTL *ioctl)
{
	struct pcidma_param param;
	struct pcidma_request req;
	struct cb_info cbinfo;
	UDIF_SIZE failed;
	UDIF_ERR ret;

	if (!ioctl) {
		return UDIF_ERR_PAR;
	}
	if (PCIDMA_TRANSFER != ioctl->cmd) {
		return UDIF_ERR_PAR;
	}
	if (!ioctl->arg) {
		return UDIF_ERR_PAR;
	}

	failed = udif_copy_from_user(&param, (UDIF_USR_RO)ioctl->arg, sizeof param);
	if (failed) {
		return UDIF_ERR_PAR;
	}

	/* check param */
	if (param.pcie_ch >= PCIDMA_PCIE_MAX) {
		return UDIF_ERR_PAR;
	}
	if (param.dmac_ch >= PCIDMA_CH_MAX) {
		return UDIF_ERR_PAR;
	}

	req.pcie_ch	= param.pcie_ch;
	req.dmac_ch	= param.dmac_ch;
	req.dir		= param.dir;
	req.size	= param.size;
	req.src		= param.src;
	req.dst		= param.dst;
	req.callback	= pcidma_ioctl_cb;
	req.cookies	= &cbinfo;
	if (!req.size) {
		return UDIF_ERR_OK;
	}
	udif_wait_init(&cbinfo.wait);
	cbinfo.cond = 0;
	cbinfo.err = 0;
	ret = pcidma_transfer_async(&req);
	if (ret) {
		return UDIF_ERR_PAR;
	}
	ret = udif_wait_event_timeout(&cbinfo.wait, cbinfo.cond, PCIDMA_TIMEOUT);
	if (UDIF_ERR_TIMEOUT == ret) {
		return UDIF_ERR_TIMEOUT;
	}
	if (cbinfo.err) {
		return UDIF_ERR_IO;
	}
	return UDIF_ERR_OK;
}

static UDIF_ERR pcidma_open(UDIF_FILE *file)
{
	return UDIF_ERR_OK;
}

static UDIF_ERR pcidma_close(UDIF_FILE *file)
{
	return UDIF_ERR_OK;
}

static UDIF_CDEV_OPS pcidma_fops = {
	.open	= pcidma_open,
	.close	= pcidma_close,
	.ioctl	= pcidma_ioctl,
};

static UDIF_DECLARE_CDEV(pcidma_cdev, udif_device_node(UDIF_NODE_PCIDMA), &pcidma_fops, NULL);

void UDIF_INIT pcidma_cdev_init(void)
{
	UDIF_ERR ret;

	ret = udif_cdev_register(&pcidma_cdev);
	if (UDIF_ERR_OK != ret) {
		printk(KERN_ERR "%s: udif_cdev_register failed: %d\n",
		       __func__, ret);
		BUG();
	}
}

void UDIF_EXIT pcidma_cdev_exit(void)
{
	udif_cdev_unregister(&pcidma_cdev);
}
