/*
 *  pcie_dma.h
 *
 *  Copyright 2018 Sony Corporation.
 *  Copyright 2019,2020 Sony Imaging Products & Solutions Inc
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */
#ifndef __PCIEDMA_PCIE_DMA_H__
#define __PCIEDMA_PCIE_DMA_H__

#include "config.h"

#include <linux/snsc_boot_time.h>
#include <linux/udif/spinlock.h>
#include <linux/udif/timer.h>
#include <linux/udif/delay.h>
#include <linux/pcidma.h>
#include <asm/io.h>

#include "log.h"
struct chlog {
	unsigned char type;
	unsigned char pad[3];
	unsigned int data;
};
#define CHLOG_DMASTART	0x01
#define CHLOG_DMAINTR	0x02
#define CHLOG_ACQ	0x03
#define CHLOG_RESTORE_B	0x04
#define CHLOG_RESTORE_E	0x05
#define CHLOG_ERR1	0x06

/* latency threshold to warn */
#include <linux/hwtimer.h>
#define PCIDMA_LATENCY_01mSEC  (TIMER_1MSEC/10)
#define PCIDMA_LATENCY_1mSEC   (TIMER_1MSEC)
#define PCIDMA_LATENCY_4mSEC   (TIMER_1MSEC*4)
#define PCIDMA_LATENCY_16mSEC  (TIMER_1MSEC*16)
#define PCIDMA_LATENCY_1SEC    (TIMER_1MSEC*1000)
#define PCIDMA_LATENCY_10SEC   (TIMER_1MSEC*10000)

#ifdef ER9001359142
#include "aspmctrl.h"
#endif

/* PCIe DMAC CHANNEL info */
struct dma_reg_t {
	volatile void __iomem *pwr;
	volatile void __iomem *ctl;
	volatile void __iomem *doorbell;
};

struct dma_stat_t {
	int ll_mode; /* current DMAC Mode (LL or non-LL) */
	int dir;     /* current direction (RD or WR) */
	int pending_cnt;
	unsigned long abort_cnt, go_cnt, done_cnt;
};

static inline void dma_stat_init(struct dma_stat_t *stat)
{
	memset(stat, 0, sizeof *stat);
}

struct dma_info_t {
	int id;
	struct pci_info_t *pci_info;
	/*--- request control ---*/
	UDIF_SPINLOCK _lock;
	int state; /* PCIe connection state */
#ifdef ER9001359142
	int ready; /* errata workaround done */
#endif
	struct list_head pending_list;
	struct pcidma_request *cur;
	struct dma_stat_t stat;  /* for proc */
	/*--- log ---*/
	struct log log;
	/*------------------------*/
	struct dma_reg_t dmawr, dmard;
};

static inline int pcidma_busy(struct dma_info_t *chan)
{
	return !!chan->cur;
}

/* PCIe HOST info */
struct pci_info_t {
	int id;
	unsigned int flag;
	/*--- PCIe CH control */
	UDIF_SPINLOCK lock;
	int state; /* PCIe connection state */
	/*--- DMAC global control */
	UDIF_SPINLOCK _lock;
#ifdef ER9001359142
	struct aspm_ctrl aspm;
#endif
	/*------------------------*/
	struct dma_info_t dma_info[PCIDMA_CH_MAX];
	const int irq;
	volatile void __iomem * const ctl;
	volatile void __iomem * const dmac;
	volatile void __iomem * const phy;
	const char *name;
	struct proc_dir_entry *proc;
	struct log log;
};
#define FLAG_IRQ  0x00000001U
#define FLAG_DMA  0x00000002U

#define LOCK(x)   udif_spin_lock_irqsave(&(x)->_lock, flags)
#define UNLOCK(x) udif_spin_unlock_irqrestore(&(x)->_lock, flags)

/* I/O access */
static inline uint32_t pcie_readl(volatile void __iomem *p)
{
	return readl_relaxed(p);
}

static inline void pcie_writel(volatile void __iomem *p, uint32_t data)
{
	writel_relaxed(data, p);
}

static inline uint16_t pcie_readw(volatile void __iomem *p)
{
	return readw_relaxed(p);
}

static inline void pcie_writew(volatile void __iomem *p, uint16_t data)
{
	writew_relaxed(data, p);
}

static inline uint8_t pcie_readb(volatile void __iomem *p)
{
	return readb_relaxed(p);
}

static inline void pcie_writeb(volatile void __iomem *p, uint8_t data)
{
	writeb_relaxed(data, p);
}

static inline void pcie_mb(void)
{
	mb();
}

static inline void pcie_wmb(void)
{
	dma_wmb();
}

extern struct pci_info_t pcie_dmacs[];
extern void pcidma_irq_setup(struct pci_info_t *p);
extern void pcidma_irq_remove(struct pci_info_t *p);
extern void pcidma_shutdown(struct pci_info_t *p);
extern void pcidma_host_cb(int ch, int state);
extern void pcidma_cdev_init(void);
extern void pcidma_cdev_exit(void);

extern void pcidma_latency_mon_init(void);

#ifdef CONFIG_PROC_FS
#if 0
extern void pcidma_proc_dir_create(void);
extern void pcidma_proc_dir_remove(void);
#endif
extern void pcidma_proc_create(struct pci_info_t *p);
extern void pcidma_proc_remove(struct pci_info_t *p);
#endif /* CONFIG_PROC_FS */

extern void logdump(struct pci_info_t *p);
extern void chanlog(struct dma_info_t *chan, unsigned char type, unsigned int data);
extern void pcidma_start(struct dma_info_t *chan);

#ifdef CONFIG_PCIEDMA_ER3425265
extern int stallmon_init(void);
extern void stallmon_exit(void);
#endif

#endif /* __PCIEDMA_PCIE_DMA_H__ */
