/*
 * udif/mach-cxd900xx/pcie.c
 *
 *
 * Copyright 2018,2019,2020 Sony Imaging Products & Solutions Inc
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */
#include <linux/kernel.h>
#include <linux/pci.h>
#include <linux/module.h>
#include <linux/mutex.h>
#include <linux/rtmutex.h>
#include <linux/fs.h>
#include <asm/uaccess.h>
#include <linux/proc_fs.h>
#include <linux/snsc_boot_time.h>
#include <linux/pci-aspm.h>
#include <mach/pcie_export.h>

#include <linux/udif/module.h>
#include <linux/udif/io.h>
#include <linux/udif/delay.h>
#include <linux/udif/timer.h>
#include <mach/regs-pci.h>
#include <mach/regs-srst.h>
#include <mach/regs-gpio.h>
#include <mach/platform.h>
#include <mach/irqs.h>
#include <mach/moduleparam.h>

#define N_PCIE	UDIF_NR_PCIE

#define VID1 0x1556
#define VID2 0x16c3
#define BAR_SIZE 0x1000000000ULL /* 64GB */

#ifdef CONFIG_PCI
int pcibios_assign_resource(struct pci_dev *dev, int resno, resource_size_t size, resource_size_t min_align)
{
	struct resource *res = dev->resource + resno;

	if (((PCI_VENDOR_ID_SONY == dev->vendor
	      && PCI_DEVICE_ID_CXD90058 == dev->device)
	     || VID1 == dev->vendor
	     || VID2 == dev->vendor)
	    && BAR_SIZE == size) {
		printk(KERN_ERR "%s: %x %d %llx\n", __func__, dev->vendor,
		       resno, size);
		switch (resno) {
		case 2:
			res->start = 0x000000000UL;
			res->end   = res->start + size - 1;
			break;
		}
		return 0;
	}
	if (PCI_VENDOR_ID_XILINX == dev->vendor
	    && 0x9031 == dev->device) {
		printk(KERN_ERR "%s: %04x:%04x %d %llx\n", __func__,
		       dev->vendor, dev->device, resno, size);
		switch (resno) {
		case 0:
			if (0x1000000000ULL == size) { /* 64GB */
				res->start = 0x1000000000UL;
				res->end   = res->start + size - 1;
				return 0;
			}
			break;
		case 2:
			if (0x0100000000ULL == size) { /* 4GB */
				res->start = 0x2000000000UL;
				res->end   = res->start + size - 1;
				return 0;
			}
			break;
		}
	}
	return -1;
}


static char *pcie_ch_name[N_PCIE] = {
	"pcie_i", "pcie_m0", "pcie_m1", "pcie_c", "pcie_n"
};

/* pci_lock */
static raw_spinlock_t pci_lock[N_PCIE] = {
	__RAW_SPIN_LOCK_UNLOCKED(pci_lock[0]),
	__RAW_SPIN_LOCK_UNLOCKED(pci_lock[1]),
	__RAW_SPIN_LOCK_UNLOCKED(pci_lock[2]),
	__RAW_SPIN_LOCK_UNLOCKED(pci_lock[3]),
	__RAW_SPIN_LOCK_UNLOCKED(pci_lock[4])
};

static int get_domain_nr(struct pci_bus *bus, const char *name)
{
	int domain = pci_domain_nr(bus);

	if (domain < 0 || N_PCIE <= domain) {
		printk(KERN_ERR "ERROR:%s:domain=%d\n", name, domain);
		domain = 0;
	}
	return domain;
}

void pci_lock_irqsave(struct pci_bus *bus, unsigned long *flags)
	__acquires(pci_lock)
{
	int n = get_domain_nr(bus, __func__);
	raw_spin_lock_irqsave(&pci_lock[n], *flags);
}

void pci_unlock_irqrestore(struct pci_bus *bus, unsigned long flags)
	__releases(pci_lock)
{
	int n = get_domain_nr(bus, __func__);
	raw_spin_unlock_irqrestore(&pci_lock[n], flags);
}

void pci_lock_irq(struct pci_bus *bus)
	__acquires(pci_lock)
{
	int n = get_domain_nr(bus, __func__);
	raw_spin_lock_irq(&pci_lock[n]);
}

void pci_unlock_irq(struct pci_bus *bus)
	__releases(pci_lock)
{
	int n = get_domain_nr(bus, __func__);
	raw_spin_unlock_irq(&pci_lock[n]);
}

/* AER */
static int noaer[N_PCIE] = { 0, 0, 0, 0, 0 };
module_param_named(noaer0, noaer[0], int, S_IRUSR|S_IWUSR);
module_param_named(noaer1, noaer[1], int, S_IRUSR|S_IWUSR);
module_param_named(noaer2, noaer[2], int, S_IRUSR|S_IWUSR);
module_param_named(noaer3, noaer[3], int, S_IRUSR|S_IWUSR);
module_param_named(noaer4, noaer[4], int, S_IRUSR|S_IWUSR);

int pci_aer_disable(int ch)
{
	if (ch < 0 || N_PCIE <= ch)
		return 0;
	return noaer[ch];
}

/* ASPM */
static int aspm[N_PCIE] = { 1, 1, 1, 1, 1 }; // POLICY_PERFORMANCE

static int aspm_set(const char *val, const struct kernel_param *kp)
{
	int domain, policy, ret;

	domain = (int *)kp->arg - aspm;
	if (domain < 0 || N_PCIE <= domain) {
		printk(KERN_ERR "ERROR:%s:internal error\n", __func__);
		return -EFAULT;
	}

	ret = kstrtoint(val, 0, &policy);
	if (ret)
		return ret;

	ret = pcie_aspm_set_domain_policy(domain, policy);
	if (ret) {
		printk(KERN_ERR "ERROR:%s:Invalid policy:%d\n",
		       __func__, policy);
		return -EINVAL;
	}
	return 0;
}

static const struct kernel_param_ops aspm_ops = {
	.set	= aspm_set,
	.get	= param_get_int,
};
module_param_cb(aspm0, &aspm_ops, &aspm[0], S_IRUSR|S_IWUSR);
module_param_cb(aspm1, &aspm_ops, &aspm[1], S_IRUSR|S_IWUSR);
module_param_cb(aspm2, &aspm_ops, &aspm[2], S_IRUSR|S_IWUSR);
module_param_cb(aspm3, &aspm_ops, &aspm[3], S_IRUSR|S_IWUSR);
module_param_cb(aspm4, &aspm_ops, &aspm[4], S_IRUSR|S_IWUSR);

int pcie_aspm_policy(int domain)
{
	if (domain < 0 || N_PCIE <= domain)
		return 1; // POLICY_PERFORMANCE
	return aspm[domain];
}
EXPORT_SYMBOL(pcie_aspm_policy);

/* caller must hold aspm_lock */
int pcie_aspm_change_policy(int domain, int policy)
{
	if (domain < 0 || N_PCIE <= domain)
		return -EINVAL;

	if (policy == aspm[domain])
		return 0;

	aspm[domain] = policy;
	return 1;
}

static struct rt_mutex aspm_ctrl_mutex[N_PCIE] = {
	__RT_MUTEX_INITIALIZER(aspm_ctrl_mutex[0]),
	__RT_MUTEX_INITIALIZER(aspm_ctrl_mutex[1]),
	__RT_MUTEX_INITIALIZER(aspm_ctrl_mutex[2]),
	__RT_MUTEX_INITIALIZER(aspm_ctrl_mutex[3]),
	__RT_MUTEX_INITIALIZER(aspm_ctrl_mutex[4])
};

int pcie_aspm_ctrl_trylock(int domain)
{
	if (domain < 0 || N_PCIE <= domain)
		return -EINVAL;
	return rt_mutex_trylock(&aspm_ctrl_mutex[domain]);
}

int pcie_aspm_ctrl_lock(int domain)
{
	if (domain < 0 || N_PCIE <= domain)
		return -EINVAL;
	rt_mutex_lock(&aspm_ctrl_mutex[domain]);
	return 0;
}

int pcie_aspm_ctrl_unlock(int domain)
{
	if (domain < 0 || N_PCIE <= domain)
		return -EINVAL;
	rt_mutex_unlock(&aspm_ctrl_mutex[domain]);
	return 0;
}

EXPORT_SYMBOL_GPL(pcie_aspm_ctrl_trylock);
EXPORT_SYMBOL_GPL(pcie_aspm_ctrl_lock);
EXPORT_SYMBOL_GPL(pcie_aspm_ctrl_unlock);

/* HOST callback */
static DEFINE_MUTEX(pcie_cb_mutex);
static void (*pcie_cb)(int ch, int state);
static int pcie_host_state[N_PCIE];

void pcie_dwc_host_notify(int ch, int state)
{
	if (ch < 0 || N_PCIE <= ch)
		return;
        mutex_lock(&pcie_cb_mutex);
	pcie_host_state[ch] = state;
	if (pcie_cb) {
		(*pcie_cb)(ch, state);
	}
	mutex_unlock(&pcie_cb_mutex);
}

void pcie_host_register_cb(void (*func)(int ch, int state))
{
	int i;

        mutex_lock(&pcie_cb_mutex);
	if (pcie_cb) {
		printk(KERN_ERR "%s: already registered\n", __func__);
		goto err;
	}
	pcie_cb = func;

	for (i = 0; i < N_PCIE; i++) {
		if (pcie_host_state[i]) {
			(*pcie_cb)(i, pcie_host_state[i]);
		}
	}
err:
	mutex_unlock(&pcie_cb_mutex);
}
EXPORT_SYMBOL(pcie_host_register_cb);

void pcie_host_unregister_cb(void)
{
        mutex_lock(&pcie_cb_mutex);
	if (!pcie_cb) {
		printk(KERN_ERR "%s: empty\n", __func__);
	}
	pcie_cb = NULL;
	mutex_unlock(&pcie_cb_mutex);
}
EXPORT_SYMBOL(pcie_host_unregister_cb);


int pcie_dwc_write_config(struct pci_dev *dev, int pos, int size,
			  unsigned int val)
{
	extern int dw_pcie_bus_write_config(struct pci_bus *bus,
		unsigned int devfn, int pos, int size, unsigned int val);
	return dw_pcie_bus_write_config(dev->bus, dev->devfn,
					pos, size, val);
}
EXPORT_SYMBOL(pcie_dwc_write_config);

static int vboost[N_PCIE] = { -1, -1, -1, -1, -1 };
module_param_named(vboost0, vboost[0], int, S_IRUSR|S_IWUSR);
module_param_named(vboost1, vboost[1], int, S_IRUSR|S_IWUSR);
module_param_named(vboost2, vboost[2], int, S_IRUSR|S_IWUSR);
module_param_named(vboost3, vboost[3], int, S_IRUSR|S_IWUSR);
module_param_named(vboost4, vboost[4], int, S_IRUSR|S_IWUSR);

int pcie_dwc_vboost(int ch)
{
	if (ch < 0 || N_PCIE <= ch)
		return -1;
	return vboost[ch];
}

static int tmo[N_PCIE] = { -1, -1, -1, -1, -1 };
module_param_named(tmo0, tmo[0], int, S_IRUSR|S_IWUSR);
module_param_named(tmo1, tmo[1], int, S_IRUSR|S_IWUSR);
module_param_named(tmo2, tmo[2], int, S_IRUSR|S_IWUSR);
module_param_named(tmo3, tmo[3], int, S_IRUSR|S_IWUSR);
module_param_named(tmo4, tmo[4], int, S_IRUSR|S_IWUSR);

int pcie_dwc_tmo(int ch)
{
	if (ch < 0 || N_PCIE <= ch)
		return -1;
	return tmo[ch];
}

static int axitmo[N_PCIE] = { -1, -1, -1, -1, -1 };
module_param_named(axitmo0, axitmo[0], int, S_IRUSR|S_IWUSR);
module_param_named(axitmo1, axitmo[1], int, S_IRUSR|S_IWUSR);
module_param_named(axitmo2, axitmo[2], int, S_IRUSR|S_IWUSR);
module_param_named(axitmo3, axitmo[3], int, S_IRUSR|S_IWUSR);
module_param_named(axitmo4, axitmo[4], int, S_IRUSR|S_IWUSR);

int pcie_dwc_axi_tmo(int ch)
{
	if (ch < 0 || N_PCIE <= ch)
		return -1;
	return axitmo[ch];
}

#define PORT_UNDEF (~0U)
/* /PRSNT port */
static unsigned int prsnt[N_PCIE] = {
	PORT_UNDEF, PORT_UNDEF, PORT_UNDEF, PORT_UNDEF, PORT_UNDEF,
};
module_param_named(prsnt0, prsnt[0], port, S_IRUSR|S_IWUSR);
module_param_named(prsnt1, prsnt[1], port, S_IRUSR|S_IWUSR);
module_param_named(prsnt2, prsnt[2], port, S_IRUSR|S_IWUSR);
module_param_named(prsnt3, prsnt[3], port, S_IRUSR|S_IWUSR);
module_param_named(prsnt4, prsnt[4], port, S_IRUSR|S_IWUSR);

int pcie_has_prsnt(int ch)
{
	if (ch < 0 || N_PCIE <= ch)
		return 0;
	if (PORT_UNDEF == prsnt[ch])
		return 0;
	return 1;
}

int pcie_prsnt(int ch)
{
	unsigned int port, bit;
	int val;

	if (ch < 0 || N_PCIE <= ch)
		return -1;
	if (PORT_UNDEF == prsnt[ch])
		return -1;
	port = prsnt[ch] & 0xff;
	bit  = (prsnt[ch] >> 8) & 0xff;
	val = (readl_relaxed(VA_GPIO(port)+GPIO_RDATA) >> bit) & 0x1;
	return !val;
}

int pcie_prsnt_irq(int ch)
{
	unsigned int port, bit;
	int irq;

	if (ch < 0 || N_PCIE <= ch)
		return -1;
	if (PORT_UNDEF == prsnt[ch])
		return -1;
	port = prsnt[ch] & 0xff;
	bit  = (prsnt[ch] >> 8) & 0xff;
	irq = gpiopin_to_irq(port, bit);
	return irq;
}

static int prsntmo[N_PCIE] = { 0, 0, 0, 0, 0 };
module_param_named(prsntmo0, prsntmo[0], int, S_IRUSR|S_IWUSR);
module_param_named(prsntmo1, prsntmo[1], int, S_IRUSR|S_IWUSR);
module_param_named(prsntmo2, prsntmo[2], int, S_IRUSR|S_IWUSR);
module_param_named(prsntmo3, prsntmo[3], int, S_IRUSR|S_IWUSR);
module_param_named(prsntmo4, prsntmo[4], int, S_IRUSR|S_IWUSR);

int pcie_prsnt_tmo(int ch)
{
	if (ch < 0 || N_PCIE <= ch)
		return 0;
	return udif_usecs_to_cycles(prsntmo[ch]);
}

static int prsntH[N_PCIE] = { 0, 0, 0, 0, 0 };
module_param_named(prsntH_0, prsntH[0], int, S_IRUSR|S_IWUSR);
module_param_named(prsntH_1, prsntH[1], int, S_IRUSR|S_IWUSR);
module_param_named(prsntH_2, prsntH[2], int, S_IRUSR|S_IWUSR);
module_param_named(prsntH_3, prsntH[3], int, S_IRUSR|S_IWUSR);
module_param_named(prsntH_4, prsntH[4], int, S_IRUSR|S_IWUSR);

int pcie_prsnt_high(int ch)
{
	if (ch < 0 || N_PCIE <= ch)
		return 0;
	return udif_usecs_to_cycles(prsntH[ch]);
}

static int prsntL1[N_PCIE] = { 0, 0, 0, 0, 0 };
module_param_named(prsntL1_0, prsntL1[0], int, S_IRUSR|S_IWUSR);
module_param_named(prsntL1_1, prsntL1[1], int, S_IRUSR|S_IWUSR);
module_param_named(prsntL1_2, prsntL1[2], int, S_IRUSR|S_IWUSR);
module_param_named(prsntL1_3, prsntL1[3], int, S_IRUSR|S_IWUSR);
module_param_named(prsntL1_4, prsntL1[4], int, S_IRUSR|S_IWUSR);

int pcie_prsnt_low1(int ch)
{
	if (ch < 0 || N_PCIE <= ch)
		return 0;
	return udif_usecs_to_cycles(prsntL1[ch]);
}

static int prsntL2[N_PCIE] = { 0, 0, 0, 0, 0 };
module_param_named(prsntL2_0, prsntL2[0], int, S_IRUSR|S_IWUSR);
module_param_named(prsntL2_1, prsntL2[1], int, S_IRUSR|S_IWUSR);
module_param_named(prsntL2_2, prsntL2[2], int, S_IRUSR|S_IWUSR);
module_param_named(prsntL2_3, prsntL2[3], int, S_IRUSR|S_IWUSR);
module_param_named(prsntL2_4, prsntL2[4], int, S_IRUSR|S_IWUSR);

int pcie_prsnt_low2(int ch)
{
	if (ch < 0 || N_PCIE <= ch)
		return 0;
	return udif_usecs_to_cycles(prsntL2[ch]);
}

/* /PERST port */
static unsigned int perst[N_PCIE] = {
	PORT_UNDEF, PORT_UNDEF, PORT_UNDEF, PORT_UNDEF, PORT_UNDEF,
};
module_param_named(perst0, perst[0], port, S_IRUSR|S_IWUSR);
module_param_named(perst1, perst[1], port, S_IRUSR|S_IWUSR);
module_param_named(perst2, perst[2], port, S_IRUSR|S_IWUSR);
module_param_named(perst3, perst[3], port, S_IRUSR|S_IWUSR);
module_param_named(perst4, perst[4], port, S_IRUSR|S_IWUSR);

void pcie_set_perst(int ch, int assert)
{
	unsigned int port, bit;

	if (ch < 0 || N_PCIE <= ch)
		return;
	if (PORT_UNDEF == perst[ch])
		return;
	port = perst[ch] & 0xff;
	bit  = (perst[ch] >> 8) & 0xff;
	if (assert)
		writel_relaxed(BIT(bit), VA_GPIO(port)+GPIO_WDATA+GPIO_CLR);
	else
		writel_relaxed(BIT(bit), VA_GPIO(port)+GPIO_WDATA+GPIO_SET);
}

/* /PERST READ port */
static unsigned int perstR[N_PCIE] = {
	PORT_UNDEF, PORT_UNDEF, PORT_UNDEF, PORT_UNDEF, PORT_UNDEF,
};
module_param_named(perstR0, perstR[0], port, S_IRUSR|S_IWUSR);
module_param_named(perstR1, perstR[1], port, S_IRUSR|S_IWUSR);
module_param_named(perstR2, perstR[2], port, S_IRUSR|S_IWUSR);
module_param_named(perstR3, perstR[3], port, S_IRUSR|S_IWUSR);
module_param_named(perstR4, perstR[4], port, S_IRUSR|S_IWUSR);

int pcie_get_perst(int ch)
{
	unsigned int port, bit;
	int val;

	if (ch < 0 || N_PCIE <= ch)
		return -1;
	if (PORT_UNDEF == perstR[ch])
		return -1;
	port = perstR[ch] & 0xff;
	bit  = (perstR[ch] >> 8) & 0xff;
	val = (readl_relaxed(VA_GPIO(port)+GPIO_WDATA) >> bit) & 0x1;
	return !val;
}

/* Tperst-clk */
#define PERST_DELAY 110
static unsigned int perst_delay[N_PCIE] = {
	PERST_DELAY, PERST_DELAY, PERST_DELAY, PERST_DELAY, PERST_DELAY,
};
module_param_named(perst_delay0, perst_delay[0], uint, S_IRUSR|S_IWUSR);
module_param_named(perst_delay1, perst_delay[1], uint, S_IRUSR|S_IWUSR);
module_param_named(perst_delay2, perst_delay[2], uint, S_IRUSR|S_IWUSR);
module_param_named(perst_delay3, perst_delay[3], uint, S_IRUSR|S_IWUSR);
module_param_named(perst_delay4, perst_delay[4], uint, S_IRUSR|S_IWUSR);

void pcie_perst_delay(int ch)
{
	if (ch < 0 || N_PCIE <= ch)
		return;
	if (!perst_delay[ch])
		return;
	usleep_range(perst_delay[ch], perst_delay[ch] + 10);
}


static DEFINE_MUTEX(pcie_ctrl_mutex);

void pcie_root_ctrl(int ch, int enable)
{
	char buf[64];
	int ret = 0;
	extern int dt_node_en_ctrl(char *name, bool enabled);

	scnprintf(buf, sizeof buf, "B PCI%d pcie_root_ctrl %d", ch, enable);
        BOOT_TIME_ADD1(buf);
        mutex_lock(&pcie_ctrl_mutex);

	if (ch < 0 || N_PCIE <= ch) {
		printk(KERN_ERR "ERROR:%s:ch=%d\n", __func__, ch);
		ret = -EINVAL;
		goto end;
	}
	ret = dt_node_en_ctrl(pcie_ch_name[ch], enable);
	if (ret) {
		printk(KERN_ERR "ERROR:%s:ch=%d,enable=%d:ret=%d\n",
		       __func__, ch, enable, ret);
		goto end;
	}

end:
	scnprintf(buf, sizeof buf, "E PCI%d pcie_root_ctrl %d", ch, ret);
	BOOT_TIME_ADD1(buf);
	mutex_unlock(&pcie_ctrl_mutex);
	return;
}
EXPORT_SYMBOL(pcie_root_ctrl);

#define PCIE_SLOT_FMT "/sys/bus/pci/slots/%d/power"

int pcie_slot_ctrl(int id, int enable)
{
#if 0
	struct file *file;
	mm_segment_t old_fs;
	char path[64];
	char cmd[1];
#endif
	int ret = 0;
	char buf[64];

	scnprintf(buf, sizeof buf, "B PCI pcie_slot_ctrl %d %d", id, enable);
	BOOT_TIME_ADD1(buf);
#if 0
	mutex_lock(&pcie_ctrl_mutex);

	scnprintf(path, sizeof path, PCIE_SLOT_FMT, id);
	cmd[0] = (enable) ? '1' : '0';

	file = filp_open(path, O_WRONLY, 0);
	if (IS_ERR(file)) {
		printk(KERN_ERR "PCI:%s:can not open %s: err=%ld\n",
		       __func__, path, PTR_ERR(file));
		ret = PTR_ERR(file);
		goto end;
	}
	old_fs = get_fs();
	set_fs(get_ds());
	ret = vfs_write(file, (char __user *)cmd, sizeof cmd, &file->f_pos);
	set_fs(old_fs);
	filp_close(file, NULL);

	if (ret < 0) {
		printk(KERN_ERR "PCI:%s:write: err=%d\n", __func__, ret);
		goto end;
	}
	if (ret != sizeof cmd) {
		printk(KERN_ERR "PCI:%s:write: ret=%d\n", __func__, ret);
		ret = -EIO;
		goto end;
	}
	ret = 0;

end:
#endif
	buf[0] = 'E';
	BOOT_TIME_ADD1(buf);
#if 0
	mutex_unlock(&pcie_ctrl_mutex);
#endif
	return ret;
}
EXPORT_SYMBOL(pcie_slot_ctrl);

static inline int pcie_link_ok(struct pci_dev *pdev)
{
	u16 lnksta;

	if (pci_read_config_word(pdev, pci_pcie_cap(pdev)+PCI_EXP_LNKSTA,
                                 &lnksta)) {
		return 0;
	}
	return !!(lnksta & PCI_EXP_LNKSTA_DLLLA);
}

int pcie_wait_datalink(int ch, unsigned int msec)
{
	char buf[64];

	scnprintf(buf, sizeof buf, "B %s %d", __func__, ch);
	BOOT_TIME_ADD1(buf);
	buf[0] = 'E';
	BOOT_TIME_ADD1(buf);
	return 0;
}
EXPORT_SYMBOL(pcie_wait_datalink);

int pcie_hot_add(int ch)
{
	char buf[64];

	scnprintf(buf, sizeof buf, "B %s %d", __func__, ch);
	BOOT_TIME_ADD1(buf);
	pcie_root_ctrl(ch, 1);
	buf[0] = 'E';
	BOOT_TIME_ADD1(buf);
        return 0;
}
EXPORT_SYMBOL(pcie_hot_add);

int pcie_hot_remove(int ch)
{
	char buf[64];

	scnprintf(buf, sizeof buf, "B %s %d", __func__, ch);
        BOOT_TIME_ADD1(buf);
	pcie_root_ctrl(ch, 0);
	buf[0] = 'E';
	BOOT_TIME_ADD1(buf);
        return 0;
}
EXPORT_SYMBOL(pcie_hot_remove);

/* Link Width, Speed */
struct pcie_link {
	const char *name;
	int (*read)(struct pci_dev *);
	int (*change)(struct pci_dev *, int, unsigned int, unsigned int);
	const int *delay_us;
	const int *sleep_us;
	const int *timeout_us;
};

static DEFINE_MUTEX(pcie_link_mutex);

static int set_link_width_delay_us   =    1; /*   1 us */
static int set_link_width_sleep_us   =   15; /*  15 us */
static int set_link_width_timeout_us = 1100; /* 1.1 ms (> 1,050,864ns@4->2, 1,050,902ns@4->1, 12,864ns@2->4, 12,862ns@1->4) */
module_param_named(set_link_width_delay_us  , set_link_width_delay_us  , int, S_IRUSR|S_IWUSR);
module_param_named(set_link_width_sleep_us  , set_link_width_sleep_us  , int, S_IRUSR|S_IWUSR);
module_param_named(set_link_width_timeout_us, set_link_width_timeout_us, int, S_IRUSR|S_IWUSR);

static int set_link_speed_delay_us   =    1; /*   1 us */
static int set_link_speed_sleep_us   =   15; /*  15 us */
static int set_link_speed_timeout_us = 1000; /*   1 ms */
module_param_named(set_link_speed_delay_us  , set_link_speed_delay_us  , int, S_IRUSR|S_IWUSR);
module_param_named(set_link_speed_sleep_us  , set_link_speed_sleep_us  , int, S_IRUSR|S_IWUSR);
module_param_named(set_link_speed_timeout_us, set_link_speed_timeout_us, int, S_IRUSR|S_IWUSR);


extern int pcie_aspm_set_domain_policy(int domain, int policy);
extern int dw_plat_pcie_lanes_dynamic_change(struct pci_dev *dev, unsigned int target_lanes);
extern int dw_plat_pcie_negotiated_link_width(struct pci_dev *dev);

static int pcie_get_link(int ch, const struct pcie_link *link)
{
	struct pci_dev *dev = NULL;
	int current_link;

	if ((ch < 0) || (N_PCIE <= ch)) {
		printk(KERN_ERR "ERROR:%s: unexpected ch: ch=%d\n", __func__, ch);
		return -EINVAL;
	}

	mutex_lock(&pcie_link_mutex);

	/* find a PCI device and increment the reference count */
	dev = pci_get_domain_bus_and_slot(ch, 0, 0);
	if (!dev) {
		printk(KERN_ERR "ERROR:%s: could not find such a device: ch=%d\n", __func__, ch);
		current_link = -ENODEV;
		goto end;
	}
	/* read current Link status */
	current_link = link->read(dev);
	if (current_link < 0) {
		printk(KERN_ERR "ERROR:%s: failed to read current Link %s : ch=%d, err=%d\n", __func__, link->name, ch, current_link);
	}

end:
	/* release a use of the pci device */
	if (dev)
		pci_dev_put(dev);

	mutex_unlock(&pcie_link_mutex);
	return current_link;
}

static int pcie_set_link(int ch, unsigned int target_link, const struct pcie_link *link)
{
	struct pci_dev *dev = NULL;
	int current_link;
	int delay_us;
	unsigned int sleep_us;
	unsigned long long timeout;
	int err = 0;

	if ((ch < 0) || (N_PCIE <= ch)) {
		printk(KERN_ERR "ERROR:%s: unexpected ch: ch=%d\n", __func__, ch);
		return -EINVAL;
	}

	mutex_lock(&pcie_link_mutex);

	/* find a PCI device and increment the reference count */
	dev = pci_get_domain_bus_and_slot(ch, 0, 0);
	if (!dev) {
		printk(KERN_ERR "ERROR:%s: could not find such a device: ch=%d\n", __func__, ch);
		err = -ENODEV;
		goto end;
	}
	/* read current Link status */
	current_link = link->read(dev);
	if (current_link < 0) {
		printk(KERN_ERR "ERROR:%s: failed to read current Link %s: ch=%d, err=%d\n", __func__, link->name, ch, current_link);
		err = current_link;
		goto end;
	}
	/* confirm Link status */
	if ((unsigned int)current_link == target_link) {
		err = 0;
		goto end; /* success */
	}
	/* change Link status */
	err = link->change(dev, ch, (unsigned int)current_link, target_link);
	if (err) {
		goto end;
	}
	delay_us = *(link->delay_us);
	if (delay_us > 0)
		udif_udelay(delay_us);

	/* wait */
	sleep_us = (unsigned int)(*(link->sleep_us));
	if (sleep_us < 1)
		sleep_us = 1;
	timeout = udif_read_freerun() + udif_usecs_to_cycles((unsigned long long)(*(link->timeout_us)));

	do {
		current_link = link->read(dev);
		if ((current_link < 0) || ((unsigned int)current_link == target_link)) {
			break;
		}
		if (udif_read_freerun() > timeout) {
			current_link = link->read(dev);
			break;
		}
		usleep_range(sleep_us, sleep_us * 2);
	} while (1);
	/* error */
	if (current_link < 0) {
		printk(KERN_ERR "ERROR:%s: failed to read current Link %s: ch=%d, err=%d\n", __func__, link->name, ch, current_link);
		err = current_link;
		goto end;
	}
	/* timeout */
	if ((unsigned int)current_link != target_link) {
		printk(KERN_ERR "ERROR:%s: current Link %s does not match the target: ch=%d, current=%d, target=%d\n", __func__, link->name, ch, current_link, target_link);
		err = -EIO;
		goto end;
	}
	/* success */
	err = 0;

end:
	/* release a use of the pci device */
	if (dev)
		pci_dev_put(dev);

	mutex_unlock(&pcie_link_mutex);
	return err;
}

static int pcie_change_link_width(struct pci_dev *dev, int ch, unsigned int current_link_width, unsigned int target_link_width)
{
	const int aspm_policy = pcie_aspm_policy(ch);
	int err;

	/* ASPM OFF */
	if (aspm_policy != 1) {
		int aspm_err;

		pcie_aspm_ctrl_lock(ch);
		aspm_err = pcie_aspm_set_domain_policy(ch, 1);
		if (aspm_err) {
			printk(KERN_ERR "ERROR:%s: ASPM OFF: ch=%d, policy=%d, err=%d\n", __func__, ch, aspm_policy, aspm_err);
			pcie_aspm_ctrl_unlock(ch);
			return aspm_err;
		}
	}
	/* change Link Width */
	err = dw_plat_pcie_lanes_dynamic_change(dev, target_link_width);
	if (err) {
		printk(KERN_ERR "ERROR:%s: failed to change the Link Width: ch=%d, current=%d, target=%d, err=%d\n", __func__, ch, current_link_width, target_link_width, err);
	}
	/* ASPM ON */
	if (aspm_policy != 1) {
		int aspm_err = pcie_aspm_set_domain_policy(ch, aspm_policy);
		pcie_aspm_ctrl_unlock(ch);
		if (aspm_err) {
			printk(KERN_ERR "ERROR:%s: ASPM ON: ch=%d, policy=%d, err=%d\n", __func__, ch, aspm_policy, aspm_err);
			return aspm_err;
		}
	}

	return err;
}

static const struct pcie_link link_width = {
	.name = "Width",
	.read   = dw_plat_pcie_negotiated_link_width,
	.change = pcie_change_link_width,
	.delay_us   = &set_link_width_delay_us,
	.sleep_us   = &set_link_width_sleep_us,
	.timeout_us = &set_link_width_timeout_us,
};

int pcie_get_link_width(int ch)
{
	return pcie_get_link(ch, &link_width);
}
EXPORT_SYMBOL(pcie_get_link_width);

int pcie_set_link_width(int ch, unsigned int target_link_width)
{
	return pcie_set_link(ch, target_link_width, &link_width);
}
EXPORT_SYMBOL(pcie_set_link_width);

/* reads current Link Speed */
static int read_link_speed(struct pci_dev *dev)
{
	int ret = 0;
	u16 val = 0;
	u16 cls = 0;; /* Current Link Speed */

	/* read Link Speed (PCI_EXP_LNKSTA : bit[3:0] LINK_SPEED) */
	pcie_capability_read_word(dev, PCI_EXP_LNKSTA, &val);
	cls = (val & PCI_EXP_LNKSTA_CLS);
	switch (cls) {
	case PCI_EXP_LNKSTA_CLS_2_5GB:
		ret = (int)PCIE_LINK_SPEED_GEN1;
		break;
	case PCI_EXP_LNKSTA_CLS_5_0GB:
		ret = (int)PCIE_LINK_SPEED_GEN2;
		break;
	case PCI_EXP_LNKSTA_CLS_8_0GB:
		ret = (int)PCIE_LINK_SPEED_GEN3;
		break;
	default:
		ret = -EFAULT;
		break;
	}
	if (ret < 0) {
		printk(KERN_ERR "ERROR:%s:%d: unexpected Link Speed (val=0x%04X, cls=0x%04X)\n", __func__, __LINE__, val, cls);
	}

	return ret;
}

/* changes Link Speed */
static int change_link_speed(struct pci_dev *dev, int ch, unsigned int current_link_speed, unsigned int target_link_speed)
{
	int err = 0;
	u16 val = 0;
	u16 tls = 0; /* Target Link Speed */

	/* check parameters */
	switch (target_link_speed) {
	case PCIE_LINK_SPEED_GEN1:
		tls = PCI_EXP_LNKCTL2_TLS_2_5GB;
		break;
	case PCIE_LINK_SPEED_GEN2:
		tls = PCI_EXP_LNKCTL2_TLS_5_0GB;
		break;
	case PCIE_LINK_SPEED_GEN3:
		tls = PCI_EXP_LNKCTL2_TLS_8_0GB;
		break;
	default:
		err = -EINVAL;
		break;
	}
	if (err) {
		printk(KERN_ERR "ERROR:%s:%d: unexpected parameter (link_speed=%d)\n", __func__, __LINE__, target_link_speed);
		return err;
	}

	/* set TARGET_SPEED (PCI_EXP_LNKCTL2 : bit[3:0] TARGET_LINK_SPEED) */
	pcie_capability_read_word(dev, PCI_EXP_LNKCTL2, &val);
	val &= ~PCI_EXP_LNKCTL2_TLS;
	val |= tls;
	pcie_capability_write_word(dev, PCI_EXP_LNKCTL2, val);

	/* re-train link (PCI_EXP_LNKCTL : bit5 RETRAIN_LINK) */
	pcie_capability_read_word(dev, PCI_EXP_LNKCTL, &val);
	val |= PCI_EXP_LNKCTL_RL;
	pcie_capability_write_word(dev, PCI_EXP_LNKCTL, val);

	return 0;
}

/* Link Speed: parameters */
static const struct pcie_link link_speed = {
	.name = "Speed",
	.read   = read_link_speed,
	.change = change_link_speed,
	.delay_us   = &set_link_speed_delay_us,
	.sleep_us   = &set_link_speed_sleep_us,
	.timeout_us = &set_link_speed_timeout_us,
};

/* Link Speed API: set */
int pcie_set_link_speed(int ch, PCIE_LINK_SPEED target_link_speed)
{
	return pcie_set_link(ch, (unsigned int)target_link_speed, &link_speed);
}
EXPORT_SYMBOL(pcie_set_link_speed);

/* Link Speed API: get */
int pcie_get_link_speed(int ch)
{
	return pcie_get_link(ch, &link_speed);
}
EXPORT_SYMBOL(pcie_get_link_speed);

static int check_dev_state_ch0(void)
{
	const int ch = 0; /* only ch0 is supported */
	int err = 0;
	struct pci_dev *dev_rc = pci_get_domain_bus_and_slot(ch, 0, 0);
	struct pci_dev *dev_ep = pci_get_domain_bus_and_slot(ch, 1, 0);

	if (!dev_rc || !dev_ep)
		err = -ENODEV;

	if (dev_rc)
		pci_dev_put(dev_rc);

	if (dev_ep)
		pci_dev_put(dev_ep);

	return err; /* 0: ready to get/set LinkWidth, !0: not ready */
}

int pcie_link_width_check_dev_state_ch0(void)
{
	return check_dev_state_ch0();
}
EXPORT_SYMBOL(pcie_link_width_check_dev_state_ch0);

int pcie_link_speed_check_dev_state_ch0(void)
{
	return check_dev_state_ch0();
}
EXPORT_SYMBOL(pcie_link_speed_check_dev_state_ch0);

/* proc */
static ssize_t proc_pcie_write(struct file *file, const char __user *buffer,
			       size_t count, loff_t *ppos)
{
	int ch = (uintptr_t)PDE_DATA(file_inode(file));
	char arg[16];
	long en;

	if (count >= sizeof arg)
		return -EOVERFLOW;
	if (copy_from_user(arg, buffer, count))
		return -EFAULT;
	arg[count] = '\0';
	en = simple_strtol(arg, NULL, 0);

	pcie_root_ctrl(ch, en);

	return count;
}

static ssize_t proc_pcie_link_read(struct file *file, char __user *buffer,
			       size_t count, loff_t *ppos, int (*get_link)(int))
{
	int ch = (uintptr_t)PDE_DATA(file_inode(file));
	int current_link;
	char buf[4]; /* xxx\0 */
	int len;

	if (!buffer || (count < sizeof(buf)))
		return -EOVERFLOW;
	if (!ppos)
		return -EINVAL;

	if (*ppos > 0)
		return 0;

	current_link = get_link(ch);
	if (current_link < 0)
		return current_link;

	len = num_to_str(buf, sizeof(buf) - 1, current_link);
	if ((len == 0) || (sizeof(buf) - 1) < len)
		return -EOVERFLOW;

	buf[len++] = '\0';

	if (copy_to_user(buffer, buf, len))
		return -EFAULT;

	*ppos += len;
	return len;
}

static ssize_t proc_pcie_link_write(struct file *file, const char __user *buffer,
			       size_t count, loff_t *ppos, int (*set_link)(int, unsigned int))
{
	int ch = (uintptr_t)PDE_DATA(file_inode(file));
	char arg[4]; /* xxx\0 */
	unsigned int target_link;
	int err;

	if (!buffer || (count >= sizeof(arg)))
		return -EOVERFLOW;
	if (copy_from_user(arg, buffer, count))
		return -EFAULT;
	arg[count] = '\0';
	target_link = simple_strtol(arg, NULL, 0);

	err = set_link(ch, target_link);
	if (err)
		return err;

	return count;
}

static ssize_t proc_pcie_link_dev_state_read(struct file *file, char __user *buffer,
			       size_t count, loff_t *ppos, int (*check_dev_state_ch0)(void))
{
	int ch = (uintptr_t)PDE_DATA(file_inode(file));
	int err;
	const char ready[] = "READY";
	const char nodev[] = "NODEV";
	const char *buf = NULL;
	int len;

	if (!buffer || (count < sizeof(ready)) || (count < sizeof(nodev)))
		return -EOVERFLOW;
	if (!ppos)
		return -EINVAL;
	if (ch !=0) {
		printk(KERN_ERR "ERROR:%s: only ch0 is supported\n", __func__);
		return -ENODEV;
	}

	if (*ppos > 0)
		return 0;

	err = check_dev_state_ch0();
	buf = (!err) ? ready : nodev;
	len = strlen(buf) + 1;

	if (copy_to_user(buffer, buf, len))
		return -EFAULT;

	*ppos += len;
	return len;
}

static ssize_t proc_pcie_link_width_read(struct file *file, char __user *buffer,
			       size_t count, loff_t *ppos)
{
	return proc_pcie_link_read(file, buffer, count, ppos, pcie_get_link_width);
}

static ssize_t proc_pcie_link_width_write(struct file *file, const char __user *buffer,
			       size_t count, loff_t *ppos)
{
	return proc_pcie_link_write(file, buffer, count, ppos, pcie_set_link_width);
}

static ssize_t proc_pcie_link_width_dev_state_read(struct file *file, char __user *buffer,
			       size_t count, loff_t *ppos)
{
	return proc_pcie_link_dev_state_read(file, buffer, count, ppos, pcie_link_width_check_dev_state_ch0);
}

static ssize_t proc_pcie_link_speed_read(struct file *file, char __user *buffer,
			       size_t count, loff_t *ppos)
{
	return proc_pcie_link_read(file, buffer, count, ppos, pcie_get_link_speed);
}

static ssize_t proc_pcie_link_speed_write(struct file *file, const char __user *buffer,
			       size_t count, loff_t *ppos)
{
	return proc_pcie_link_write(file, buffer, count, ppos, pcie_set_link_speed);
}

static ssize_t proc_pcie_link_speed_dev_state_read(struct file *file, char __user *buffer,
			       size_t count, loff_t *ppos)
{
	return proc_pcie_link_dev_state_read(file, buffer, count, ppos, pcie_link_speed_check_dev_state_ch0);
}

static struct proc_dir_entry *proc_pcie;
#define PROC_PCIE_DIR "driver/pcie"
#define PROC_PCIE_RC "root_ctrl"
#define PROC_PCIE_LW "link_width"
#define PROC_PCIE_LWDS "link_width_dev_state"
#define PROC_PCIE_LS "link_speed"
#define PROC_PCIE_LSDS "link_speed_dev_state"

static const struct file_operations proc_fops = {
	.write = proc_pcie_write,
};

static const struct file_operations proc_link_width_fops = {
	.read  = proc_pcie_link_width_read,
	.write = proc_pcie_link_width_write,
};

static const struct file_operations proc_link_width_dev_state_fops = {
	.read  = proc_pcie_link_width_dev_state_read,
};

static const struct file_operations proc_link_speed_fops = {
	.read  = proc_pcie_link_speed_read,
	.write = proc_pcie_link_speed_write,
};

static const struct file_operations proc_link_speed_dev_state_fops = {
	.read  = proc_pcie_link_speed_dev_state_read,
};

static void pcie_proc_create(UDIF_CH ch)
{
	struct proc_dir_entry *dir, *ent;
	char name[16];

	if (!proc_pcie)
		return;
	scnprintf(name, sizeof name, "%u", ch);
	dir = proc_mkdir(name, proc_pcie);
	if (!dir)
		return;
	ent = proc_symlink(pcie_ch_name[ch], proc_pcie, name);
	ent = proc_create_data(PROC_PCIE_RC, 0, dir, &proc_fops, (void *)(uintptr_t)ch);
	ent = proc_create_data(PROC_PCIE_LW, 0, dir, &proc_link_width_fops, (void *)(uintptr_t)ch);
	ent = proc_create_data(PROC_PCIE_LS, 0, dir, &proc_link_speed_fops, (void *)(uintptr_t)ch);

	/* only ch0 is supported */
	if (ch == 0) {
		ent = proc_create_data(PROC_PCIE_LWDS, 0, dir, &proc_link_width_dev_state_fops, (void *)(uintptr_t)ch);
		ent = proc_create_data(PROC_PCIE_LSDS, 0, dir, &proc_link_speed_dev_state_fops, (void *)(uintptr_t)ch);
	}
}

static inline int srst_stat(const UDIF_DEVICE *dev, UDIF_CH ch)
{
	UDIF_VA va = dev->chs[ch].clock.dclk.set;
	UDIF_U32 bit = dev->chs[ch].clock.dclk.shift;
	UDIF_U32 val;

	if (!va)
		return 0;
	val = udif_ioread32(va - SRST_SET + SRST_DATA);
	return !!(val & BIT(bit));
}

static inline void srst_ctrl(const UDIF_DEVICE *dev, UDIF_CH ch, int assert)
{
	udif_devio_devclk(dev, ch, assert, 0);
}

static void pcie_clkreq_init(const UDIF_DEVICE *dev, UDIF_CH ch)
{
	UDIF_VA phy = udif_devio_virt(dev, ch);
	UDIF_U32 val;

	if (!phy)
		return;

	/* APB_RESET is already negate ? */
	if (!srst_stat(dev, ch))
		return;

	/* enable APB_PCLK */
	udif_devio_hclk(dev, ch, 1);

	/* negate APB_RESET */
	srst_ctrl(dev, ch, 0);
	wmb();
	udif_udelay(1);

	/* CLKREQ# HiZ */
	val = udif_ioread32(phy + GENERAL_SIGNAL_CTRL);
	val |= CLKREQ_FW;
	udif_iowrite32(val, phy + GENERAL_SIGNAL_CTRL);
	wmb();
	udif_udelay(1);

	/* disable APB_PCLK */
	udif_devio_hclk(dev, ch, 0);
}
#endif /* CONFIG_PCI */

static UDIF_ERR pcie_probe(const UDIF_DEVICE *dev, UDIF_CH ch, UDIF_VP data)
{
#ifdef CONFIG_PCI
	pcie_clkreq_init(dev, ch);
	pcie_proc_create(ch);
#endif /* CONFIG_PCI */
	return UDIF_ERR_OK;
}

static UDIF_ERR pcie_resume(const UDIF_DEVICE *dev, UDIF_CH ch, UDIF_VP data)
{
#ifdef CONFIG_PCI
	pcie_clkreq_init(dev, ch);
#endif /* CONFIG_PCI */
	return UDIF_ERR_OK;
}

static UDIF_ERR pcie_init(UDIF_VP data)
{
#ifdef CONFIG_PCI
	proc_pcie = proc_mkdir(PROC_PCIE_DIR, NULL);
#endif /* CONFIG_PCI */
	return UDIF_ERR_OK;
}

static UDIF_DRIVER_OPS pcie_ops = {
	.init		= pcie_init,
	.probe		= pcie_probe,
	.resume		= pcie_resume,
};

UDIF_IDS(pcie_devs) = {
	UDIF_ID(UDIF_ID_PCIE, UDIF_CH_MASK_DEFAULT),
};
UDIF_DEPS(pcie_deps) = {};

static UDIF_MODULE(pcie, "pciephy", "1.0", pcie_ops, pcie_devs, pcie_deps, NULL);
