/*
 * arch/arm/mach-cxd900x0/hwbp_ctrl.c
 *
 * Copyright 2016 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */
#include <linux/module.h>
#include <linux/cpu.h>
#include <linux/init.h>
#include <linux/io.h>
#include <asm/cputype.h>
#include <asm/system.h>
#include <mach/regs-dap.h>
#include <mach/regs-cp14.h>
#include <mach/hwbp.h>

#define N_CONTROLLER 2

struct hwbp_controller {
	unsigned long bitmap;
	int ini_state;
	char *start_str;
	char *stop_str;
	unsigned long start_pc;
	unsigned long stop_pc;
};
static struct hwbp_controller hwbpctl[N_CONTROLLER];

module_param_named(0_wp, hwbpctl[0].bitmap, ulongH, S_IRUSR|S_IWUSR);
module_param_named(0_init, hwbpctl[0].ini_state, int, S_IRUSR|S_IWUSR);
module_param_named(0_start_str, hwbpctl[0].start_str, charp, S_IRUSR|S_IWUSR);
module_param_named(0_stop_str, hwbpctl[0].stop_str, charp, S_IRUSR|S_IWUSR);
module_param_named(0_start_pc, hwbpctl[0].start_pc, ulongH, S_IRUSR|S_IWUSR);
module_param_named(0_stop_pc, hwbpctl[0].stop_pc, ulongH, S_IRUSR|S_IWUSR);

module_param_named(1_wp, hwbpctl[1].bitmap, ulongH, S_IRUSR|S_IWUSR);
module_param_named(1_init, hwbpctl[1].ini_state, int, S_IRUSR|S_IWUSR);
module_param_named(1_start_str, hwbpctl[1].start_str, charp, S_IRUSR|S_IWUSR);
module_param_named(1_stop_str, hwbpctl[1].stop_str, charp, S_IRUSR|S_IWUSR);
module_param_named(1_start_pc, hwbpctl[1].start_pc, ulongH, S_IRUSR|S_IWUSR);
module_param_named(1_stop_pc, hwbpctl[1].stop_pc, ulongH, S_IRUSR|S_IWUSR);

static DEFINE_RAW_SPINLOCK(hwbp_ctrl_lock);

static void bp_setup(uint slot, u32 addr)
{
	_hwbp_bp_setup(slot, addr, HWBP_BYTE4|HWBP_ACCANY|HWBP_ENABLE);
}

/* Enable/disable watch point(s) */
static void wp_ctrl(struct hwbp_controller *ctl, int enable)
{
	int i;
	unsigned long bitmap;

	for (bitmap = ctl->bitmap, i = 0; bitmap; bitmap >>= 1, i++) {
		if (bitmap & 1)
			_hwbp_wp_ctrl(i, enable);
	}
}

static int hwbp_handler(int id, unsigned long addr, unsigned int fsr,
			struct pt_regs *regs)
{
	int ret, slot;
	struct hwbp_controller *ctl;

	ret = 0;
	/* Interrupt disabled */
	raw_spin_lock(&hwbp_ctrl_lock);

	if (HWBP_TYPE_PC != id) {
		ret = 1; /* Unhandled (not a breakpoint) */
		goto skip;
	}

	slot = hwbp_find_slot(addr);
	if (slot < 0) {
		ret = 0; /* continue (spurious) */
		goto skip;
	}
	if (slot >= N_CONTROLLER) {
		ret = 1; /* Unhandled (Controller does not use this) */
		goto skip;
	}

	ctl = hwbpctl + slot; /* slot == controller# */

	if (!ctl->bitmap || (!ctl->start_pc && !ctl->stop_pc)) {
		ret = 1; /* Unhandled (Controller does not use this) */
		goto skip;
	}

	/* at first, disable this breakpoint */
	hwbp_disable(HWBP_PC(slot));

	if (addr == ctl->start_pc) {
		wp_ctrl(ctl, 1); /* Enable watch point(s) */
		if (ctl->stop_pc) {
			bp_setup(slot, ctl->stop_pc);
		}
		hwbp_update();
		ret = 0; /* continue */
	} else if (addr == ctl->stop_pc) {
		wp_ctrl(ctl, 0); /* Disable watch point(s) */
		if (ctl->start_pc) {
			bp_setup(slot, ctl->start_pc);
		}
		hwbp_update();
		ret = 0; /* continue */
	} else {
		ret = 1; /* Unhandled */
	}
skip:
	raw_spin_unlock(&hwbp_ctrl_lock);
	return ret;
}

static void hwbp_ctrl(struct hwbp_controller *ctl, int enable)
{
	unsigned long flags;

	raw_spin_lock_irqsave(&hwbp_ctrl_lock, flags);
	wp_ctrl(ctl, enable);
	hwbp_update();
	raw_spin_unlock_irqrestore(&hwbp_ctrl_lock, flags);
}

/* Called from snsc_boot_time */
void hwbp_boot_time_notify(char *comment)
{
	int i;
	struct hwbp_controller *ctl;

	if (!comment)
		return;
	for (i = 0, ctl = hwbpctl; i < N_CONTROLLER; i++, ctl++) {
		if (!ctl->bitmap)
			continue;
		if (!ctl->start_str && !ctl->stop_str)
			continue;

		if (ctl->start_str && !strncmp(comment, ctl->start_str,
					       strlen(ctl->start_str))) {
			hwbp_ctrl(ctl, 1); /* enable watchpoint(s) */
		} else if (ctl->stop_str && !strncmp(comment, ctl->stop_str,
						     strlen(ctl->stop_str))) {
			hwbp_ctrl(ctl, 0); /* disable watchpoint(s) */
		}
	}
}

static void hwbp_ctrl_setup(void)
{
	int i;
	struct hwbp_controller *ctl;

	/* initialize watchpoint and breakpoint */
	for (i = 0, ctl = hwbpctl; i < N_CONTROLLER; i++, ctl++) {
		if (!ctl->bitmap)
			continue;
		if (ctl->ini_state) {
			/* initial state is ON */
			if (ctl->stop_pc)
				bp_setup(i, ctl->stop_pc);
		} else {
			/* initial state is OFF */
			if (ctl->start_pc)
				bp_setup(i, ctl->start_pc);
		}
		/* hwbp_update() is not required */
	}
}

void __init hwbp_ctrl_init(void)
{
	if (hwbpctl[0].bitmap & hwbpctl[1].bitmap) {
		printk(KERN_ERR "ERROR:%s: bitmaps overwrap\n", __func__);
		return;
	}

	hwbp_ctrl_setup();
	hwbp_set_handler(hwbp_handler);
}
