/*
 * arch/arm/mach-cxd90014/pcie_ctl.c
 *
 * Copyright (C) 2011-2012 FUJITSU SEMICONDUCTOR LIMITED
 * Copyright 2013 Sony Corporation
 *
 * ALL RIGHTS RESERVED, COPYRIGHT (C) SOCIONEXT INC. 2015
 * LICENSED MATERIAL - PROGRAM PROPERTY OF SOCIONEXT INC.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/signal.h>
#include <linux/delay.h>
#include <asm/io.h>
#include <linux/interrupt.h>
#include <linux/snsc_boot_time.h>
#include <linux/pci.h>
#include <linux/pcieport_if.h>
#include "pcie.h"
#include "pcie_ctl.h"
#include "reg_PCIE.h"
#include "reg_WPCIE.h"
#include "pci_platform.h"
#include <mach/pcie_export.h>
#include <mach/regs-gpio.h>

static pcie_remap_reg_t     pcie_reg[PCIE_CHANNEL_MAX];


static irqreturn_t  pcie_err_interrupt      (int irq, void *dev_id);
static irqreturn_t  pcie_other_interrupt    (int irq, void *dev_id);
static void         pcie_ctl_get_reg_base   ( int channel, pcie_remap_reg_t *pcie_reg );
static void         PCIeRC_Initialize       ( int channel, pcie_remap_reg_t *pcie_reg );
static void         PCIeEP_Initialize       ( int channel, pcie_remap_reg_t *pcie_reg );
static void         PCIe_RC_Configuration   ( int channel, pcie_remap_reg_t *pcie_reg );
static void         pcie_set_bit_w          ( uint16_t data, volatile void __iomem *addr );
static void         pcie_set_bit_l          ( uint32_t data, volatile void __iomem *addr );
static void         pcie_clr_bit_l          ( uint32_t data, volatile void __iomem *addr );
static void pcie_rc_setLimitLinkSpeed(reg_pcie_rcxh_t __iomem *reg, PCIE_EPRC_UN_PCIECR_LC2R_TLS speed);
static void pcie_rc_setLinkSpeed(reg_pcie_rcxh_t __iomem *reg, PCIE_EPRC_UN_PCIECR_LC2R_TLS speed);
static unsigned int pcie_rc_getSpeedChangeStatus(reg_pcie_rcxh_t __iomem *reg);
static unsigned int pcie_rc_getCurrentLinskSpeed(reg_pcie_rcxh_t __iomem *reg);
static void pcie_rc_changeLinkSpeed(pcie_remap_reg_t *pcie_reg, PCIE_EPRC_UN_PCIECR_LC2R_TLS speed);

u32 pcie_read(volatile void __iomem *addr)
{
	return readl_relaxed(addr);
}

void pcie_writel(u32 val, volatile void __iomem *addr)
{
	writel_relaxed(val, addr);
}

void pcie_writew(u16 val, volatile void __iomem *addr)
{
	writew_relaxed(val, addr);
}

void pcie_writeb(u8 val, volatile void __iomem *addr)
{
	writeb_relaxed(val, addr);
}

/* get config reg. base address */
volatile void __iomem *pcie_ctl_root_config(int channel, int slot)
{
	if (!slot) { /* RootComplex */
		return pcie_reg[channel].pcie_base;
	}
	return NULL;
}

void pcie_ctl_cfg_io_set(int channel, uint8_t type,
						 uint8_t bus_num, uint8_t dev_num, uint8_t func_num)
{
	uint32_t __iomem *reg;
	uint32_t wdata = 0;

	reg = (uint32_t __iomem *)( pcie_reg[channel].pcie_base );
	pcie_assert_NULL( reg );

	wdata = ((type << PCIE_TLP_TYPE_BOUNDARY) | (bus_num << PCIE_BUS_BOUNDARY) |
			 (dev_num << PCIE_DEV_BOUNDARY) | (func_num << PCIE_FUNC_BOUNDARY));

	writel_relaxed(wdata, (void __iomem *)reg + PCIE_AXISTR_AXISCTID); /* AXI Slave Config Target ID(1110h) */

	return;
}

int pcie_ctl_cfg_read(int channel, volatile void __iomem *addr, u32 *val)
{
	volatile void __iomem *reg;
	u32 stat;

	reg = (volatile void __iomem *)pcie_reg[channel].pcie_base;
	pcie_assert_NULL(reg);

	/* Clear AXI Slave IO/Config Interrupt Status */
	pcie_writel(PCIE_AXISTR_AXISIOIS_ALL, reg + PCIE_AXISTR_AXISIOIS);
	wmb();

	*val = pcie_read(addr);

	/* Error Check */
	stat = pcie_read(reg + PCIE_AXISTR_AXISIOIS);
	if (stat) {
		/* Clear error status */
		pcie_writel(stat, reg + PCIE_AXISTR_AXISIOIS);
		return PCIBIOS_DEVICE_NOT_FOUND;
	}

	return 0;
}

int pcie_ctl_cfg_write(int channel, void __iomem *config_reg,
					   int where, u32 size, u32 val)
{
	volatile void __iomem *reg;
	volatile void __iomem *addr;
	u32 stat;

	reg = (volatile void __iomem *)pcie_reg[channel].pcie_base;
	pcie_assert_NULL(reg);

	/* Clear AXI Slave IO/Config Interrupt Status */
	pcie_writel(PCIE_AXISTR_AXISIOIS_ALL, reg + PCIE_AXISTR_AXISIOIS);
	wmb();

	/* Write */
	switch( size ) {
	case PCIE_LONG_ACCESS:
		addr = (volatile void __iomem *)((uint32_t __iomem *)config_reg + ( where / sizeof(uint32_t)));
		pcie_writel( (u32)val , addr );
		break;
	case PCIE_SHORT_ACCESS:
		addr = (volatile void __iomem *)((uint16_t __iomem *)config_reg + ( where / sizeof(uint16_t)));
		pcie_writew( (u16)val , addr );
		break;
	case PCIE_BYTE_ACCESS:
		addr = (volatile void __iomem *)((uint8_t __iomem *)config_reg + ( where / sizeof(uint8_t)));
		pcie_writeb( (u8)val , addr );
		break;
	default:
		PCIE_PRINTK_ERR(KERN_ERR "PCI%d: size error = (%d)\n", channel, size );
		return PCIBIOS_BAD_REGISTER_NUMBER;
	}

	/* Error Check */
	stat = pcie_read(reg + PCIE_AXISTR_AXISIOIS);
	if (stat) {
		/* Clear error status */
		pcie_writel(stat, reg + PCIE_AXISTR_AXISIOIS);
		return PCIBIOS_DEVICE_NOT_FOUND;
	}

	return PCIBIOS_SUCCESSFUL;
}

#ifdef CONFIG_PM
void pcie_ctl_suspend(int ch, uint32_t pcie_select)
{
}

void pcie_ctl_resume(int ch, uint32_t pcie_select)
{
}
#endif /* CONFIG_PM */

void pcie_ctl_start(int channel, uint32_t pcie_select)
{
	BOOT_TIME_ADD1(channel ? "B PCI1 pcie_ctl_start" : "B PCI0 pcie_ctl_start");

	if( pcie_select == PCIE_MODE_ROOT ) {
	    PCIE_PRINTK_DBG(KERN_INFO " PCIe_Initialize start... \n");
		PCIeRC_Initialize( channel, &pcie_reg[channel] );
		PCIE_PRINTK_DBG(KERN_INFO " Pcie_config_Config start... \n");
		PCIe_RC_Configuration( channel, &pcie_reg[channel] );
		PCIE_PRINTK_DBG(KERN_INFO " Pcie_config_Config finish... \n");
	} else {
		PCIE_PRINTK_DBG(KERN_INFO " PCIe_Initialize start... \n");
		PCIeEP_Initialize( channel, &pcie_reg[channel] );
	}

	BOOT_TIME_ADD1(channel ? "E PCI1 pcie_ctl_start" : "E PCI0 pcie_ctl_start");
	return;

}

void pcie_ctl_stop(int channel, uint32_t pcie_select)
{
	reg_wpcie_t __iomem     *w_reg;

	w_reg = (reg_wpcie_t __iomem *)( pcie_reg[channel].wpcie_base );
	/* Power On Reset */
	pcie_clr_bit_l(PCIE_SET_CTRL_RST_PONX, &w_reg->CONTROL.DATA);
	/* PHY Reset */
	pcie_clr_bit_l(PCIE_SET_PHY_CTRL_PHY_RSTX, &w_reg->PHY_CONTROL);
}

void pcie_ctl_init(int channel, uint32_t pcie_select)
{
	int err;

	/* Get register base address */
	pcie_ctl_get_reg_base( channel, &pcie_reg[channel] );

	if (PCIE_MODE_ROOT == pcie_select) {
		/* RootComplex */
		/* ST_ERR_INT_O(Correctable, Non-Fata, Fatal) */
		err = request_irq(IRQ_PCIE_ERR(channel), pcie_err_interrupt,
						  IRQF_DISABLED, "PCIe_COM_ERR", NULL);
		if (err) {
			PCIE_PRINTK_ERR(KERN_ERR "Failed to request interrupt IRQ: %d\n",
							IRQ_PCIE_ERR(channel));
			/* request_irq error */
			PCIE_PRINTK_ERR(KERN_ERR "BUG %s(%d)\n",__FUNCTION__,__LINE__);
			BUG();
		}
	}

	/* PCIE_ERR_OTHER */
	err = request_irq(IRQ_PCIE_OTHER(channel), pcie_other_interrupt, IRQF_DISABLED,
					  "PCIe_OTHER", NULL);
	if (err) {
		PCIE_PRINTK_ERR(KERN_ERR "Failed to request interrupt IRQ: %d\n",
						IRQ_PCIE_OTHER(channel));
		/* request_irq error */
		PCIE_PRINTK_ERR(KERN_ERR "BUG %s(%d)\n",__FUNCTION__,__LINE__);
		BUG();
	}

	return;
}

static irqreturn_t pcie_other_interrupt(int irq, void *dev_id)
{
	reg_wpcie_t __iomem *wreg;
	reg_pcie_rcxh_t __iomem *reg;
	u32 stat;
	int channel;

	channel = (irq == IRQ_PCIE_OTHER(0)) ? 0 : 1;

	wreg = (reg_wpcie_t __iomem *)( pcie_reg[channel].wpcie_base );
	pcie_assert_NULL( wreg );
	reg = (reg_pcie_rcxh_t __iomem *)( pcie_reg[channel].pcie_base );
	pcie_assert_NULL( reg );

	/* Check System Reset(PERST#) Status */
	/* Read EXTINT_STS */
	stat = pcie_read(&wreg->EXTINT_STS.DATA);
	/* clear */
	pcie_writel(stat, &wreg->EXTINT_CLR.DATA);
	wmb();
	if (stat & PCIE_EXTINT_ST_DL_UP_POS) {
		/* Enable TRS_IF */
		pcie_set_bit_l(PCIE_AXIBCSR_AXISTER_TRS_IF_EN,
					   &reg->AXIBCSR.AXISTER.DATA ); /* AXI Slave Transfer Eanble Register(884h) */
	}
	if (stat & PCIE_EXTINT_ST_GRGMX_POS) {
		if (!pcie_read(&wreg->EXTINT_MON.DATA) & PCIE_EXTINT_RST_PEX_NEG) { /* PERST==L */
			/* PHY Reset */
			pcie_clr_bit_l(PCIE_SET_PHY_CTRL_PHY_RSTX, &wreg->PHY_CONTROL);
		} else { /* PERST==H */
			/* PHY Reset */
			pcie_clr_bit_l(PCIE_SET_PHY_CTRL_PHY_RSTX, &wreg->PHY_CONTROL);
			/* Clear PHY Reset */
			pcie_set_bit_l(PCIE_SET_PHY_CTRL_PHY_RSTX, &wreg->PHY_CONTROL );
		}
	}

	/* Check AXI Master I/F Error */
	stat = pcie_read(&wreg->ST_TRM_INT_O);
	if (stat == PCIE_ST_TRM_INT_O) {
		/* Read AXI Master Error Interrupt Status2 Register (908h) */
		stat = pcie_read(&reg->AXIBCSR.AXIEIS2R.DATA);
		/* clear */
		pcie_writel(stat, &reg->AXIBCSR.AXIEIS2R.DATA);
		wmb();

		PCIE_PRINTK_ERR(KERN_ERR "PCIe%d:ERROR:AXI:status=0x%x\n", channel, stat);
		BUG();
	}

	return IRQ_HANDLED;
}

static irqreturn_t pcie_err_interrupt(int irq, void *dev_id)
{
	reg_pcie_rcxh_t __iomem *reg;
	volatile void __iomem *aer;
	uint32_t stat, src, id, multi;
	uint32_t cor = 0;
	uint32_t unc = 0;
	int channel;

	channel = (irq == IRQ_PCIE_ERR(0)) ? 0 : 1;

	reg = (reg_pcie_rcxh_t __iomem *)( pcie_reg[channel].pcie_base );
	pcie_assert_NULL( reg );

	aer = (volatile void __iomem *)reg + PCIE_RC_AER_OFFSET;
	stat = pcie_read(aer + PCI_ERR_ROOT_STATUS);
	src = pcie_read(aer + PCI_ERR_ROOT_ERR_SRC);
	/* clear */
	pcie_writel(stat, aer + PCI_ERR_ROOT_STATUS);

	if (stat & PCI_ERR_ROOT_UNCOR_RCV) {
		unc = pcie_read(aer + PCI_ERR_UNCOR_STATUS);
		/* clear */
		pcie_writel(unc, aer + PCI_ERR_UNCOR_STATUS);
	}
	if (stat & PCI_ERR_ROOT_COR_RCV) {
		cor = pcie_read(aer + PCI_ERR_COR_STATUS);
		/* clear */
		pcie_writel(cor, aer + PCI_ERR_COR_STATUS);
	}
	wmb();

	PCIE_PRINTK_ERR(KERN_ERR "PCIe%d:ERROR:AER:status=0x%x\n", channel, stat);

	if (stat & PCI_ERR_ROOT_COR_RCV) {
		multi = stat & PCI_ERR_ROOT_MULTI_COR_RCV;
		id = PCIE_RC_AER_SRC_COR(src);
		PCIE_PRINTK_ERR(KERN_ERR "PCIe%d:ERROR:Correctable%s:0x%x:id=0x%04x\n", channel, multi?"(multi)":"", cor, id);
	}
	if (stat & PCI_ERR_ROOT_UNCOR_RCV) {
		multi = stat & PCI_ERR_ROOT_MULTI_UNCOR_RCV;
		id = PCIE_RC_AER_SRC_UNC(src);
		PCIE_PRINTK_ERR(KERN_ERR "PCIe%d:ERROR:UnCorrectable%s:0x%x:id=0x%04x\n", channel, multi?"(multi)":"", unc, id);
	}

	if (stat & PCI_ERR_ROOT_NONFATAL_RCV) {
		PCIE_PRINTK_ERR(KERN_ERR "PCIe%d:ERROR:Non-Fatal\n", channel);
	}
	if (stat & PCI_ERR_ROOT_FATAL_RCV) {
		PCIE_PRINTK_ERR(KERN_ERR "PCIe%d:ERROR:Fatal\n", channel);
#if 0
		BUG();
#endif
	}

	return IRQ_HANDLED;
}

static void pcie_ctl_get_reg_base( int channel, pcie_remap_reg_t *pcie_reg )
{
	pcie_assert_channel( channel );
	pcie_assert_NULL( pcie_reg );

	if (channel == 0) {
		pcie_reg->pcie_base  = (void __iomem *)IO_ADDRESSP((unsigned long)PCIE0_LINK_REG_BASE);
		pcie_reg->wpcie_base = (void __iomem *)IO_ADDRESSP((unsigned long)WPCIE0_REG_BASE);
	} else {
		pcie_reg->pcie_base  = (void __iomem *)IO_ADDRESSP((unsigned long)PCIE1_LINK_REG_BASE);
		pcie_reg->wpcie_base = (void __iomem *)IO_ADDRESSP((unsigned long)WPCIE1_REG_BASE);
	}
}

static void    PCIeRC_Initialize( int channel, pcie_remap_reg_t *pcie_reg )
{
	reg_pcie_rcxh_t __iomem *reg;
	reg_wpcie_t __iomem     *w_reg;
	volatile void __iomem   *p;
	uint32_t            count;

	pcie_assert_NULL( pcie_reg );

	reg = (reg_pcie_rcxh_t __iomem *)( pcie_reg->pcie_base );
	pcie_assert_NULL( reg );

	w_reg = (reg_wpcie_t __iomem *)( pcie_reg->wpcie_base );
	pcie_assert_NULL( w_reg );

	p = &w_reg->CONTROL.DATA;
	/* Power On Reset */
	pcie_writel( PCIE_SET_CTRL_INIT, p );

	/* Set PRV_RE_TYPE to RootComplex */
	pcie_set_bit_l( PCIE_SET_CTRL_PRV_RE_TYPE_RC, p );                         /* +000H RootComplex */

#ifdef CONFIG_CXD900X0_FPGA
	/* Set PRV_SUPPORT_LINK_SPD to Gen1(2.5GT/s) */
	pcie_clr_bit_l( PCIE_SET_CTRL_PRV_SUPPORT_LINK_SPD, p );                   /* +000H Gen1(2.5GT/s) */
#else
	/* Set PRV_SUPPORT_LINK_SPD to Gen2(5.0GT/s) */
	pcie_set_bit_l( PCIE_SET_CTRL_PRV_SUPPORT_LINK_SPD, p );                   /* +000H Gen2(5.0GT/s) */
#endif

	/* Clear Power On Reset */
	pcie_set_bit_l( PCIE_SET_CTRL_RST_PONX, p );                               /* +000H Reset "10" */

	/* Clear PHY Reset */
	pcie_set_bit_l( PCIE_SET_PHY_CTRL_PHY_RSTX, &w_reg->PHY_CONTROL );    /* +100H Reset */

#ifdef CONFIG_CXD900X0_FPGA
	/* Wait for PHY to become stable */
	mdelay(1);
#endif

	/* Wait RootComplex Mode */
	PCIE_PRINTK_DBG(KERN_INFO " WAIT Dualrole Link \n");
	for( count = 0; count < PCIE_WAIT_MAX; count++ ) {
		if( pcie_read( p ) & PCIE_SET_CTRL_ST_RE_TYPE ) {
			break;
		}
		udelay(1);
	}
	if( count == PCIE_WAIT_MAX ) {
		PCIE_PRINTK_ERR(KERN_ERR "PCIe%d timeout_error: %d\n", channel, count );
		PCIE_PRINTK_ERR(KERN_ERR "BUG %s(%d)\n",__FUNCTION__,__LINE__);
		BUG();
	}
	PCIE_PRINTK_DBG(KERN_INFO " Dualrole Link ACTIVE \n");

	pcie_rc_setLimitLinkSpeed(reg, PCIE_EPRC_EN_TSSR_TSCRR_RCS_2_5GT);

	/* Enable MSG_INTx */
	pcie_writel( PCIE_SET_EXTINT_CLR, &w_reg->EXTINT_CLR.DATA );
	pcie_writel( PCIE_SET_EXTINT_ENB, &w_reg->EXTINT_ENB.DATA );

	return;
}

static void PCIeEP_Initialize( int channel, pcie_remap_reg_t *pcie_reg )
{
	reg_pcie_epxh_t __iomem *reg;
	reg_wpcie_t     __iomem *w_reg;
	volatile void   __iomem *p;

	pcie_assert_NULL( pcie_reg );

	reg = (reg_pcie_epxh_t __iomem *)( pcie_reg->pcie_base );
	pcie_assert_NULL( reg );

	w_reg = (reg_wpcie_t __iomem *)( pcie_reg->wpcie_base );
	pcie_assert_NULL( w_reg );

	PCIE_PRINTK_DBG(KERN_INFO " PCIeEP_Initialize start... \n");

	/* Mask/Clear Interrupt */
	pcie_writel(0x00000000, &w_reg->EXTINT_ENB.DATA);
	pcie_writel(PCIE_EXTINT_ALL, &w_reg->EXTINT_CLR.DATA);

	/* PHY Reset */
	pcie_clr_bit_l(PCIE_SET_PHY_CTRL_PHY_RSTX, &w_reg->PHY_CONTROL);

	p = &w_reg->CONTROL.DATA;
	/* Power On Reset */
	pcie_writel( PCIE_SET_CTRL_INIT, p );

#ifdef CONFIG_CXD900X0_FPGA
	/* Set PRV_SUPPORT_LINK_SPD to Gen1(2.5GT/s) */
	pcie_clr_bit_l( PCIE_SET_CTRL_PRV_SUPPORT_LINK_SPD, p );                   /* +000H Gen1(2.5GT/s) */
#else
	/* Set PRV_SUPPORT_LINK_SPD to Gen2(5.0GT/s) */
	pcie_set_bit_l( PCIE_SET_CTRL_PRV_SUPPORT_LINK_SPD, p );                   /* +000H Gen2(5.0GT/s) */
#endif

	/* Clear Power On Reset */
	pcie_set_bit_l( PCIE_SET_CTRL_RST_PONX, p );                                /* +000H Reset "10" */

#ifdef CONFIG_CXD900X0_FPGA
	/* Wait for PHY to become stable */
	mdelay(1);
#endif

	/*---------------- 4GB window ------------------*/
	/*----- BAR -----*/
	pcie_writel(0, &reg->PCIEPR.FNSBAR.DATA); /* func#0 */
	/* BAR0: 32bit,MEM,non-prefetchable,2GB */
	pcie_writel(0x80000000, &reg->PCIEPR.BA0);
	/* BAR1: 32bit,MEM,non-prefetchable,2GB */
	pcie_writel(0x80000000, &reg->PCIEPR.BA1);
	/* BAR2: unuse */
	pcie_writel(0, &reg->PCIEPR.BA2);
	/* BAR bitmap */
	pcie_writel(0x3, &reg->PCIEPR.BARE.DATA);
	/*----- AXI remap -----*/
	pcie_writel(0, &reg->AXIBCSR.FNSBARRS.DATA);
	pcie_writel(0x00000000, &reg->AXIBCSR.AXIRABAR[0]);
	pcie_writel(0x80000000, &reg->AXIBCSR.AXIRABAR[1]);

	/*--- DL write enable ---*/
	pcie_set_bit_l( PCIE_AXIBCSR_AXIBMR_DLWM, &reg->AXIBCSR.AXIBMR.DATA );
	/* Clear Interrupt Pin */
	pcie_clr_bit_l( PCIE_COMP_IP, &reg->T0CSH.ILIPMGML.DATA );
	/* Set Interrupt Pin */
	pcie_set_bit_l( PCIE_T0CSH_ILIPMGML_IP, &reg->T0CSH.ILIPMGML.DATA );
	/*--- DL write protect ---*/
	pcie_clr_bit_l( PCIE_AXIBCSR_AXIBMR_DLWM, &reg->AXIBCSR.AXIBMR.DATA );

	/* Enable interrupt */
	pcie_writel(PCIE_EXTINT_ST_GRGMX_POS|PCIE_EXTINT_ST_DL_UP_POS,
				&w_reg->EXTINT_ENB.DATA);

	/* PHY_RST is cleared in the interrupt handler. */

	PCIE_PRINTK_DBG(KERN_INFO " PCIeEP_Initialize end... \n");
	return;
}

static void  PCIe_RC_Configuration( int channel, pcie_remap_reg_t *pcie_reg )
{
	reg_pcie_rcxh_t __iomem *reg;
	uint32_t dat;
	uint32_t addr, size, ch = channel;

	pcie_assert_NULL( pcie_reg );

	reg = (reg_pcie_rcxh_t __iomem *)( pcie_reg->pcie_base );
	pcie_assert_NULL( reg );

	/*--- write enable ---*/
	pcie_set_bit_l( PCIE_AXIBCSR_AXIBMR_DLWM, &reg->AXIBCSR.AXIBMR.DATA );
	/* Set device class to PCI_CLASS_BRIDGE_PCI */
	writew_relaxed(PCI_CLASS_BRIDGE_PCI, (void __iomem *)&reg->T1CSH.CC+1);
	/* PCIe capabilities */
	dat = readw_relaxed((void __iomem *)&reg->PCIECR.CR+2);
	dat |= PCI_EXP_FLAGS_SLOT;
	writew_relaxed(dat, (void __iomem *)&reg->PCIECR.CR+2);
	/* Slot Capabilities */
	dat = (channel ? PCIE1_ROOT_PSN : PCIE0_ROOT_PSN) << PCI_EXP_SLTCAP_PSN_SHIFT;
	dat |= PCI_EXP_SLTCAP_HPC|PCI_EXP_SLTCAP_HPS;
	writel_relaxed(dat, (void __iomem *)&reg->PCIECR.SCR);
	/*--- write protect ---*/
	pcie_clr_bit_l( PCIE_AXIBCSR_AXIBMR_DLWM, &reg->AXIBCSR.AXIBMR.DATA );

#if !defined(CONFIG_PCIEAER)
	pcie_set_bit_l( PCIE_SET_RECR, &reg->AERC.RECR.DATA );              /* +12CH Root Error Command Register  */
	pcie_set_bit_w( PCIE_SET_BC,   &reg->T1CSH.BC );                    /* +03EH Root Error Command Register  */
	pcie_set_bit_l( PCIE_SET_DCSR, &reg->PCIECR.DCSR.DATA );            /* +088H Device Control Register  */
#endif /* !CONFIG_PCIEAER */

	/* Set Primary Bus Number(018h) */
	writeb_relaxed(pcie_port[channel].root_busnr, (void __iomem *)&reg->T1CSH.PBN);
	/* Set Command Register(004h) */
	writel_relaxed(PCI_COMMAND_IO|PCI_COMMAND_MEMORY|PCI_COMMAND_MASTER,
				   (void __iomem *)&reg->T1CSH.CRSR);

	/* Setup Configure space */
	if (ch == 0) {
		addr = PCIE0_CONFIG_0;
	} else {
		addr = PCIE1_CONFIG_0;
	}
	writel_relaxed(addr, (void __iomem *)reg + PCIE_AXISTR_AXISCBA); /* AXI Slave Config Base Address(1100h) */
	writel_relaxed(0, (void __iomem *)reg + PCIE_AXISTR_AXISCTID); /* AXI Slave Config Target ID(1110h) */

	/* Setup IO space */
	if (ch == 0) {
		addr = PCIE0_MEM_IO;
		size = (uint32_t)(~(PCIE0_MEM_IO_SIZE - 1));
	} else {
		addr = PCIE1_MEM_IO;
		size = (uint32_t)(~(PCIE1_MEM_IO_SIZE - 1));
	}
	writel_relaxed(addr, (void __iomem *)reg + PCIE_AXISTR_AXISIBA);
	writel_relaxed(size, (void __iomem *)reg + PCIE_AXISTR_AXISIRS);
	writel_relaxed(addr, (void __iomem *)reg + PCIE_AXISTR_AXISIRA);

	/* Setup Memory space */
	if (ch == 0) {
		addr = PCIE0_MEM_NONPREF;
		size = (uint32_t)(~(PCIE0_MEM_NONPREF_SIZE - 1));
	} else {
		addr = PCIE1_MEM_NONPREF;
		size = (uint32_t)(~(PCIE1_MEM_NONPREF_SIZE - 1));
	}
	writel_relaxed(addr, (void __iomem *)reg + PCIE_AXISTR_AXISMBA);
	writel_relaxed(size, (void __iomem *)reg + PCIE_AXISTR_AXISMRS);
	writel_relaxed(addr, (void __iomem *)reg + PCIE_AXISTR_AXISMRA);

	/* Enable AXI Slave IO/Config Interrupt */
	writel_relaxed(0, (void __iomem *)reg + PCIE_AXISTR_AXISIOIM);

	return;
}

static void pcie_rc_setLimitLinkSpeed(reg_pcie_rcxh_t __iomem *reg, PCIE_EPRC_UN_PCIECR_LC2R_TLS speed)
{
	uint16_t	wdata;

	wdata = pcie_read(&reg->PCIECR.LC2R.DATA);
	wdata &= ~PCIE_PCIECR_LC2R_TLS;			/* clear limit speed */
	wdata |= speed;					/* set limit speed */
	pcie_writew(wdata, &reg->PCIECR.LC2R.DATA);
}

static void pcie_rc_setLinkSpeed(reg_pcie_rcxh_t __iomem *reg, PCIE_EPRC_UN_PCIECR_LC2R_TLS speed)
{
	uint32_t	wdata;

	wdata = pcie_read(&reg->PCIECR.LCSR.DATA);
	wdata |= PCIE_PCIECR_LCSR_RL;		/* re-train link enable */
	wdata |= PCIE_PCIECR_LCSR_LABS;		/* clear link autonomous bandwidth status */
	wdata |= PCIE_PCIECR_LCSR_LBMS;		/* clear bandwidth manangement status */
	pcie_writel(wdata , &reg->PCIECR.LCSR.DATA);
}

static unsigned int pcie_rc_getSpeedChangeStatus(reg_pcie_rcxh_t __iomem *reg)
{
	return pcie_read(&reg->TSSR.TSSR.DATA) & PCIE_RCEP_TSSR_SCS;
}

static unsigned int pcie_rc_getCurrentLinskSpeed(reg_pcie_rcxh_t __iomem *reg)
{
	return (pcie_read(&reg->PCIECR.LCSR.DATA) & PCIE_RCEP_TSSR_CS) >> PCIE_RCEP_TSSR_CS_BIT;
}

static void pcie_rc_changeLinkSpeed(pcie_remap_reg_t *pcie_reg, PCIE_EPRC_UN_PCIECR_LC2R_TLS speed)
{
	reg_pcie_rcxh_t	__iomem *reg;
	uint32_t		count;

	pcie_assert_NULL(pcie_reg);
	reg = (reg_pcie_rcxh_t __iomem *)(pcie_reg->pcie_base);
	pcie_assert_NULL(reg);

	PCIE_PRINTK_DBG(KERN_INFO "pcie_rc_changeLinkSpeed  change rc's limit speed.\n");
	pcie_rc_setLimitLinkSpeed(reg, speed);

	if(pcie_read( &reg->TSSR.TSSR.DATA ) & PCIE_RCEP_TSSR_SCR) {
		/* link speed change available */
		;
	} else {
		/* link speed change not available */
        PCIE_PRINTK_DBG(KERN_INFO "pcie_rc_changeLinkSpeed  can not change link speed.\n");
		return;
	}

	PCIE_PRINTK_DBG(KERN_INFO "pcie_rc_changeLinkSpeed  change link speed start.\n");
	pcie_rc_setLinkSpeed(reg, speed);

	for(count=0; count<PCIE_WAIT_MAX; count++) {
		if(pcie_rc_getSpeedChangeStatus(reg) == PCIE_EPRC_EN_TSSR_TSSR_SCS_COMPLETED) {
			PCIE_PRINTK_DBG(KERN_INFO "pcie_rc_changeLinkSpeed  change link speed completed.\n");
			break;
		} else {
			udelay(1);
		}
	}
	if(count == PCIE_WAIT_MAX) {
		PCIE_PRINTK_DBG(KERN_INFO "pcie_rc_changeLinkSpeed  can not change link speed.\n");
	}

	switch(pcie_rc_getCurrentLinskSpeed(reg)) {
	case PCIE_EPRC_EN_TSSR_TSCRR_RCS_2_5GT:
		PCIE_PRINTK_DBG(KERN_INFO "pcie_rc_changeLinkSpeed  current link speed : 2.5GT/s.\n");
		break;
	case PCIE_EPRC_EN_TSSR_TSCRR_RCS_5_0GT:
		PCIE_PRINTK_DBG(KERN_INFO "pcie_rc_changeLinkSpeed  current link speed : 5.0GT/s.\n");
		break;
	default:
		PCIE_PRINTK_DBG(KERN_INFO "pcie_rc_changeLinkSpeed  current link speed : unknown.\n");
		break;
	}
}

void pcie_ctl_speed(int channel, int speed)
{
	reg_pcie_rcxh_t __iomem *reg;
	int timeout = 1000; /* msec */
	u32 stat = 0;

	reg = (reg_pcie_rcxh_t __iomem *)( pcie_reg[channel].pcie_base );
	pcie_assert_NULL( reg );

	if (!speed) {
		pcie_rc_setLimitLinkSpeed(reg, PCIE_EPRC_EN_TSSR_TSCRR_RCS_2_5GT);
		return;
	}

	/* Try Gen2 */
	pcie_rc_setLimitLinkSpeed(reg, PCIE_EPRC_UN_PCIECR_LC2R_TLS_5_0GT);
	if (!(pcie_read(&reg->TSSR.TSSR.DATA) & PCIE_RCEP_TSSR_SCR)) {
		PCIE_PRINTK_DBG(KERN_INFO "PCI%d: pcie_ctl_speed: can not change link speed.\n", channel);
		return;
	}
	pcie_rc_setLinkSpeed(reg, PCIE_EPRC_UN_PCIECR_LC2R_TLS_5_0GT);
	while (timeout > 0) {
		stat = pcie_rc_getSpeedChangeStatus(reg);
		if (stat & (PCIE_TSSR_SCS_FAIL|PCIE_TSSR_SCS_SUCCESS))
			break;
		msleep(10);
		timeout -= 10;
	}
	if (stat & PCIE_TSSR_SCS_FAIL) {
		PCIE_PRINTK_ERR(KERN_ERR "PCI%d: ERROR:pcie_ctl_speed:FAILED\n", channel);
	} else if (stat & PCIE_TSSR_SCS_SUCCESS) {
		PCIE_PRINTK_DBG(KERN_INFO "PCI%d: pcie_ctl_speed:COMPLETED\n", channel);
	} else {
		PCIE_PRINTK_ERR(KERN_ERR "PCI%d: ERROR:pcie_ctl_speed:TIMEOUT\n", channel);
	}
}

int pcie_ctl_wait_dl_active(int channel)
{
	reg_pcie_rcxh_t __iomem *reg;
	int timeout = 1000; /* msec */

	pcie_assert_channel( channel );

	reg = (reg_pcie_rcxh_t __iomem *)( pcie_reg[channel].pcie_base );
	pcie_assert_NULL( reg );

	/* wait DL_Active */
	while (timeout > 0) {
		if (pcie_read(&reg->TSSR.TSSR.DATA) & PCIE_RCEP_TSSR_DLA) {
			/* Enable ICS_IF/TRS_IF
			 *  After PCIe link down, ICS_IF/TRS_IF will automatically disable. */
			pcie_set_bit_l(PCIE_AXIBCSR_AXISTER_ICS_IF_EN | PCIE_AXIBCSR_AXISTER_TRS_IF_EN,
						   &reg->AXIBCSR.AXISTER.DATA ); /* AXI Slave Transfer Eanble Register(884h) */
			return 1;
		}
		msleep(10);
		timeout -= 10;
	}
	return 0;
}

int pcie_ctl_linkstart(int channel)
{
	reg_pcie_rcxh_t __iomem *reg;
	uint32_t count;

	pcie_assert_channel( channel );
	if (!pcie_prsnt(channel))
		return 0;

	reg = (reg_pcie_rcxh_t __iomem *)( pcie_reg[channel].pcie_base );
	pcie_assert_NULL( reg );

	/* negate PERST */
	pcie_clr_bit_l( PCIE_ARB_PCS_PERST, &reg->ARB.PCS.DATA );
	BOOT_TIME_ADD1(channel ? "PCI1:bus#1:PERST:negate" : "PCI0:bus#1:PERST:negate");

	PCIE_PRINTK_DBG(KERN_INFO "WAIT DL ACTIVE \n");

	/* wait DL_Active */
	for( count = 0; count < PCIE_WAIT_MAX; count++ ) {
		if( pcie_read( &reg->TSSR.TSSR.DATA ) & PCIE_RCEP_TSSR_DLA ) {
			/*  After PCIe link down, ICS_IF/TRS_IF will automatically disable. */
			pcie_set_bit_l(PCIE_AXIBCSR_AXISTER_ICS_IF_EN | PCIE_AXIBCSR_AXISTER_TRS_IF_EN,
						   &reg->AXIBCSR.AXISTER.DATA ); /* AXI Slave Transfer Eanble Register(884h) */
			break;
		}
		udelay(1);
	}
	if( count == PCIE_WAIT_MAX ) {
		PCIE_PRINTK_ERR(KERN_ERR "PCI%d: ERROR: PCIe: pcie_ctl_rc_wait_DLActive  TIME OUT ERROR\n", channel);
#if 0
		PCIE_PRINTK_ERR(KERN_ERR "BUG %s(%d)\n",__FUNCTION__,__LINE__);
		BUG();
#endif
		return 0;
	}

#ifndef CONFIG_CXD900X0_FPGA
	pcie_rc_changeLinkSpeed( &pcie_reg[channel], PCIE_EPRC_EN_TSSR_TSCRR_RCS_5_0GT );
#endif
	PCIE_PRINTK_DBG(KERN_INFO " Pcie_change_link_speed finish... \n");

	return 1;
}

static void pcie_set_bit_w( uint16_t data, volatile void __iomem *addr )
{
	uint16_t                wdata;

	pcie_assert_NULL( addr );

	wdata   =  readw_relaxed( addr );
	wdata   |= data;
	writew_relaxed( wdata, addr );
	return;
}

static void pcie_set_bit_l( uint32_t data, volatile void __iomem *addr )
{
	uint32_t                wdata;

	pcie_assert_NULL( addr );

	wdata   =  readl_relaxed( addr );
	wdata   |= data;
	writel_relaxed( wdata, addr );
	return;
}

static void pcie_clr_bit_l( uint32_t data, volatile void __iomem *addr )
{
	uint32_t                wdata;

	pcie_assert_NULL( addr );

	wdata   =  readl_relaxed( addr );
	wdata   &= ~data;
	writel_relaxed( wdata, addr );
	return;
}

/*---------------------------------------------------------------------------
  END
---------------------------------------------------------------------------*/
