/**
 * host.c - DesignWare USB3 DRD Controller Host Glue
 *
 * Copyright (C) 2011 Texas Instruments Incorporated - http://www.ti.com
 *
 * ALL RIGHTS RESERVED, COPYRIGHT (C) SOCIONEXT INC. 2015
 * LICENSED MATERIAL - PROGRAM PROPERTY OF SOCIONEXT INC.
 *
 * Authors: Felipe Balbi <balbi@ti.com>,
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2  of
 * the License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/platform_device.h>

#include <linux/usb/f_usb/usb_otg_control.h>

#include "core.h"

int dwc3_host_otg_probe(void *data);
int dwc3_host_otg_remove(void *data);

static struct usb_otg_module_ops dwc3_host_otg_ops = {
	.probe = dwc3_host_otg_probe,
	.remove = dwc3_host_otg_remove,
};

static int dwc3_host_plat_init(struct dwc3 *dwc)
{
	struct platform_device	*xhci;
	int ret;

	xhci = platform_device_alloc("f_usb-xhci-hcd", -1);
	if (!xhci) {
		dev_err(dwc->dev, "couldn't allocate xHCI device\n");
		ret = -ENOMEM;
		goto err0;
	}

	dma_set_coherent_mask(&xhci->dev, dwc->dev->coherent_dma_mask);

	xhci->dev.parent	= dwc->dev;
	xhci->dev.dma_mask	= dwc->dev->dma_mask;
	xhci->dev.dma_parms	= dwc->dev->dma_parms;

	ret = platform_device_add_resources(xhci, dwc->xhci_resources,
						DWC3_XHCI_RESOURCES_NUM);
	if (ret) {
		dev_err(dwc->dev, "couldn't add resources to xHCI device\n");
		goto err1;
	}

	ret = platform_device_add(xhci);
	if (ret) {
		dev_err(dwc->dev, "failed to register xHCI device\n");
		goto err1;
	}

	dwc->xhci = xhci;
	return 0;

err1:
	platform_device_put(xhci);

err0:
	return ret;

}

int dwc3_host_init(struct dwc3 *dwc)
{
	int ret = 0;

	dwc->xhci = NULL;

	dwc3_host_otg_ops.pdev = container_of((dwc->dev), struct platform_device, dev);
	dwc3_host_otg_ops.data = dwc;
	fusb_otg_host_bind(&dwc3_host_otg_ops);

	return ret;
}

void dwc3_host_exit(struct dwc3 *dwc)
{
	/*
	 * Before remove this driver,
	 * upper layer must stop host mode.
	 */
	BUG_ON(dwc->xhci);
}

int dwc3_host_otg_probe(void *data)
{
	struct dwc3 *dwc = (struct dwc3 *)data;
	int ret;

	dwc3_core_soft_reset(dwc);
	dwc3_set_mode(dwc, DWC3_GCTL_PRTCAP_HOST);
	dwc3_event_buffers_setup(dwc);

	ret = dwc3_host_plat_init(dwc);
	if (!ret)
		dwc->current_mode = USB_DR_MODE_HOST;
	else
		dwc3_event_buffers_cleanup(dwc);

	return ret;
}

int dwc3_host_otg_remove(void *data)
{
	struct dwc3 *dwc = (struct dwc3 *)data;

	platform_device_unregister(dwc->xhci);
	dwc3_event_buffers_cleanup(dwc);
	dwc->xhci = NULL;
	dwc->current_mode = USB_DR_MODE_UNKNOWN;

	return 0;
}
