/*
 * arch/arm/mach-cxd900x0/pcie.c
 *
 * Copyright (C) 2011-2012 FUJITSU SEMICONDUCTOR LIMITED
 * Copyright 2013 Sony Corporation
 *
 * ALL RIGHTS RESERVED, COPYRIGHT (C) SOCIONEXT INC. 2015
 * LICENSED MATERIAL - PROGRAM PROPERTY OF SOCIONEXT INC.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#include <linux/kernel.h>
#include <linux/pci.h>
#include <linux/pcieport_if.h>
#include <linux/proc_fs.h>
#include <asm/mach/pci.h>
#include <asm/mach/arch.h>
#include <asm/setup.h>
#include <asm/delay.h>

#include <mach/irqs.h>
#include <linux/ioport.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/spinlock.h>
#include <linux/init.h>
#include <linux/io.h>
#include <linux/snsc_boot_time.h>
#include <linux/delay.h>

#include <linux/fs.h>
#include <asm/uaccess.h>

#include <mach/hardware.h>
#include <asm/irq.h>
#include <asm/system.h>
#include <asm/mach/pci.h>
#include <mach/pcie_export.h>
#include <mach/moduleparam.h>
#include <mach/cfg.h>
#include <mach/regs-gpio.h>

#define PCI_MAX_BUSNR 255

#include "pcie.h"
#include "pcie_ctl.h"
#include "pci_platform.h"

/* Enable PCIe IP core */
static int pcie_enable[PCIE_CHANNEL_MAX] = { 0, 0 };
module_param_named(ch0_enable, pcie_enable[0], bool, S_IRUSR);
module_param_named(ch1_enable, pcie_enable[1], bool, S_IRUSR);

/* Override INTx trigger type */
static int pcie_irq_trigger[PCIE_CHANNEL_MAX][PCIE_INT_MASK+1]; /* zeroed */
#define PCIE_IRQ_TRIGGER_NONE	0 /* do not override */
#define PCIE_IRQ_TRIGGER_EDGE	1 /* edge  trigger */
#define PCIE_IRQ_TRIGGER_LEVEL	2 /* level trigger */
module_param_named(ch0_inta, pcie_irq_trigger[0][0], int, S_IRUSR);
module_param_named(ch0_intb, pcie_irq_trigger[0][1], int, S_IRUSR);
module_param_named(ch0_intc, pcie_irq_trigger[0][2], int, S_IRUSR);
module_param_named(ch0_intd, pcie_irq_trigger[0][3], int, S_IRUSR);
module_param_named(ch1_inta, pcie_irq_trigger[1][0], int, S_IRUSR);
module_param_named(ch1_intb, pcie_irq_trigger[1][1], int, S_IRUSR);
module_param_named(ch1_intc, pcie_irq_trigger[1][2], int, S_IRUSR);
module_param_named(ch1_intd, pcie_irq_trigger[1][3], int, S_IRUSR);

/* Alternate pin of PRSNT pin */
static unsigned long pcie_prsnt_port[PCIE_CHANNEL_MAX] = {
	CXD900X0_PORT_UNDEF, CXD900X0_PORT_UNDEF,
};
module_param_named(ch0_prsnt, pcie_prsnt_port[0], port, S_IRUSR|S_IWUSR);
module_param_named(ch1_prsnt, pcie_prsnt_port[1], port, S_IRUSR|S_IWUSR);

/* The polarity of the PRSNT pin */
static int pcie_prsnt_pol[PCIE_CHANNEL_MAX] = { 1, 1 };
module_param_named(ch0_prsnt_pol, pcie_prsnt_pol[0], int, S_IRUSR|S_IWUSR);
module_param_named(ch1_prsnt_pol, pcie_prsnt_pol[1], int, S_IRUSR|S_IWUSR);

/* Start mode(demand or always-on) */
static int pcie_startmode[PCIE_CHANNEL_MAX]; /* 0:demand mode, 1:always-on mode */
module_param_named(ch0_start, pcie_startmode[0], int, S_IRUSR);
module_param_named(ch1_start, pcie_startmode[1], int, S_IRUSR);

/* Role(RootComplex of Endpoint) */
static int pcie_select[PCIE_CHANNEL_MAX] = { PCIE_MODE_SELECT, PCIE_MODE_SELECT };
module_param_named(ch0_select, pcie_select[0], int, S_IRUSR);
module_param_named(ch1_select, pcie_select[1], int, S_IRUSR);

static uint pcie_pon_delay;
module_param_named(pon_delay, pcie_pon_delay, uint, S_IRUSR|S_IWUSR);
static uint pcie_cold_delay;
module_param_named(cold_delay, pcie_cold_delay, uint, S_IRUSR|S_IWUSR);

extern void cxd900x0_pcie_powersave(int channel, int save);

static void                 __pci_addr(struct pci_bus *bus,unsigned int devfn, int offset);
static int          f_pcie2_dme_read_config(struct pci_bus *bus, unsigned int devfn, int where, int size, u32 *val);
static int          f_pcie2_dme_write_config(struct pci_bus *bus, unsigned int devfn, int where, int size, u32 val);
static void         pcie_get_config_base(int channel, pcie_remap_config_t *pcie_config);
static int          __init  f_pcie2_dme_setup_resources(int channel, struct resource **resource);
static int          __init  f_pcie2_dme_setup(int nr, struct pci_sys_data *sys);
static int          f_pcie2_dme_map_irq(struct pci_dev *dev, u8 slot, u8 pin);
static struct pci_bus *    __init  f_pcie2_dme_scan_bus(int nr, struct pci_sys_data *sys);
static void         __init  f_pcie2_dme_preinit(void);

/* Config space base address */
static pcie_remap_config_t  pcie_config[PCIE_CHANNEL_MAX];

static int root_channel[PCIE_CHANNEL_MAX];

struct f_pcie_port pcie_port[PCIE_CHANNEL_MAX];

static struct pci_ops f_pcie2_dme_ops = {
	.read   = f_pcie2_dme_read_config,
	.write  = f_pcie2_dme_write_config,
};

static struct resource io_mem[] = {
	{
		.name   = "PCIe0 I/O space",
		.start  = PCIE0_MEM_IO,
		.end    = PCIE0_MEM_IO+PCIE0_MEM_IO_SIZE-1,
		.flags  = IORESOURCE_IO,
	},
	{
		.name   = "PCIe1 I/O space",
		.start  = PCIE1_MEM_IO,
		.end    = PCIE1_MEM_IO+PCIE1_MEM_IO_SIZE-1,
		.flags  = IORESOURCE_IO,
	}
};

static struct resource non_mem[] = {
	{
		.name   = "PCIe0 non-prefetchable",
		.start  = PCIE0_MEM_NONPREF,
		.end    = PCIE0_MEM_NONPREF+PCIE0_MEM_NONPREF_SIZE-1,
		.flags  = IORESOURCE_MEM,
	},
	{
		.name   = "PCIe1 non-prefetchable",
		.start  = PCIE1_MEM_NONPREF,
		.end    = PCIE1_MEM_NONPREF+PCIE1_MEM_NONPREF_SIZE-1,
		.flags  = IORESOURCE_MEM,
	}
};

static struct hw_pci pcie2_dme __initdata = {
	.swizzle        = pci_std_swizzle,
	.map_irq        = f_pcie2_dme_map_irq,
	.nr_controllers = 0, /* Number of RootComplex */
	.setup          = f_pcie2_dme_setup,
	.scan           = f_pcie2_dme_scan_bus,
	.preinit        = f_pcie2_dme_preinit,
};

static struct f_pcie_port *bus_to_port(struct pci_bus *bus)
{
	struct pci_sys_data *sys = bus->sysdata;
	return sys->private_data;
}

static void __pci_addr(struct pci_bus *bus,
		       unsigned int devfn, int offset)
{
	unsigned int busnr = bus->number;
	struct f_pcie_port *pp = bus_to_port(bus);

	/*
	 * Trap out illegal values
	 */
	if (offset > PCIE_DEV_OFFSET_MAX) {
		PCIE_PRINTK_ERR(KERN_ERR "PCI%d: offset error = (%d)\n", pp->channel, offset);
		PCIE_PRINTK_ERR(KERN_ERR "BUG %s(%d)\n",__FUNCTION__,__LINE__);
		BUG();
	}
	if (busnr > PCIE_BUS_MAX) {
		PCIE_PRINTK_ERR(KERN_ERR "PCI%d: busnr error = (%d)\n", pp->channel, busnr);
		PCIE_PRINTK_ERR(KERN_ERR "BUG %s(%d)\n",__FUNCTION__,__LINE__);
		BUG();
	}
	if (devfn > PCIE_DEVFN_MAX) {
		PCIE_PRINTK_ERR(KERN_ERR "PCI%d: devfn error = (%d)\n", pp->channel, devfn);
		PCIE_PRINTK_ERR(KERN_ERR "BUG %s(%d)\n",__FUNCTION__,__LINE__);
		BUG();
	}
	return;
}

static int f_pcie2_dme_read_config(struct pci_bus *bus, unsigned int devfn, int where,
				 int size, u32 *val)
{
	uint32_t __iomem *config_reg;
	u32 v;
	volatile void __iomem *addr;
	int slot = PCI_SLOT(devfn);
	int func = PCI_FUNC(devfn);
	struct f_pcie_port *pp = bus_to_port(bus);
	uint8_t type;
	unsigned long flags;

	__pci_addr(bus, devfn, where & ~PCIE_MASK_ACCESS);

	spin_lock_irqsave(&pp->lock, flags);

	if (pp->root_busnr == bus->number) {
		/* Virtual bus for connecting Root Complexes. */
		config_reg = (uint32_t __iomem *)pcie_ctl_root_config(pp->channel, slot);
		if (!config_reg) {
			*val = 0xFFFFFFFFU;
			spin_unlock_irqrestore(&pp->lock, flags);
			return PCIBIOS_DEVICE_NOT_FOUND;
		}
	} else {
		/* PCIe config data read */
		if (pp->root_busnr == bus->primary) {
			type = PCIE_TLP_TYPE0;
		} else {
			type = PCIE_TLP_TYPE1;
		}
		pcie_ctl_cfg_io_set(pp->channel, type, (u8) bus->number,(u8) slot,(u8) func );

		config_reg = (uint32_t __iomem *)( pcie_config[pp->channel].pcie_config_base_0 );
		pcie_assert_NULL( config_reg );
	}

	addr = (volatile void __iomem *)((uint32_t __iomem *)config_reg + ( where / sizeof(uint32_t)));
	if (pcie_ctl_cfg_read(pp->channel, addr, &v) != PCIBIOS_SUCCESSFUL) { /* ERROR ? */
		PCIE_PRINTK_DBG(KERN_INFO "PCI%d: %02x:%02x:%d: Read Config Error\n",
						pp->channel, bus->number, slot, func);
		*val = 0xFFFFFFFFU;
		spin_unlock_irqrestore(&pp->lock, flags);
		return PCIBIOS_DEVICE_NOT_FOUND;
	}
	spin_unlock_irqrestore(&pp->lock, flags);

	switch( size ) {
	case PCIE_LONG_ACCESS:
		*val = v;
		break;
	case PCIE_SHORT_ACCESS:
		*val = (v >> (PCIE_ACCESS_BOUNDARY * (where & PCIE_MASK_ACCESS))) & 0xffff;
		break;
	case PCIE_BYTE_ACCESS:
		*val = (v >> (PCIE_ACCESS_BOUNDARY * (where & PCIE_MASK_ACCESS))) & 0xff;
		break;
	default:
		PCIE_PRINTK_ERR(KERN_ERR "PCI%d: size error = (%d)\n", pp->channel, size );
		return PCIBIOS_BAD_REGISTER_NUMBER;
	}
	return PCIBIOS_SUCCESSFUL;
}

static int f_pcie2_dme_write_config(struct pci_bus *bus, unsigned int devfn, int where,
				  int size, u32 val)
{
	void __iomem *config_reg;
	int slot = PCI_SLOT(devfn);
	int func = PCI_FUNC(devfn);
	struct f_pcie_port *pp = bus_to_port(bus);
	uint8_t type;
	int ret;
	unsigned long flags;

	__pci_addr(bus, devfn, where );

	spin_lock_irqsave(&pp->lock, flags);

	/* PCIe config data write */
	if (pp->root_busnr == bus->number) {
		/* Virtual bus for connecting Root Complexes. */
		config_reg = (uint32_t __iomem *)pcie_ctl_root_config(pp->channel, slot);
		if (!config_reg) {
			spin_unlock_irqrestore(&pp->lock, flags);
			return PCIBIOS_DEVICE_NOT_FOUND;
		}
	} else {
		/* PCIe config data write */
		if (pp->root_busnr == bus->primary) {
			type = PCIE_TLP_TYPE0;
		} else {
			type = PCIE_TLP_TYPE1;
		}
		pcie_ctl_cfg_io_set(pp->channel, type, (u8) bus->number,(u8) slot,(u8) func );

		config_reg = (uint32_t __iomem *)( pcie_config[pp->channel].pcie_config_base_0 );
		pcie_assert_NULL( config_reg );
	}

	ret = pcie_ctl_cfg_write(pp->channel, config_reg, where, size, val);

	spin_unlock_irqrestore(&pp->lock, flags);

	return ret;
}

static void pcie_get_config_base( int channel, pcie_remap_config_t *pcie_config )
{
	unsigned long pcie_config_addr[] = { PCIE0_CONFIG_0, PCIE1_CONFIG_0 };

	pcie_assert_NULL( pcie_config );

	pcie_config->pcie_config_base_0   = ioremap( (unsigned long)pcie_config_addr[channel],
												 (size_t)PCIE_CONFIG_SIZE );
	if( pcie_config->pcie_config_base_0 == NULL ) {
		PCIE_PRINTK_ERR(KERN_ERR "PCI%d: pcie_config_base_0 error = (%p)\n", channel,
						pcie_config->pcie_config_base_0);
		PCIE_PRINTK_ERR(KERN_ERR "BUG %s(%d)\n",__FUNCTION__,__LINE__);
		BUG();
	}
	return;
}

static int __init f_pcie2_dme_setup_resources(int channel, struct resource **resource)
{
	int ret;

	ret = request_resource(&ioport_resource, &io_mem[channel]);
	if (ret) {
		PCIE_PRINTK_ERR(KERN_ERR "PCI%d: unable to allocate I/O "
						"memory region (%d)\n", channel, ret);
		goto _OUT;
	}
	ret = request_resource(&iomem_resource, &non_mem[channel]);
	if (ret) {
		PCIE_PRINTK_ERR(KERN_ERR "PCI%d: unable to allocate non-prefetchable "
						"memory region (%d)\n", channel, ret);
		goto _RELEASE_IO_MEM;
	}

	/*
	 * bus->resource[0] is the IO resource for this bus
	 * bus->resource[1] is the mem resource for this bus
	 * bus->resource[2] is none
	 */
	resource[0] = &io_mem[channel];
	resource[1] = &non_mem[channel];
	resource[2] = NULL;

	goto _OUT;

_RELEASE_IO_MEM:
	release_resource(&io_mem[channel]);
_OUT:
	return ret;
}

static int pcie_start(int channel)
{
	int start;

	switch (pcie_startmode[channel]) {
	case 0: /* demand mode */
		start = 0;
		break;
	case 1: /* always-on mode */
		start = 1;
		break;
	default:
		start = 0;
		break;
	}
	return start;
}

/* Status of PCIe IP core */
static int pcie2_dme_enable[PCIE_CHANNEL_MAX];

static void pcie2_dme_ctrl(int channel, int on)
{
	pcie_assert_channel(channel);
	if (on) {
		pcie2_dme_enable[channel] = 1;
		cxd900x0_pcie_powersave(channel, false);
		pcie_ctl_start(channel, pcie_select[channel]);
	} else {
		pcie2_dme_enable[channel] = 0;
		pcie_ctl_stop(channel, pcie_select[channel]);
		cxd900x0_pcie_powersave(channel, true);
	}
}

static void pcie2_dme_init(int ch)
{
	pcie2_dme_enable[ch] = 0;

	if (pcie_start(ch)) {
		/* always-on mode */
		/* We can start DME now. */
		pcie2_dme_ctrl(ch, 1);
		if (pcie_ctl_linkstart(ch)) {
			mdelay(100); /* wait after DL active */
		}
	} else {
		/* demand mode */
		pcie2_dme_ctrl(ch, 0);
	}
}

static int __init f_pcie2_dme_setup(int nr, struct pci_sys_data *sys)
{
	int ret = 0;
	int channel = nr;

	if ((nr != 0) && (nr != 1)) {
		PCIE_PRINTK_ERR(KERN_ERR "f_pcie2_dme_setup: resources... nr == 0/1??\n");
		goto _OUT;
	}

	channel = root_channel[nr];

	ret = f_pcie2_dme_setup_resources(channel, sys->resource);
	if (ret < 0) {
		PCIE_PRINTK_ERR(KERN_ERR "f_pcie2_dme_setup: resources... oops?\n");
		goto _OUT;
	}

	pcie_get_config_base( channel, &pcie_config[channel] );

	sys->private_data = &pcie_port[channel];
	pcie_port[channel].channel = channel;
	/* Set root bus number */
	pcie_port[channel].root_busnr = sys->busnr;

	ret = 1;

_OUT:
	return ret;
}

static struct pci_bus * __init f_pcie2_dme_scan_bus(int nr, struct pci_sys_data *sys)
{
	struct pci_bus *bus;
	struct f_pcie_port *pp = (struct f_pcie_port *)sys->private_data;

	BOOT_TIME_ADD1(pp->channel ? "B PCI1 f_pcie2_dme_scan_bus" : "B PCI0 f_pcie2_dme_scan_bus");
	pcie_assert_channel(pp->channel);
	if (pcie2_dme_enable[pp->channel]) {
		bus = pci_scan_bus(sys->busnr, &f_pcie2_dme_ops, sys);
	} else {
		/* Create BUS, because we can not access PCIe devices */
		bus = pci_create_bus(NULL, sys->busnr, &f_pcie2_dme_ops, sys);
	}
	BOOT_TIME_ADD1(pp->channel ? "E PCI1 f_pcie2_dme_scan_bus" : "E PCI0 f_pcie2_dme_scan_bus");
	return bus;
}

static void __init f_pcie2_dme_preinit(void)
{
	int ch;
	for (ch=0; ch<PCIE_CHANNEL_MAX; ch++) {
		if (pcie_enable[ch] && (pcie_select[ch] == PCIE_MODE_ROOT)) {
			/* Initialize RootComplex */
			pcie2_dme_init(ch);
		}
	}

	return;
}

static int f_pcie2_dme_map_irq(struct pci_dev *dev, u8 slot, u8 pin)
{
	u8 intx;
	int irq;
	struct f_pcie_port *pp = bus_to_port(dev->bus);
	pcie_assert_channel( pp->channel );

	/* slot,  pin,  irq
	 *  XX     1     PCIEx_MSG_INTA_O
	 *  XX+1   1     PCIEx_MSG_INTB_O
	 *  XX+2   1     PCIEx_MSG_INTC_O
	 *  XX+3   1     PCIEx_MSG_INTD_O
	 */
	intx = (slot + pin - 1) & PCIE_INT_MASK;
	irq = IRQ_PCIE_INTA(pp->channel) + intx;

	if (pcie_irq_trigger[pp->channel][intx]) { /* override trigger type */
		if (PCIE_IRQ_TRIGGER_EDGE == pcie_irq_trigger[pp->channel][intx]) {
			/* INTx: edge interrupt */
			irq_set_irq_type(irq, IRQ_TYPE_EDGE_RISING);
		} else {
			/* INTx: level interrupt */
			irq_set_irq_type(irq, IRQ_TYPE_LEVEL_HIGH);
		}
	}

	PCIE_PRINTK_INFO(KERN_INFO "PCI%d map irq: slot %d, pin %d, irq %d \n",
					 pp->channel, slot, pin, irq);

	return irq;
}

#ifdef CONFIG_PM
void f_pcie2_dme_suspend(void)
{
	int ch;
	for (ch=0; ch<PCIE_CHANNEL_MAX; ch++) {
		if (pcie_enable[ch]) {
			pcie_ctl_suspend(ch, pcie_select[ch]);
		}
	}
}

void f_pcie2_dme_resume(void)
{
	int ch;

	if (pcie_pon_delay) {
		mdelay(pcie_pon_delay);
	}

	for (ch=0; ch<PCIE_CHANNEL_MAX; ch++) {
		if (pcie_enable[ch]) {
			if (PCIE_MODE_ROOT != pcie_select[ch]) {
				/* Initialize Endpoint */
				pcie2_dme_ctrl(ch, 1);
			} else {
				/* Initialize RootComplex */
				pcie2_dme_init(ch);
			}
		} else { /* PCIe Channel OFF */
			/* Stop the clock of disabled channel */
			cxd900x0_pcie_powersave(ch, true);
		}
	}
}
#endif /* CONFIG_PM */

int pcie_root_port_max_busnr(struct pci_dev *bridge)
{
	struct f_pcie_port *pp = bus_to_port(bridge->bus);
	int max = PCI_MAX_BUSNR;

	pcie_assert_channel(pp->channel);

	if (!pp->channel) {
		max >>= 1;
	}
	return max;
}

int pcie_prsnt(int channel)
{
	unsigned int ch, bit, dat;
	int ret;

	if (CXD900X0_PORT_UNDEF == pcie_prsnt_port[channel]) {
		/* always present */
		return 1;
	}
	ch = pcie_prsnt_port[channel] & 0xff;
	bit = (pcie_prsnt_port[channel] >> 8) & 0xff;
	dat = readl_relaxed(VA_GPIO(ch,RDATA));
	dat = (dat >>bit) & 0x00000001;
	ret = (dat == pcie_prsnt_pol[channel]);
	return ret;
}

int pcie_root_port_prsnt(struct pcie_device *dev)
{
	int ret;
	struct f_pcie_port *pp = bus_to_port(dev->port->bus);
	pcie_assert_channel(pp->channel);

	ret = pcie_prsnt(pp->channel);
	PCIE_PRINTK_INFO(KERN_DEBUG "PCI%d: pcie_root_port_prsnt: %d\n", pp->channel, ret);

	return ret;
}

int pcie_root_port_pdc(struct pcie_device *dev)
{
	struct f_pcie_port *pp = bus_to_port(dev->port->bus);
	pcie_assert_channel(pp->channel);

	if (CXD900X0_PORT_UNDEF == pcie_prsnt_port[pp->channel]) {
		/* Return 0, if no kernel patermeter */
		return 0;
	}
	return 1;
}


static struct pcie_hook *cxd900x0_pcie_hook = NULL;

void set_pcie_hook(struct pcie_hook *hook)
{
	cxd900x0_pcie_hook = hook;
}
EXPORT_SYMBOL(set_pcie_hook);

int pcie_port_perst(struct pci_dev *bridge, int assert)
{
	u16 p2p_ctrl;
	int changed, now;
	struct f_pcie_port *pp = bus_to_port(bridge->bus);

	changed = 0;
	pci_read_config_word(bridge, PCI_BRIDGE_CONTROL, &p2p_ctrl);
	now = !!(p2p_ctrl & PCI_BRIDGE_CTL_BUS_RESET);
	if (now != assert) {
		p2p_ctrl ^= PCI_BRIDGE_CTL_BUS_RESET;
		changed = 1;
		pci_write_config_word(bridge, PCI_BRIDGE_CONTROL, p2p_ctrl);
	}

	if (PCI_EXP_TYPE_ROOT_PORT == bridge->pcie_type) {
		u32 pwr_csr;

		changed = 0; /* override */
		pci_read_config_dword(bridge, PCIE_ARB_PCS, &pwr_csr);
		now = !!(pwr_csr & PCIE_ARB_PCS_PERST);
		if (now != assert) {
			pwr_csr ^= PCIE_ARB_PCS_PERST;
			changed = 1;
			if (!assert) {
				/* setup 2.5GT */
				pcie_ctl_speed(pp->channel, 0);
			}
			pci_write_config_dword(bridge, PCIE_ARB_PCS, pwr_csr);
			if (!assert) {
				if (pcie_ctl_wait_dl_active(pp->channel)) {
#ifndef CONFIG_CXD900X0_FPGA
					/* setup 5.0GT */
					pcie_ctl_speed(pp->channel, 2);
#endif
				}
			}
		}
	}
	if (changed) {
		char buf[64];
		scnprintf(buf, sizeof buf, "PCI%d:bus#%u:PERST:%s", pp->channel,
			  bridge->subordinate->number,
			  assert ? "assert":"negate");
		BOOT_TIME_ADD1(buf);
	}
	if (cxd900x0_pcie_hook && cxd900x0_pcie_hook->perst) {
		(*cxd900x0_pcie_hook->perst)(bridge, assert, changed);
	}
	return changed;
}

void pcibios_configure_slot(struct pci_dev *dev)
{
	u8 pin, slot;
	struct pci_sys_data *sys = dev->sysdata;

	/* setup legacy IRQ */
	pci_read_config_byte(dev, PCI_INTERRUPT_PIN, &pin);
	if (pin && sys && sys->map_irq) {
		if (pin > 4)
			pin = 1;
		slot = 0;
		if (sys->swizzle)
			slot = sys->swizzle(dev, &pin);
		dev->irq = sys->map_irq(dev, slot, pin);
		pcibios_update_irq(dev, dev->irq);
	}
}

int pcibios_assign_resource(struct pci_dev *dev, int resno, int size, resource_size_t min_align)
{
	struct resource *res = dev->resource + resno;

	if (PCI_VENDOR_ID_SONY == dev->vendor
	    && 0x80000000UL == size) {
		switch (resno) {
		case 0:
			res->start = 0x00000000;
			res->end   = 0x7fffffff;
			break;
		case 1:
			res->start = 0x80000000;
			res->end   = 0xffffffff;
			break;
		}
		return 0;
	}
	return -1;
}

int arch_pcie_service_irqs(struct pci_dev *dev, int *irqs, int mask)
{
	int slot = PCI_SLOT(dev->devfn);
	struct f_pcie_port *pp = bus_to_port(dev->bus);
	pcie_assert_channel( pp->channel );

	if (PCI_EXP_TYPE_ROOT_PORT == dev->pcie_type && !slot) { /* DME */
		/* AER interrupt */
		irqs[PCIE_PORT_SERVICE_AER_SHIFT] = IRQ_PCIE_ERR(pp->channel);
		/* Hot-Plug */
		irqs[PCIE_PORT_SERVICE_HP_SHIFT] = IRQ_PCIE_EXPCAP(pp->channel);
		return 0;
	}
	return -1;
}

#ifdef CONFIG_HOTPLUG_PCI_SELECT_NOTIFICATION
static uint pcie_psn_notify[PCI_MAX_BUSNR];
static uint pcie_psn_notify_num;
module_param_array_named(notify, pcie_psn_notify, uint, &pcie_psn_notify_num, S_IRUSR|S_IWUSR);

bool pcie_use_notification(u32 psn)
{
	int i;

	for (i = 0; i < pcie_psn_notify_num; i++) {
		if (pcie_psn_notify[i] == psn)
			return true;
	}
	return false;
}
#endif /* CONFIG_HOTPLUG_PCI_SELECT_NOTIFICATION */

static ssize_t proc_pcie_write(struct file *file, const char __user *buffer,
			       unsigned long count, void *data)
{
	int ch = (int)data;
	char arg[16];
	long en;

	if (count >= sizeof arg)
		return -EOVERFLOW;
	if (copy_from_user(arg, buffer, count))
		return -EFAULT;
	arg[count] = '\0';
	en = simple_strtol(arg, NULL, 0);

	pcie_root_ctrl(ch, en);

	return count;
}

static struct proc_dir_entry *proc_pcie;
#define PROC_PCIE_DIR "driver/pcie"
#define PROC_PCIE_RC "root_ctrl"

static int __init f_pcie2_dme_init(void)
{
	int ch;
	struct proc_dir_entry *dir, *ent;
	char name[16];

	proc_pcie = proc_mkdir(PROC_PCIE_DIR, NULL);

	pcibios_min_io = PCIE0_MEM_IO;
	pcibios_min_mem = PCIE0_MEM_NONPREF;

	/* Initialize PCIe IP core */
	for (ch=0; ch<PCIE_CHANNEL_MAX; ch++) {
		if (pcie_enable[ch]) { /* PCIe Channel ON */
			dir = NULL;
			if (proc_pcie) {
				scnprintf(name, sizeof name, "%d", ch);
				dir = proc_mkdir(name, proc_pcie);
			}
			pcie_ctl_init(ch, pcie_select[ch]);
			if( pcie_select[ch] == PCIE_MODE_ROOT ) {
				PCIE_PRINTK_DBG(KERN_INFO "PCI%d:%s: RootComplex mode\n", ch, __FUNCTION__);
				root_channel[pcie2_dme.nr_controllers] = ch;
				pcie2_dme.nr_controllers++;
				if (dir) {
					ent = create_proc_entry(PROC_PCIE_RC, 0, dir);
					if (ent) {
						ent->write_proc = proc_pcie_write;
						ent->data = (void *)ch;
					}
				}
			} else {
				PCIE_PRINTK_DBG(KERN_INFO "PCI%d:%s: Endpoint Mode\n", ch, __FUNCTION__);
				pcie2_dme_ctrl(ch, 1);
			}
		} else { /* PCIe Channel OFF */
			/* Stop the clock of disabled channel */
			cxd900x0_pcie_powersave(ch, true);
		}
	}

	/* Initialize RootComplex */
	if( pcie2_dme.nr_controllers ) {
		if (pcie_cold_delay) {
			mdelay(pcie_cold_delay);
		}
		pci_common_init(&pcie2_dme);
	}

	return 0;
}

static DEFINE_MUTEX(pcie_ctrl_mutex);

void pcie_root_ctrl(int ch, int enable)
{
	struct pci_bus *b = NULL;
	char buf[64];

	scnprintf(buf, sizeof buf, "B PCI%d pcie_root_ctrl %d", ch, enable);
	BOOT_TIME_ADD1(buf);
	mutex_lock(&pcie_ctrl_mutex);

	if (enable == pcie2_dme_enable[ch]) /* nothing to do */
		goto end;

	b = pci_find_bus(0, pcie_port[ch].root_busnr);
	if (!b) {
		scnprintf(buf, sizeof buf, "BUG:PCI%d:%s: BUS#%d not found",
			  ch, __func__, pcie_port[ch].root_busnr);
		BOOT_TIME_ADD1(buf);
		printk(KERN_ERR "%s\n", buf);
		BUG();
	}

	if (enable) {
		/* Start RootComplex */
		pcie2_dme_ctrl(ch, 1);
		b->subordinate = pci_rescan_bus(b);
	} else {
		/* Stop RootComplex */
		struct pci_dev *pdev;

		pdev = pci_get_slot(b, PCI_DEVFN(0, 0));
		if (!pdev) {
			scnprintf(buf, sizeof buf, "BUG:PCI%d:%s: RC not found",
				  ch, __func__);
			BOOT_TIME_ADD1(buf);
			printk(KERN_ERR "%s\n", buf);
			BUG();
		}
		scnprintf(buf, sizeof buf, "B PCI%d: remove", ch);
		BOOT_TIME_ADD1(buf);
		pci_remove_bus_device(pdev);
		scnprintf(buf, sizeof buf, "E PCI%d: remove", ch);
		BOOT_TIME_ADD1(buf);
		 /* Stop PCIe IP core */
		pcie2_dme_ctrl(ch, 0);
	}
end:
	scnprintf(buf, sizeof buf, "E PCI%d pcie_root_ctrl %d", ch, enable);
	BOOT_TIME_ADD1(buf);
	mutex_unlock(&pcie_ctrl_mutex);
}
EXPORT_SYMBOL(pcie_root_ctrl);

#define PCIE_SLOT_FMT "/sys/bus/pci/slots/%d/power"

int pcie_slot_ctrl(int id, int enable)
{
	struct file *file;
	mm_segment_t old_fs;
	char path[64];
	char cmd[1];
	int ret = 0;
	char buf[64];

	scnprintf(buf, sizeof buf, "B PCI pcie_slot_ctrl %d %d", id, enable);
	BOOT_TIME_ADD1(buf);
	mutex_lock(&pcie_ctrl_mutex);

	scnprintf(path, sizeof path, PCIE_SLOT_FMT, id);
	cmd[0] = (enable) ? '1' : '0';

	file = filp_open(path, O_WRONLY, 0);
	if (IS_ERR(file)) {
		printk(KERN_ERR "PCI:%s:can not open %s: err=%ld\n",
		       __func__, path, PTR_ERR(file));
		ret = PTR_ERR(file);
		goto end;
	}
	old_fs = get_fs();
	set_fs(get_ds());
	ret = vfs_write(file, (char __user *)cmd, sizeof cmd, &file->f_pos);
	set_fs(old_fs);
	filp_close(file, NULL);

	if (ret < 0) {
		printk(KERN_ERR "PCI:%s:write: err=%d\n", __func__, ret);
		goto end;
	}
	if (ret != sizeof cmd) {
		printk(KERN_ERR "PCI:%s:write: ret=%d\n", __func__, ret);
		ret = -EIO;
		goto end;
	}
	ret = 0;

end:
	buf[0] = 'E';
	BOOT_TIME_ADD1(buf);
	mutex_unlock(&pcie_ctrl_mutex);
	return ret;
}
EXPORT_SYMBOL(pcie_slot_ctrl);

/*----------- PCIe Hot-Plug API w/o Presence Detect Change ----------*/
static inline int pcie_link_ok(struct pci_dev *pdev)
{
	u16 lnksta;

	if (pci_read_config_word(pdev, pci_pcie_cap(pdev)+PCI_EXP_LNKSTA,
				 &lnksta)) {
		return 0;
	}
	return !!(lnksta & PCI_EXP_LNKSTA_DLLLA);
}

int pcie_wait_datalink(int busnr, unsigned int msec)
{
	struct pci_bus *b;
	struct pci_dev *pdev;
	unsigned long timeout;
	char buf[64];
	int lnk;
	int ret = 0;

	mutex_lock(&pcie_ctrl_mutex);

	b = pci_find_bus(0, busnr);
	if (!b) {
		printk(KERN_ERR "ERROR:%s:BUS#%d not found.\n",
		       __func__, busnr);
		ret = -EINVAL;
		goto out;
	}
	pdev = b->self;

	scnprintf(buf, sizeof buf, "B %s BUS#%d", __func__, busnr);
	BOOT_TIME_ADD1(buf);
	timeout = jiffies + msecs_to_jiffies(msec);
	while (!(lnk = pcie_link_ok(pdev)) && time_before(jiffies, timeout)) {
		msleep(1);
	}
	buf[0] = 'E';
	if (!lnk) {
		ret = -ETIMEDOUT;
		scnprintf(buf, sizeof buf, "E %s BUS#%d TMO", __func__, busnr);
	}
	BOOT_TIME_ADD1(buf);

out:
	mutex_unlock(&pcie_ctrl_mutex);
	return ret;
}
EXPORT_SYMBOL(pcie_wait_datalink);

int pcie_hot_add(int busnr)
{
	struct pci_bus *b;
	struct pci_dev *dev;
	int num, fn;
	int ret = 0;
	char buf[64];

	scnprintf(buf, sizeof buf, "B pcie_hot_add %d", busnr);
	BOOT_TIME_ADD1(buf);
	mutex_lock(&pcie_ctrl_mutex);

	b = pci_find_bus(0, busnr);
	if (!b) {
		printk(KERN_ERR "ERROR:%s:BUS#%d not found.\n",
		       __func__, busnr);
		ret = -EINVAL;
		goto out;
	}

	dev = pci_get_slot(b, PCI_DEVFN(0, 0));
	if (dev) {
		printk(KERN_ERR "ERROR:%s:BUS#%d:already exists\n",
		       __func__, busnr);
		pci_dev_put(dev);
		ret = -EBUSY;
		goto out;
	}

	num = pci_scan_slot(b, PCI_DEVFN(0, 0));
	if (!num) {
		printk(KERN_ERR "ERROR:%s:BUS#%d:no device found.\n",
		       __func__, busnr);
		ret = -ENODEV;
		goto out;
	}

	pci_assign_unassigned_bridge_resources(b->self);

	for (fn = 0; fn < 8; fn++) {
		dev = pci_get_slot(b, PCI_DEVFN(0, fn));
		if (!dev)
			continue;
		pcibios_configure_slot(dev);
		pci_dev_put(dev);
	}
	pci_bus_add_devices(b);

out:
	buf[0] = 'E';
	BOOT_TIME_ADD1(buf);
	mutex_unlock(&pcie_ctrl_mutex);
	return ret;
}
EXPORT_SYMBOL(pcie_hot_add);

int pcie_hot_remove(int busnr)
{
	struct pci_bus *b;
	int fn;
	int ret = 0;
	char buf[64];

	scnprintf(buf, sizeof buf, "B pcie_hot_remove %d", busnr);
	BOOT_TIME_ADD1(buf);
	mutex_lock(&pcie_ctrl_mutex);

	b = pci_find_bus(0, busnr);
	if (!b) {
		printk(KERN_ERR "ERROR:%s:BUS#%d not found.\n",
		       __func__, busnr);
		ret = -EINVAL;
		goto out;
	}

	for (fn = 0; fn < 8; fn++) {
		struct pci_dev *dev;
		dev = pci_get_slot(b, PCI_DEVFN(0, fn));
		if (!dev)
			continue;
		pci_remove_bus_device(dev);
		pci_dev_put(dev);
	}

	if (cxd900x0_pcie_hook && cxd900x0_pcie_hook->remove) {
		(*cxd900x0_pcie_hook->remove)(b->self);
	}
out:
	buf[0] = 'E';
	BOOT_TIME_ADD1(buf);
	mutex_unlock(&pcie_ctrl_mutex);
	return ret;
}
EXPORT_SYMBOL(pcie_hot_remove);
/*-------------------------------------------------------------------*/

subsys_initcall(f_pcie2_dme_init);
/*---------------------------------------------------------------------------
  END
---------------------------------------------------------------------------*/
void (*pcie_xhci_reset)(int);
EXPORT_SYMBOL(pcie_xhci_reset);
