/*
 * xhci-f_usb.c - F_USB30DR xHCI host controller driver.
 *
 * Copyright (C) 2012 Texas Instruments Incorporated - http://www.ti.com
 *
 * ALL RIGHTS RESERVED, COPYRIGHT (C) SOCIONEXT INC. 2015
 * LICENSED MATERIAL - PROGRAM PROPERTY OF SOCIONEXT INC.
 *
 * Author: Sebastian Andrzej Siewior <bigeasy@linutronix.de>
 *
 * A lot of code borrowed from the Linux xHCI driver.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 */

#include <linux/platform_device.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/dma-mapping.h>
#include <linux/usb/f_usb/f_usb30dr_hub.h>
#include <linux/usb/f_usb/usb_otg_control.h>

#include "xhci.h"

#define F_USB30DR_PORT_TEST_MASK		(0xfU << 28U)				/**< TEST MODE MASK		    */
#define F_USB30DR_PORT_TEST_J_STATE		(1U << 28U)				/**< set TEST J 		    */
#define F_USB30DR_PORT_TEST_K_STATE		(2U << 28U)				/**< set TEST K 		    */
#define F_USB30DR_PORT_TEST_SE0_NAK		(3U << 28U)				/**< set TEST SE0 NAK		    */
#define F_USB30DR_PORT_TEST_PACKET		(4U << 28U)				/**< set TEST PACKET		    */
#define F_USB30DR_PORT_TEST_FORCE_ENABLE	(5U << 28U)				/**< set TEST FORCE ENABE	    */


static int xhci_f_usb_get_ports(struct usb_hcd *hcd, __le32 __iomem ***port_array)
{
	int max_ports;
	struct xhci_hcd	*xhci = hcd_to_xhci(hcd);

	if (hcd->speed == HCD_USB3) {
		max_ports = xhci->num_usb3_ports;
		*port_array = xhci->usb3_ports;
	} else {
		max_ports = xhci->num_usb2_ports;
		*port_array = xhci->usb2_ports;
	}

	return max_ports;
}

static int xhci_f_usb_hub_control(struct usb_hcd *hcd, u16 typeReq, u16 wValue,
		u16 wIndex, char *buf, u16 wLength)
{
	struct xhci_hcd	*xhci = hcd_to_xhci(hcd);
	int max_ports;
	unsigned long flags;
	u32 temp;
	int retval = 0;
	__le32 __iomem **port_array;

	if (typeReq != F_USB30DR_SetPortFeature && typeReq != F_USB30DR_ClearPortFeature)
		return xhci_hub_control(hcd, typeReq, wValue, wIndex, buf, wLength);

	switch (wValue) {
	case F_USB30DR_PORT_FEAT_TEST_SE0:
	case F_USB30DR_PORT_FEAT_TEST_J:
	case F_USB30DR_PORT_FEAT_TEST_K:
	case F_USB30DR_PORT_FEAT_TEST_PACKET:
	case F_USB30DR_PORT_FEAT_TEST_FORCE_ENABLE:
		/* Only USB2.0 roothub supports test mode */
		if (hcd->speed != HCD_USB2)
			return -EPIPE;

	default:
		break;
	}

	max_ports = xhci_f_usb_get_ports(hcd, &port_array);

	if (!wIndex || wIndex > max_ports)
		return -EPIPE;

	wIndex--;

	spin_lock_irqsave(&xhci->lock, flags);
	switch (typeReq) {
	case F_USB30DR_ClearPortFeature:
		switch (wValue) {
		case F_USB30DR_PORT_FEAT_TEST_SUSPEND:
			/* port resume for Host Resume Timing Test */
			xhci_set_link_state(xhci, port_array, wIndex, XDEV_U0);
			break;
		case F_USB30DR_PORT_FEAT_EP_NO_DMA:
		case F_USB30DR_PORT_FEAT_TEST_IN:
		case F_USB30DR_PORT_FEAT_TEST_SETUP:
			break;
		case F_USB30DR_PORT_FEAT_TEST_SE0:
		case F_USB30DR_PORT_FEAT_TEST_J:
		case F_USB30DR_PORT_FEAT_TEST_K:
		case F_USB30DR_PORT_FEAT_TEST_PACKET:
		case F_USB30DR_PORT_FEAT_TEST_FORCE_ENABLE:
			/* clear test feature */
			temp = readl(port_array[wIndex] + PORTPMSC);
			temp &= ~F_USB30DR_PORT_TEST_MASK;
			writel(temp, port_array[wIndex] + PORTPMSC);
			break;
		default:
			goto error;
		}
		break;
	case F_USB30DR_SetPortFeature:
		temp = readl(port_array[wIndex] + PORTPMSC);
		switch (wValue) {
		case F_USB30DR_PORT_FEAT_EP_NO_DMA:
			break;
		case F_USB30DR_PORT_FEAT_TEST_SE0:
			temp = (temp & ~F_USB30DR_PORT_TEST_MASK) | F_USB30DR_PORT_TEST_SE0_NAK;
			writel(temp, port_array[wIndex] + PORTPMSC);
			break;
		case F_USB30DR_PORT_FEAT_TEST_J:
			temp = (temp & ~F_USB30DR_PORT_TEST_MASK) | F_USB30DR_PORT_TEST_J_STATE;
			writel(temp, port_array[wIndex] + PORTPMSC);
			break;
		case F_USB30DR_PORT_FEAT_TEST_K:
			temp = (temp & ~F_USB30DR_PORT_TEST_MASK) | F_USB30DR_PORT_TEST_K_STATE;
			writel(temp, port_array[wIndex] + PORTPMSC);
			break;
		case F_USB30DR_PORT_FEAT_TEST_PACKET:
			temp = (temp & ~F_USB30DR_PORT_TEST_MASK) | F_USB30DR_PORT_TEST_PACKET;
			writel(temp, port_array[wIndex] + PORTPMSC);
			break;
		case F_USB30DR_PORT_FEAT_TEST_SETUP:
			break;
		case F_USB30DR_PORT_FEAT_TEST_IN:
			break;
		case F_USB30DR_PORT_FEAT_TEST_FORCE_ENABLE:
			temp = (temp & ~F_USB30DR_PORT_TEST_MASK) | F_USB30DR_PORT_TEST_FORCE_ENABLE;
			writel(temp, port_array[wIndex] + PORTPMSC);
			break;
		case F_USB30DR_PORT_FEAT_TEST_SUSPEND:
			/* port suspend for Host Resume Timing Test */
			xhci_set_link_state(xhci, port_array, wIndex, XDEV_U3);
			break;
		default:
			goto error;
		}
		break;
	default:
error:
		/* "stall" on error */
		retval = -EPIPE;
	}
	spin_unlock_irqrestore(&xhci->lock, flags);
	return retval;
}

/*
 * Returns 0 if the status hasn't changed, or the number of bytes in buf.
 * Ports are 0-indexed from the HCD point of view,
 * and 1-indexed from the USB core pointer of view.
 *
 * Note that the status change bits will be cleared as soon as a port status
 * change event is generated, so we use the saved status from that event.
 */
static int xhci_f_usb_hub_status_data(struct usb_hcd *hcd, char *buf)
{
	unsigned long flags;
	u32 temp, status;
	u32 mask;
	int i, retval;
	struct xhci_hcd	*xhci = hcd_to_xhci(hcd);
	int max_ports;
	__le32 __iomem **port_array;
	struct xhci_bus_state *bus_state;
	bool reset_change = false;

	max_ports = xhci_f_usb_get_ports(hcd, &port_array);
	bus_state = &xhci->bus_state[hcd_index(hcd)];

	/* Initial status is no changes */
	retval = (max_ports + 8) / 8;
	memset(buf, 0, retval);

	/*
	 * Inform the usbcore about resume-in-progress by returning
	 * a non-zero value even if there are no status changes.
	 */
	status = bus_state->resuming_ports;

	mask = PORT_CSC | PORT_PEC | PORT_OCC | PORT_PLC | PORT_WRC;

	spin_lock_irqsave(&xhci->lock, flags);
	/* For each port, did anything change?  If so, set that bit in buf. */
	for (i = 0; i < max_ports; i++) {
		temp = readl(port_array[i]);
		if (temp == 0xffffffff) {
			retval = -ENODEV;
			break;
		}
		if ((temp & mask) != 0 ||
			(bus_state->port_c_suspend & 1 << i) ||
			(bus_state->resume_done[i] && time_after_eq(
			    jiffies, bus_state->resume_done[i]))) {
			buf[(i + 1) / 8] |= 1 << (i + 1) % 8;
			status = 1;
		}
		if ((temp & PORT_RC))
			reset_change = true;

		/* notify */
		if (temp & (PORT_OCC | PORT_OC)){
			fusb_otg_notify_vbuserror();
		}

		if (temp & PORT_CSC){
			if (temp & PORT_CONNECT)
				fusb_otg_notify_connect(FUSB_EVENT_CONNECT);
			else
				fusb_otg_notify_connect(FUSB_EVENT_DISCONNECT);
		}
	}
	if (!status && !reset_change) {
		xhci_dbg(xhci, "%s: stopping port polling.\n", __func__);
		clear_bit(HCD_FLAG_POLL_RH, &hcd->flags);
	}
	spin_unlock_irqrestore(&xhci->lock, flags);
	return status ? retval : 0;
}

static void xhci_f_usb_quirks(struct device *dev, struct xhci_hcd *xhci)
{
	/*
	 * As of now platform drivers don't provide MSI support so we ensure
	 * here that the generic code does not try to make a pci_dev from our
	 * dev struct in order to setup MSI
	 */
	xhci->quirks |= XHCI_PLAT;
}

/* called during probe() after chip reset completes */
static int xhci_f_usb_setup(struct usb_hcd *hcd)
{
	return xhci_gen_setup(hcd, xhci_f_usb_quirks);
}

static const struct hc_driver xhci_f_usb_xhci_driver = {
	.description =		"f_usb-xhci-hcd",
	.product_desc =		"F_USB30DR xHCI Host Controller",
	.hcd_priv_size =	sizeof(struct xhci_hcd *),

	/*
	 * generic hardware linkage
	 */
	.irq =			xhci_irq,
	.flags =		HCD_MEMORY | HCD_USB3 | HCD_SHARED,

	/*
	 * basic lifecycle operations
	 */
	.reset =		xhci_f_usb_setup,
	.start =		xhci_run,
	.stop =			xhci_stop,
	.shutdown =		xhci_shutdown,

	/*
	 * managing i/o requests and associated device resources
	 */
	.urb_enqueue =		xhci_urb_enqueue,
	.urb_dequeue =		xhci_urb_dequeue,
	.alloc_dev =		xhci_alloc_dev,
	.free_dev =		xhci_free_dev,
	.alloc_streams =	xhci_alloc_streams,
	.free_streams =		xhci_free_streams,
	.add_endpoint =		xhci_add_endpoint,
	.drop_endpoint =	xhci_drop_endpoint,
	.endpoint_reset =	xhci_endpoint_reset,
	.check_bandwidth =	xhci_check_bandwidth,
	.reset_bandwidth =	xhci_reset_bandwidth,
	.address_device =	xhci_address_device,
	.enable_device =	xhci_enable_device,
	.update_hub_device =	xhci_update_hub_device,
	.reset_device =		xhci_discover_or_reset_device,

	/*
	 * scheduling support
	 */
	.get_frame_number =	xhci_get_frame,

	/* Root hub support */
	.hub_control =		xhci_f_usb_hub_control,
	.hub_status_data =	xhci_f_usb_hub_status_data,
	.bus_suspend =		xhci_bus_suspend,
	.bus_resume =		xhci_bus_resume,
};

static int xhci_f_usb_probe(struct platform_device *pdev)
{
	const struct hc_driver	*driver;
	struct xhci_hcd		*xhci;
	struct resource         *res;
	struct usb_hcd		*hcd;
	int			ret;
	int			irq;

	if (usb_disabled())
		return -ENODEV;

	driver = &xhci_f_usb_xhci_driver;

	irq = platform_get_irq(pdev, 0);
	if (irq < 0)
		return -ENODEV;

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res)
		return -ENODEV;

	/* Initialize dma_mask and coherent_dma_mask to 32-bits */
	ret = dma_set_coherent_mask(&pdev->dev, DMA_BIT_MASK(32));
	if (ret)
		return ret;
	if (!pdev->dev.dma_mask)
		pdev->dev.dma_mask = &pdev->dev.coherent_dma_mask;
	else
		dma_set_mask(&pdev->dev, DMA_BIT_MASK(32));

	hcd = usb_create_hcd(driver, &pdev->dev, dev_name(&pdev->dev));
	if (!hcd)
		return -ENOMEM;

	hcd->rsrc_start = res->start;
	hcd->rsrc_len = resource_size(res);

	if (!request_mem_region(hcd->rsrc_start, hcd->rsrc_len,
				driver->description)) {
		dev_dbg(&pdev->dev, "controller already in use\n");
		ret = -EBUSY;
		goto put_hcd;
	}

	hcd->regs = ioremap_nocache(hcd->rsrc_start, hcd->rsrc_len);
	if (!hcd->regs) {
		dev_dbg(&pdev->dev, "error mapping memory\n");
		ret = -EFAULT;
		goto release_mem_region;
	}

	ret = usb_add_hcd(hcd, irq, IRQF_SHARED);
	if (ret)
		goto unmap_registers;
	device_wakeup_enable(hcd->self.controller);

	/* USB 2.0 roothub is stored in the platform_device now. */
	hcd = platform_get_drvdata(pdev);
	xhci = hcd_to_xhci(hcd);
	xhci->shared_hcd = usb_create_shared_hcd(driver, &pdev->dev,
			dev_name(&pdev->dev), hcd);
	if (!xhci->shared_hcd) {
		ret = -ENOMEM;
		goto dealloc_usb2_hcd;
	}

	/*
	 * Set the xHCI pointer before xhci_plat_setup() (aka hcd_driver.reset)
	 * is called by usb_add_hcd().
	 */
	*((struct xhci_hcd **) xhci->shared_hcd->hcd_priv) = xhci;

	ret = usb_add_hcd(xhci->shared_hcd, irq, IRQF_SHARED);
	if (ret)
		goto put_usb3_hcd;

	return 0;

put_usb3_hcd:
	usb_put_hcd(xhci->shared_hcd);

dealloc_usb2_hcd:
	usb_remove_hcd(hcd);

unmap_registers:
	iounmap(hcd->regs);

release_mem_region:
	release_mem_region(hcd->rsrc_start, hcd->rsrc_len);

put_hcd:
	usb_put_hcd(hcd);

	return ret;
}

static int xhci_f_usb_remove(struct platform_device *dev)
{
	struct usb_hcd	*hcd = platform_get_drvdata(dev);
	struct xhci_hcd	*xhci = hcd_to_xhci(hcd);

	usb_remove_hcd(xhci->shared_hcd);
	usb_put_hcd(xhci->shared_hcd);

	usb_remove_hcd(hcd);
	iounmap(hcd->regs);
	release_mem_region(hcd->rsrc_start, hcd->rsrc_len);
	usb_put_hcd(hcd);
	kfree(xhci);

	return 0;
}

#ifdef CONFIG_PM
static int xhci_f_usb_suspend(struct device *dev)
{
	struct usb_hcd	*hcd = dev_get_drvdata(dev);
	struct xhci_hcd	*xhci = hcd_to_xhci(hcd);

	/*
	 * xhci_suspend() needs `do_wakeup` to know whether host is allowed
	 * to do wakeup during suspend. Since xhci_plat_suspend is currently
	 * only designed for system suspend, device_may_wakeup() is enough
	 * to dertermine whether host is allowed to do wakeup. Need to
	 * reconsider this when xhci_plat_suspend enlarges its scope, e.g.,
	 * also applies to runtime suspend.
	 */
	return xhci_suspend(xhci, device_may_wakeup(dev));
}

static int xhci_f_usb_resume(struct device *dev)
{
	struct usb_hcd	*hcd = dev_get_drvdata(dev);
	struct xhci_hcd	*xhci = hcd_to_xhci(hcd);

	/* re-initialize the HC during resume */
	return xhci_resume(xhci, true);
}

static const struct dev_pm_ops xhci_f_usb_pm_ops = {
	SET_SYSTEM_SLEEP_PM_OPS(xhci_f_usb_suspend, xhci_f_usb_resume)
};
#define DEV_PM_OPS	(&xhci_f_usb_pm_ops)
#else
#define DEV_PM_OPS	NULL
#endif /* CONFIG_PM */

#ifdef CONFIG_OF
static const struct of_device_id usb_xhci_of_match[] = {
	{ .compatible = "xhci-platform" },
	{ },
};
MODULE_DEVICE_TABLE(of, usb_xhci_of_match);
#endif

static struct platform_driver f_usb_xhci_driver = {
	.probe	= xhci_f_usb_probe,
	.remove	= xhci_f_usb_remove,
	.driver	= {
		.name = "f_usb-xhci-hcd",
		.pm = DEV_PM_OPS,
#ifdef CONFIG_OF
		.of_match_table = usb_xhci_of_match,
#endif
	},
};
MODULE_ALIAS("platform:f_usb-xhci-hcd");

int xhci_register_f_usb(void)
{
	return platform_driver_register(&f_usb_xhci_driver);
}

void xhci_unregister_f_usb(void)
{
	platform_driver_unregister(&f_usb_xhci_driver);
}
