/*
 * Copyright 2018 Sony Imaging Products & Solutions Inc.
 * Copyright 2018 Sony Corporation.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

/*
 * Gated clock implementation
 */

#include <linux/clk-provider.h>
#include <linux/regmap.h>
#include <linux/slab.h>

#include "cxd-clk-define.h"
#include "cxd-clk-gate.h"

#define to_cxd_clk_gate(_hw) container_of(_hw, struct cxd_clk_gate, hw)

static void _cxd_clk_gate_enable(struct cxd_clk_gate *gate);
static int cxd_clk_gate_enable(struct clk_hw *hw)
{
        struct cxd_clk_gate *gate = to_cxd_clk_gate(hw);
        unsigned long flags = 0;

        if (gate->lock)
	{
                spin_lock_irqsave(gate->lock, flags);
		_cxd_clk_gate_enable(gate);
                spin_unlock_irqrestore(gate->lock, flags);
	} else {
		_cxd_clk_gate_enable(gate);
	}

        return 0;
}

static void _cxd_clk_gate_enable(struct cxd_clk_gate *gate)
{
        regmap_write(gate->regmap, gate->set_reg, BIT(gate->bit));

#ifdef CONFIG_ARCH_CXD900XX_FPGA
        {
                u32 reg_val;
                regmap_read(gate->regmap, gate->sta_reg, &reg_val);
                reg_val |= BIT(gate->bit);
                regmap_write(gate->regmap, gate->sta_reg, reg_val);
        }
#endif
}

static void _cxd_clk_gate_disable(struct cxd_clk_gate *gate);
static void cxd_clk_gate_disable(struct clk_hw *hw)
{
        struct cxd_clk_gate *gate = to_cxd_clk_gate(hw);
        unsigned long flags = 0;

        if (gate->lock)
	{
                spin_lock_irqsave(gate->lock, flags);
		_cxd_clk_gate_disable(gate);
                spin_unlock_irqrestore(gate->lock, flags);
	} else {
		_cxd_clk_gate_disable(gate);
	}
}

static void _cxd_clk_gate_disable(struct cxd_clk_gate *gate)
{
        regmap_write(gate->regmap, gate->clr_reg, BIT(gate->bit));

#ifdef CONFIG_ARCH_CXD900XX_FPGA
        {
                u32 reg_val;

                regmap_read(gate->regmap, gate->sta_reg, &reg_val);
                reg_val &= ~(BIT(gate->bit));
                regmap_write(gate->regmap, gate->sta_reg, reg_val);
        }
#endif
}

static int cxd_clk_gate_is_enabled(struct clk_hw *hw)
{
        struct cxd_clk_gate *gate = to_cxd_clk_gate(hw);
        unsigned int reg_val;

        regmap_read(gate->regmap, gate->sta_reg, &reg_val);

        reg_val &= BIT(gate->bit);

        return reg_val ? 1 : 0;
}

static struct clk_ops cxd_clk_gate_ops = {
        .enable = cxd_clk_gate_enable,
        .disable = cxd_clk_gate_disable,
        .is_enabled = cxd_clk_gate_is_enabled,
};

struct clk *cxd_clk_register_gate(struct device *dev, const char *name,
                const char *parent_name, unsigned long flags, struct regmap *regmap,
                u32 sta_ofs, u32 set_ofs, u32 clr_ofs,  u8 bit_idx,
                u8 clk_gate_flags, spinlock_t *lock)
{
        struct cxd_clk_gate *gate;
        struct clk *clk;
        struct clk_init_data init;

        gate = kzalloc(sizeof(struct cxd_clk_gate), GFP_KERNEL);
        if (!gate) {
                err_print("memory allocate fail\n");
                return ERR_PTR(-ENOMEM);
        }

        /* struct cxd_clk_gate assignments */
        gate->regmap = regmap;
        gate->sta_reg = sta_ofs;
        gate->set_reg = set_ofs;
        gate->clr_reg = clr_ofs;
        gate->bit = bit_idx;
        gate->flags = flags;
        gate->lock = lock;

        init.name = name;
        init.ops = &cxd_clk_gate_ops;
        init.flags = flags | CLK_IGNORE_UNUSED;
        init.parent_names = parent_name ? &parent_name : NULL;
        init.num_parents = parent_name ? 1 : 0;

        gate->hw.init = &init;

        clk = clk_register(dev, &gate->hw);
        if (IS_ERR(clk)) {
                err_print("register gate clock error\n");
                kfree(gate);
        }

        return clk;
}

