/*
 *  pcie_dma.c
 *
 *  Copyright 2018 Sony Corporation.
 *  Copyright 2019,2020 Sony Imaging Products & Solutions Inc
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */
#include "pcie_dma.h"

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/tmonitor.h>
#include "regs.h"
#include <mach/noncache.h>

#include <linux/module.h>
#if 0
static uint32_t pcidma_ch_mask = 0x00;
module_param_named(mask, pcidma_ch_mask, uint, S_IRUSR);
#endif
static uint32_t pcidma_warn_thr = PCIDMA_LATENCY_4mSEC;
module_param_named(warn_thr, pcidma_warn_thr, uint, S_IRUSR);

static int pciedma_warn_ch = 0;
static uint32_t pciedma_warn_accm_thr = PCIDMA_LATENCY_1SEC;

/*#define PCIEDMA_TMON*/

#define IRQFLAG (IRQF_NO_THREAD|IRQF_SHARED)

#define PCIDMAC_LL_ALIGN 3  /* 2^3 byte align */

#ifdef PCIEDMA_TMON
static inline void pcie_dma_tmon(char *name, long val)
{
	char s[32];
	snprintf(s, sizeof(s), "PCIeDMA:%s,%016lx", name, val);
	tmonitor_write(s);
}
#endif

void chanlog(struct dma_info_t *chan, unsigned char typ, unsigned int dat)
{
	struct chlog ent = {
		.type = typ,
		.data = dat,
	};
	log_put(&chan->log, &ent, sizeof (struct chlog));
}

static inline void pcidma_enqueue(struct dma_info_t *chan, struct pcidma_request *req)
{
	list_add_tail(&req->__drv.list, &chan->pending_list);
	chan->stat.pending_cnt++;
}

static inline struct pcidma_request *pcidma_dequeue(struct dma_info_t *chan)
{
	struct pcidma_request *next;

	if (list_empty(&chan->pending_list))
		return NULL;
	next = list_first_entry(&chan->pending_list,
				struct pcidma_request, __drv.list);
	list_del(&next->__drv.list);
	chan->stat.pending_cnt--;
	return next;
}

static inline int pcidma_llmode(struct dma_info_t *chan)
{
	return !chan->cur->size;
}

static inline int pci_is_connected(struct pci_info_t *p)
{
	return p->state;
}

static void dma_start(struct dma_info_t *chan, struct dma_reg_t *reg)
{
	struct pcidma_request *req = chan->cur;
#ifdef PCIEDMA_DEBUG
	char buf[64];
	scnprintf(buf, sizeof buf, "PCI%d:dma_start %d", chan->pci_info->id, chan->id);
	BOOT_TIME_ADD1(buf);
#endif

	pcie_writel(reg->pwr, DMA_CH_PWR_EN);
	chan->stat.dir = req->dir;
	if (pcidma_llmode(chan)) {
		/* LL mode */
		chan->stat.ll_mode = 1;
		pcie_writel(reg->ctl + DMA_LLP_L, lower_32_bits(req->src));
		pcie_writel(reg->ctl + DMA_LLP_H, upper_32_bits(req->src));
		pcie_writel(reg->ctl + DMA_CTRL1, DMA_CTRL1_LLE|DMA_CTRL1_CCS);
	} else {
		/* non-LL mode */
		chan->stat.ll_mode = 0;
		pcie_writel(reg->ctl + DMA_SIZE, req->size);
		pcie_writel(reg->ctl + DMA_CTRL1, DMA_CTRL1_LIE);
		pcie_writel(reg->ctl + DMA_SRC_L, lower_32_bits(req->src));
		pcie_writel(reg->ctl + DMA_SRC_H, upper_32_bits(req->src));
		pcie_writel(reg->ctl + DMA_DST_L, lower_32_bits(req->dst));
		pcie_writel(reg->ctl + DMA_DST_H, upper_32_bits(req->dst));
	}
	pcie_writel(reg->doorbell, chan->id);
	chan->stat.go_cnt++;
	chanlog(chan, CHLOG_DMASTART, (chan->stat.dir << 28)|(chan->stat.ll_mode << 24)|chan->state);
	pcie_wmb();
}

struct pcidma_latency_mon_t {
	int flag;
	int count;
	uint32_t ts;
	uint32_t accm;
};
static struct pcidma_latency_mon_t pcidma_latency_mon;

void pcidma_latency_mon_init(void)
{
	memset(&pcidma_latency_mon, 0, sizeof(pcidma_latency_mon));
	pcidma_latency_mon.accm = pciedma_warn_accm_thr;
	printk(KERN_INFO "pcidma_warn_thr=%u, pciedma_warn_accm_thr=%u\n", pcidma_warn_thr, pciedma_warn_accm_thr);
}

void pcidma_start(struct dma_info_t *chan)
{
	if (pcidma_busy(chan))
		return;
	if (!chan->state)
		return;
#ifdef ER9001359142
	if (!chan->ready)
		return;
#endif /* ER9001359142 */

	chan->cur = pcidma_dequeue(chan);
	if (chan->cur) {
#ifdef PCIEDMA_TMON
		pcie_dma_tmon("st", (long)(chan->cur));
#endif
		if (chan->id == pciedma_warn_ch) {
			pcidma_latency_mon.ts = hwtimer_read_clock_cpu(0);
			pcidma_latency_mon.flag = 1;
		}
		if (PCIDMA_WR == chan->cur->dir) { /* local to remote */
			dma_start(chan, &chan->dmawr);
		} else { /* remote to local */
			dma_start(chan, &chan->dmard);
		}
	}
}

static inline void pcidma_intr_enable(struct pci_info_t *p)
{
	pcie_writel(p->dmac + DMAWR_INTMASK, ~DMA_INT_ALL);
	pcie_writel(p->dmac + DMARD_INTMASK, ~DMA_INT_ALL);
}

static inline void pcidma_intr_disable(struct pci_info_t *p)
{
	pcie_writel(p->dmac + DMAWR_INTMASK, DMA_INT_ALL);
	pcie_writel(p->dmac + DMARD_INTMASK, DMA_INT_ALL);
	pcie_wmb();
}

#ifdef CONFIG_PCIEDMA_IRQ_THREAD
static irqreturn_t pcidma_intr0(int irq, void *dev_id)
{
	uint32_t stat_wr, stat_rd;
	struct pci_info_t *p = (struct pci_info_t *)dev_id;

	stat_wr = pcie_readl(p->dmac + DMAWR_INTSTAT);
	stat_rd = pcie_readl(p->dmac + DMARD_INTSTAT);
	if (!(stat_wr|stat_rd)) {
		return IRQ_NONE;
	}
	pcidma_intr_disable(p);
	return IRQ_WAKE_THREAD;
}
#endif /* CONFIG_PCIEDMA_IRQ_THREAD */

#ifdef FPU_SAVE
static inline int cb_use_fpu(struct pcidma_request *req)
{
	return (req->__drv.info & INFO_SUBSYSTEM);
}
#endif /* FPU_SAVE */

#define MAX_POLL_ISTAT 10

static void pcidma_warn_latency(struct dma_info_t *chan, int count, uint32_t elapsed, int f)
{
	char buf[28];
	snprintf(buf, sizeof(buf), "pcidma%d,%u,%d", chan->id, elapsed, count);
	if (f) BOOT_TIME_ADD1(buf);
#if !defined(PCIEDMA_TMON)
	else   tmonitor_write(buf);
#endif
}

static irqreturn_t pcidma_intr(int irq, void *dev_id)
{
	uint32_t stat, stat_wr, stat_rd, mask, stat2;
	uint32_t wr_err, rd_err[2];
	struct pci_info_t *p = (struct pci_info_t *)dev_id;
	struct dma_info_t *chan;
	irqreturn_t status = IRQ_NONE;
	int i;
	char buf[64];
#ifdef FPU_SAVE
	struct fpsimd_state fpu;
	int fpu_saved = 0;
#endif /* FPU_SAVE */

	stat_wr = pcie_readl(p->dmac + DMAWR_INTSTAT);
	stat_rd = pcie_readl(p->dmac + DMARD_INTSTAT);
	stat = stat_wr | stat_rd;
	if (!stat) {
		return IRQ_NONE;
	}
	status = IRQ_HANDLED;
 	stat2 = (stat_wr << 8) | stat_rd;
	wr_err    = pcie_readl(p->dmac + DMAWR_ERR);
	rd_err[0] = pcie_readl(p->dmac + DMARD_ERR_L);
	rd_err[1] = pcie_readl(p->dmac + DMARD_ERR_H);

	mask = BIT(DMA_INT_DONE_BIT) | BIT(DMA_INT_ABORT_BIT);
	for (i = 0, chan = p->dma_info; i < PCIDMA_CH_MAX; i++, chan++, mask <<= 1) {
		int err;
		struct pcidma_request *cur;
		uint32_t ctrl1_sav, ctrl1_tmp;
		uint32_t ctrl1, istat;
		int j;
		unsigned long flags;

		if (!(stat & mask))
			continue;

		chanlog(chan, CHLOG_DMAINTR, stat2);
		/* clear IRQ */
		if (stat_wr & mask) {
			/* clear LIE */
			ctrl1_sav = pcie_readl(chan->dmawr.ctl + DMA_CTRL1);
			ctrl1_tmp = ctrl1_sav & ~DMA_CTRL1_LIE;
			pcie_writel(chan->dmawr.ctl + DMA_CTRL1, ctrl1_tmp);
			/* clear IRQ */
			pcie_writel(p->dmac + DMAWR_INTCLR, mask);
			/* Read INTSTAT
			   The regression test is required to eliminate
			   the unnecessary loop.
			*/
			for (j = 0; j < MAX_POLL_ISTAT; j++) {
				ctrl1 = pcie_readl(chan->dmawr.ctl + DMA_CTRL1);
				istat = pcie_readl(p->dmac + DMAWR_INTSTAT);
			}
			/* retry */
			for (j = 0; j < MAX_POLL_ISTAT; j++) {
				if (!(istat & mask))
					break;
				pcie_writel(p->dmac + DMAWR_INTCLR, mask);
				ctrl1 = pcie_readl(chan->dmawr.ctl + DMA_CTRL1);
				istat = pcie_readl(p->dmac + DMAWR_INTSTAT);
			}
		} else {
			/* clear LIE */
			ctrl1_sav = pcie_readl(chan->dmard.ctl + DMA_CTRL1);
			ctrl1_tmp = ctrl1_sav & ~DMA_CTRL1_LIE;
			pcie_writel(chan->dmard.ctl + DMA_CTRL1, ctrl1_tmp);
			/* clear IRQ */
			pcie_writel(p->dmac + DMARD_INTCLR, mask);
			/* Read INTSTAT
			   The regression test is required to eliminate
			   the unnecessary loop.
			*/
			for (j = 0; j < MAX_POLL_ISTAT; j++) {
				ctrl1 = pcie_readl(chan->dmard.ctl + DMA_CTRL1);
				istat = pcie_readl(p->dmac + DMARD_INTSTAT);
			}
			/* retry */
			for (j = 0; j < MAX_POLL_ISTAT; j++) {
				if (!(istat & mask))
					break;
				pcie_writel(p->dmac + DMARD_INTCLR, mask);
				ctrl1 = pcie_readl(chan->dmard.ctl + DMA_CTRL1);
				istat = pcie_readl(p->dmac + DMARD_INTSTAT);
			}
		}
		if (istat & mask) {
			scnprintf(buf, sizeof buf, "ERR:PCI%d:DMA#%d:INTCLR:%x:%x:%x:%x", p->id, i, stat2, istat, ctrl1, mask);
			BOOT_TIME_ADD1(buf);
			log_stop();
			logdump(p);
			BUG();
		}

		err = 0;
		if (stat & mask & DMA_INT_ABORT) {
			scnprintf(buf, sizeof buf, "ERR:PCI%d:DMA#%d:abort:%x,%x,%x:",
				  p->id, i, wr_err, rd_err[0], rd_err[1]);
			BOOT_TIME_ADD1(buf);
			err = -1;
		}

		LOCK(chan);
		chan->stat.done_cnt++;
		if (err) {
			chan->stat.abort_cnt++;
		}
		cur = chan->cur;
		UNLOCK(chan);
		if (unlikely(!cur)) {
			scnprintf(buf, sizeof buf, "ERR:PCI%d:DMA#%d:cur=NULL\n",
				  p->id, i);
			BOOT_TIME_ADD1(buf);
			chanlog(chan, CHLOG_ERR1, chan->state);
			log_stop();
			logdump(p);
			BUG();
		} else if (unlikely(!cur->callback)) {
			scnprintf(buf, sizeof buf, "ERR:PCI%d:DMA#%d:cb=NULL\n",
				  p->id, i);
			BOOT_TIME_ADD1(buf);
		} else  {
#ifdef CB_TIMEOUT
			unsigned long long dt = udif_read_freerun();
#endif /* CB_TIMEOUT */
			if (err) {
				scnprintf(buf, sizeof buf, "ERR:PCI%d:DMA#%d:abort2:req=%d,%d,%d,%x,%p", p->id, i, cur->pcie_ch, cur->dmac_ch, cur->dir, cur->size, cur->callback);
				BOOT_TIME_ADD1(buf);
				scnprintf(buf, sizeof buf, "ERR:PCI%d:DMA#%d:abort3:%llx,%llx", p->id, i, cur->src, cur->dst);
				BOOT_TIME_ADD1(buf);
			}
#ifdef FPU_SAVE
			if (cb_use_fpu(cur) && !fpu_saved) {
				fpsimd_save_state(&fpu);
				fpu_saved = 1;
			}
#endif /* FPU_SAVE */
#ifdef PCIEDMA_TMON
			pcie_dma_tmon("in", (long)(chan->cur));
#endif /* PCIEDMA_TMON */
			if (pciedma_warn_ch == i && pcidma_latency_mon.flag) {
				uint32_t curts, elapsed;
				int logf;
				logf = 0;
				curts = hwtimer_read_clock_cpu(0);
				elapsed = curts - pcidma_latency_mon.ts;
				if (elapsed > pcidma_warn_thr) {
					pcidma_latency_mon.count ++;
					pcidma_latency_mon.accm += elapsed;
					if (pcidma_latency_mon.accm > pciedma_warn_accm_thr) {
						logf = 1;
						pcidma_latency_mon.accm -= pciedma_warn_accm_thr;
					}
					pcidma_warn_latency(chan, pcidma_latency_mon.count, elapsed, logf);
				}
				pcidma_latency_mon.flag = 0;
			}
			(*cur->callback)(err, cur->cookies);
#ifdef CB_TIMEOUT
			dt = udif_read_freerun() - dt;
			dt = udif_cycles_to_usecs(dt);
			if (dt >= CB_TIMEOUT) {
				scnprintf(buf, sizeof buf, "ERR:PCI%d:DMA#%d:cb=%p:dt=%llu", p->id, i, cur->callback, dt);
				BOOT_TIME_ADD1(buf);
				tmonitor_dumper_wakeup();
			}
#endif /* CB_TIMEOUT */
		}

#ifdef ER9001359142
		aspmctl_relinquish(chan);
#endif

		LOCK(chan);
		chan->cur = NULL;
		pcidma_start(chan);
		UNLOCK(chan);
	}
#ifdef FPU_SAVE
	if (fpu_saved) {
		fpsimd_load_state(&fpu);
	}
#endif /* FPU_SAVE */
#ifdef CONFIG_PCIEDMA_IRQ_THREAD
	pcidma_intr_enable(p);
#endif
	pcie_wmb();
	return status;
}


static inline int pcidma_check_args(struct pcidma_request *req)
{
	if (!req)
		return -1;
	if (req->pcie_ch >= PCIDMA_PCIE_MAX)
		return -1;
#ifdef CONFIG_ARCH_CXD900XX_FPGA
	if (PCIDMA_PCIE_I != req->pcie_ch &&
	    PCIDMA_PCIE_N != req->pcie_ch)
		return -1;
#endif /* CONFIG_ARCH_CXD900XX_FPGA */
	if (req->dmac_ch >= PCIDMA_CH_MAX)
		return -1;
	if (!req->callback)
		return -1;
#if 0
	if (pcidma_ch_mask & BIT(req->dmac_ch))
		return -1;
#endif
	if (!req->size) { /* LL-mode */
		if (req->src & ((1ULL << PCIDMAC_LL_ALIGN) - 1))
			return -1;
	}

	return 0;
}

static void pcidma_enable(struct pci_info_t *p)
{
	p->flag |= FLAG_DMA;
	/* Enable DMA engines */
	pcie_writel(p->dmac + DMAWR_EN, DMA_EN_ENABLE);
	pcie_writel(p->dmac + DMAWR_INTCLR, DMA_INT_ALL);
	pcie_writel(p->dmac + DMARD_EN, DMA_EN_ENABLE);
	pcie_writel(p->dmac + DMARD_INTCLR, DMA_INT_ALL);
	pcie_mb();
	pcie_writel(p->dmac + DMAWR_LLERR_EN, DMA_LLLAIE);
	pcie_writel(p->dmac + DMARD_LLERR_EN, DMA_LLLAIE);
	pcidma_intr_enable(p);
}

static void pcidma_disable(struct pci_info_t *p)
{
	if (!(p->flag & FLAG_DMA))
		return;
	p->flag &= ~FLAG_DMA;
	pcidma_intr_disable(p);
	/* Disable DMA engines */
	pcie_writel(p->dmac + DMAWR_EN, DMA_EN_DISABLE);
	pcie_writel(p->dmac + DMARD_EN, DMA_EN_DISABLE);
}

void pcidma_irq_setup(struct pci_info_t *p)
{
	int ret;

#ifdef CONFIG_PCIEDMA_IRQ_THREAD
	ret = request_threaded_irq(p->irq, pcidma_intr0, pcidma_intr, IRQFLAG, "PCIeDMAC", (void *)p);
#else
	ret = request_irq(p->irq, pcidma_intr, IRQFLAG, "PCIeDMAC", (void *)p);
#endif
	if (ret) {
		printk(KERN_ERR "%s: request_irq %u failed: %d\n",
		       __func__, p->irq, ret);
		//BUG();
		return;
	}
	p->flag |= FLAG_IRQ;
}

void pcidma_irq_remove(struct pci_info_t *p)
{
	if (!(p->flag & FLAG_IRQ))
		return;
	p->flag &= ~FLAG_IRQ;
	free_irq(p->irq, p);
}

static void pcidma_run(struct pci_info_t *p)
{
	int i;
	struct dma_info_t *chan;

	for (i = 0, chan = p->dma_info; i < PCIDMA_CH_MAX; i++, chan++) {
		unsigned long flags;

		LOCK(chan);
		if (chan->state || chan->cur) {
			UNLOCK(chan);
			printk(KERN_ERR "%s:PCIDMA:%d:%d:state=%d,cur=%p\n",
			       __func__, p->id, i, chan->state, chan->cur);
			BUG();
		}
		chan->state = 1;
		pcidma_start(chan);
		UNLOCK(chan);
	}
}

static void pcidma_stop(struct pci_info_t *p)
{
	int i;
	struct dma_info_t *chan;

	for (i = 0, chan = p->dma_info; i < PCIDMA_CH_MAX; i++, chan++) {
		unsigned long flags;

		LOCK(chan);
		chan->state = 0;
		UNLOCK(chan);
	}
}

void pcidma_shutdown(struct pci_info_t *p)
{
	int i;
	struct dma_info_t *chan;
	unsigned long flags;

	udif_spin_lock(&p->lock);
	if (p->state) {
		p->state = 0;
		pcidma_stop(p);
	}

	pcidma_disable(p);

	for (i = 0, chan = p->dma_info; i < PCIDMA_CH_MAX; i++, chan++) {
		LOCK(chan);
		chan->cur = NULL;
#ifdef ER9001359142
		aspmctl_chan_init(chan);
#endif
		UNLOCK(chan);
	}
#ifdef ER9001359142
	aspmctl_shutdown(p);
#endif
	udif_spin_unlock(&p->lock);
}

/* Callback from PCIe HOST driver */
void pcidma_host_cb(int pci_ch, int state)
{
	struct pci_info_t *p;

	if (pci_ch < 0 || PCIDMA_PCIE_MAX <= pci_ch) {
		printk(KERN_ERR "%s: illegal pci_ch:%d\n", __func__, pci_ch);
		BUG();
	}
	p = &pcie_dmacs[pci_ch];
	if (state) { /* RC start */
		pcidma_enable(p);
	} else { /* RC shutdown */
		pcidma_shutdown(p);
	}
}

/* Kernel API */
int pcidma_transfer_async(struct pcidma_request *req)
{
	struct dma_info_t *chan; /* channel info */
	unsigned long flags;
#ifdef PCIEDMA_TMON
	void *lr = __builtin_return_address(0);
#endif

	if (pcidma_check_args(req) < 0) {
		return UDIF_ERR_PAR;
	}

#ifdef PCIEDMA_TMON
	pcie_dma_tmon("xf", (long)req);
	pcie_dma_tmon("lr", (long)lr);
	if (req->size) {
		if (PCIDMA_WR == req->dir) { /* local to remote */
			pcie_dma_tmon("wr", (long)(req->size));
		}
		else { /* remote to local */
			pcie_dma_tmon("rd", (long)(req->size));
		}
	}
	else {
		struct pcidma_lli *p_lli = (struct pcidma_lli *)PHYS_TO_VA(req->src);
		if (PCIDMA_WR == req->dir) { /* local to remote */
			while (p_lli->size) {
				pcie_dma_tmon("WR", (long)(p_lli->size));
				p_lli++;
			}
		}
		else { /* remote to local */
			while (p_lli->size) {
				pcie_dma_tmon("RD", (long)(p_lli->size));
				p_lli++;
			}
		}
	}
#endif

	chan = &pcie_dmacs[req->pcie_ch].dma_info[req->dmac_ch];
	req->__drv.pid = task_pid_nr(current);
	req->__drv.info = 0;
#ifdef CONFIG_SUBSYSTEM
	if (current->itron_tid >= 0) {
		req->__drv.info |= INFO_SUBSYSTEM;
	}
#endif /* CONFIG_SUBSYSTEM */
#ifdef ER9001359142
	aspmctl_request(chan);
#endif

	LOCK(chan);
	pcidma_enqueue(chan, req);
	pcidma_start(chan);
	UNLOCK(chan);

	return UDIF_ERR_OK;
}
EXPORT_SYMBOL(pcidma_transfer_async);

void pcidma_connect(int pci_ch, int state)
{
	struct pci_info_t *p;

	if (pci_ch < 0 || PCIDMA_PCIE_MAX <= pci_ch) {
		printk(KERN_ERR "%s: illegal pci_ch:%d\n", __func__, pci_ch);
		BUG();
	}
	p = &pcie_dmacs[pci_ch];
	udif_spin_lock(&p->lock);
	if (!p->state && state) {
#ifdef ER9001359142
		aspmctl_connect(p);
#endif
		p->state = 1;
		pcidma_run(p);
	} else if (p->state && !state) {
		p->state = 0;
		pcidma_stop(p);
	}
	udif_spin_unlock(&p->lock);
}
EXPORT_SYMBOL(pcidma_connect);

void logdump(struct pci_info_t *p)
{
	int ch;

	printk(KERN_ERR "--- PCIe log ---------------\n");
	smp_rmb();
	log_dump(&p->log);
	for (ch = 0; ch < PCIDMA_CH_MAX; ch++) {
		printk(KERN_ERR "CH%d:\n", ch);
		log_dump(&p->dma_info[ch].log);
	}
}
