/**
 * debugfs.c - DesignWare PCIe Controller DebugFS file
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2  of
 * the License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/debugfs.h>
#include <linux/seq_file.h>
#include <linux/device.h>
#include <linux/pci.h>
#include <linux/list.h>
#include "debugfs.h"

static struct debugfs_reg32 *ptr_pcie_hdr_reg = NULL;
static struct debugfs_reg32 *ptr_side_band_reg = NULL;
static struct debugfs_reg32 *ptr_pci_exp_cap_reg = NULL;
static const struct debugfs_reg32 outbound_reg[iATU_REGION_MAX][iATU_DUMP_SIZE] =
{
    {dump_iatu_register_set(0,0)},
    {dump_iatu_register_set(1,0)},
    {dump_iatu_register_set(2,0)},
    {dump_iatu_register_set(3,0)},
    {dump_iatu_register_set(4,0)},
    {dump_iatu_register_set(5,0)},
    {dump_iatu_register_set(6,0)},
    {dump_iatu_register_set(7,0)},
};

static const struct debugfs_reg32 inbound_reg[iATU_REGION_MAX][iATU_DUMP_SIZE] =
{
    {dump_iatu_register_set(0,1)},
    {dump_iatu_register_set(1,1)},
    {dump_iatu_register_set(2,1)},
    {dump_iatu_register_set(3,1)},
    {dump_iatu_register_set(4,1)},
    {dump_iatu_register_set(5,1)},
    {dump_iatu_register_set(6,1)},
    {dump_iatu_register_set(7,1)},
};

static struct debugfs_reg32 * _debugfs_reg_data_create( int reg_size, char* reg_name_fmt, unsigned long pos)
{
    int i;
    unsigned long offset;
    char** name, **regs_name;
    struct debugfs_reg32 * ptr, *ptr_reg32;

    ptr_reg32 = kzalloc(sizeof(*ptr_reg32)*reg_size, GFP_KERNEL);
    regs_name = kzalloc(sizeof(char**)*reg_size, GFP_KERNEL);
    if(!ptr_reg32 || !regs_name)
        goto fail;

    for(i=0;i<reg_size;i++)
    {
        ptr = (ptr_reg32 + i);
        name = (regs_name+i);
        offset = (unsigned long)(i*REG32BYTES) + pos;
        *name = kzalloc(sizeof(char*)*MAX_STR_LEN, GFP_KERNEL);
        sprintf(*name, reg_name_fmt, offset);
        ptr->name =  (*name);
        ptr->offset =  offset;
    }

    return ptr_reg32;

fail:
    kfree(ptr_reg32);
    kfree(regs_name);
    return NULL;
}

static int _debugfs_create_hdr_dump( struct dw_pcie *pci,  struct dentry *parent)
{
    struct dentry  *file;

    pci->pcie_hdr_regset = kzalloc(sizeof(*pci->pcie_hdr_regset), GFP_KERNEL);

    if (!pci->pcie_hdr_regset)
        return -ENOMEM;

    if(!ptr_pcie_hdr_reg)
    {
        ptr_pcie_hdr_reg = _debugfs_reg_data_create(PCIE_HDR_SIZE, "hdr_0x%02x", 0);
        if(!ptr_pcie_hdr_reg)
            return -ENOMEM;
    }

    pci->pcie_hdr_regset->regs = ptr_pcie_hdr_reg;
    pci->pcie_hdr_regset->nregs = PCIE_HDR_SIZE;
    pci->pcie_hdr_regset->base = pci->dbi_base;

    file= debugfs_create_regset32("dump_pcie_hdr", S_IRUGO, parent, pci->pcie_hdr_regset);
    if (!file)
        dev_dbg(pci->dev, "Can't create debugfs regdump\n");

    return 0;
}

static int _debugfs_create_sideband_dump( struct dw_pcie *pci,  struct dentry *parent)
{
    struct dentry  *file;

    pci->side_band_regset = kzalloc(sizeof(*pci->side_band_regset), GFP_KERNEL);

    if (!pci->side_band_regset)
        return -ENOMEM;

    if(!ptr_side_band_reg)
    {
        ptr_side_band_reg = _debugfs_reg_data_create(SIDE_BAND_REG32_SIZE, "side_band_0x%05x", (SIDE_BAND_BASE_OFF+SIDE_BAND_DUMP_START));
        if(!ptr_side_band_reg)
            return -ENOMEM;
    }

    pci->side_band_regset->regs = ptr_side_band_reg;
    pci->side_band_regset->nregs = SIDE_BAND_REG32_SIZE;
    pci->side_band_regset->base = pci->phy_base;

    file= debugfs_create_regset32("dump_side_band_registers", S_IRUGO, parent, pci->side_band_regset);
    if (!file)
        dev_dbg(pci->dev, "Can't create debugfs regdump\n");

    return 0;
}

static int _debugfs_create_pcie_cap_dump( struct dw_pcie *pci,  struct dentry *parent)
{
    struct dentry  *file;
    int pos;

    pci->pci_express_cap = kzalloc(sizeof(*pci->pci_express_cap), GFP_KERNEL);
    if (!pci->pci_express_cap)
        return -ENOMEM;

    pos = dw_pcie_find_capability(pci, (u8)PCI_CAP_ID_EXP);
    if(pos)
    {
        if(!ptr_pci_exp_cap_reg)
        {
            ptr_pci_exp_cap_reg = _debugfs_reg_data_create(PCI_EXP_CAP_SIZE, "pci_express_cap_0x%02x", (unsigned long)pos);
            if(!ptr_pci_exp_cap_reg)
                return -ENOMEM;
        }

        pci->pci_express_cap->regs = ptr_pci_exp_cap_reg;
        pci->pci_express_cap->nregs = PCI_EXP_CAP_SIZE;
        pci->pci_express_cap->base = pci->dbi_base;

        file= debugfs_create_regset32("dump_pci_express_cap", S_IRUGO,parent, pci->pci_express_cap);
        if (!file)
            dev_dbg(pci->dev, "Can't create debugfs regdump\n");
    }

    return 0;
}

static int _debugfs_create_iatu_dump( struct dw_pcie *pci,  struct dentry *parent)
{
    struct dentry  *file;
    u32 i, num_ib, num_ob;
    char** file_name, **name_ptr;

    if((dw_pcie_readl_phy(pci, GENERAL_CORE_CTRL_OFF) & 0xfUL) == DW_PCIE_AS_RC)
    {
        num_ib = pci->num_viewport;
        num_ob = pci->num_viewport;
    }
    else
    {
        num_ib = pci->ep.num_ib_windows;
        num_ob = pci->ep.num_ob_windows;
    }

    name_ptr = kzalloc(sizeof(char**)*(num_ib+num_ob), GFP_KERNEL);

    for(i=0;i<num_ob;i++)
    {
        file_name = name_ptr + i;
        *file_name = kzalloc(sizeof(char*)*MAX_STR_LEN, GFP_KERNEL);
        pci->ob_regset[i] = kzalloc(sizeof(struct debugfs_reg32), GFP_KERNEL);
        if (!pci->ob_regset[i]) {
            return -ENOMEM;
        }

        pci->ob_regset[i]->regs = &outbound_reg[i][0];
        pci->ob_regset[i]->nregs = iATU_DUMP_SIZE;
        pci->ob_regset[i]->base = pci->dbi_base;

        sprintf(*file_name, "dump_iatu_ob_%d", (int)i);
        file= debugfs_create_regset32(*file_name, S_IRUGO, parent, pci->ob_regset[i]);
        if (!file)
            dev_dbg(pci->dev, "Can't create debugfs regdump\n");
    }

    for(i=0;i<num_ib;i++)
    {
        file_name = name_ptr +num_ob + i;
        *file_name = kzalloc(sizeof(char*)*MAX_STR_LEN, GFP_KERNEL);
        pci->ib_regset[i] = kzalloc(sizeof(struct debugfs_reg32), GFP_KERNEL);
        if (!pci->ib_regset[i]) {
            return -ENOMEM;
        }

        pci->ib_regset[i]->regs = &inbound_reg[i][0];
        pci->ib_regset[i]->nregs = iATU_DUMP_SIZE;
        pci->ib_regset[i]->base = pci->dbi_base;
        sprintf(*file_name, "dump_iatu_ib_%d", (int)i);
        file= debugfs_create_regset32(*file_name, S_IRUGO, parent, pci->ib_regset[i]);
        if (!file)
            dev_dbg(pci->dev, "Can't create debugfs regdump\n");
    }

    return 0;
}

void dw_pcie_debugfs_init( struct dw_pcie *pci)
{
    pci->debugfs_root = debugfs_create_dir(dev_name(pci->dev), NULL);
    if (IS_ERR_OR_NULL(pci->debugfs_root)) {
        if (!pci->debugfs_root)
            dev_err(pci->dev, "Can't create debugfs root\n");
        return;
    }

    if(_debugfs_create_hdr_dump(pci,  pci->debugfs_root) ||\
            _debugfs_create_sideband_dump(pci,  pci->debugfs_root)||\
            _debugfs_create_pcie_cap_dump(pci,  pci->debugfs_root)||\
            _debugfs_create_iatu_dump(pci,  pci->debugfs_root))
        goto fail;

    return;

fail:
    dw_pcie_debugfs_exit(pci);
    return;
}

void dw_pcie_debugfs_exit( struct dw_pcie *pci)
{
    u8 i;
    debugfs_remove_recursive(pci->debugfs_root );
    kfree(pci->pcie_hdr_regset);
    kfree(pci->side_band_regset);
    kfree(pci->pci_express_cap);

    for(i=0;i<iATU_REGION_MAX;i++)
    {
        kfree(pci->ob_regset[i]);
        kfree(pci->ib_regset[i]);
    }
}

