/*
 * arch/arm/mach-cxd900x0/clock.c
 *
 * clock manager
 *
 * Copyright 2016 Sony Corporation
 *
 * This code is based on arch/arm/mach-integrator/clock.c.
 */
/*
 *  linux/arch/arm/mach-integrator/clock.c
 *
 *  Copyright (C) 2004 ARM Limited.
 *  Written by Deep Blue Solutions Limited.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/list.h>
#include <linux/errno.h>
#include <linux/err.h>
#include <linux/string.h>
#include <linux/clkdev.h>
#include <linux/mutex.h>

#include <asm/clkdev.h>
#include <mach/hardware.h>
#include <asm/io.h>
#include <mach/timex.h>
#include <mach/regs-clk.h>
#include <mach/cfg.h>

/* initialize CRG */
#define CRG11_40 VA_CLKCRG4(0)
#define CRG11_41 VA_CLKCRG4(1)

static void cxd900x0_set_crdm(const void __iomem *base, int clk, u32 div)
{
#if !defined(CONFIG_CXD900X0_FPGA) && !defined(CONFIG_CXD900X0_QEMU)
	writel_relaxed(div, base + CRG_CRDM(clk));
#endif /* !CONFIG_CXD900X0_FPGA && !CONFIG_CXD900X0_QEMU */
}

static void cxd900x0_update_crdm(const void __iomem *base)
{
#if !defined(CONFIG_CXD900X0_FPGA) && !defined(CONFIG_CXD900X0_QEMU)
	int n;

	writel_relaxed(1, base + CRG_CRCDC);
	dsb();

	n = CRCDC_UPDATE_WAIT;
	while (readl_relaxed(base + CRG_CRCDC) & 1) {
		if (--n <= 0)
			break;
	}
	if (n <= 0) {
		printk(KERN_ERR "ERROR:%s:CRCDC timeout:addr=%p\n",
		       __func__, base);
	}
#endif /* !CONFIG_CXD900X0_FPGA && !CONFIG_CXD900X0_QEMU */
}

/* UART baud rate clock */
#define CK_UART(ch)	(9-(ch))
#define UART_DIV	0xf  /* 1/16 (48MHz) */

void cxd900x0_clock_init(void)
{
	int i, update;

	/*--- CRG11_40 ---*/
	update = 0;
	/*   UART2,3: baud rate clock */
	for (i = 2; i <= 3; i++) {
		if (cxd900x0_uart_bitmap & (1 << i)) {
			cxd900x0_set_crdm(CRG11_40, CK_UART(i), UART_DIV);
			update = 1;
		}
	}
	if (update) {
		cxd900x0_update_crdm(CRG11_40);
	}
}


#include "clock.h"

static int cxd900x0_enable_uart(struct clk *clk, int enable)
{
#if !defined(CONFIG_CXD900X0_FPGA) && !defined(CONFIG_CXD900X0_QEMU)
	int bit;
	u32 bitmask;

	/*
	 *  Enable/Disable UART BAUD clock.
	 *    (note) UART PCLK should be always enabled.
	 */
	switch (clk->chan) {
	case 0: case 1: case 2:
		bit = CXD900X0_UART_CLK_SHIFT(clk->chan);
		break;
	case 3:
		bit = CXD900X0_UART_CLK_SHIFT3;
		break;
	default:
		return -1;
	}
	bitmask = 1 << bit;
	if (enable)
		writel_relaxed(bitmask, VA_CLKRST4(IPCLKEN4)+CLKRST_SET);
	else
		writel_relaxed(bitmask, VA_CLKRST4(IPCLKEN4)+CLKRST_CLR);
#endif /* !CONFIG_CXD900X0_FPGA && !CONFIG_CXD900X0_QEMU */

	return 0;
}

static unsigned long cxd900x0_getrate_uart(struct clk *clk)
{
	return CXD900X0_UART_CLK_HIGH;
}

static struct clk uart_clk[] = {
	/*-------------------- UART --------------------*/
	{
		.name		= "UARTCLK0",
		.chan		= 0,
		.get_rate       = cxd900x0_getrate_uart,
		.enable		= cxd900x0_enable_uart,
	},
	{
		.name		= "UARTCLK1",
		.chan		= 1,
		.get_rate       = cxd900x0_getrate_uart,
		.enable		= cxd900x0_enable_uart,
	},
	{
		.name		= "UARTCLK2",
		.chan		= 2,
		.get_rate       = cxd900x0_getrate_uart,
		.enable		= cxd900x0_enable_uart,
	},
	{
		.name		= "UARTCLK3",
		.chan		= 3,
		.get_rate       = cxd900x0_getrate_uart,
		.enable		= cxd900x0_enable_uart,
	},
	/*----------------------------------------------*/
};

static struct clk_lookup lookups[] = {
	{	/* UART0 */
		.dev_id		= "uart0",
		.clk		= &uart_clk[0],
	},
	{	/* UART1 */
		.dev_id		= "uart1",
		.clk		= &uart_clk[1],
	},
	{	/* UART2 */
		.dev_id		= "uart2",
		.clk		= &uart_clk[2],
	},
	{	/* UART3 */
		.dev_id		= "uart3",
		.clk		= &uart_clk[3],
	},
};

int clk_enable(struct clk *clk)
{
	(clk->enable)(clk, 1);
	return 0;
}
EXPORT_SYMBOL(clk_enable);

void clk_disable(struct clk *clk)
{
	(clk->enable)(clk, 0);
}
EXPORT_SYMBOL(clk_disable);

unsigned long clk_get_rate(struct clk *clk)
{
	if (clk->rate != 0)
		return clk->rate;
	if (clk->get_rate != NULL)
		return (clk->get_rate)(clk);
	return clk->rate;
}
EXPORT_SYMBOL(clk_get_rate);

long clk_round_rate(struct clk *clk, unsigned long rate)
{
	return rate;
}
EXPORT_SYMBOL(clk_round_rate);

int clk_set_rate(struct clk *clk, unsigned long rate)
{
	return 0;
}
EXPORT_SYMBOL(clk_set_rate);

static int __init clk_init(void)
{
	int i;

	for (i = 0; i < ARRAY_SIZE(lookups); i++)
		clkdev_add(&lookups[i]);
	return 0;
}
arch_initcall(clk_init);
