/*
 * arch/arm/mach-cxd900x0/cxd900x0.c
 *
 * CXD900x0 devices
 *
 * Copyright 2015 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */
#include <linux/device.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/dma-mapping.h>
#include <linux/platform_device.h>
#include <linux/amba/bus.h>
#include <mach/hardware.h>
#include <asm/irq.h>
#include <asm/mach/arch.h>
#include <asm/mach-types.h>
#include <mach/uart.h>
#include <asm/io.h>
#include <asm/pmu.h>
#include <mach/gic_export.h>
#include <mach/cxd900x0_board.h>
#include <mach/cfg.h>
#include <mach/regs-misc.h>
#include <mach/bam.h>
#include <mach/dap_export.h>

#if defined(CONFIG_SMSC911X) || defined(CONFIG_SMSC911X_MODULE)
#include <linux/smsc911x.h>
#include <mach/regs-sramc.h>
#include <mach/regs-gpio.h>
#include <mach/regs-octrl.h>
#endif

/*
 * NAME
 * 	gpiopin_to_irq
 * SYNOPSIS
 *	#include <linux/interrupt.h>
 *	int gpiopin_to_irq(unsigned int port, unsigned int bit);
 * INPUT
 *	port: GPIO port number (0=GPIO_S, 1=GPIO_SB)
 *       bit: GPIO bit number (0..31)
 * RETURN VALUE
 *      IRQ number
 *	  If input is out of range, -1 shall be returned.
 */
int gpiopin_to_irq(unsigned int port, unsigned int bit)
{
	int irq = -1;

	switch (port) {
	case 0: /* GPIO_S */
		if (bit < NR_GPIO_S_LOW_IRQ) {	/* GPIO_S_0 .. 15 */
			irq = IRQ_GPIO_BASE + bit;
		} else if (16 == bit) {		/* GPIO_S_16 */
			irq = IRQ_GPIO_S_16;
		}
		break;
	case 1: /* GPIO_SB */
		if (bit < NR_GPIO_SB_IRQ) {	/* GPIO_SB_0 .. 6 */
			irq = IRQ_GPIO_SB_BASE + bit;
		}
		break;
	default:
		break;
	}
	return irq;
}
EXPORT_SYMBOL(gpiopin_to_irq);

/* List of edge trigger IRQs, except GPIO */
static u8 edge_irqs[] __initdata = {
#ifdef CONFIG_MPCORE_GIC_INIT
	IRQ_NANDC,
	IRQ_MS0_INS_RISE, IRQ_MS0_INS_FALL,
	IRQ_USB_VBUS, IRQ_USB_ID,
#endif /* CONFIG_MPCORE_GIC_INIT */
};

/* List of GPIO IRQ definitions */
#define GPIOIRQ_EDGE	0x01 /* edge */
#define GPIOIRQ_INV	0x02 /* active L */
#define G_RISE	GPIOIRQ_EDGE
#define G_FALL	(GPIOIRQ_EDGE|GPIOIRQ_INV)
static u8 gpio_irqs[] __initdata = {
#ifdef CONFIG_MPCORE_GIC_INIT
	/* 0 */	G_RISE,	G_RISE,	G_RISE,	G_FALL,
		G_RISE,	G_RISE,	G_RISE,	G_RISE,
	/* 8 */ G_RISE,	G_RISE,	G_RISE,	G_RISE,
		G_RISE,	G_RISE, G_RISE,	G_RISE,
	/*16 */	G_RISE,	G_RISE,	G_RISE, G_RISE,
		G_RISE,	G_RISE,	G_RISE,	G_RISE,
#endif /* CONFIG_MPCORE_GIC_INIT */
};

void __init cxd900x0_irq_type_init(void)
{
	int i;
	u8 *p;

	/* setup edge trigger IRQs, except GPIO */
	for (i = 0, p = edge_irqs; i < sizeof edge_irqs; i++, p++) {
		gic_config_edge((unsigned int)*p);
	}
	/* setup GPIO IRQs */
	for (i = 0, p = gpio_irqs; i < sizeof gpio_irqs; i++, p++) {
		u8 typ;
		typ = *p;
		if (typ & GPIOIRQ_EDGE) { /* edge */
			gic_config_edge((unsigned int)(IRQ_GPIO_BASE + i));
			/* set EXINT */
			if (typ & GPIOIRQ_INV) { /* fall edge */
				writel_relaxed(1 << i, VA_MISC+MISC_GPIOINTLE+MISCREG_SET);
			} else { /* rise edge */
				writel_relaxed(1 << i, VA_MISC+MISC_GPIOINTHE+MISCREG_SET);
			}

		} else { /* level */
			/* set EXINT */
			writel_relaxed(1 << i, VA_MISC+MISC_GPIOINTS+MISCREG_SET);
			if (typ & GPIOIRQ_INV) { /* active Low */
				writel_relaxed(1 << 1, VA_MISC+MISC_GPIOINTINV+MISCREG_SET);
			}
		}
	}
}

static struct resource pmu_resources[] = {
	[0] = {
		.start	= IRQ_PMU(0),
		.end	= IRQ_PMU(0),
		.flags	= IORESOURCE_IRQ,
	},
	[1] = {
		.start	= IRQ_PMU(1),
		.end	= IRQ_PMU(1),
		.flags	= IORESOURCE_IRQ,
	},
	[2] = {
		.start	= IRQ_PMU(2),
		.end	= IRQ_PMU(2),
		.flags	= IORESOURCE_IRQ,
	},
	[3] = {
		.start	= IRQ_PMU(3),
		.end	= IRQ_PMU(3),
		.flags	= IORESOURCE_IRQ,
	},
};

static irqreturn_t cxd900x0_pmu_handler(int irq, void *dev, irq_handler_t handler)
{
	cxd900x0_cti_ack(irq - IRQ_PMU_BASE);
	return handler(irq, dev);
}

static struct arm_pmu_platdata cxd900x0_pmu_platdata = {
	.handle_irq	= cxd900x0_pmu_handler,
};

static struct platform_device pmu_device = {
	.name		= "arm-pmu",
	.id		= ARM_PMU_DEVICE_CPU,
	.num_resources	= ARRAY_SIZE(pmu_resources),
	.resource	= pmu_resources,
	.dev.platform_data	= &cxd900x0_pmu_platdata,
};


static struct uart_platform_data uart0_data = {
	.chan		= 0,
	.fifosize	= 32,
	.clock		= "UARTCLK0",
};

static struct uart_platform_data uart1_data = {
	.chan		= 1,
	.fifosize	= 32,
	.clock		= "UARTCLK1",
};

static struct uart_platform_data uart2_data = {
	.chan		= 2,
	.fifosize	= 32,
	.clock		= "UARTCLK2",
};

static struct uart_platform_data uart3_data = {
	.chan		= 3,
	.fifosize	= 32,
	.clock		= "UARTCLK3",
};

static struct amba_device uart0_device = {
	.dev = {
		.init_name = "uart0",
		.platform_data	= &uart0_data,
	},
	.res = {
		 .start = CXD900X0_UART(0),
		 .end   = CXD900X0_UART(0)+SZ_4K-1,
		 .flags = IORESOURCE_MEM,
	 },
	.irq = { IRQ_UART0, },
};

static struct amba_device uart1_device = {
	.dev = {
		.init_name = "uart1",
		.platform_data	= &uart1_data,
	},
	.res = {
		 .start = CXD900X0_UART(1),
		 .end   = CXD900X0_UART(1)+SZ_4K-1,
		 .flags = IORESOURCE_MEM,
	 },
	.irq = { IRQ_UART1, },
};

static struct amba_device uart2_device = {
	.dev = {
		.init_name = "uart2",
		.platform_data	= &uart2_data,
	},
	.res = {
		 .start = CXD900X0_UART(2),
		 .end   = CXD900X0_UART(2)+SZ_4K-1,
		 .flags = IORESOURCE_MEM,
	 },
	.irq = { IRQ_UART2, },
};

static struct amba_device uart3_device = {
	.dev = {
		.init_name = "uart3",
		.platform_data	= &uart3_data,
	},
	.res = {
		 .start = CXD900X0_UART(3),
		 .end   = CXD900X0_UART(3)+SZ_4K-1,
		 .flags = IORESOURCE_MEM,
	 },
	.irq = { IRQ_UART3, },
};

static struct amba_device *uart_devs[] __initdata = {
	&uart0_device,
	&uart1_device,
	&uart2_device,
	&uart3_device,
};

#if !defined(CONFIG_CXD900X0_QEMU)
/* USB Dual-Role Device Controller */
static u64 f_usb30dr_dma_mask = DMA_BIT_MASK(32);

static struct resource f_usb30dr_resources[] = {
	[0] = {
		.start  = CXD900X0_USB_DRD_BASE,
		.end	= CXD900X0_USB_DRD_BASE+CXD900X0_USB_DRD_SIZE-1,
		.flags	= IORESOURCE_MEM,
	},
	[1] = {
		.start  = IRQ_USB_INTR,
		.end	= IRQ_USB_INTR,
		.flags	= IORESOURCE_IRQ,
	}
};

/*
 * It is not guaranteed to set a value except USB_SPEED_SUPER
 * in dwc3_platform_data.maximum_speed.
 * (If dwc3_platform_data is omitted, USB_SPEED_SUPER is set in maximum_speed)
 */

/*
static struct dwc3_platform_data dwc3_pdata = {
	.maximum_speed = USB_SPEED_SUPER,
};
*/
static struct platform_device f_usb30dr_device = {
	.name		= "dwc3",
	.id		= 0,
	.dev = {
		.dma_mask			= &f_usb30dr_dma_mask,
		.coherent_dma_mask	= DMA_BIT_MASK(32),
/*		.platform_data		= &dwc3_pdata, */
	},
	.num_resources	= ARRAY_SIZE(f_usb30dr_resources),
	.resource	= f_usb30dr_resources,
};

static struct platform_device *fusb_devices[] __initdata = {
	&f_usb30dr_device,
};
#endif /* !CONFIG_CXD900X0_QEMU */

static int bam_enable = 1;
module_param_named(bam, bam_enable, bool, S_IRUSR);

static struct resource bam_resources[] = {
	[0] = {
		.start  = CXD900X0_BAM_BASE,
		.end	= CXD900X0_BAM_BASE+CXD900X0_BAM_SIZE-1,
		.flags	= IORESOURCE_MEM,
	},
	[1] = {
		.start  = IRQ_AVB,
		.end	= IRQ_AVB,
		.flags	= IORESOURCE_IRQ,
	}
};

static struct platform_device bam_device = {
	.name		= "bam",
	.id		= 0,
	.num_resources	= ARRAY_SIZE(bam_resources),
	.resource	= bam_resources,
};

#if defined(CONFIG_SMSC911X) || defined(CONFIG_SMSC911X_MODULE)

#define SMC911X_INT		CONFIG_CXD900X0_SMC911X_INT
#ifdef CONFIG_CXD900X0_FPGA
#define SMC9118_MEM_START	CXD900X0_SRAMC_CS0_BASE
#else
#define SMC9118_MEM_START	CXD900X0_SRAMC_CS2_BASE
#endif

static struct resource smc9118_resources[] = {
	[0] = {
		.start	= SMC9118_MEM_START,
		.end	= SMC9118_MEM_START + SZ_4K-1,
		.flags	= IORESOURCE_MEM,
	},
	[1] = {
		.start  = IRQ_GPIO(SMC911X_INT),
		.end	= IRQ_GPIO(SMC911X_INT),
		.flags	= IORESOURCE_IRQ | IRQF_TRIGGER_RISING,
	}
};

static struct smsc911x_platform_config smsc911x_config = {
#ifdef CONFIG_CXD900X0_QEMU
	.irq_polarity   = SMSC911X_IRQ_POLARITY_ACTIVE_HIGH,
#else
	.irq_polarity   = SMSC911X_IRQ_POLARITY_ACTIVE_LOW,
#endif
#if defined(CONFIG_CXD900X0_FPGA) && !defined(CONFIG_CXD900X0_QEMU)
	.irq_type       = SMSC911X_IRQ_TYPE_OPEN_DRAIN,
#else
	.irq_type       = SMSC911X_IRQ_TYPE_PUSH_PULL,
#endif
	.flags		= SMSC911X_USE_32BIT,
};

static struct platform_device smc9118_device = {
	.name		= "smsc911x",
	.id		= 0,
	.num_resources	= ARRAY_SIZE(smc9118_resources),
	.resource	= smc9118_resources,
	.dev = {
		.platform_data = &smsc911x_config,
	},
};
#endif /* CONFIG_SMSC911X */

#if defined(CONFIG_SMSC911X) || defined(CONFIG_SMSC911X_MODULE)
#define VA_SRAMC(ch,reg)	IO_ADDRESSP(CXD900X0_SRAMC_BASE + SRAMC_REG(ch,reg))
/* config CS0 as DW=16bit */
#define SRAMC_ETHER_MODE	(SRAMC_MODE_DW16)

#ifdef CONFIG_CXD900X0_FPGA
/* SRAMC_CS# of ETHER */
#define ETHER_CS 0
/* CLK=20MHz (period=50ns) */
/* RIDLEC+1=1, FRADC=16(fix), RADC=2, RACC=6 */
#define SRAMC_ETHER_RDTIM	0x00100206
/* WIDLEC+1=1, WWEC+1=1, WADC+1=2, WACC+1=5 */
#define SRAMC_ETHER_WRTIM	0x00000104
/* WCSMWC+1=4, RCSMWC+1=4 */
#define SRAMC_ETHER_CSMWC	0x01030103
/* RADHDC=1, RADVC=1, RADSUC=0 */
#define SRAMC_ETHER_RADV	0x00010100
/* WADHDC=1, WADVC=1, WADSUC=0 */
#define SRAMC_ETHER_WADV	0x00010100

#else /* ASIC */

/* SRAMC_CS# of ETHER */
#define ETHER_CS 2
/* CLK=132MHz (period(min)=7.5ns) */
/* RIDLEC+1=1, FRADC=16(fix), RADC=4, RACC+1=16 */
#define SRAMC_ETHER_RDTIM	0x0010040F
/* WIDLEC+1=1, WWEC+1=7, WADC+1=4, WACC+1=15 */
#define SRAMC_ETHER_WRTIM	0x0006030E
/* WCSMWC+1=13, RCSMWC+1=13 */
#define SRAMC_ETHER_CSMWC	0x010C010C
/* RADHDC=1, RADVC=2, RADSUC=0 */
#define SRAMC_ETHER_RADV	0x00010200
/* WADHDC=1, WADVC=2, WADSUC=0 */
#define SRAMC_ETHER_WADV	0x00010200
#endif /* CONFIG_CXD900X0_FPGA */

static unsigned int eth_port, eth_bit;

#endif /* CONFIG_SMSC911X */

void cxd900x0_board_init(void)
{
	extern void cxd900x0_clock_init(void);

	cxd900x0_clock_init();

#if defined(CONFIG_HW_PERF_EVENTS)
	cxd900x0_cti_setup();
#endif /* CONFIG_HW_PERF_EVENTS */
	/* Powersave */
#if !defined(CONFIG_CXD900X0_FPGA) && !defined(CONFIG_CXD900X0_QEMU)
	bam_powersave(true);
#endif /* !CONFIG_CXD900X0_FPGA && !CONFIG_CXD900X0_QEMU */

#if defined(CONFIG_SMSC911X) || defined(CONFIG_SMSC911X_MODULE)
	{
	unsigned int exint_bit;

	/*--- setup SRAMC for ETHER ---*/
	/* MODE   */
	writel_relaxed(SRAMC_ETHER_MODE, VA_SRAMC(ETHER_CS,MODE));
	/* async read timing */
	writel_relaxed(SRAMC_ETHER_RDTIM, VA_SRAMC(ETHER_CS,RTIM));
	/* async write timing */
	writel_relaxed(SRAMC_ETHER_WRTIM, VA_SRAMC(ETHER_CS,WTIM));
#ifndef CONFIG_CXD900X0_SRAMC_CSMASK_ERRATA
	/* async csmask timinig */
	writel_relaxed(SRAMC_ETHER_CSMWC, VA_SRAMC(ETHER_CS,CSMWC));
#endif /* !CONFIG_CXD900X0_SRAMC_CSMASK_ERRATA */
	/* async read adv timing */
	writel_relaxed(SRAMC_ETHER_RADV, VA_SRAMC(ETHER_CS,RADV));
	/* async write adv timing */
	writel_relaxed(SRAMC_ETHER_WADV, VA_SRAMC(ETHER_CS,WADV));
	/* dummy read */
	readl_relaxed(VA_SRAMC(ETHER_CS,WADV));

	/*--- setup INT pin ---*/
	/* DIC (setup by request_irq) */
	/* GPIO */
	/*   fall edge (EXINT) */
	exint_bit = eth_port * NR_GPIO_S_IRQ + eth_bit;
	writel_relaxed(BIT(exint_bit), VA_MISC+MISC_GPIOINTINV+MISCREG_CLR);
	writel_relaxed(BIT(exint_bit), VA_MISC+MISC_GPIOINTS+MISCREG_CLR);
	writel_relaxed(BIT(exint_bit), VA_MISC+MISC_GPIOINTHE+MISCREG_CLR);
	writel_relaxed(BIT(exint_bit), VA_MISC+MISC_GPIOINTLE+MISCREG_SET);
#if !defined(CONFIG_CXD900X0_FPGA)
	/* pull up */
	writel_relaxed(BIT(eth_bit), VA_OCTRL(eth_port)+OCTRL_PUD2+OCTRL_SET);
	/* pull on */
	writel_relaxed(BIT(eth_bit), VA_OCTRL(eth_port)+OCTRL_PUD1+OCTRL_SET);
#endif /* !CONFIG_CXD900X0_FPGA */
	/*   input enable */
	writel_relaxed(BIT(eth_bit), VA_GPIO(eth_port,INEN)+GPIO_SET);
	}
#endif /* CONFIG_SMSC911X */
}

static void __init cxd900x0_fixup(struct machine_desc *desc,
				 struct tag *tags, char **cmdline,
				 struct meminfo *mi)
{
}

static void __init cxd900x0_init(void)
{
	int i, ret;
	struct amba_device **dev;

#if defined(CONFIG_SMSC911X) || defined(CONFIG_SMSC911X_MODULE)
	eth_port = (cxd900x0_ether_irq >> 8) & 0xff;
	eth_bit  = cxd900x0_ether_irq & 0xff;
#endif
	cxd900x0_board_init();
	ret = cxd900x0_core_init();
	for (i = 0, dev = uart_devs; i < ARRAY_SIZE(uart_devs); i++, dev++) {
		if (cxd900x0_uart_bitmap & (1 << i)) {
			ret = amba_device_register(*dev, &iomem_resource);
			if (ret) {
				printk(KERN_ERR "cxd900x0_init: amba_device_register(uart_devs[%d]) failed.\n", i);
			}
		}
	}
	platform_device_register(&pmu_device);

#if !defined(CONFIG_CXD900X0_QEMU)
	/* USB devices */
	ret = platform_add_devices(fusb_devices, ARRAY_SIZE(fusb_devices));
	if (ret) {
		printk(KERN_ERR "f_usb30dr_device add failed.\n");
	}
#endif /* !CONFIG_CXD900X0_QEMU */

	/* BAM */
	if (bam_enable) {
		ret = platform_device_register(&bam_device);
		if (ret) {
			printk(KERN_ERR "cxd900x0_init: platform_device_register bam failed.\n");
		}
	}
#if defined(CONFIG_SMSC911X) || defined(CONFIG_SMSC911X_MODULE)
	{ /* Override IRQ pin */
		smc9118_resources[1].start = gpiopin_to_irq(eth_port, eth_bit);
		smc9118_resources[1].end   = smc9118_resources[1].start;
	}
	ret = platform_device_register(&smc9118_device);
	if (ret) {
		printk(KERN_ERR "cxd900x0_init: platform_device_register smc9118 failed.\n");
	}
#endif /* CONFIG_SMSC911X */
}

MACHINE_START(CXD900X0, "ARM-CXD900X0")
	/* Maintainer: Sony Corporation */
	.map_io		= cxd900x0_map_io,
	.init_irq	= cxd900x0_init_irq,
	.timer		= &cxd4115_timer,
	.init_machine	= cxd900x0_init,
	.fixup		= cxd900x0_fixup,
MACHINE_END
