/*
 * arch/arm/mach-cxd900x0/hwbp.c
 *
 * Copyright 2015,2016 Sony Corporation
 *
 * This code is based on arch/arm/kernel/hw_breakpoint.c
 */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Copyright (C) 2009, 2010 ARM Limited
 *
 * Author: Will Deacon <will.deacon@arm.com>
 */
#include <linux/module.h>
#include <linux/cpu.h>
#include <linux/init.h>
#include <linux/uaccess.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/signal.h>
#include <linux/io.h>
#include <asm/cputype.h>
#include <asm/system.h>
#include <mach/regs-dap.h>
#include <mach/regs-cp14.h>
#include <mach/hwbp.h>

#define HWBP_PROCNAME "hwbp"

#define HWBP_MAX_BP DAP_N_BP
#define HWBP_MAX_WP DAP_N_WP

/* HWBP mode */
#define HWBP_OFF 0
#define HWBP_ON  1
static int hwbp_mode = HWBP_OFF;
module_param_named(mode, hwbp_mode, int, S_IRUSR|S_IWUSR);

static unsigned long bp_addr[HWBP_MAX_BP];
module_param_named(bpaddr0, bp_addr[0], ulongH, S_IRUSR|S_IWUSR);
module_param_named(bpaddr1, bp_addr[1], ulongH, S_IRUSR|S_IWUSR);
module_param_named(bpaddr2, bp_addr[2], ulongH, S_IRUSR|S_IWUSR);
static unsigned long bp_ctrl[HWBP_MAX_BP];
module_param_named(bpctrl0, bp_ctrl[0], ulongH, S_IRUSR|S_IWUSR);
module_param_named(bpctrl1, bp_ctrl[1], ulongH, S_IRUSR|S_IWUSR);
module_param_named(bpctrl2, bp_ctrl[2], ulongH, S_IRUSR|S_IWUSR);

static unsigned long wp_addr[HWBP_MAX_WP];
module_param_named(wpaddr0, wp_addr[0], ulongH, S_IRUSR|S_IWUSR);
module_param_named(wpaddr1, wp_addr[1], ulongH, S_IRUSR|S_IWUSR);
static unsigned long wp_ctrl[HWBP_MAX_WP];
module_param_named(wpctrl0, wp_ctrl[0], ulongH, S_IRUSR|S_IWUSR);
module_param_named(wpctrl1, wp_ctrl[1], ulongH, S_IRUSR|S_IWUSR);


static DEFINE_RAW_SPINLOCK(hwbp_dap_lock);

static int hwbp_init_done;

/* Number of BRP/WRP registers on this CPU. */
static int core_num_brps, num_brps;
static int core_num_wrps, num_wrps;

/* DAP register list */
struct reginfo {
	u32 dap_reg;
	unsigned long *data;
};
#define HWBP_IDX_BP(slot)	(0 + ((slot) << 1))
#define HWBP_IDX_WP(slot)	(HWBP_MAX_BP*2 + ((slot) << 1))
#define HWBP_IDX_ADR		0
#define HWBP_IDX_CTL		1
#define HWBP_IDX_ADR_MSK	(1 << HWBP_IDX_ADR)
#define HWBP_IDX_CTL_MSK	(1 << HWBP_IDX_CTL)
#define HWBP_IDX_BOTH		(HWBP_IDX_ADR_MSK|HWBP_IDX_CTL_MSK)
static const struct reginfo hwbp_regs[(HWBP_MAX_BP+HWBP_MAX_WP)*2] = {
	[HWBP_IDX_BP(0)+HWBP_IDX_ADR] = { DAP_DBGBVR(0), &bp_addr[0] },
	[HWBP_IDX_BP(0)+HWBP_IDX_CTL] = { DAP_DBGBCR(0), &bp_ctrl[0] },
	[HWBP_IDX_BP(1)+HWBP_IDX_ADR] = { DAP_DBGBVR(1), &bp_addr[1] },
	[HWBP_IDX_BP(1)+HWBP_IDX_CTL] = { DAP_DBGBCR(1), &bp_ctrl[1] },
	[HWBP_IDX_BP(2)+HWBP_IDX_ADR] = { DAP_DBGBVR(2), &bp_addr[2] },
	[HWBP_IDX_BP(2)+HWBP_IDX_CTL] = { DAP_DBGBCR(2), &bp_ctrl[2] },
	[HWBP_IDX_WP(0)+HWBP_IDX_ADR] = { DAP_DBGWVR(0), &wp_addr[0] },
	[HWBP_IDX_WP(0)+HWBP_IDX_CTL] = { DAP_DBGWCR(0), &wp_ctrl[0] },
	[HWBP_IDX_WP(1)+HWBP_IDX_ADR] = { DAP_DBGWVR(1), &wp_addr[1] },
	[HWBP_IDX_WP(1)+HWBP_IDX_CTL] = { DAP_DBGWCR(1), &wp_ctrl[1] },
};

/*----------------------- cp14 I/F ------------------------*/
#define READ_WB_REG_CASE(OP2, M, VAL)		\
	case ((OP2 << 4) + M):			\
		ARM_DBG_READ(c ## M, OP2, VAL); \
		break

#define WRITE_WB_REG_CASE(OP2, M, VAL)		\
	case ((OP2 << 4) + M):			\
		ARM_DBG_WRITE(c ## M, OP2, VAL);\
		break

#define GEN_READ_WB_REG_CASES(OP2, VAL)		\
	READ_WB_REG_CASE(OP2, 0, VAL);		\
	READ_WB_REG_CASE(OP2, 1, VAL);		\
	READ_WB_REG_CASE(OP2, 2, VAL);		\
	READ_WB_REG_CASE(OP2, 3, VAL);		\
	READ_WB_REG_CASE(OP2, 4, VAL);		\
	READ_WB_REG_CASE(OP2, 5, VAL);		\
	READ_WB_REG_CASE(OP2, 6, VAL);		\
	READ_WB_REG_CASE(OP2, 7, VAL);		\
	READ_WB_REG_CASE(OP2, 8, VAL);		\
	READ_WB_REG_CASE(OP2, 9, VAL);		\
	READ_WB_REG_CASE(OP2, 10, VAL);		\
	READ_WB_REG_CASE(OP2, 11, VAL);		\
	READ_WB_REG_CASE(OP2, 12, VAL);		\
	READ_WB_REG_CASE(OP2, 13, VAL);		\
	READ_WB_REG_CASE(OP2, 14, VAL);		\
	READ_WB_REG_CASE(OP2, 15, VAL)

#define GEN_WRITE_WB_REG_CASES(OP2, VAL)	\
	WRITE_WB_REG_CASE(OP2, 0, VAL);		\
	WRITE_WB_REG_CASE(OP2, 1, VAL);		\
	WRITE_WB_REG_CASE(OP2, 2, VAL);		\
	WRITE_WB_REG_CASE(OP2, 3, VAL);		\
	WRITE_WB_REG_CASE(OP2, 4, VAL);		\
	WRITE_WB_REG_CASE(OP2, 5, VAL);		\
	WRITE_WB_REG_CASE(OP2, 6, VAL);		\
	WRITE_WB_REG_CASE(OP2, 7, VAL);		\
	WRITE_WB_REG_CASE(OP2, 8, VAL);		\
	WRITE_WB_REG_CASE(OP2, 9, VAL);		\
	WRITE_WB_REG_CASE(OP2, 10, VAL);	\
	WRITE_WB_REG_CASE(OP2, 11, VAL);	\
	WRITE_WB_REG_CASE(OP2, 12, VAL);	\
	WRITE_WB_REG_CASE(OP2, 13, VAL);	\
	WRITE_WB_REG_CASE(OP2, 14, VAL);	\
	WRITE_WB_REG_CASE(OP2, 15, VAL)

static u32 read_wb_reg(int n)
{
	u32 val = 0;

	switch (n) {
		GEN_READ_WB_REG_CASES(ARM_OP2_BVR, val);
		GEN_READ_WB_REG_CASES(ARM_OP2_BCR, val);
		GEN_READ_WB_REG_CASES(ARM_OP2_WVR, val);
		GEN_READ_WB_REG_CASES(ARM_OP2_WCR, val);
	default:
		printk(KERN_ERR "attempt to read from unknown breakpoint "
		       "register %d\n", n);
	}

	return val;
}

static void write_wb_reg(int n, u32 val)
{
	switch (n) {
		GEN_WRITE_WB_REG_CASES(ARM_OP2_BVR, val);
		GEN_WRITE_WB_REG_CASES(ARM_OP2_BCR, val);
		GEN_WRITE_WB_REG_CASES(ARM_OP2_WVR, val);
		GEN_WRITE_WB_REG_CASES(ARM_OP2_WCR, val);
	default:
		printk(KERN_ERR "attempt to write to unknown breakpoint "
		       "register %d\n", n);
	}
	isb();
}
/*---------------------------------------------------------*/

static int cp14_enable_monitor_mode(void)
{
	u32 dscr;
	int ret = 0;

	ARM_DBG_READ(c1, 0, dscr);

	/* Ensure that halting mode is disabled. */
	if (dscr & ARM_DSCR_HDBGEN) {
		ret = -EPERM;
		goto out;
	}

	/* If monitor mode is already enabled, just return. */
	if (dscr & ARM_DSCR_MDBGEN)
		goto out;

	/* clear VCR */
	asm volatile("mcr p14, 0, %0, c0, c7, 0" : : "r" (0));
	isb();

	/* Write to the corresponding DSCR. */
	ARM_DBG_WRITE(c2, 2, (dscr | ARM_DSCR_MDBGEN));

	/* Check that the write made it through. */
	ARM_DBG_READ(c1, 0, dscr);
	if (!(dscr & ARM_DSCR_MDBGEN))
		ret = -EPERM;

out:
	return ret;
}

static inline void dap_lock(int cpu)
{
	const void __iomem *p = VA_DAPDBG(cpu);

	writel(0, p + DAP_DBGLAR);
}

static inline void dap_unlock(int cpu)
{
	const void __iomem *p = VA_DAPDBG(cpu);

	writel(DAP_MAGIC, p + DAP_DBGLAR);
}

static int dap_enable_monitor_mode(int cpu)
{
	const void __iomem *p = VA_DAPDBG(cpu);
	u32 dscr;
	int ret = 0;

	dscr = readl_relaxed(p + DAP_DBGDSCR);

	/* Ensure that halting mode is disabled. */
	if (dscr & ARM_DSCR_HDBGEN) {
		ret = -EPERM;
		goto out;
	}

	/* If monitor mode is already enabled, just return. */
	if (dscr & ARM_DSCR_MDBGEN)
		goto out;

	/* clear VCR */
	writel(0, p + DAP_DBGVCR);

	/* Write to the corresponding DSCR. */
	writel(dscr | ARM_DSCR_MDBGEN, p + DAP_DBGDSCR);

	/* Check that the write made it through. */
	dscr = readl_relaxed(p + DAP_DBGDSCR);
	if (!(dscr & ARM_DSCR_MDBGEN))
		ret = -EPERM;

out:
	return ret;
}

struct hwbp_info {
	u32 type;
	u32 slot;
	u32 vreg;
	u32 creg;
	u32 dap_vreg;
	u32 dap_creg;
	u32 addr;
	u32 ctrl;
};

static int hwbp_chkparam(struct hwbp_info *info, u32 id)
{
	uint type, slot;

	if (HWBP_OFF == hwbp_mode || !hwbp_init_done)
		return -1;

	type = HWBP_TYPE(id);
	slot = HWBP_SLOT(id);
	info->type = type;
	info->slot = slot;
	switch (type) {
	case HWBP_TYPE_PC:
		if (slot >= num_brps) {
			printk(KERN_ERR "ERROR:%s:BP:id=%u\n", __func__, id);
			return -1;
		}
		info->vreg = ARM_BASE_BVR + slot;
		info->creg = ARM_BASE_BCR + slot;
		info->dap_vreg = DAP_DBGBVR(slot);
		info->dap_creg = DAP_DBGBCR(slot);
		break;
	case HWBP_TYPE_MEM:
		if (slot >= num_wrps) {
			printk(KERN_ERR "ERROR:%s:WP:id=%u\n", __func__, id);
			return -1;
		}
		info->vreg = ARM_BASE_WVR + slot;
		info->creg = ARM_BASE_WCR + slot;
		info->dap_vreg = DAP_DBGWVR(slot);
		info->dap_creg = DAP_DBGWCR(slot);
		break;
	default:
		printk(KERN_ERR "ERROR:%s:id=%u\n", __func__, id);
		return -1;
	}
	return 0;
}

static void hwbp_prepare(struct hwbp_info *info, u32 addr, u32 ctrl)
{
	info->addr = HWBP_ADDR(addr);
	info->ctrl = ctrl;
	if (HWBP_TYPE_PC == info->type) {
		bp_addr[info->slot] = addr;
		bp_ctrl[info->slot] = ctrl;
	} else {
		wp_addr[info->slot] = addr;
		wp_ctrl[info->slot] = ctrl;
	}
}


static void cpu_hwbp_setup(int cpu, void *data)
{
	struct hwbp_info *info = (struct hwbp_info *)data;
	const void __iomem *p;

	p = VA_DAPDBG(cpu);
	writel(info->addr, p + info->dap_vreg);
	writel(info->ctrl, p + info->dap_creg);
}

static void cpu_hwbp_ctrl(int cpu, void *data)
{
	struct hwbp_info *info = (struct hwbp_info *)data;
	const void __iomem *p;

	p = VA_DAPDBG(cpu);
	writel(info->ctrl, p + info->dap_creg);
}

static void cpu_hwbp_update(int cpu, void *data)
{
	unsigned int update = (unsigned int)data;
	const void __iomem *p;
	const struct reginfo *reg;

	p = VA_DAPDBG(cpu);
	reg = hwbp_regs;
	while (update) {
		if (update & 1) {
			writel(*reg->data, p + reg->dap_reg);
		}
		update >>= 1;
		reg++;
	}
}

static void hwbp_each_cpu(void (*func)(int cpu, void *data), void *data)
{
	int cpu;
	unsigned long flags;

	smp_wmb();
	raw_spin_lock_irqsave(&hwbp_dap_lock, flags);
	for_each_online_cpu(cpu) {
		dap_unlock(cpu);
		if (dap_enable_monitor_mode(cpu)) {
			printk(KERN_ERR "%s:cpu%d can not enter monitor debug mode\n",
			       __func__, cpu);
			continue;
		}
		(*func)(cpu, data);
		dap_lock(cpu);
	}
	raw_spin_unlock_irqrestore(&hwbp_dap_lock, flags);
}

/*---------------- kernel API -----------------*/
static unsigned int update_flag;

void hwbp_update(void)
{
	hwbp_each_cpu(cpu_hwbp_update, (void *)update_flag);
	update_flag = 0;
}
EXPORT_SYMBOL(hwbp_update);

void _hwbp_bp_setup(uint slot, u32 addr, u32 ctrl)
{
	bp_addr[slot] = HWBP_ADDR(addr);
	bp_ctrl[slot] = ctrl;
	update_flag |= HWBP_IDX_BOTH << HWBP_IDX_BP(slot);
}
EXPORT_SYMBOL(_hwbp_bp_setup);

void _hwbp_bp_ctrl(uint slot, int enable)
{
	if (enable)
		bp_ctrl[slot] |= HWBP_ENABLE;
	else
		bp_ctrl[slot] &= ~HWBP_ENABLE;
	update_flag |= HWBP_IDX_CTL_MSK << HWBP_IDX_BP(slot);
}
EXPORT_SYMBOL(_hwbp_bp_ctrl);

void _hwbp_wp_setup(uint slot, u32 addr, u32 ctrl)
{
	wp_addr[slot] = HWBP_ADDR(addr);
	wp_ctrl[slot] = ctrl;
	update_flag |= HWBP_IDX_BOTH << HWBP_IDX_WP(slot);
}
EXPORT_SYMBOL(_hwbp_wp_setup);

void _hwbp_wp_ctrl(uint slot, int enable)
{
	if (enable)
		wp_ctrl[slot] |= HWBP_ENABLE;
	else
		wp_ctrl[slot] &= ~HWBP_ENABLE;
	update_flag |= HWBP_IDX_CTL_MSK << HWBP_IDX_WP(slot);
}
EXPORT_SYMBOL(_hwbp_wp_ctrl);

int hwbp_setup(u32 id, u32 addr, u32 ctrl)
{
	struct hwbp_info info;

	if (hwbp_chkparam(&info, id) < 0)
		return -1;
	hwbp_prepare(&info, addr, ctrl);
	hwbp_each_cpu(cpu_hwbp_setup, &info);
	return 0;
}
EXPORT_SYMBOL(hwbp_setup);

void hwbp_ctrl_local(u32 id, int enable)
{
	struct hwbp_info info;
	u32 old, new;
	ulong flags;

	if (hwbp_chkparam(&info, id) < 0)
		return;

	if (cp14_enable_monitor_mode())
		return;

	local_irq_save(flags);
	old = read_wb_reg(info.creg);
	if (enable)
		new = old | HWBP_ENABLE;
	else
		new = old & ~HWBP_ENABLE;
	if (new != old) {
		write_wb_reg(info.creg, new);
	}
	local_irq_restore(flags);
}
EXPORT_SYMBOL(hwbp_ctrl_local);

int hwbp_enable(u32 id)
{
	struct hwbp_info info;
	unsigned long *p;

	if (hwbp_chkparam(&info, id) < 0)
		return -1;
	if (HWBP_TYPE_PC == info.type) {
		p = &bp_ctrl[info.slot];
	} else {
		p = &wp_ctrl[info.slot];
	}
	*p |= HWBP_ENABLE;
	info.ctrl = *p;
	hwbp_each_cpu(cpu_hwbp_ctrl, &info);
	return 0;
}
EXPORT_SYMBOL(hwbp_enable);

int hwbp_disable(u32 id)
{
	struct hwbp_info info;
	unsigned long *p;

	if (hwbp_chkparam(&info, id) < 0)
		return -1;
	if (HWBP_TYPE_PC == info.type) {
		p = &bp_ctrl[info.slot];
	} else {
		p = &wp_ctrl[info.slot];
	}
	info.ctrl = *p & ~HWBP_ENABLE;
	hwbp_each_cpu(cpu_hwbp_ctrl, &info);
	*p = info.ctrl;
	return 0;
}
EXPORT_SYMBOL(hwbp_disable);

int hwbp_find_slot(unsigned long addr)
{
	int i;

	for (i = 0; i < HWBP_MAX_BP; i++) {
		if (bp_addr[i] == addr)
			return i;
	}
	return -1;
}
EXPORT_SYMBOL(hwbp_find_slot);
/*---------------------------------------------------------------*/

static void cpu_hwbp_setup_full(void *info)
{
	int i;

	if (cp14_enable_monitor_mode())
		return;

	/* break point */
	for (i = 0; i < num_brps; i++) {
		write_wb_reg(ARM_BASE_BVR + i, HWBP_ADDR(bp_addr[i]));
		write_wb_reg(ARM_BASE_BCR + i, bp_ctrl[i]);
	}
	/* watch point */
	for (i = 0; i < num_wrps; i++) {
		write_wb_reg(ARM_BASE_WVR + i, HWBP_ADDR(wp_addr[i]));
		write_wb_reg(ARM_BASE_WCR + i, wp_ctrl[i]);
	}
}

static int __cpuinit
cpu_callback(struct notifier_block *nfb, unsigned long action, void *hcpu)
{
	switch (action) {
	case CPU_STARTING:
	case CPU_STARTING_FROZEN:
		cpu_hwbp_setup_full(NULL);
		break;
	}
	return NOTIFY_OK;
}

static struct notifier_block __cpuinitdata cpu_nfb = {
	.notifier_call = cpu_callback,
};

/* proc I/F
   Usage
   	show:  cat /proc/hwbp
   	setup: echo 1 > /proc/hwbp
   	clear: echo 0 > /proc/hwbp
*/
static ssize_t hwbp_write(struct file *file , const char __user *buf,
			  size_t len, loff_t *ppos)
{
	char arg[16];
	int i;

	if (len > sizeof arg - 1) {
		printk(KERN_ERR "/proc/%s: too long\n", HWBP_PROCNAME);
		return -1;
	}
	if (copy_from_user(arg, buf, len)) {
		return -EFAULT;
	}
	arg[len] = '\0';

	if (HWBP_OFF == hwbp_mode)
		return len;

	if ('0' == arg[0]) { /* clear all */
		for (i = 0; i < HWBP_MAX_BP; i++) {
			bp_addr[i] = 0;
			bp_ctrl[i] = 0;
		}
		for (i = 0; i < HWBP_MAX_WP; i++) {
			wp_addr[i] = 0;
			wp_ctrl[i] = 0;
		}
	}

	if (!hwbp_init_done) {
		hwbp_init_done = 1;
		register_cpu_notifier(&cpu_nfb);
	}

	on_each_cpu(cpu_hwbp_setup_full, NULL, 1);
	return len;
}

#define HWBP_NR_SEQ (1 + num_brps * 2 + num_wrps * 2)

static void *hwbp_seq_start(struct seq_file *seq, loff_t *pos)
{
	if (*pos < HWBP_NR_SEQ)
		return pos;
	return NULL;
}

static void *hwbp_seq_next(struct seq_file *seq, void *v, loff_t *pos)
{
	(*pos)++;
	if (*pos < HWBP_NR_SEQ)
		return pos;
	return NULL;
}

static void hwbp_seq_stop(struct seq_file *seq, void *v)
{
}

struct hwbp_read_info {
	int idx;
	u32 *data;
};

static void cpu_hwbp_read(int cpu, void *data)
{
	struct hwbp_read_info *info = (struct hwbp_read_info *)data;
	const void __iomem *p;

	p = VA_DAPDBG(cpu);
	info->data[cpu] = readl(p + hwbp_regs[info->idx].dap_reg);
}

static void hwbp_show_regs(struct seq_file *seq, int type, int slot, int ctrl)
{
	struct hwbp_read_info info;
	u32 data[NR_CPUS];
	int i;

	info.idx = (HWBP_TYPE_PC == type) ? HWBP_IDX_BP(slot):HWBP_IDX_WP(slot);
	info.idx += ctrl;
	info.data = data;
	memset(data, 0, sizeof data);
	hwbp_each_cpu(cpu_hwbp_read, (void *)&info);

	seq_printf(seq, ":");
	for (i = 0; i < NR_CPUS; i++) {
		seq_printf(seq, " 0x%08x", data[i]);
	}
}

static int hwbp_seq_show(struct seq_file *seq, void *v)
{
	int i = *(loff_t *)v;
	u32 dscr;
	int slot, ctrl;

	if (HWBP_OFF == hwbp_mode) {
                if (!i) {
                        seq_printf(seq, "HWBP: off\n");
                }
                return 0;
        }
        if (!i) {
		ARM_DBG_READ(c1, 0, dscr);
                seq_printf(seq, "HWBP: ");
		if (dscr & ARM_DSCR_HDBGEN)
			seq_printf(seq, "halting-mode");
		else if (dscr & ARM_DSCR_MDBGEN)
			seq_printf(seq, "monitor-mode");
		seq_printf(seq, ", %u bp, %u wp (%u bp, %u wp)",
			   num_brps, num_wrps,
			   core_num_brps, core_num_wrps);
		seq_printf(seq, "\n");
        } else if (i < 1 + num_brps * 2) {
		slot = (i - 1) >> 1;
		ctrl = (i - 1) & 1;
		if (!ctrl) {
			seq_printf(seq, "     BP%d: 0x%08lx", slot,
				   bp_addr[slot]);
		} else {
			seq_printf(seq, "          0x%08lx", bp_ctrl[slot]);
		}
		hwbp_show_regs(seq, HWBP_TYPE_PC, slot, ctrl);
		seq_printf(seq, "\n");
	} else {
		slot = (i - (1 + num_brps * 2)) >> 1;
		ctrl = (i - (1 + num_brps * 2)) & 1;
		if (!ctrl) {
			seq_printf(seq, "     WP%d: 0x%08lx", slot,
				   wp_addr[slot]);
		} else {
			seq_printf(seq, "          0x%08lx", wp_ctrl[slot]);
		}
		hwbp_show_regs(seq, HWBP_TYPE_MEM, slot, ctrl);
		seq_printf(seq, "\n");
	}
        return 0;
}

static struct seq_operations hwbp_seq_ops = {
        .start  = hwbp_seq_start,
        .next   = hwbp_seq_next,
        .stop   = hwbp_seq_stop,
        .show   = hwbp_seq_show,
};

static int hwbp_seq_open(struct inode *inode, struct file *file)
{
	return seq_open(file, &hwbp_seq_ops);
}

static struct file_operations proc_hwbp_fops = {
	.owner		= THIS_MODULE,
	.open           = hwbp_seq_open,
	.read           = seq_read,
	.llseek         = seq_lseek,
	.release        = seq_release,
	.write		= hwbp_write,
};

static int (*hwbp_handler)(int type, unsigned long addr, unsigned int fsr, struct pt_regs *regs);
void hwbp_set_handler(int (*fn)(int, unsigned long, unsigned int, struct pt_regs *))
{
	hwbp_handler = fn;
}
EXPORT_SYMBOL(hwbp_set_handler);

/*
 * Called from either the Data Abort Handler [watchpoint] or the
 * Prefetch Abort Handler [breakpoint] with interrupts disabled.
 */
static int hwbp_exception(unsigned long addr, unsigned int fsr,
			  struct pt_regs *regs)
{
	int ret = 0;
	u32 dscr;

	if (!hwbp_handler)
		return 1; /* Unhandled */

	preempt_disable();

	ARM_DBG_READ(c1, 0, dscr);
	switch (ARM_DSCR_MOE(dscr)) {
	case ARM_ENTRY_BREAKPOINT:
		ret = (*hwbp_handler)(HWBP_TYPE_PC, addr, fsr, regs);
		break;
	case ARM_ENTRY_SYNC_WATCHPOINT:
		ret = (*hwbp_handler)(HWBP_TYPE_MEM, addr, fsr, regs);
		break;
	default:
		ret = 1;
		break;
	}

	preempt_enable();

	return ret;
}

static struct proc_dir_entry *proc_hwbp = NULL;

/* pre_smp_initcalls */
static int __init hwbp_init(void)
{
	u32 didr;
	extern void hwbp_ctrl_init(void);

	proc_hwbp = create_proc_entry(HWBP_PROCNAME, 0, NULL);
	if (!proc_hwbp) {
		printk(KERN_ERR "ERROR:%s:can not create proc entry\n",
		       __func__);
	} else {
		proc_hwbp->proc_fops = &proc_hwbp_fops;
	}

	/* resource info. */
	ARM_DBG_READ(c0, 0, didr);
	num_brps = core_num_brps = ((didr >> 24) & 0xf) + 1;
	num_wrps = core_num_wrps = ((didr >> 28) & 0xf) + 1;
	if (num_brps > HWBP_MAX_BP) {
		num_brps = HWBP_MAX_BP;
	}
	if (num_wrps > HWBP_MAX_WP) {
		num_wrps = HWBP_MAX_WP;
	}

	/* Register debug fault handler. */
	hook_fault_code(2, hwbp_exception, SIGTRAP, TRAP_HWBKPT,
			"watchpoint debug exception");
	hook_ifault_code(2, hwbp_exception, SIGTRAP, TRAP_HWBKPT,
			"breakpoint debug exception");

	if (HWBP_OFF == hwbp_mode)
		return 0;

	/* before cpu_callback */
	hwbp_ctrl_init();
	/*--------------------------------------------*/
	update_flag = 0;
	cpu_callback(&cpu_nfb, CPU_STARTING, (void *)0);
	register_cpu_notifier(&cpu_nfb);
	hwbp_init_done = 1;

	return 0;
}
early_initcall(hwbp_init);

void cxd900x0_hwbp_resume(void)
{
	if (HWBP_OFF == hwbp_mode || !hwbp_init_done)
		return;

	cpu_callback(&cpu_nfb, CPU_STARTING, (void *)0);
}
