/* 2012-06-22: File changed by Sony Corporation */
/*
 *  linux/arch/arm/mm/pgd.c
 *
 *  Copyright (C) 1998-2005 Russell King
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */
#include <linux/mm.h>
#include <linux/gfp.h>
#include <linux/highmem.h>
#include <linux/slab.h>

#include <asm/pgalloc.h>
#include <asm/page.h>
#include <asm/tlbflush.h>

#include "mm.h"

#ifndef CONFIG_EJ_SPLIT_PAGETABLE
#define PGD_ORDER 2
#else /* CONFIG_EJ_SPLIT_PAGETABLE */
#if CONFIG_EJ_SPLIT_SHIFT < 3
#define PGD_ORDER (2 - CONFIG_EJ_SPLIT_SHIFT)
#else
#define PGD_SIZE (PAGE_SIZE)/(1 << (CONFIG_EJ_SPLIT_SHIFT - 2))
#endif
#endif

#ifdef PGD_ORDER
#define PGD_SIZE (PAGE_SIZE * (1 << PGD_ORDER))

# ifdef CONFIG_EJ_PGD_PREALLOC
#  include <linux/mempool.h>
#  include <linux/module.h>
#  include <linux/snsc_boot_time.h>

static int nr_pgd;
module_param_named(n, nr_pgd, int, S_IRUSR);
static int pgd_debug;
module_param_named(debug, pgd_debug, int, S_IRUSR|S_IWUSR);
static unsigned int pgd_min = ~0U;
static mempool_t *pgd_pool;

static inline pgd_t *mempool_alloc_pgd(void)
{
	struct page *page;
	pgd_t *pgd = NULL;

	if (!pgd_pool) {
		return (pgd_t *)__get_free_pages(GFP_KERNEL, PGD_ORDER);
	}
	page = (struct page *)mempool_alloc(pgd_pool, GFP_KERNEL|__GFP_PREALLOC);
	if (likely(page)) {
		pgd = (pgd_t *)page_address(page);
	}
	if (unlikely(pgd_debug)) {
		int remain;
		char buf[64];
		smp_rmb();
		remain = pgd_pool->curr_nr;
		/* update pgd_min w/o LOCK. */
		if (remain < pgd_min) {
			pgd_min = remain;
			smp_wmb();
			scnprintf(buf, sizeof buf, "pgd_pool:%u", pgd_min);
			BOOT_TIME_ADD1(buf);
		}
	}
	return pgd;
}

static inline void mempool_free_pgd(pgd_t *pgd)
{
	if (!pgd_pool) {
		free_pages((unsigned long)pgd, PGD_ORDER);
		return;
	}
	mempool_free(virt_to_page((void *)pgd), pgd_pool);
}

static int pgd_init(void)
{
	if (nr_pgd <= 0)
		return 0;
	pgd_pool = mempool_create_page_pool(nr_pgd, PGD_ORDER);
	if (!pgd_pool) {
		printk(KERN_ERR "%s: mempool_create_page_pool(%d,%d) failed\n",
		       __func__, nr_pgd, PGD_ORDER);
		BUG();
	}
	return 0;
}
late_initcall(pgd_init);

#  define ALLOC_PGD() mempool_alloc_pgd()
#  define FREE_PGD(p) mempool_free_pgd(p)

# else /* CONFIG_EJ_PGD_PERALLOC */

#define ALLOC_PGD() (pgd_t *)__get_free_pages(GFP_KERNEL, PGD_ORDER)
#define FREE_PGD(p) free_pages((unsigned long)(p), PGD_ORDER)

# endif /* CONFIG_EJ_PGD_PREALLOC */
#else
#define ALLOC_PGD() ({BUG_ON(!pgd_cachep);(pgd_t *)kmem_cache_alloc(pgd_cachep, GFP_KERNEL);})
#define FREE_PGD(p) kmem_cache_free(pgd_cachep, p)

static struct kmem_cache *pgd_cachep;
static int pgd_init(void)
{
    pgd_cachep = kmem_cache_create("pgd", PGD_SIZE, PGD_SIZE, SLAB_PANIC, NULL);
    return 0;
}

late_initcall(pgd_init);
#endif

/*
 * need to get a 16k page for level 1
 */
pgd_t *pgd_alloc(struct mm_struct *mm)
{
	pgd_t *new_pgd, *init_pgd;
	pud_t *new_pud, *init_pud;
	pmd_t *new_pmd, *init_pmd;
	pte_t *new_pte, *init_pte;

	new_pgd = ALLOC_PGD();
	if (!new_pgd)
		goto no_pgd;

	memset(new_pgd, 0, USER_PTRS_PER_PGD * sizeof(pgd_t));

	/*
	 * Copy over the kernel and IO PGD entries
	 */
	init_pgd = pgd_offset_k(0);
#ifndef CONFIG_EJ_SPLIT_PAGETABLE
	memcpy(new_pgd + USER_PTRS_PER_PGD, init_pgd + USER_PTRS_PER_PGD,
		       (PTRS_PER_PGD - USER_PTRS_PER_PGD) * sizeof(pgd_t));
#else /* CONFIG_EJ_SPLIT_PAGETABLE */
# if TASK_SIZE < CONFIG_PAGE_OFFSET
	memcpy(new_pgd + USER_PTRS_PER_PGD, init_pgd + USER_PTRS_PER_PGD,
	       (CONFIG_PAGE_OFFSET-TASK_SIZE)/PGDIR_SIZE * sizeof(pgd_t));
# endif
#endif
	clean_dcache_area(new_pgd, PGD_SIZE);
	if (!vectors_high()) {
		/*
		 * On ARM, first page must always be allocated since it
		 * contains the machine vectors.
		 */
		new_pud = pud_alloc(mm, new_pgd, 0);
		if (!new_pud)
			goto no_pud;

		new_pmd = pmd_alloc(mm, new_pud, 0);
		if (!new_pmd)
			goto no_pmd;

		new_pte = pte_alloc_map(mm, NULL, new_pmd, 0);
		if (!new_pte)
			goto no_pte;

		init_pud = pud_offset(init_pgd, 0);
		init_pmd = pmd_offset(init_pud, 0);
		init_pte = pte_offset_map(init_pmd, 0);
		set_pte_ext(new_pte, *init_pte, 0);
		pte_unmap(init_pte);
		pte_unmap(new_pte);
	}

	return new_pgd;

no_pte:
	pmd_free(mm, new_pmd);
no_pmd:
	pud_free(mm, new_pud);
no_pud:
	FREE_PGD(new_pgd);
no_pgd:
	return NULL;
}

void pgd_free(struct mm_struct *mm, pgd_t *pgd_base)
{
	pgd_t *pgd;
	pud_t *pud;
	pmd_t *pmd;
	pgtable_t pte;

	if (!pgd_base)
		return;

	pgd = pgd_base + pgd_index(0);
	if (pgd_none_or_clear_bad(pgd))
		goto no_pgd;

	pud = pud_offset(pgd, 0);
	if (pud_none_or_clear_bad(pud))
		goto no_pud;

	pmd = pmd_offset(pud, 0);
	if (pmd_none_or_clear_bad(pmd))
		goto no_pmd;

	pte = pmd_pgtable(*pmd);
	pmd_clear(pmd);
	pte_free(mm, pte);
no_pmd:
	pud_clear(pud);
	pmd_free(mm, pmd);
no_pud:
	pgd_clear(pgd);
	pud_free(mm, pud);
no_pgd:
	FREE_PGD(pgd);
}
