/* 2015-08-31: File added by Sony Corporation */
/*
 *  Copyright 2013 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/sched.h>
#include <linux/stacktrace.h>
#include <linux/kallsyms.h>
#include <linux/page-event.h>

static struct page_event_trace *get_page_event_trace(struct page *page)
{
	return &page->event_trace;
}

static void init_event_record(struct page_event_record *record,
			      unsigned long event)
{
	struct stack_trace trace;

	record->event = event;
	record->tgid  = current->tgid;
	record->pid   = current->pid;

	trace.nr_entries  = 0;
	trace.entries     = record->backtrace;
	trace.max_entries = PAGE_EVENT_MAX_BACKTRACE;
	trace.skip        = PAGE_EVENT_SKIP_BACKTRACE;

	save_stack_trace(&trace);
}

static void set_page_event(struct page *page,
			   struct page_event_record *record)
{
	struct page_event_trace *trace = get_page_event_trace(page);

	if (!trace)
		return;

	trace->record[trace->next] = *record;
	trace->next = (trace->next + 1) % PAGE_EVENT_MAX_HISTROY;
}

static char* get_comm_by_pid(pid_t pid, char *comm)
{
	struct task_struct *task;

	/*
	 * To reduce memory consumption, do not record comm string,
	 * and try to restore it from pid at memory corruption
	 * detected. Because a culprit is likely still alive at the
	 * detection, this approach works well in most cases.
	 */
	task = find_task_by_pid_ns(pid, &init_pid_ns);
	if (!task)
		return "N/A";

	return get_task_comm(comm, task);
}

static char *get_event_string(unsigned long event, char *str, size_t size)
{
	switch (event) {
	case PAGE_EVENT_ALLOC : return "alloc";
	case PAGE_EVENT_FREE  : return "free";
	}

	snprintf(str, size, "0x%08lx", event);

	return str;
}

static void print_event_record(struct page_event_record *record, int age)
{
	int i;
	char evstr[32];
	char tgid_comm[sizeof(current->comm)];
	char pid_comm[sizeof(current->comm)];

	pr_info("%s : age=%d, tgid=%d(%s), pid=%d(%s)\n",
		get_event_string(record->event, evstr, sizeof(evstr)), age,
		record->tgid, get_comm_by_pid(record->tgid, tgid_comm),
		record->pid, get_comm_by_pid(record->pid, pid_comm));

	for (i = 0; i < PAGE_EVENT_MAX_BACKTRACE; i++) {
		if (record->backtrace[i] != ULONG_MAX)
			print_ip_sym(record->backtrace[i]);
	}
}

void dump_page_event(struct page *page)
{
	int i;
	int age;
	int index;
	struct page_event_record *record;
	struct page_event_trace *trace = get_page_event_trace(page);

	if (!trace)
		return;

	pr_info("page evnet trace : phys=0x%05ux, smaller age is older\n",
		page_to_phys(page));

	for (i = 0, age = 0; i < PAGE_EVENT_MAX_HISTROY; i++) {
		index = (trace->next + i) % PAGE_EVENT_MAX_HISTROY;
		record = &trace->record[index];

		if (!record->tgid)
			continue;

		print_event_record(record, age++);
	}
}

void trace_page_event(struct page *page, int numpages, unsigned long event)
{
	int i;
	struct page_event_record record;

	init_event_record(&record, event);

	for (i = 0; i < numpages; i++)
		set_page_event(page + i, &record);
}

static int __init init_page_event_trace(void)
{
	int size, perc;

	size = sizeof(struct page_event_trace);
	perc = (size * 100 * 10 + (PAGE_SIZE - 1)) / PAGE_SIZE;

	pr_info("Memory usage of page event trace : "
		"%d.%d%% of RAM (%dbytes/page)\n",
		perc / 10, perc % 10, size);

	return 0;
}
module_init(init_page_event_trace);
