/*
 * Copyright (c) 2015 Elliptic Technologies Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <elpspacc.h>


int spacc_clone_handle(spacc_device *spacc, int old_handle, void *cbdata)
{
	int new_handle;

	new_handle = spacc_job_request(spacc, spacc->job[old_handle].ctx_idx);
	if (new_handle < 0) {
		return new_handle;
	}
	spacc->job[new_handle]          = spacc->job[old_handle];
	spacc->job[new_handle].job_used = new_handle;
	spacc->job[new_handle].cbdata   = cbdata;
	return new_handle;
}

/*
 Allocates a job for spacc module context and initialize it with an
 appropriate type.
 */

static uint32_t CTRL_SET_CIPH_ALG_GEN(spacc_device *spacc, uint32_t x)
{
	return CTRL_SET_CIPH_ALG_415(x);
}

static uint32_t CTRL_SET_CIPH_MODE_GEN(spacc_device *spacc, uint32_t x)
{
	return CTRL_SET_CIPH_MODE_415(x);
}

static uint32_t CTRL_SET_HASH_ALG_GEN(spacc_device *spacc, uint32_t x)
{
	return CTRL_SET_HASH_ALG_415(x);
}

static uint32_t CTRL_SET_HASH_MODE_GEN(spacc_device *spacc, uint32_t x)
{
	return CTRL_SET_HASH_MODE_415(x);
}

int spacc_open (spacc_device *spacc, int enc, int hash, int ctxid, int secure_mode, spacc_callback cb, void *cbdata)
{
	int ret = CRYPTO_OK;
	int job_idx = 0;
	spacc_job *job = NULL;
	uint32_t ctrl = 0;

	if ((job_idx = spacc_job_request (spacc, ctxid)) < 0) {
		ret = CRYPTO_FAILED;
	} else {
		job = &spacc->job[job_idx];

		job->auxinfo_cs_mode = 0;
		job->icv_len = 0;

		switch (enc) {
		case CRYPTO_MODE_NULL:
			break;
		case CRYPTO_MODE_AES_CBC:
			ctrl |= CTRL_SET_CIPH_ALG_GEN (spacc,C_AES);
			ctrl |= CTRL_SET_CIPH_MODE_GEN (spacc,CM_CBC);
			break;
		case CRYPTO_MODE_AES_CTR:
			ctrl |= CTRL_SET_CIPH_ALG_GEN (spacc,C_AES);
			ctrl |= CTRL_SET_CIPH_MODE_GEN (spacc,CM_CTR);
			break;
		case CRYPTO_MODE_AES_GCM:
			ctrl |= CTRL_SET_CIPH_ALG_GEN (spacc,C_AES);
			ctrl |= CTRL_SET_CIPH_MODE_GEN (spacc,CM_GCM);
			break;
		case CRYPTO_MODE_AES_XTS:
			ctrl |= CTRL_SET_CIPH_ALG_GEN (spacc,C_AES);
			ctrl |= CTRL_SET_CIPH_MODE_GEN (spacc,CM_XTS);
			break;
		default:
			ret = CRYPTO_INVALID_ALG;
			break;
		}
		switch (hash) {
		case CRYPTO_MODE_NULL:
			ctrl |= CTRL_SET_HASH_ALG_GEN (spacc,H_NULL);
			break;

		case CRYPTO_MODE_HMAC_SHA256:
			ctrl |= CTRL_SET_HASH_ALG_GEN (spacc,H_SHA256);
			ctrl |= CTRL_SET_HASH_MODE_GEN (spacc,HM_HMAC);
			break;

		default:
			ret = CRYPTO_INVALID_ALG;
			break;
		}
	}

	ctrl |= (1UL << _SPACC_CTRL_MSG_BEGIN) | (1UL << _SPACC_CTRL_MSG_END);

	if (ret != CRYPTO_OK) {
		spacc_job_release (spacc, job_idx);
	} else {
		ret = job_idx;
		job->first_use   = 1;
		job->enc_mode    = enc;
		job->hash_mode   = hash;
		job->ckey_sz     = 0;
		job->hkey_sz     = 0;
		job->job_done    = 0;
		job->job_swid    = 0;
		job->job_secure  = ! !secure_mode;
		job->auxinfo_dir = 0;
		job->auxinfo_bit_align = 0;
		job->job_err     = CRYPTO_INPROGRESS;
		job->ctrl        = ctrl | CTRL_SET_CTX_IDX (job->ctx_idx);
		job->cb          = cb;
		job->cbdata      = cbdata;
	}
	return ret;
}
