// ------------------------------------------------------------------------
//
//                (C) COPYRIGHT 2011 - 2015 SYNOPSYS, INC.
//                          ALL RIGHTS RESERVED
//
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  version 2 as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, see <https://gnu.org/licenses/>.
//
// ------------------------------------------------------------------------

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>

#include <linux/io.h>

#include <linux/fs.h>
#include <asm/uaccess.h>
#include <asm/param.h>
#include <linux/interrupt.h>
#include <linux/platform_device.h>
#include <linux/list.h>
#include <linux/miscdevice.h>

#include <elppdu.h>
#include <elpspacc.h>
#include <elpspaccdrv.h>

//extern int spacc_dev_init (void);
//extern void spacc_dev_exit (void);
int dump_ddt;
EXPORT_SYMBOL(dump_ddt);
int dump_pkt_info;
EXPORT_SYMBOL(dump_pkt_info);
int dump_key_context_memory; //dump IV only work for non-IV-imported FPGA
EXPORT_SYMBOL(dump_key_context_memory);

int spacc_enqueue_all;
EXPORT_SYMBOL (spacc_enqueue_all);
int pop_jobs_in_isr;
EXPORT_SYMBOL (pop_jobs_in_isr);

#ifdef SPACC_DEBUG_PERF
int perf;
EXPORT_SYMBOL (perf);
volatile s64 spacc_kt[SPACC_PERF_MAX_ROUNDS][13];
EXPORT_SYMBOL (spacc_kt);
volatile int kt_idx;
EXPORT_SYMBOL (kt_idx);
#endif

struct platform_device * get_spacc_platdev_by_epn(uint32_t epn, uint32_t virt)
{
	char name[256];
	struct device *dev;

	snprintf(name, sizeof(name), "spacc.%d", (epn << 16) | virt);
	dev = bus_find_device_by_name(&platform_bus_type, NULL, name);
	if (!dev) {
		printk(KERN_ERR "failed to find device for %s\n", name);
		return NULL;
	}
	return to_platform_device(dev);
}
EXPORT_SYMBOL(get_spacc_platdev_by_epn);

/* This is used by RE and KEP to get the spacc device */
spacc_device * get_spacc_device_by_epn(uint32_t epn, uint32_t virt)
{
	struct platform_device *plat;
	struct spacc_priv *priv;

	plat = get_spacc_platdev_by_epn(epn, virt);
	if (!plat)
		return NULL;

	priv = platform_get_drvdata(plat);
	return &priv->spacc;
}
EXPORT_SYMBOL(get_spacc_device_by_epn);

/* a function to run callbacks in the IRQ handler */
static void spacc_pop_jobs (unsigned long data);
static irqreturn_t spacc_irq_handler(int irq, void *dev)
{
	struct spacc_priv *priv = platform_get_drvdata(to_platform_device(dev));
	spacc_device *spacc = &priv->spacc;

#ifdef SPACC_DEBUG_PERF
	if(perf)
		spacc_kt[kt_idx][11] =ktime_get().tv64;
#endif
	/* check irq flags and process as required */
	if (!spacc_process_irq(spacc)) {
		return IRQ_NONE;
	}
	if(pop_jobs_in_isr)
		spacc_pop_jobs((unsigned long)priv);

#ifdef SPACC_DEBUG_PERF
	if(perf)
		spacc_kt[kt_idx][12] =ktime_get().tv64;
#endif
	return IRQ_HANDLED;
}

/* callback function to initialize tasklet running */
static void spacc_stat_process(spacc_device *spacc)
{
	struct spacc_priv *priv = container_of(spacc, struct spacc_priv, spacc);

	/* run tasklet to pop jobs off fifo */
	tasklet_schedule(&priv->pop_jobs);
}

static void spacc_pop_jobs (unsigned long data)
{
	struct spacc_priv * priv =  (struct spacc_priv *)data;
	spacc_device *spacc = &priv->spacc;
	int num;

	spacc_pop_packets(spacc, &num);
}

#define HW_ENTRY(x) { x, #x }
static const struct { unsigned addr; char *name; } reg_names[] = {
	HW_ENTRY(SPACC_REG_IRQ_EN),
	HW_ENTRY(SPACC_REG_IRQ_STAT),
	HW_ENTRY(SPACC_REG_IRQ_CTRL),
	HW_ENTRY(SPACC_REG_FIFO_STAT),
	HW_ENTRY(SPACC_REG_SDMA_BRST_SZ),
	HW_ENTRY(SPACC_REG_HSM_CMD_REQ),
	HW_ENTRY(SPACC_REG_HSM_CMD_GNT),
	HW_ENTRY(SPACC_REG_SRC_PTR),
	HW_ENTRY(SPACC_REG_DST_PTR),
	HW_ENTRY(SPACC_REG_OFFSET),
	HW_ENTRY(SPACC_REG_PRE_AAD_LEN),
	HW_ENTRY(SPACC_REG_POST_AAD_LEN),
	HW_ENTRY(SPACC_REG_PROC_LEN),
	HW_ENTRY(SPACC_REG_ICV_LEN),
	HW_ENTRY(SPACC_REG_ICV_OFFSET),
	HW_ENTRY(SPACC_REG_IV_OFFSET),
	HW_ENTRY(SPACC_REG_SW_CTRL),
	HW_ENTRY(SPACC_REG_AUX_INFO),
	HW_ENTRY(SPACC_REG_CTRL),
	HW_ENTRY(SPACC_REG_STAT_POP),
	HW_ENTRY(SPACC_REG_STATUS),
	HW_ENTRY(SPACC_REG_STAT_WD_CTRL),
	HW_ENTRY(SPACC_REG_KEY_SZ),
	HW_ENTRY(SPACC_REG_VIRTUAL_RQST),
	HW_ENTRY(SPACC_REG_VIRTUAL_ALLOC),
	HW_ENTRY(SPACC_REG_VIRTUAL_PRIO),
	HW_ENTRY(SPACC_REG_VIRTUAL_RC4_KEY_RQST),
	HW_ENTRY(SPACC_REG_VIRTUAL_RC4_KEY_GNT),
	HW_ENTRY(SPACC_REG_ID),
	HW_ENTRY(SPACC_REG_CONFIG),
	HW_ENTRY(SPACC_REG_CONFIG2),
	HW_ENTRY(SPACC_REG_HSM_VERSION),
	HW_ENTRY(SPACC_REG_SECURE_CTRL),
	HW_ENTRY(SPACC_REG_SECURE_RELEASE),
	HW_ENTRY(SPACC_REG_SK_LOAD),
	HW_ENTRY(SPACC_REG_SK_STAT),
	HW_ENTRY(SPACC_REG_SK_KEY),
	HW_ENTRY(SPACC_REG_HSM_CTX_CMD),
	HW_ENTRY(SPACC_REG_HSM_CTX_STAT),

	{ 0, NULL },
};
#undef HW_ENTRY

static uint32_t reg_epn=0, reg_virt=0;

static ssize_t show_reg(struct device *dev, struct device_attribute *devattr, char *buf)
{
	char *name, out[128], work_buf[128];
	unsigned x, reg_addr;
	spacc_device *spacc = get_spacc_device_by_epn(reg_epn, reg_virt);

	work_buf[0] = 0;
	buf[0] = 0;
	if (!spacc) {
		snprintf( work_buf, sizeof( work_buf ),
				 "Could not find SPAcc device (EPN=%lx,%lu), please write to this file first in the form <epn,virt>\n",
				 (unsigned long)reg_epn, (unsigned long)reg_virt);
		strncat( buf, work_buf, sizeof( work_buf ) );
		return strlen(buf);
	}

	out[0] = 0;
	for (reg_addr = 0; reg_addr < 0x300; reg_addr += 4) {
		name = NULL;
		for (x = 0; reg_names[x].name != NULL; x++) {
			if (reg_names[x].addr == reg_addr) {
				name = reg_names[x].name;
				break;
			}
		}
		if (name == NULL) { continue; }
		snprintf( out, sizeof( out ), "%-35s = %08lx\n", name, (unsigned long)pdu_io_read32(spacc->regmap + reg_addr) );
		strncat( buf, out, sizeof( out ) );
	}
	return strlen(buf);
}

static ssize_t store_reg(struct device *dev, struct device_attribute *devattr, const char *buf, size_t count)
{
	unsigned long x, y;

	if (sscanf(buf, "%lx,%lu", &x, &y) != 2)
		return -EINVAL;

	reg_epn = x;
	reg_virt = y;
	return count;
}

static DEVICE_ATTR(reg, 0600, show_reg, store_reg);
static const struct attribute_group spacc_attr_group = {
	.attrs = (struct attribute *[]) {
		&dev_attr_reg.attr,
		NULL
	}
};

static int __devinit spacc_probe(struct platform_device *pdev)
{
	void *baseaddr;
	struct resource *mem, *irq;
	int x, err;
	struct spacc_priv   *priv;
	pdu_info     info;

	dev_info(&pdev->dev, "probe called!\n");

	/* Setup DMA masks */
        err = dma_set_mask(&pdev->dev, DMA_BIT_MASK(64));
        if (err)
                return err;
        err = dma_set_coherent_mask(&pdev->dev, DMA_BIT_MASK(64));
        if (err)
                return err;

	/* Initialize DDT DMA pools based on this device's resources */
	if (pdu_mem_init(&pdev->dev)) {
		dev_err(&pdev->dev, "Could not initialize DMA pools\n");
		return -ENOMEM;
	}

	mem = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	irq = platform_get_resource(pdev, IORESOURCE_IRQ, 0);
	if (!mem || !irq) {
		dev_err(&pdev->dev, "no memory/irq resource for spacc\n");
		return -ENXIO;
	}

	priv = devm_kzalloc(&pdev->dev, sizeof *priv, GFP_KERNEL);
	if (!priv) {
		dev_err(&pdev->dev, "no memory for spacc private data\n");
		return -ENOMEM;
	}

	printk("spacc_probe: Device at %pR\n", mem);

	baseaddr = devm_ioremap_nocache(&pdev->dev, mem->start, resource_size(mem));
	if (!baseaddr) {
		dev_err(&pdev->dev, "unable to map iomem\n");
		return -ENXIO;
	}

#ifdef SPACC_DEBUG_INIT
	print_hex_dump(KERN_INFO, "SPAcc: ", DUMP_PREFIX_ADDRESS, 16, 4, baseaddr, 0x300, false); //0x300: ref. the show_reg()
#endif

	x = pdev->id;
	dev_info(&pdev->dev, "EPN %04X : virt [%d] \n", (x >> 16) & 0xFFFF, x & 0xF);

	pdu_get_version(baseaddr, &info);
	if (pdev->dev.platform_data) {
		pdu_info *parent_info = pdev->dev.platform_data;
		memcpy(&info.pdu_config, &parent_info->pdu_config, sizeof info.pdu_config);
	}

	err = spacc_init (baseaddr, &priv->spacc, &info);
	if (err != CRYPTO_OK) {
		printk("spacc_probe::Failed to initialize device %d...\n", x);
		return -ENXIO;
	}

	err = sysfs_create_group(&pdev->dev.kobj, &spacc_attr_group);
	if (err) {
		spacc_fini(&priv->spacc);
		return -1;
	}

	spin_lock_init(&priv->hw_lock);
	spacc_irq_glbl_disable (&priv->spacc);
	tasklet_init(&priv->pop_jobs, spacc_pop_jobs, (unsigned long)priv);
	platform_set_drvdata(pdev, priv);

	/* Determine configured maximum message length. */
	priv->max_msg_len = priv->spacc.config.max_msg_size;

	if (devm_request_irq(&pdev->dev, irq->start, spacc_irq_handler, IRQF_SHARED, dev_name(&pdev->dev), &pdev->dev)) {
		dev_err(&pdev->dev, "failed to request IRQ\n");
		return -EBUSY;
	}

	priv->spacc.irq_cb_stat = spacc_stat_process;

	// unlock device by default
	pdu_io_write32(baseaddr + SPACC_REG_SECURE_CTRL, 0);

	return err;
}

static int __devexit spacc_remove(struct platform_device *pdev)
{
	spacc_device *spacc;

	// free test vector memory
	spacc = &((struct spacc_priv *)platform_get_drvdata(pdev))->spacc;
	spacc_fini(spacc);
	sysfs_remove_group(&pdev->dev.kobj, &spacc_attr_group);

	pdu_mem_deinit(&pdev->dev);

	/* devm functions do proper cleanup */
	dev_info(&pdev->dev, "removed!\n");

	return 0;
}

#include <elpspacc_irq.h>

static struct platform_driver spacc_driver = {
	.probe  = spacc_probe,
	.remove = __devexit_p(spacc_remove),
	.driver = {
		.name  = "spacc",
		.owner = THIS_MODULE
	},
};

static int __init spacc_mod_init (void)
{
	int err;

	err = platform_driver_register(&spacc_driver);
	return err;
}

static void __exit spacc_mod_exit (void)
{
	platform_driver_unregister(&spacc_driver);
}

MODULE_LICENSE ("GPL");
MODULE_AUTHOR ("Elliptic Technologies Inc.");
module_init (spacc_mod_init);
module_exit (spacc_mod_exit);

// export wrapped library functions
EXPORT_SYMBOL (spacc_open);
EXPORT_SYMBOL (spacc_clone_handle);
EXPORT_SYMBOL (spacc_close);
EXPORT_SYMBOL (spacc_write_context);
EXPORT_SYMBOL (spacc_read_context);
EXPORT_SYMBOL (spacc_error_msg);
EXPORT_SYMBOL (spacc_set_operation);
EXPORT_SYMBOL (spacc_set_key_exp);
EXPORT_SYMBOL (spacc_set_auxinfo);
EXPORT_SYMBOL (spacc_packet_enqueue_ddt);
EXPORT_SYMBOL (spacc_packet_dequeue);
EXPORT_SYMBOL (spacc_pop_packets);
EXPORT_SYMBOL (spacc_dump_ctx);
EXPORT_SYMBOL (spacc_set_wd_count);

EXPORT_SYMBOL(spacc_irq_cmdx_enable);
EXPORT_SYMBOL(spacc_irq_cmdx_disable);
EXPORT_SYMBOL(spacc_irq_stat_enable);
EXPORT_SYMBOL(spacc_irq_stat_disable);
EXPORT_SYMBOL(spacc_irq_stat_wd_enable);
EXPORT_SYMBOL(spacc_irq_stat_wd_disable);
EXPORT_SYMBOL(spacc_irq_glbl_enable);
EXPORT_SYMBOL(spacc_irq_glbl_disable);
EXPORT_SYMBOL(spacc_process_irq);

// used by RE/KEP
EXPORT_SYMBOL (spacc_ctx_request);
EXPORT_SYMBOL (spacc_ctx_release);

int spacc_endian;
module_param(spacc_endian, int, 0);
MODULE_PARM_DESC(spacc_endian, "Endianess of data transfers (0==little)");
EXPORT_SYMBOL(spacc_endian);

module_param(dump_ddt, int, 0600);
MODULE_PARM_DESC(dump_ddt, "dump src_ddt[] dst_ddt[] for each packet");

module_param(dump_pkt_info, int, 0600);
MODULE_PARM_DESC(dump_pkt_info, "dump struct __pkt_info");

module_param(dump_key_context_memory, int, 0600);
MODULE_PARM_DESC(dump_key_context_memory, "dump key context memory"); //dump IV only work for non-IV-imported FPGA

#ifdef SPACC_DEBUG_PERF
module_param(perf, int, 0600);
MODULE_PARM_DESC(perf, "Measure performance");
#endif

module_param(spacc_enqueue_all, int, 0600);
MODULE_PARM_DESC(spacc_enqueue_all, "Use spin_lock_irqsave to protect enqueuing all pkts");

module_param(pop_jobs_in_isr, int, 0600);
MODULE_PARM_DESC(pop_jobs_in_isr, "Pop all jobs in ISR");

