// ------------------------------------------------------------------------
//
//                (C) COPYRIGHT 2011 - 2015 SYNOPSYS, INC.
//                          ALL RIGHTS RESERVED
//
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  version 2 as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, see <https://gnu.org/licenses/>.
//
// ------------------------------------------------------------------------
//
//  This file incorporates zero-copy routines from Linux CryptoDev,
//  covered by the following copyright and permission notice:
//
//    Copyright (c) 2009-2011 Nikos Mavrogiannopoulos <nmav@gnutls.org>
//    Copyright (c) 2010 Phil Sutter
//    Copyright (c) 2011, 2012 OpenSSL Software Foundation, Inc.
//
//    This program is free software; you can redistribute it and/or
//    modify it under the terms of the GNU General Public License
//    as published by the Free Software Foundation; either version 2
//    of the License, or (at your option) any later version.
//
// ------------------------------------------------------------------------

#include <linux/kernel.h>
#include <linux/scatterlist.h>
#include <linux/pagemap.h>
#include <linux/dma-mapping.h>
#include <linux/completion.h>
#include <linux/vmalloc.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/miscdevice.h>
#include <linux/interrupt.h>
#include <linux/ratelimit.h>
#include <linux/jiffies.h>
#include <asm/uaccess.h>
#include <linux/cdev.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/pipe_fs_i.h>
#include <linux/slab.h>
#include <linux/sizes.h>
#include <linux/list.h>
#include <linux/delay.h>
#include <mach/noncache.h>

#include <elpspaccdrv.h>
#include <params.h>

#define HW_LOCK_SI (11) //0 for kapi, 1~10 for uapi, 11 for HW locking
#define MAX_PACKET_NUM	255
#define SPACC_UAPI_PACKET_SIZE (SZ_2K)
#define FP_FOR_KAPI (0xdeadbeef)
#define INODE_FOR_KAPI (0xdeadbeef)
extern int dump_ddt;
extern int dump_pkt_info;
extern int dump_key_context_memory; //dump IV only work for non-IV-imported FPGA
extern int spacc_enqueue_all;
extern int pop_jobs_in_isr;

#ifdef SPACC_DEBUG_PERF
extern int perf;
extern volatile s64 spacc_kt[SPACC_PERF_MAX_ROUNDS][13];
extern volatile int kt_idx;
#endif
// this is the maximum number of DDT entries (not including the NULL) for
// either direction of the engine.  Don't set below 8 unless you are really
// constrained for memory.  Don't set above 64 unless you really need to support
// highly fragmented data.
#ifndef SPACC_DEFAULT_DDT_LENGTH
	#define SPACC_DEFAULT_DDT_LENGTH 8
#endif

#ifndef KERNEL
#define KERNEL
#endif
#include <elpspaccusr.h>

#define Error(fmt, ...) \
		do { \
		pr_err("FAIL %s():%i " fmt, __func__, __LINE__, ##__VA_ARGS__); \
		}while (0)
#define Info(fmt, ...) \
		do { \
		pr_info("Info %s():%i " fmt, __func__, __LINE__, ##__VA_ARGS__); \
		}while (0)

/* pr_devel() should produce zero code unless DEBUG is defined */
#define Debug(fmt, ...) \
		do { \
		pr_devel("Debug %s():%i " fmt, __func__, __LINE__, ##__VA_ARGS__); \
		}while (0)


struct platform_device *spacc_pdev;
static spacc_device *spacc;

static struct cryptEngine_encParam *encParam; //need DMA address for IV
static struct cryptEngine_decParam *decParam; //need DMA address for IV
static struct cryptEngine_authParam _authParam;
static struct cryptEngine_authParam *authParam = &_authParam;
static void *_cipherParam;
static PDU_DMA_ADDR_T _cipherParam_dma;
static PDU_DMA_ADDR_T cipherParam_dma;
static int handle_hash;

static pdu_ddt src_ddt[MAX_PACKET_NUM], dst_ddt[MAX_PACKET_NUM];

static struct State
{
	struct device *dev;
	struct file *fp;
	int spacc_handle;

	struct semaphore lock;
	struct completion comp;

	pid_t pid;
	int cancel;

} *states = NULL;

static int state_num = 0;
static int spacc_epn = 0x0001;  // EPN of hardware to look up (TODO: change this to your own core)

static void spaccdev_callback (void *sprof_dev, void *data)
{
	complete (data);
}

// look up a file handle in our state table and return the index if any
static int spacc_fp_lookup (struct file *fp)
{
	int x;
	for (x = 0; x < state_num; x++) {
		if (states[x].fp == fp) {
			return x;
		}
	}
	return -1;
}

static struct semaphore dev_lock;

static int spacc_dev_open (struct inode *in, struct file *fp)
{
	int x;

	// Info("spaccdev::open %p\n", fp);

	down(&dev_lock);

	// find a hole
	for (x = 0; x < state_num; x++) {
		if (states[x].fp == NULL) {
			break;
		}
	}
	if (x == state_num) {
		Error("spacc_dev_open::No more open slots for a new SPAcc handle\n");
		up(&dev_lock);
		return -1;
	}
	// we allocate the spacc handle when a job is fired
	states[x].dev          = get_device(&spacc_pdev->dev);
	states[x].spacc_handle = -1;
	states[x].fp           = fp;
	states[x].pid         = current->pid;
	states[x].cancel       = 0;

	up(&dev_lock);

	return 0;
}

static int spacc_dev_release (struct inode *in, struct file *fp)
{
	int x;

	// Info("spaccdev::close %p\n", fp);
	down(&dev_lock);

	x = spacc_fp_lookup (fp);
	if (x == -1) {
		Error("spacc_dev_close::invalid 'struct file *' pointer\n");
		up(&dev_lock);
		return -1;
	}

	states[x].fp = NULL;
	if (states[x].spacc_handle >= 0) {
		spacc_close (spacc, states[x].spacc_handle);
		states[x].spacc_handle = -1;
	}

	// complete the lock in case there was a pending job
	complete(&states[x].comp);
	put_device(states[x].dev);

	up(&dev_lock);

	return 0;
}

static void spacc_packet_enqueue_ddt2(int si, pdu_ddt * src, pdu_ddt * dst,
				      int proc_len, uint32_t aad_offset, int pre_aad_sz,
				      int post_aad_sz, int iv_offset)
{
	spacc_job *job;
	job = &spacc->job[states[si].spacc_handle];

	pdu_io_cached_write32(spacc->regmap + SPACC_REG_SRC_PTR,
			      (uint32_t) src->phys, &spacc->cache.src_ptr);
	pdu_io_cached_write32(spacc->regmap + SPACC_REG_DST_PTR,
			      (uint32_t) dst->phys, &spacc->cache.dst_ptr);
	pdu_io_cached_write32(spacc->regmap + SPACC_REG_PROC_LEN,
			      proc_len - post_aad_sz, &spacc->cache.proc_len);
	pdu_io_cached_write32(spacc->regmap + SPACC_REG_PRE_AAD_LEN, pre_aad_sz,
			      &spacc->cache.pre_aad);
	pdu_io_cached_write32(spacc->regmap + SPACC_REG_POST_AAD_LEN,
			      post_aad_sz, &spacc->cache.post_aad);

	pdu_io_cached_write32 (spacc->regmap + SPACC_REG_OFFSET,       aad_offset                 , &spacc->cache.offset);
	pdu_io_cached_write32 (spacc->regmap + SPACC_REG_IV_OFFSET,    iv_offset == -1 ? 0 : (iv_offset | 0x80000000UL), &spacc->cache.iv_offset);

//According DWC spacc databook 2.10a, No need to write SW ID every time, HW inc by one for each cmd/job
#if 0 //debug only
	/* write the job ID to the core, we keep track of it in software now to avoid excessive port I/O */
	pdu_io_write32(spacc->regmap + SPACC_REG_SW_CTRL,
		       SPACC_SW_CTRL_ID_SET(spacc->job_tally++));
#endif

	pdu_io_write32(spacc->regmap + SPACC_REG_CTRL, job->ctrl);

}

static struct __pkt_info {
	uint32_t src, dst;	// begin of each packet
	uint32_t end_src;	// including all source contents
	uint32_t end_dst;	// end of all HW output for this packet
	int srclen, dstlen,	// despite the DDTs above these need to be filled in to make things simpler inside the kernel
	 outlen,		// size including all HW output+offset for this packet
	 src_offset, dst_offset,	// offsets into buffers
	 pre_aad_len, post_aad_len,	// AAD lengths before and after plaintext (if any)
	 src_total_len,		// including all source contents and offset, but no ICV from uapi, and no IV(imported), this value count from beginning of the data packet
	 ivoffset;		// offset in packet for IV, or -1 to not use import

} _pkt_info;

static struct __pkt_info *pkt_info = &_pkt_info;

//idx: packet#
static int _calculate_packet_params(struct elp_spacc_ioctl *io, int idx)	//use input of uapi to do packet formatting in p.286 of DWC spacc databook 2.10a
{
	int tmp;
	int outlen;
	u64 src64addr;
	u64 dst64addr;

	memset(pkt_info, 0, sizeof(*pkt_info));

	if (unlikely(!io->cipher_mode && !io->hash_mode)) {
		Error("cipher_mode and hash_mode are all CRYPTO_MODE_NULL!\n");
		return -1;
	}

	src64addr =
		((phys_addr_t) (io->cryptParam.src_addr_upper) << 32) | io->cryptParam.src_addr_lower;
	pkt_info->src = PHYS64_TO_PHYS32(src64addr) + io->pkt_size * idx;

	dst64addr =
		((phys_addr_t) (io->cryptParam.dst_addr_upper) << 32) | io->cryptParam.dst_addr_lower;
	pkt_info->dst = PHYS64_TO_PHYS32(dst64addr) + io->pkt_size * idx;

	if (io->encrypt) {
		if (io->cipher_mode && !handle_hash) {
			pkt_info->src_offset = encParam->encrypt_offset[idx];
			pkt_info->srclen = encParam->encrypt_size[idx];	//proc_len
			pkt_info->src_total_len = pkt_info->src_offset + pkt_info->srclen;
		}
		if (!io->cipher_mode && handle_hash) {
			pkt_info->src_offset = authParam->auth_offset[idx];
			pkt_info->srclen = authParam->auth_size[idx];	//proc_len
			pkt_info->src_total_len = pkt_info->src_offset + pkt_info->srclen;
			if(!io->aad_copy)
				pkt_info->pre_aad_len = pkt_info->srclen;
		}
		if (io->cipher_mode && handle_hash) {
			tmp =
			    encParam->encrypt_offset[idx] -
			    authParam->auth_offset[idx];
			if (tmp < 0) {
				Error("SPAcc does not support sha_offset > aes_offset, i.e. pre_aad < 0\n");
				return -1;
			}
			pkt_info->src_offset = authParam->auth_offset[idx];
			pkt_info->pre_aad_len = tmp;
			pkt_info->srclen = pkt_info->pre_aad_len + encParam->encrypt_size[idx];	//proc_len
			pkt_info->post_aad_len = authParam->auth_size[idx] - pkt_info->srclen;	// post_aad_len = sha_size - proc_len
			if (pkt_info->post_aad_len < 0) {
				Error("SPAcc does not support post_aad < 0\n");
				Error
					("pkt#%03d, src=%08x, dst=%08x\n, end_src=%08x, end_dst=%08x\n, srclen=%08x, dstlen=%08x, outlen=%08x\n, src_offset=%08x, dst_offset=%08x, pre_aad_len=%08x, post_aad_len=%08x\n, src_total_len=%08x, ivoffset=%08x",
					 idx, pkt_info->src, pkt_info->dst, pkt_info->end_src, pkt_info->end_dst,
					 pkt_info->srclen, pkt_info->dstlen, pkt_info->outlen, pkt_info->src_offset,
					 pkt_info->dst_offset, pkt_info->pre_aad_len, pkt_info->post_aad_len,
					 pkt_info->src_total_len, pkt_info->ivoffset);
				return -1;
			}
			pkt_info->srclen += pkt_info->post_aad_len;
			pkt_info->src_total_len =
			    authParam->auth_offset[idx] +
			    authParam->auth_size[idx];
		}
	} else {
		if (io->cipher_mode && !handle_hash) {
			pkt_info->src_offset = decParam->decrypt_offset[idx];
			pkt_info->srclen = decParam->decrypt_size[idx];	//proc_len
			pkt_info->src_total_len = pkt_info->src_offset + pkt_info->srclen;
		}
		if (!io->cipher_mode && handle_hash) {
								//According QASheet's item40, remove "message authentication" feature
				Error("This driver does not support \"message authentication\"\n");
				return -1;
		}

		if (io->cipher_mode && handle_hash) {
			if (io->cipher_mode != CRYPTO_MODE_AES_GCM)
			{
				//According QASheet's item45
				Error("This driver does not support decryption + message authentication\n");
				return -1;
			}
			tmp =
			    decParam->decrypt_offset[idx] -
			    authParam->auth_offset[idx];
			if (tmp < 0) {
				Error
				    ("SPAcc does not support sha_offset > aes_offset, i.e. pre_aad < 0\n");
				return -1;
			}
			pkt_info->src_offset = authParam->auth_offset[idx];
			pkt_info->pre_aad_len = tmp;
			pkt_info->srclen = pkt_info->pre_aad_len + decParam->decrypt_size[idx];	//proc_len
			pkt_info->post_aad_len = authParam->auth_size[idx] - pkt_info->srclen;	// post_aad_len = sha_size - proc_len
			// peter: todo, not sure post_aad in decrypt
			if (pkt_info->post_aad_len < 0) {
				Error("SPAcc does not support post_aad < 0\n");
				return -1;
			}
			pkt_info->src_total_len =
			    authParam->auth_offset[idx] +
			    authParam->auth_size[idx];
		}
	}

	pkt_info->dst_offset = pkt_info->src_offset; //dst buffer use the same layout as src (except IV)
	if (io->cipher_mode && handle_hash)
		if(!io->aad_copy)
			pkt_info->dst_offset += pkt_info->pre_aad_len; //keep space for pre-aad in dst side

	if (io->cipher_mode)
		pkt_info->src_offset += io->civlen;	//reserved space for IV in src_DDT[0] of each packet

	pkt_info->end_src = pkt_info->src + pkt_info->src_total_len;

	if (io->cipher_mode && (io->cipher_mode == CRYPTO_MODE_AES_GCM) && pkt_info->post_aad_len)
	{
		Error("AES-GCM does not support POST_AAD\n"); //DWC spacc databook 2.10a, p.289, p.286
								//comfimed in QASheet's item42
				return -1;
	}

	// length depends on mode, in !aad_copy mode with !cipher we treat the data as all AAD regardless of how the SPAcc is actually programmed
	if (io->encrypt) {
		if (io->aad_copy) {
			outlen = pkt_info->srclen + io->icv_len;
		} else {
			if (io->cipher_mode && handle_hash) { //need to keep layout for enc+auth mode
				outlen = pkt_info->srclen + io->icv_len;
			} else {
				outlen =
					pkt_info->srclen - pkt_info->pre_aad_len -
					pkt_info->post_aad_len + io->icv_len;
			}
		}
	} else {
		if (io->aad_copy) {
			outlen = pkt_info->srclen;	// we don't subtract icvlen since it's not added by the user
		} else {
			outlen =
			    pkt_info->srclen - pkt_info->pre_aad_len -
			    pkt_info->post_aad_len;
		}
	}
	// add offsets to mapping since we map from the start of the buffer
	outlen += pkt_info->dst_offset;
	pkt_info->outlen = outlen;
	pkt_info->end_dst = pkt_info->dst + outlen;

	// sanity check output length
	if (io->cryptParam.data_num != 1) {
		if (unlikely(io->pkt_size < pkt_info->outlen)) {
			Error("dst buffer shorter than output length %d < %d\n",
			     io->pkt_size, pkt_info->outlen);
			return -1;
		}
	} else {
		if (unlikely(spacc->config.max_msg_size < pkt_info->outlen)) {
			Error
			    ("dst buffer shorter than output length %d < %d\n",
			     spacc->config.max_msg_size, pkt_info->outlen);
			return -1;
		}
	}

	if(dump_pkt_info)
		Info
			("pkt#%03d, src=%08x, dst=%08x\n, end_src=%08x, end_dst=%08x\n, srclen=%08x, dstlen=%08x, outlen=%08x\n, src_offset=%08x, dst_offset=%08x, pre_aad_len=%08x, post_aad_len=%08x\n, src_total_len=%08x, ivoffset=%08x",
			 idx, pkt_info->src, pkt_info->dst, pkt_info->end_src, pkt_info->end_dst,
			 pkt_info->srclen, pkt_info->dstlen, pkt_info->outlen, pkt_info->src_offset,
			 pkt_info->dst_offset, pkt_info->pre_aad_len, pkt_info->post_aad_len,
			 pkt_info->src_total_len, pkt_info->ivoffset);

	return 0;
}

//idx: packet#
//DDT need 8-byte alignment, but each entry requires no word-alignment in address and length in p.300 of DWC spacc databook 2.10a
static int _construct_ddt(struct elp_spacc_ioctl *io, int idx)
{
	pdu_ddt_reset(&src_ddt[idx]);
	if (io->cipher_mode) {
		if (io->encrypt)
		{
			if(!io->is_kapi)
				memcpy(((struct cryptEngine_encParam *)_cipherParam)->encrypt_iv[idx], encParam->encrypt_iv[idx], io->civlen);
			pdu_ddt_add(&src_ddt[idx], (PDU_DMA_ADDR_T) (((struct cryptEngine_encParam *)cipherParam_dma)->encrypt_iv[idx]), io->civlen);	//for IV in cipher mode
		}
		else
		{
			if(!io->is_kapi)
				memcpy(((struct cryptEngine_decParam *)_cipherParam)->decrypt_iv[idx], decParam->decrypt_iv[idx], io->civlen);
			pdu_ddt_add(&src_ddt[idx], (PDU_DMA_ADDR_T) (((struct cryptEngine_decParam *)cipherParam_dma)->decrypt_iv[idx]), io->civlen);	//for IV in cipher mode
		}
	}
	if (io->cipher_mode == CRYPTO_MODE_AES_GCM) { // Set last 4 bytes of IV for AES-GCM packet
		u8 *addr;
		if (io->encrypt)
			addr = encParam->encrypt_iv[idx];
		else
			addr = decParam->decrypt_iv[idx];

		addr += 12; // Set last 4 bytes of IV for AES-GCM packet
		*addr++ = 0;
		*addr++ = 0;
		*addr++ = 0;
		*addr   = 1;

	}
	pdu_ddt_add(&src_ddt[idx], pkt_info->src, pkt_info->src_total_len);


	pdu_ddt_reset(&dst_ddt[idx]);

	if (!io->aad_copy && (!io->cipher_mode && handle_hash) ) // !aad_copy && hash_only
		pdu_ddt_add(&dst_ddt[idx], (PDU_DMA_ADDR_T) pkt_info->dst, pkt_info->dst_offset);  //need space for offset
	else
		pdu_ddt_add(&dst_ddt[idx], (PDU_DMA_ADDR_T) pkt_info->dst, pkt_info->src_total_len - pkt_info->post_aad_len); //dst buffer use the same layout as src (except IV)


	if (handle_hash && authParam->hash_offset[idx]) {
		if (io->encrypt) { //encrypt, HW write ICV to dst; for uapi spec., AES-GCM need 16B, other need 32B
			pdu_ddt_add(&dst_ddt[idx],
					pkt_info->dst + authParam->hash_offset[idx],
					io->icv_len);
		} else { //decrypt, HW read ICV from src; for uapi spec., AES-GCM need 16B, other need 32B
			pdu_ddt_add(&src_ddt[idx],
					pkt_info->src + authParam->hash_offset[idx],
					io->icv_len);
		}
	}

   if(dump_ddt)
	{
		Info("\nDump src_ddt[%d]\n",idx);
                print_hex_dump(KERN_INFO, "src_ddt[]: ", DUMP_PREFIX_ADDRESS, 16, 4,src_ddt[idx].virt, (src_ddt[idx].idx+1)*8, false);
		Info("\nDump dst_ddt[%d]\n",idx);
                print_hex_dump(KERN_INFO, "dst_ddt[]: ", DUMP_PREFIX_ADDRESS, 16, 4,dst_ddt[idx].virt, (dst_ddt[idx].idx+1)*8, false);
	}

	return 0;
}
void _dump_key_context_memory(int si, struct elp_spacc_ioctl* io)
{
	unsigned char keybuf[64], ivbuf[16];
	unsigned char hmackeybuf[64], hmacivbuf[16];
	spacc_read_context (spacc, states[si].spacc_handle, SPACC_CRYPTO_OPERATION, keybuf, io->ckeylen, ivbuf, io->civlen);
	spacc_read_context (spacc, states[si].spacc_handle, SPACC_HASH_OPERATION, hmackeybuf, io->hkeylen, hmacivbuf, 0);
	print_hex_dump(KERN_INFO, "ckeybuf[]: ", DUMP_PREFIX_ADDRESS, 16, 1,keybuf, io->ckeylen, false);
	print_hex_dump(KERN_INFO, "civbuf[]: ", DUMP_PREFIX_ADDRESS, 16, 1,ivbuf, io->civlen, false);
	print_hex_dump(KERN_INFO, "hkeybuf[]: ", DUMP_PREFIX_ADDRESS, 16, 1,hmackeybuf, io->hkeylen, false);
	print_hex_dump(KERN_INFO, "hivbuf[]: ", DUMP_PREFIX_ADDRESS, 16, 1,hmacivbuf, io->hivlen, false);
}
static long spacc_proc(int si, struct elp_spacc_ioctl* io)
{
	int i;
	uint32_t fifo_stat;
	int jobs;
	int ret, outlen = 0;
	unsigned long lock_flags = 0;


	if(spacc_enqueue_all)
		PDU_LOCK(&spacc->lock, lock_flags);

	spacc->ch_mode=1; //channel mode

	fifo_stat = pdu_io_read32(spacc->regmap + SPACC_REG_FIFO_STAT);
	if(unlikely(SPACC_FIFO_STAT_CMD0_CNT_GET(fifo_stat) || SPACC_FIFO_STAT_STAT_CNT_GET(fifo_stat))) {
		Error("start a new task, but FIFO status is not empty(0x%08x)\n",fifo_stat);
		Error("cmd count = %d, stat count = %d\n",SPACC_FIFO_STAT_CMD0_CNT_GET(fifo_stat),SPACC_FIFO_STAT_STAT_CNT_GET(fifo_stat));

		while(!(fifo_stat & SPACC_FIFO_STAT_STAT_EMPTY)) {
			fifo_stat = pdu_io_read32(spacc->regmap + SPACC_REG_FIFO_STAT);
			jobs = SPACC_FIFO_STAT_STAT_CNT_GET (fifo_stat);
			while (jobs-- > 0)
				pdu_io_write32 (spacc->regmap + SPACC_REG_STAT_POP, 1);
			Error("wait cmd to be finished and pop off all stats\n");
		}
	}

	if(unlikely(!(fifo_stat & SPACC_FIFO_STAT_STAT_EMPTY))) {
		Error("start a new task, but FIFO status is not empty(0x%08x)\n",fifo_stat);
		Error("cmd count = %d, stat count = %d\n",SPACC_FIFO_STAT_CMD0_CNT_GET(fifo_stat),SPACC_FIFO_STAT_STAT_CNT_GET(fifo_stat));
	}
	spacc->job_tally = states[si].spacc_handle; //first cmd has the job_idx

	// program spacc
	init_completion(&states[si].comp);

	if(io->cryptParam.src_addr_lower && !io->cryptParam.dst_addr_lower)
	{
		io->cryptParam.dst_addr_lower = io->cryptParam.src_addr_lower;
		io->cryptParam.dst_addr_upper = io->cryptParam.src_addr_upper;
	}

	if(!io->cryptParam.src_addr_lower && io->cryptParam.dst_addr_lower)
	{
		io->cryptParam.src_addr_lower = io->cryptParam.dst_addr_lower;
		io->cryptParam.src_addr_upper = io->cryptParam.dst_addr_upper;
	}

#ifdef SPACC_DEBUG_PERF
	if(perf)
	{
		if(kt_idx > SPACC_PERF_MAX_ROUNDS)
			kt_idx = 0 ;

		spacc_kt[kt_idx][2] =ktime_get().tv64;
	}
#endif
	if(_calculate_packet_params(io, 0))
	{
		ret = -1;
		goto error1;
	}

#ifdef SPACC_DEBUG_PERF
	if(perf)
	{
		spacc_kt[kt_idx][3] =ktime_get().tv64;
	}
#endif
	outlen += pkt_info->outlen;
	if(_construct_ddt(io, 0))
	{
		ret = -1;
		goto error1;
	}
#ifdef SPACC_DEBUG_PERF
	if(perf)
	{
		spacc_kt[kt_idx][4] =ktime_get().tv64;
	}
#endif

	if(dump_key_context_memory) //dump IV only work for non-IV-imported FPGA
		_dump_key_context_memory(si,io);


	// set MSG flags
	spacc->job[states[si].spacc_handle].ctrl |= ((1UL << _SPACC_CTRL_MSG_BEGIN) | (1UL << _SPACC_CTRL_MSG_END));
	spacc->job[states[si].spacc_handle].job_err = 0;

	spacc_irq_stat_enable(spacc, io->cryptParam.data_num);

#ifdef SPACC_DEBUG_PERF
	if(perf)
	{
		spacc_kt[kt_idx][5] =ktime_get().tv64;
	}
#endif
		io->err = spacc_packet_enqueue_ddt (spacc, states[si].spacc_handle, &src_ddt[0], &dst_ddt[0],
						pkt_info->srclen,
						(pkt_info->src_offset<<SPACC_OFFSET_SRC_O)|(pkt_info->dst_offset<<SPACC_OFFSET_DST_O),     // aad_offset
						pkt_info->pre_aad_len | (io->aad_copy ? SPACC_AADCOPY_FLAG : 0),                     // pre_aad_sz + flag for AAD COPY
						pkt_info->post_aad_len,                                                             // post_aad_sz
						pkt_info->ivoffset == -1 ? 0 : (pkt_info->ivoffset | 0x80000000UL),                        // iv_offset
						SPACC_SW_CTRL_PRIO_HI);
#ifdef SPACC_DEBUG_PERF
	if(perf)
	{
		spacc_kt[kt_idx][6] =ktime_get().tv64;
	}
#endif

		if (unlikely(io->err != CRYPTO_OK)) {
			Error("Could not enqueue packet: %d\n", io->err);
			ret = -EIO;
			goto error1;
		}
		for (i = 1; i < io->cryptParam.data_num; i++) {
			if(_calculate_packet_params(io, i))
			{
				ret = -1;
				goto error1;
			}

			outlen += pkt_info->outlen;

			if(_construct_ddt(io, i))
			{
				ret = -1;
				goto error1;
			}

#if 0 //debug only
			if(pdu_io_read32(spacc->regmap + SPACC_REG_FIFO_STAT) & SPACC_FIFO_STAT_CMD0_FULL)
				Error("CMD0 FIFO FULL while processing pkt#%d\n",i);
#endif
			if(dump_key_context_memory)
				_dump_key_context_memory(si,io);

			spacc_packet_enqueue_ddt2(si, &src_ddt[i], &dst_ddt[i], pkt_info->srclen, (pkt_info->src_offset<<SPACC_OFFSET_SRC_O)|(pkt_info->dst_offset<<SPACC_OFFSET_DST_O), pkt_info->pre_aad_len, pkt_info->post_aad_len, pkt_info->ivoffset);
		}

#ifdef SPACC_DEBUG_PERF
	if(perf)
	{
		spacc_kt[kt_idx][7] =ktime_get().tv64;
	}
#endif

	if(spacc_enqueue_all)
		PDU_UNLOCK(&spacc->lock, lock_flags);

	wait_for_completion(&states[si].comp);

#ifdef SPACC_DEBUG_PERF
	if(perf)
	{
		spacc_kt[kt_idx][8] =ktime_get().tv64;
	}
#endif

		io->err = spacc_packet_dequeue(spacc, states[si].spacc_handle);
#ifdef SPACC_DEBUG_PERF
	if(perf)
	{
		spacc_kt[kt_idx][9] =ktime_get().tv64;
	}
#endif

	//for debug
	spacc->ch_mode = 0;
	return outlen; //if no erros

error1:
	if(spacc_enqueue_all)
		PDU_UNLOCK(&spacc->lock, lock_flags);
	return ret;

}

static long pre_spacc_proc (int si, struct elp_spacc_ioctl* io, void __user *arg)
{
	int outlen = 0;
	handle_hash = 0;
#ifdef SPACC_DEBUG_PERF
	if(perf)
	{
		spacc_kt[kt_idx][1] =ktime_get().tv64;
	}
#endif
		if(si) //only for uapi
		{
			if(io->hash_mode || io->cipher_mode == CRYPTO_MODE_AES_GCM)
				handle_hash = 1;

			if(io->cipher_mode && io->encrypt)
			{
				encParam = io->encParam;
				cipherParam_dma = _cipherParam_dma;
			}

			if(io->cipher_mode && !io->encrypt)
			{
				decParam = io->decParam;
				cipherParam_dma = _cipherParam_dma;
			}

			if(handle_hash)
				authParam = io->authParam;
			io->pkt_size = SPACC_UAPI_PACKET_SIZE;
		}
		else //for kapi
		{
			struct kio_t *kio = (struct kio_t *)arg;

			if (unlikely(states[si].spacc_handle < 0)) {
				Error("SPAcc state not initialized properly\n");
				return -EFAULT;
			}

			io = kio->io;

			cipherParam_dma = kio->_cipherParam_dma;

			if(io->cipher_mode && io->encrypt)
				encParam = io->encParam;

			if(io->cipher_mode && !io->encrypt)
				decParam = io->decParam;

			io->pkt_size = kio->unit_size;

		}

		outlen = spacc_proc(si, io);
		if(outlen < 0)
		{
				Error("Got error(%d) from spacc_proc()\n",outlen);
				if(!io->err)
					io->err =  CRYPTO_FAILED;
		}

#ifdef SPACC_DEBUG_PERF
	if(perf)
	{
		spacc_kt[kt_idx][10] =ktime_get().tv64;
	}
#endif
		if (!io->encrypt && io->cipher_mode == CRYPTO_MODE_AES_GCM && io->err == CRYPTO_AUTHENTICATION_FAILED) //for QASheet's item45, only do decryption, no authentication for AES-GCM)
			io->err = CRYPTO_OK;

		if (likely (io->err == CRYPTO_OK)) {
			return outlen;
		} else {
			return pdu_error_code(io->err);
		}
}

static long spacc_dev_ioctl2 (int si, struct file *fp, unsigned int cmd, unsigned long arg_)
{
	void __user *arg = (void __user *) arg_;
	struct elp_spacc_ioctl *io = arg;
	int ret;

	switch (cmd) {
	case ELP_SPACC_USR_INIT:

		// sanity check key/iv
		if (io->ckeylen > sizeof(io->ckey) ||
			io->civlen  > sizeof(io->civ)  ||
			io->hkeylen > sizeof(io->hkey) ||
			io->hivlen  > sizeof(io->hiv)) {
			Error("Invalid key or IV length specified\n");
			return -EFAULT;
		}

		// open spacc handle
		states[si].spacc_handle = spacc_open (spacc, io->cipher_mode, io->hash_mode, -1, 0, spaccdev_callback, &states[si].comp);
		if (states[si].spacc_handle < 0) {
			Error("Failed to open a spacc handle\n");
			return -EBUSY;
		}

		// set context
		if (io->cipher_mode != CRYPTO_MODE_NULL) {
			spacc_write_context (spacc, states[si].spacc_handle, SPACC_CRYPTO_OPERATION, io->ckey, io->ckeylen, io->civ, io->civlen);
		}
		if (io->hash_mode != CRYPTO_MODE_NULL) {
			spacc_write_context (spacc, states[si].spacc_handle, SPACC_HASH_OPERATION, io->hkey, io->hkeylen, io->hiv, io->hivlen);
		}
		// set operation, we append the ICV
		spacc_set_operation (spacc, states[si].spacc_handle, io->encrypt ? OP_ENCRYPT : OP_DECRYPT, io->icv_mode, IP_ICV_APPEND, 0, io->icv_len, 0);

		// if we are decrypting set the expand bit (required for RC4/AES)
		if (!io->encrypt || io->cipher_mode == CRYPTO_MODE_RC4_128 || io->cipher_mode == CRYPTO_MODE_RC4_40) {
			spacc_set_key_exp (spacc, states[si].spacc_handle);
		}

		return 0;

	case ELP_SPACC_USR_PROCESS:
		Debug("get states[%d] for %sapi, ", si, si?"u":"k");
		Debug("spacc_handle = %d\n", states[si].spacc_handle);

		if(si) //only for uapi
		{
			// copy struct from user
			if (unlikely(states[si].spacc_handle < 0)) {
				Error("SPAcc state not initialized properly\n");
				return -EFAULT;
			}

			if (unlikely(down_timeout(&states[HW_LOCK_SI].lock, timespec_to_jiffies(&io->timeout)))) {
				Error("timeout while trying to lock state\n");
				return -ETIMEDOUT; //timeout
			}

			if(unlikely(states[si].cancel)) {
				states[si].cancel = 0;
				return -ECANCELED; //Operation Canceled
			}
		} else { //for kapi
			//for this cmd, only one channel is allowed to use HW
			if (unlikely(down_interruptible(&states[HW_LOCK_SI].lock))) {
				Error("Interrupted while trying to lock state\n");
				return -EBUSY;
			}
		}

		ret = pre_spacc_proc(si, io, arg);
		up(&states[HW_LOCK_SI].lock);

		return ret;

	}

	// shouldn't get here
	return -EIO;
}

//kernel/locking/semaphore.c
struct semaphore_waiter {
	struct list_head list;
	struct task_struct *task;
	bool up;
};

static noinline void __sched __up_ch(struct semaphore *sem, int si)
{
	struct semaphore_waiter *waiter;
	struct list_head *node, *next;


	list_for_each_safe(node, next, &sem->wait_list)
	{
		waiter = list_entry(node, struct semaphore_waiter, list);
		if(waiter->task->pid == states[si].pid)
		{
			states[si].cancel = 1;
			list_del(&waiter->list);
			waiter->up = true;
			wake_up_process(waiter->task);
			return;
		}
	}
}

/**
 * up_ch - release the semaphore for specific channel of uapi
 * @sem: the semaphore to release
 *
 * Release the semaphore.  Unlike mutexes, up_ch() may be called from any
 * context and even by tasks which have never called down().
 */
static void up_ch(struct semaphore *sem, int si)
{
	unsigned long flags;

	raw_spin_lock_irqsave(&sem->lock, flags);
	if (likely(list_empty(&sem->wait_list)))
		;
	else
		__up_ch(sem, si);
	raw_spin_unlock_irqrestore(&sem->lock, flags);
}

static long spacc_dev_ioctl (struct file *fp, unsigned int cmd, unsigned long arg_)
{
	long err;
	int  si;

	// look up structure associated with this file handle
	si = spacc_fp_lookup (fp);
	if (si == -1) {
		Error("invalid 'struct file *' pointer\n");
		return -1;
	}

	Debug("The process@%p is \"%s\" (pid %i, tgid %i)\n", current, current->comm, current->pid, current->tgid);

	if(cmd == ELP_SPACC_USR_CANCEL)
	{
		up_ch(&states[HW_LOCK_SI].lock, si);
		return 0;
	}

	if (unlikely(down_interruptible(&states[si].lock))) {
		Error("Interrupted trying to lock state\n");
		return -1;
	}


	err = spacc_dev_ioctl2(si, fp, cmd, arg_);

	up(&states[si].lock);

	return err;
}

long spacc_kapi_process (unsigned long arg_)
{
#ifdef SPACC_DEBUG_PERF
	if(perf)
	{
		if(kt_idx > SPACC_PERF_MAX_ROUNDS)
			kt_idx = 0 ;

		spacc_kt[kt_idx][0] =ktime_get().tv64;
	}
#endif
	return spacc_dev_ioctl ((struct file *)FP_FOR_KAPI, ELP_SPACC_USR_PROCESS, arg_);
}
EXPORT_SYMBOL (spacc_kapi_process);

static struct file_operations spacc_dev_fops = {
	.owner = THIS_MODULE,
	.open           = spacc_dev_open,
	.release        = spacc_dev_release,
	.unlocked_ioctl = spacc_dev_ioctl,
};

static struct miscdevice spaccdev_device = {
	.minor = MISC_DYNAMIC_MINOR,
	.name = "spaccusr",
	.fops = &spacc_dev_fops,
};


int spacc_dev_init (void)
{
	int err, x, y, tmp;

	sema_init(&dev_lock, 1);

	// sort out SPAcc
	spacc_pdev = get_spacc_platdev_by_epn(spacc_epn, 0);
	if (spacc_pdev == NULL) {
		Error("module not initialized\n");
		return -1;
	}
	spacc = platform_get_drvdata(spacc_pdev);

	spacc->op_mode = SPACC_OP_MODE_USER;
	spacc_irq_stat_disable(spacc);
	spacc_irq_cmdx_disable(spacc, 0);
	spacc_irq_stat_wd_disable(spacc);

	spacc_irq_glbl_enable(spacc);


	err = misc_register (&spaccdev_device);
	if (err) {
		return err;
	}
	state_num = HW_LOCK_SI + 1;

	states = vmalloc (state_num * sizeof (*states));
	if (!states) {
		Error("Out of memory\n");
		misc_deregister (&spaccdev_device);
		return -1;
	}
	memset (states, 0, sizeof (*states) * state_num);

	for (x = 0; x < state_num; x++) {
		init_completion(&(states[x].comp));
		sema_init(&states[x].lock, 1);

		states[x].spacc_handle = -1;
	}


	for (x = 0; x < MAX_PACKET_NUM; x++) {
		// 32 DDT segments allow for upto 16x4=64KB blocks + some slack for misaligned pages/scattergather from user
		if (pdu_ddt_init(&src_ddt[x], SPACC_DEFAULT_DDT_LENGTH)) {
			Error("Failed to initialize src_ddt %d\n", x);
			goto cleanup;
		}
		if (pdu_ddt_init(&dst_ddt[x], SPACC_DEFAULT_DDT_LENGTH)) {
			Error("Failed to initialize dst_ddt %d\n", x);
			goto cleanup;
		}
	}

	_cipherParam = pdu_dma_alloc(max(sizeof(struct cryptEngine_encParam),sizeof(struct cryptEngine_decParam)),&_cipherParam_dma);
	if(!_cipherParam || !_cipherParam_dma)
	{
		Error("Failed to open a pdu_dma_alloc()\n");
		goto cleanup;
	}

	//reserve a states[tmp] for kapi, it will get tmp=SPACC_MAX_JOBS+1
	if(spacc_dev_open ((struct inode *)INODE_FOR_KAPI, (struct file *)FP_FOR_KAPI))
		goto cleanup;

	tmp = spacc_fp_lookup ((struct file *)FP_FOR_KAPI);
	Debug("get states[%d] for kapi\n", tmp);
	if (tmp == -1) {
		Error("invalid 'struct file *' pointer\n");
		goto cleanup;
	}
	states[tmp].spacc_handle = 0;

	// open spacc handle for kapi in spacc_dev_init(), so kapi always use states[0] and spacc_handle = 0
	states[tmp].spacc_handle = spacc_open (spacc, CRYPTO_MODE_AES_XTS, CRYPTO_MODE_NULL, -1, 0, spaccdev_callback, &states[tmp].comp);
	if (states[tmp].spacc_handle < 0) {
		Error("Failed to open a spacc handle\n");
		goto cleanup;
	}
	Debug("get spacc_handle = %d for kapi\n", states[tmp].spacc_handle);

	return 0;
cleanup:
	Error("Could not allocate sg/pages during init out of memory\n");
	for (y = 0; y < x; y++) {
		pdu_ddt_free (&dst_ddt[y]);
		pdu_ddt_free (&src_ddt[y]);
	}

	pdu_dma_free(max(sizeof(struct cryptEngine_encParam),sizeof(struct cryptEngine_decParam)),_cipherParam,_cipherParam_dma);
	misc_deregister (&spaccdev_device);
	vfree(states);
	return -1;
}

EXPORT_SYMBOL (spacc_dev_init);

void spacc_dev_exit (void)
{
	int x;
	Info("removing /dev/spaccusr ...\n");

	//unconditionally release for kapi
	spacc_dev_release ((struct inode *)INODE_FOR_KAPI, (struct file *)FP_FOR_KAPI);

	misc_deregister (&spaccdev_device);
	if (states) {
		for (x = 0; x < state_num; x++) {
		}
		vfree (states);
	}

	for (x = 0; x < MAX_PACKET_NUM; x++) {
		pdu_ddt_free (&dst_ddt[x]);
		pdu_ddt_free (&src_ddt[x]);
	}
}

EXPORT_SYMBOL (spacc_dev_exit);

device_initcall_sync (spacc_dev_init);
module_exit (spacc_dev_exit);

module_param (spacc_epn, int, 0);
MODULE_PARM_DESC (spacc_epn, "Set SPAcc EPN number to use");


MODULE_AUTHOR ("Elliptic Technologies Inc.");
MODULE_LICENSE ("GPL");

